#include "transform_dag.h"
#include "binary_reader.h"
#include "configuration/transform_dag_definitions.h"
#include "dag_info.h"
#include <algorithm>
#include <bit>

uint32_t getChildMask_gpu(uint16_t const* pNode);
typename TransformDAG16::TransformPointer getChildPointer_gpu(TransformDAG16 dag, uint16_t const* pNode, uint32_t childLevel, uint32_t childIdx);

static void verifyTraversal_recurse(const TransformDAG16& dag, uint32_t level, uint32_t nodeIdx, uint64_t& numLeavesVisited)
{
    const uint32_t childLevel = level - 1;
    if (level == dag.leafLevel) {
        checkAlways(nodeIdx < dag.leaves.size());
        ++numLeavesVisited;
    } else {
        uint16_t const* pNode = &dag.nodes[dag.levelStarts[level] + nodeIdx];
        check(*pNode != 0);
        const uint32_t bitmask = TransformDAG16::convert_child_mask(*pNode);
        // const uint32_t bitmask = getChildMask_gpu(pNode);
        for (uint8_t childIdx = 0; childIdx < 8; ++childIdx) {
            if ((bitmask >> childIdx) & 0b1) {
                const auto tmp = dag.get_child_index(childLevel, pNode, *pNode, childIdx);
                // const auto tmp = getChildPointer_gpu(dag, pNode, childLevel, childIdx);
                verifyTraversal_recurse(dag, childLevel, tmp.index, numLeavesVisited);
            }
        }
    }
}

void testRayTraversal(const TransformDAG16& dag);

static void verifyTraversal(const TransformDAG16& dag)
{
    uint64_t numLeavesVisited = 0;
    verifyTraversal_recurse(dag, dag.levels, dag.get_first_node_index(), numLeavesVisited);
    checkAlways(numLeavesVisited >= dag.leaves.size());
}

struct CPUHuffmanDecodeLUT {
    std::vector<std::pair<uint32_t, uint32_t>> codeWords;
    uint32_t bitMask;

    inline void readFrom(BinaryReader& reader)
    {
        struct Code {
            uint64_t code;
            uint32_t bits;
        };
        std::vector<Code> codes;
        reader.read(codes);
        if (codes.empty())
            return;

        const uint32_t maxCodeLength = std::max_element(std::begin(codes), std::end(codes), [](const Code& lhs, const Code& rhs) { return lhs.bits < rhs.bits; })->bits;
        bitMask = (1u << maxCodeLength) - 1u;
        codeWords.resize(1llu << maxCodeLength);
        for (uint32_t key = 0; key < codes.size(); ++key) {
            const Code& code = codes[key];
            const auto suffixBits = (maxCodeLength - code.bits);
            for (uint32_t suffix = 0; suffix < (1u << suffixBits); ++suffix) {
                const auto combinedCode = (suffix << code.bits) | code.code;
                codeWords[combinedCode] = { key, code.bits };
            }
        }
    }
};

struct TransformDAGConfig {
    bool symmetry, axisPermutation, translation;
    uint32_t maxTranslationLevel;
};
struct TransformEncodingConfig {
    bool pointerTables, huffmanCode;
};

void TransformDAGFactory::load_dag_from_file(DAGInfo& outInfo, TransformDAG16& outDag, const std::filesystem::path& path, EMemoryType memoryType)
{
    PROFILE_FUNCTION();
    checkAlways(!outDag.is_valid());

    std::vector<std::vector<uint16_t>> nodesPerLevel;
    std::vector<Leaf> leaves;
    uint32_t resolution, rootLevel;

    BinaryReader reader { path };
    TransformDAGConfig dagConfig;
    TransformEncodingConfig encodingConfig;
    reader.read(dagConfig);
    reader.read(encodingConfig);
    checkAlways(dagConfig.symmetry == (bool)TRANSFORM_DAG_USE_SYMMETRY);
    checkAlways(dagConfig.axisPermutation == (bool)TRANSFORM_DAG_USE_AXIS_PERMUTATION);
    checkAlways(dagConfig.translation == (bool)TRANSFORM_DAG_USE_TRANSLATION);
#if TRANSFORM_DAG_USE_TRANSLATION
    checkAlways(dagConfig.maxTranslationLevel == TRANSFORM_DAG_MAX_TRANSLATION_LEVEL);
#endif
    checkAlways(encodingConfig.pointerTables == (bool)TRANSFORM_DAG_USE_POINTER_TABLES);
    checkAlways(encodingConfig.huffmanCode == (bool)TRANSFORM_DAG_USE_HUFFMAN_CODE);

    reader.read(nodesPerLevel);
    reader.read(leaves);
    reader.read(resolution);
    reader.read(rootLevel);
    checkAlways(rootLevel == SCENE_DEPTH);

    std::vector<std::vector<uint64_t>> pointerTables;
    reader.read(pointerTables);
#if TRANSFORM_DAG_USE_POINTER_TABLES
    // Combine pointer tables into a single array, from the root to the bottom.
    std::vector<uint32_t> tableStarts(nodesPerLevel.size());
    std::vector<uint64_t> pointerTable;
    for (uint32_t level = 0; level <= rootLevel; ++level) {
        const auto& levelPointers = pointerTables[level];
        checkAlways(pointerTable.size() <= (size_t)std::numeric_limits<uint32_t>::max());
        tableStarts[level] = (uint32_t)pointerTable.size();
        pointerTable.resize(pointerTable.size() + levelPointers.size());
        std::copy(std::begin(levelPointers), std::end(levelPointers), std::begin(pointerTable) + tableStarts[level]);
    }
    outDag.tableStarts = StaticArray<uint32_t>::allocate("TransformDAG16::tableStarts", tableStarts, memoryType);
    outDag.pointerTable = StaticArray<uint64_t>::allocate("TransformDAG16::pointerTable", pointerTable, memoryType);
#endif

    std::vector<CPUHuffmanDecodeLUT> cpuHuffmanDecoders;
    reader.read(cpuHuffmanDecoders);
#if TRANSFORM_DAG_USE_HUFFMAN_CODE
    std::vector<HuffmanCodeWord> codeWords;
    std::vector<typename TransformDAG16::HuffmanDecoderLUT> decoders;
    for (const auto& huffmanDecoderLUT : cpuHuffmanDecoders) {
        const size_t start = codeWords.size();
        codeWords.resize(codeWords.size() + huffmanDecoderLUT.codeWords.size());
        // Stored in file using U32's, but can be compacted into a single U32.
        std::transform(std::begin(huffmanDecoderLUT.codeWords), std::end(huffmanDecoderLUT.codeWords), std::begin(codeWords) + start,
            [&](const std::pair<uint32_t, uint32_t>& codeWord) { return HuffmanCodeWord { .code = std::get<0>(codeWord), .numBits = std::get<1>(codeWord) }; });
        decoders.push_back({ .codeWordStart = (uint32_t)start, .bitMask = huffmanDecoderLUT.bitMask });
    }
    outDag.huffmanCodeWords = StaticArray<HuffmanCodeWord>::allocate("TransformDAG16::huffmanCodeWords", codeWords, memoryType);
    outDag.huffmanDecoders = StaticArray<typename TransformDAG16::HuffmanDecoderLUT>::allocate("TransformDAG16::huffmanDecoders", decoders, memoryType);
#endif

    // Copy leaves to StaticArray.
    outDag.leaves = StaticArray<Leaf>::allocate("TransformDAG16::leaves", leaves, memoryType);

    constexpr size_t u64_in_u16 = sizeof(uint64_t) / sizeof(uint16_t);
    // Combine nodes into a single array, from the root to the bottom.
    std::vector<uint32_t> levelStarts(nodesPerLevel.size());
    std::vector<uint16_t> nodes;
    // The TransformDAG will read in 64-bit chunks (see memcpy64). Add some padding at the front of the array to prevent out-of-bounds access.
    for (size_t i = 0; i < u64_in_u16; ++i)
        nodes.emplace_back(0);
    for (uint32_t level = 0; level <= rootLevel; ++level) {
        const auto& levelNodes = nodesPerLevel[level];
        checkAlways(nodes.size() <= (size_t)std::numeric_limits<uint32_t>::max());
        const auto levelStart = levelStarts[level] = (uint32_t)nodes.size();
        nodes.resize(nodes.size() + levelNodes.size());
        std::copy(std::begin(levelNodes), std::end(levelNodes), std::begin(nodes) + levelStart);
    }
    // The TransformDAG will read in 64-bit chunks (see memcpy64). Add some padding at the vacj of the array to prevent out-of-bounds access.
    for (size_t i = 0; i < u64_in_u16; ++i)
        nodes.emplace_back(0);

    outDag.levelStarts = StaticArray<uint32_t>::allocate("TransformDAG16::levelStarts", levelStarts, memoryType);
    outDag.nodes = StaticArray<uint16_t>::allocate("TransformDAG16::nodes", nodes, memoryType);

    checkAlways(rootLevel == outDag.levels);
    outInfo.boundsAABBMin = Vector3(-1.0f, -1.0f, -1.0f);
    outInfo.boundsAABBMax = Vector3(+1.0f, +1.0f, +1.0f);

    if (memoryType == EMemoryType::CPU || memoryType == EMemoryType::GPU_Managed)
        verifyTraversal(outDag);
}

[[maybe_unused]] static uint8 next_child(uint8 order, uint8 mask)
{
    for (uint8 child = 0; child < 8; ++child) {
        uint8 childInOrder = child ^ order;
        if (mask & (1u << childInOrder))
            return childInOrder;
    }
    check(false);
    return 0;
}

TransformDAG16::TraversalConstants TransformDAG16::createTraversalConstants() const
{
    TraversalConstants out {};

    for (uint32_t mask = 1; mask < 256; ++mask) {
        for (uint32_t order = 0; order < 8; ++order) {
            out.nextChildLUT[mask][order] = next_child((uint8_t)order, (uint8_t)mask);
        }
    }

#if TRANSFORM_DAG_USE_TRANSFORMATION_ID
    out.transforms[0] = { .symmetry = 0, .axis0 = 0, .axis1 = 1, .axis2 = 2 };
    out.transforms[1] = { .symmetry = 1, .axis0 = 0, .axis1 = 1, .axis2 = 2 };
    out.transforms[2] = { .symmetry = 2, .axis0 = 0, .axis1 = 1, .axis2 = 2 };
    out.transforms[3] = { .symmetry = 3, .axis0 = 0, .axis1 = 1, .axis2 = 2 };
    out.transforms[4] = { .symmetry = 4, .axis0 = 0, .axis1 = 1, .axis2 = 2 };
    out.transforms[5] = { .symmetry = 5, .axis0 = 0, .axis1 = 1, .axis2 = 2 };
    out.transforms[6] = { .symmetry = 6, .axis0 = 0, .axis1 = 1, .axis2 = 2 };
    out.transforms[7] = { .symmetry = 7, .axis0 = 0, .axis1 = 1, .axis2 = 2 };
    out.transforms[8] = { .symmetry = 0, .axis0 = 0, .axis1 = 2, .axis2 = 1 };
    out.transforms[9] = { .symmetry = 1, .axis0 = 0, .axis1 = 2, .axis2 = 1 };
    out.transforms[10] = { .symmetry = 2, .axis0 = 0, .axis1 = 2, .axis2 = 1 };
    out.transforms[11] = { .symmetry = 3, .axis0 = 0, .axis1 = 2, .axis2 = 1 };
    out.transforms[12] = { .symmetry = 4, .axis0 = 0, .axis1 = 2, .axis2 = 1 };
    out.transforms[13] = { .symmetry = 5, .axis0 = 0, .axis1 = 2, .axis2 = 1 };
    out.transforms[14] = { .symmetry = 6, .axis0 = 0, .axis1 = 2, .axis2 = 1 };
    out.transforms[15] = { .symmetry = 7, .axis0 = 0, .axis1 = 2, .axis2 = 1 };
    out.transforms[16] = { .symmetry = 0, .axis0 = 1, .axis1 = 0, .axis2 = 2 };
    out.transforms[17] = { .symmetry = 1, .axis0 = 1, .axis1 = 0, .axis2 = 2 };
    out.transforms[18] = { .symmetry = 2, .axis0 = 1, .axis1 = 0, .axis2 = 2 };
    out.transforms[19] = { .symmetry = 3, .axis0 = 1, .axis1 = 0, .axis2 = 2 };
    out.transforms[20] = { .symmetry = 4, .axis0 = 1, .axis1 = 0, .axis2 = 2 };
    out.transforms[21] = { .symmetry = 5, .axis0 = 1, .axis1 = 0, .axis2 = 2 };
    out.transforms[22] = { .symmetry = 6, .axis0 = 1, .axis1 = 0, .axis2 = 2 };
    out.transforms[23] = { .symmetry = 7, .axis0 = 1, .axis1 = 0, .axis2 = 2 };
    out.transforms[24] = { .symmetry = 0, .axis0 = 1, .axis1 = 2, .axis2 = 0 };
    out.transforms[25] = { .symmetry = 1, .axis0 = 1, .axis1 = 2, .axis2 = 0 };
    out.transforms[26] = { .symmetry = 2, .axis0 = 1, .axis1 = 2, .axis2 = 0 };
    out.transforms[27] = { .symmetry = 3, .axis0 = 1, .axis1 = 2, .axis2 = 0 };
    out.transforms[28] = { .symmetry = 4, .axis0 = 1, .axis1 = 2, .axis2 = 0 };
    out.transforms[29] = { .symmetry = 5, .axis0 = 1, .axis1 = 2, .axis2 = 0 };
    out.transforms[30] = { .symmetry = 6, .axis0 = 1, .axis1 = 2, .axis2 = 0 };
    out.transforms[31] = { .symmetry = 7, .axis0 = 1, .axis1 = 2, .axis2 = 0 };
    out.transforms[32] = { .symmetry = 0, .axis0 = 2, .axis1 = 0, .axis2 = 1 };
    out.transforms[33] = { .symmetry = 1, .axis0 = 2, .axis1 = 0, .axis2 = 1 };
    out.transforms[34] = { .symmetry = 2, .axis0 = 2, .axis1 = 0, .axis2 = 1 };
    out.transforms[35] = { .symmetry = 3, .axis0 = 2, .axis1 = 0, .axis2 = 1 };
    out.transforms[36] = { .symmetry = 4, .axis0 = 2, .axis1 = 0, .axis2 = 1 };
    out.transforms[37] = { .symmetry = 5, .axis0 = 2, .axis1 = 0, .axis2 = 1 };
    out.transforms[38] = { .symmetry = 6, .axis0 = 2, .axis1 = 0, .axis2 = 1 };
    out.transforms[39] = { .symmetry = 7, .axis0 = 2, .axis1 = 0, .axis2 = 1 };
    out.transforms[40] = { .symmetry = 0, .axis0 = 2, .axis1 = 1, .axis2 = 0 };
    out.transforms[41] = { .symmetry = 1, .axis0 = 2, .axis1 = 1, .axis2 = 0 };
    out.transforms[42] = { .symmetry = 2, .axis0 = 2, .axis1 = 1, .axis2 = 0 };
    out.transforms[43] = { .symmetry = 3, .axis0 = 2, .axis1 = 1, .axis2 = 0 };
    out.transforms[44] = { .symmetry = 4, .axis0 = 2, .axis1 = 1, .axis2 = 0 };
    out.transforms[45] = { .symmetry = 5, .axis0 = 2, .axis1 = 1, .axis2 = 0 };
    out.transforms[46] = { .symmetry = 6, .axis0 = 2, .axis1 = 1, .axis2 = 0 };
    out.transforms[47] = { .symmetry = 7, .axis0 = 2, .axis1 = 1, .axis2 = 0 };
    out.transformChildMappingWorldToLocal[0][0] = 0;
    out.transformChildMappingWorldToLocal[0][1] = 1;
    out.transformChildMappingWorldToLocal[0][2] = 2;
    out.transformChildMappingWorldToLocal[0][3] = 3;
    out.transformChildMappingWorldToLocal[0][4] = 4;
    out.transformChildMappingWorldToLocal[0][5] = 5;
    out.transformChildMappingWorldToLocal[0][6] = 6;
    out.transformChildMappingWorldToLocal[0][7] = 7;
    out.transformChildMappingWorldToLocal[1][0] = 1;
    out.transformChildMappingWorldToLocal[1][1] = 0;
    out.transformChildMappingWorldToLocal[1][2] = 3;
    out.transformChildMappingWorldToLocal[1][3] = 2;
    out.transformChildMappingWorldToLocal[1][4] = 5;
    out.transformChildMappingWorldToLocal[1][5] = 4;
    out.transformChildMappingWorldToLocal[1][6] = 7;
    out.transformChildMappingWorldToLocal[1][7] = 6;
    out.transformChildMappingWorldToLocal[2][0] = 2;
    out.transformChildMappingWorldToLocal[2][1] = 3;
    out.transformChildMappingWorldToLocal[2][2] = 0;
    out.transformChildMappingWorldToLocal[2][3] = 1;
    out.transformChildMappingWorldToLocal[2][4] = 6;
    out.transformChildMappingWorldToLocal[2][5] = 7;
    out.transformChildMappingWorldToLocal[2][6] = 4;
    out.transformChildMappingWorldToLocal[2][7] = 5;
    out.transformChildMappingWorldToLocal[3][0] = 3;
    out.transformChildMappingWorldToLocal[3][1] = 2;
    out.transformChildMappingWorldToLocal[3][2] = 1;
    out.transformChildMappingWorldToLocal[3][3] = 0;
    out.transformChildMappingWorldToLocal[3][4] = 7;
    out.transformChildMappingWorldToLocal[3][5] = 6;
    out.transformChildMappingWorldToLocal[3][6] = 5;
    out.transformChildMappingWorldToLocal[3][7] = 4;
    out.transformChildMappingWorldToLocal[4][0] = 4;
    out.transformChildMappingWorldToLocal[4][1] = 5;
    out.transformChildMappingWorldToLocal[4][2] = 6;
    out.transformChildMappingWorldToLocal[4][3] = 7;
    out.transformChildMappingWorldToLocal[4][4] = 0;
    out.transformChildMappingWorldToLocal[4][5] = 1;
    out.transformChildMappingWorldToLocal[4][6] = 2;
    out.transformChildMappingWorldToLocal[4][7] = 3;
    out.transformChildMappingWorldToLocal[5][0] = 5;
    out.transformChildMappingWorldToLocal[5][1] = 4;
    out.transformChildMappingWorldToLocal[5][2] = 7;
    out.transformChildMappingWorldToLocal[5][3] = 6;
    out.transformChildMappingWorldToLocal[5][4] = 1;
    out.transformChildMappingWorldToLocal[5][5] = 0;
    out.transformChildMappingWorldToLocal[5][6] = 3;
    out.transformChildMappingWorldToLocal[5][7] = 2;
    out.transformChildMappingWorldToLocal[6][0] = 6;
    out.transformChildMappingWorldToLocal[6][1] = 7;
    out.transformChildMappingWorldToLocal[6][2] = 4;
    out.transformChildMappingWorldToLocal[6][3] = 5;
    out.transformChildMappingWorldToLocal[6][4] = 2;
    out.transformChildMappingWorldToLocal[6][5] = 3;
    out.transformChildMappingWorldToLocal[6][6] = 0;
    out.transformChildMappingWorldToLocal[6][7] = 1;
    out.transformChildMappingWorldToLocal[7][0] = 7;
    out.transformChildMappingWorldToLocal[7][1] = 6;
    out.transformChildMappingWorldToLocal[7][2] = 5;
    out.transformChildMappingWorldToLocal[7][3] = 4;
    out.transformChildMappingWorldToLocal[7][4] = 3;
    out.transformChildMappingWorldToLocal[7][5] = 2;
    out.transformChildMappingWorldToLocal[7][6] = 1;
    out.transformChildMappingWorldToLocal[7][7] = 0;
    out.transformChildMappingWorldToLocal[8][0] = 0;
    out.transformChildMappingWorldToLocal[8][1] = 1;
    out.transformChildMappingWorldToLocal[8][2] = 4;
    out.transformChildMappingWorldToLocal[8][3] = 5;
    out.transformChildMappingWorldToLocal[8][4] = 2;
    out.transformChildMappingWorldToLocal[8][5] = 3;
    out.transformChildMappingWorldToLocal[8][6] = 6;
    out.transformChildMappingWorldToLocal[8][7] = 7;
    out.transformChildMappingWorldToLocal[9][0] = 1;
    out.transformChildMappingWorldToLocal[9][1] = 0;
    out.transformChildMappingWorldToLocal[9][2] = 5;
    out.transformChildMappingWorldToLocal[9][3] = 4;
    out.transformChildMappingWorldToLocal[9][4] = 3;
    out.transformChildMappingWorldToLocal[9][5] = 2;
    out.transformChildMappingWorldToLocal[9][6] = 7;
    out.transformChildMappingWorldToLocal[9][7] = 6;
    out.transformChildMappingWorldToLocal[10][0] = 4;
    out.transformChildMappingWorldToLocal[10][1] = 5;
    out.transformChildMappingWorldToLocal[10][2] = 0;
    out.transformChildMappingWorldToLocal[10][3] = 1;
    out.transformChildMappingWorldToLocal[10][4] = 6;
    out.transformChildMappingWorldToLocal[10][5] = 7;
    out.transformChildMappingWorldToLocal[10][6] = 2;
    out.transformChildMappingWorldToLocal[10][7] = 3;
    out.transformChildMappingWorldToLocal[11][0] = 5;
    out.transformChildMappingWorldToLocal[11][1] = 4;
    out.transformChildMappingWorldToLocal[11][2] = 1;
    out.transformChildMappingWorldToLocal[11][3] = 0;
    out.transformChildMappingWorldToLocal[11][4] = 7;
    out.transformChildMappingWorldToLocal[11][5] = 6;
    out.transformChildMappingWorldToLocal[11][6] = 3;
    out.transformChildMappingWorldToLocal[11][7] = 2;
    out.transformChildMappingWorldToLocal[12][0] = 2;
    out.transformChildMappingWorldToLocal[12][1] = 3;
    out.transformChildMappingWorldToLocal[12][2] = 6;
    out.transformChildMappingWorldToLocal[12][3] = 7;
    out.transformChildMappingWorldToLocal[12][4] = 0;
    out.transformChildMappingWorldToLocal[12][5] = 1;
    out.transformChildMappingWorldToLocal[12][6] = 4;
    out.transformChildMappingWorldToLocal[12][7] = 5;
    out.transformChildMappingWorldToLocal[13][0] = 3;
    out.transformChildMappingWorldToLocal[13][1] = 2;
    out.transformChildMappingWorldToLocal[13][2] = 7;
    out.transformChildMappingWorldToLocal[13][3] = 6;
    out.transformChildMappingWorldToLocal[13][4] = 1;
    out.transformChildMappingWorldToLocal[13][5] = 0;
    out.transformChildMappingWorldToLocal[13][6] = 5;
    out.transformChildMappingWorldToLocal[13][7] = 4;
    out.transformChildMappingWorldToLocal[14][0] = 6;
    out.transformChildMappingWorldToLocal[14][1] = 7;
    out.transformChildMappingWorldToLocal[14][2] = 2;
    out.transformChildMappingWorldToLocal[14][3] = 3;
    out.transformChildMappingWorldToLocal[14][4] = 4;
    out.transformChildMappingWorldToLocal[14][5] = 5;
    out.transformChildMappingWorldToLocal[14][6] = 0;
    out.transformChildMappingWorldToLocal[14][7] = 1;
    out.transformChildMappingWorldToLocal[15][0] = 7;
    out.transformChildMappingWorldToLocal[15][1] = 6;
    out.transformChildMappingWorldToLocal[15][2] = 3;
    out.transformChildMappingWorldToLocal[15][3] = 2;
    out.transformChildMappingWorldToLocal[15][4] = 5;
    out.transformChildMappingWorldToLocal[15][5] = 4;
    out.transformChildMappingWorldToLocal[15][6] = 1;
    out.transformChildMappingWorldToLocal[15][7] = 0;
    out.transformChildMappingWorldToLocal[16][0] = 0;
    out.transformChildMappingWorldToLocal[16][1] = 2;
    out.transformChildMappingWorldToLocal[16][2] = 1;
    out.transformChildMappingWorldToLocal[16][3] = 3;
    out.transformChildMappingWorldToLocal[16][4] = 4;
    out.transformChildMappingWorldToLocal[16][5] = 6;
    out.transformChildMappingWorldToLocal[16][6] = 5;
    out.transformChildMappingWorldToLocal[16][7] = 7;
    out.transformChildMappingWorldToLocal[17][0] = 2;
    out.transformChildMappingWorldToLocal[17][1] = 0;
    out.transformChildMappingWorldToLocal[17][2] = 3;
    out.transformChildMappingWorldToLocal[17][3] = 1;
    out.transformChildMappingWorldToLocal[17][4] = 6;
    out.transformChildMappingWorldToLocal[17][5] = 4;
    out.transformChildMappingWorldToLocal[17][6] = 7;
    out.transformChildMappingWorldToLocal[17][7] = 5;
    out.transformChildMappingWorldToLocal[18][0] = 1;
    out.transformChildMappingWorldToLocal[18][1] = 3;
    out.transformChildMappingWorldToLocal[18][2] = 0;
    out.transformChildMappingWorldToLocal[18][3] = 2;
    out.transformChildMappingWorldToLocal[18][4] = 5;
    out.transformChildMappingWorldToLocal[18][5] = 7;
    out.transformChildMappingWorldToLocal[18][6] = 4;
    out.transformChildMappingWorldToLocal[18][7] = 6;
    out.transformChildMappingWorldToLocal[19][0] = 3;
    out.transformChildMappingWorldToLocal[19][1] = 1;
    out.transformChildMappingWorldToLocal[19][2] = 2;
    out.transformChildMappingWorldToLocal[19][3] = 0;
    out.transformChildMappingWorldToLocal[19][4] = 7;
    out.transformChildMappingWorldToLocal[19][5] = 5;
    out.transformChildMappingWorldToLocal[19][6] = 6;
    out.transformChildMappingWorldToLocal[19][7] = 4;
    out.transformChildMappingWorldToLocal[20][0] = 4;
    out.transformChildMappingWorldToLocal[20][1] = 6;
    out.transformChildMappingWorldToLocal[20][2] = 5;
    out.transformChildMappingWorldToLocal[20][3] = 7;
    out.transformChildMappingWorldToLocal[20][4] = 0;
    out.transformChildMappingWorldToLocal[20][5] = 2;
    out.transformChildMappingWorldToLocal[20][6] = 1;
    out.transformChildMappingWorldToLocal[20][7] = 3;
    out.transformChildMappingWorldToLocal[21][0] = 6;
    out.transformChildMappingWorldToLocal[21][1] = 4;
    out.transformChildMappingWorldToLocal[21][2] = 7;
    out.transformChildMappingWorldToLocal[21][3] = 5;
    out.transformChildMappingWorldToLocal[21][4] = 2;
    out.transformChildMappingWorldToLocal[21][5] = 0;
    out.transformChildMappingWorldToLocal[21][6] = 3;
    out.transformChildMappingWorldToLocal[21][7] = 1;
    out.transformChildMappingWorldToLocal[22][0] = 5;
    out.transformChildMappingWorldToLocal[22][1] = 7;
    out.transformChildMappingWorldToLocal[22][2] = 4;
    out.transformChildMappingWorldToLocal[22][3] = 6;
    out.transformChildMappingWorldToLocal[22][4] = 1;
    out.transformChildMappingWorldToLocal[22][5] = 3;
    out.transformChildMappingWorldToLocal[22][6] = 0;
    out.transformChildMappingWorldToLocal[22][7] = 2;
    out.transformChildMappingWorldToLocal[23][0] = 7;
    out.transformChildMappingWorldToLocal[23][1] = 5;
    out.transformChildMappingWorldToLocal[23][2] = 6;
    out.transformChildMappingWorldToLocal[23][3] = 4;
    out.transformChildMappingWorldToLocal[23][4] = 3;
    out.transformChildMappingWorldToLocal[23][5] = 1;
    out.transformChildMappingWorldToLocal[23][6] = 2;
    out.transformChildMappingWorldToLocal[23][7] = 0;
    out.transformChildMappingWorldToLocal[24][0] = 0;
    out.transformChildMappingWorldToLocal[24][1] = 2;
    out.transformChildMappingWorldToLocal[24][2] = 4;
    out.transformChildMappingWorldToLocal[24][3] = 6;
    out.transformChildMappingWorldToLocal[24][4] = 1;
    out.transformChildMappingWorldToLocal[24][5] = 3;
    out.transformChildMappingWorldToLocal[24][6] = 5;
    out.transformChildMappingWorldToLocal[24][7] = 7;
    out.transformChildMappingWorldToLocal[25][0] = 2;
    out.transformChildMappingWorldToLocal[25][1] = 0;
    out.transformChildMappingWorldToLocal[25][2] = 6;
    out.transformChildMappingWorldToLocal[25][3] = 4;
    out.transformChildMappingWorldToLocal[25][4] = 3;
    out.transformChildMappingWorldToLocal[25][5] = 1;
    out.transformChildMappingWorldToLocal[25][6] = 7;
    out.transformChildMappingWorldToLocal[25][7] = 5;
    out.transformChildMappingWorldToLocal[26][0] = 4;
    out.transformChildMappingWorldToLocal[26][1] = 6;
    out.transformChildMappingWorldToLocal[26][2] = 0;
    out.transformChildMappingWorldToLocal[26][3] = 2;
    out.transformChildMappingWorldToLocal[26][4] = 5;
    out.transformChildMappingWorldToLocal[26][5] = 7;
    out.transformChildMappingWorldToLocal[26][6] = 1;
    out.transformChildMappingWorldToLocal[26][7] = 3;
    out.transformChildMappingWorldToLocal[27][0] = 6;
    out.transformChildMappingWorldToLocal[27][1] = 4;
    out.transformChildMappingWorldToLocal[27][2] = 2;
    out.transformChildMappingWorldToLocal[27][3] = 0;
    out.transformChildMappingWorldToLocal[27][4] = 7;
    out.transformChildMappingWorldToLocal[27][5] = 5;
    out.transformChildMappingWorldToLocal[27][6] = 3;
    out.transformChildMappingWorldToLocal[27][7] = 1;
    out.transformChildMappingWorldToLocal[28][0] = 1;
    out.transformChildMappingWorldToLocal[28][1] = 3;
    out.transformChildMappingWorldToLocal[28][2] = 5;
    out.transformChildMappingWorldToLocal[28][3] = 7;
    out.transformChildMappingWorldToLocal[28][4] = 0;
    out.transformChildMappingWorldToLocal[28][5] = 2;
    out.transformChildMappingWorldToLocal[28][6] = 4;
    out.transformChildMappingWorldToLocal[28][7] = 6;
    out.transformChildMappingWorldToLocal[29][0] = 3;
    out.transformChildMappingWorldToLocal[29][1] = 1;
    out.transformChildMappingWorldToLocal[29][2] = 7;
    out.transformChildMappingWorldToLocal[29][3] = 5;
    out.transformChildMappingWorldToLocal[29][4] = 2;
    out.transformChildMappingWorldToLocal[29][5] = 0;
    out.transformChildMappingWorldToLocal[29][6] = 6;
    out.transformChildMappingWorldToLocal[29][7] = 4;
    out.transformChildMappingWorldToLocal[30][0] = 5;
    out.transformChildMappingWorldToLocal[30][1] = 7;
    out.transformChildMappingWorldToLocal[30][2] = 1;
    out.transformChildMappingWorldToLocal[30][3] = 3;
    out.transformChildMappingWorldToLocal[30][4] = 4;
    out.transformChildMappingWorldToLocal[30][5] = 6;
    out.transformChildMappingWorldToLocal[30][6] = 0;
    out.transformChildMappingWorldToLocal[30][7] = 2;
    out.transformChildMappingWorldToLocal[31][0] = 7;
    out.transformChildMappingWorldToLocal[31][1] = 5;
    out.transformChildMappingWorldToLocal[31][2] = 3;
    out.transformChildMappingWorldToLocal[31][3] = 1;
    out.transformChildMappingWorldToLocal[31][4] = 6;
    out.transformChildMappingWorldToLocal[31][5] = 4;
    out.transformChildMappingWorldToLocal[31][6] = 2;
    out.transformChildMappingWorldToLocal[31][7] = 0;
    out.transformChildMappingWorldToLocal[32][0] = 0;
    out.transformChildMappingWorldToLocal[32][1] = 4;
    out.transformChildMappingWorldToLocal[32][2] = 1;
    out.transformChildMappingWorldToLocal[32][3] = 5;
    out.transformChildMappingWorldToLocal[32][4] = 2;
    out.transformChildMappingWorldToLocal[32][5] = 6;
    out.transformChildMappingWorldToLocal[32][6] = 3;
    out.transformChildMappingWorldToLocal[32][7] = 7;
    out.transformChildMappingWorldToLocal[33][0] = 4;
    out.transformChildMappingWorldToLocal[33][1] = 0;
    out.transformChildMappingWorldToLocal[33][2] = 5;
    out.transformChildMappingWorldToLocal[33][3] = 1;
    out.transformChildMappingWorldToLocal[33][4] = 6;
    out.transformChildMappingWorldToLocal[33][5] = 2;
    out.transformChildMappingWorldToLocal[33][6] = 7;
    out.transformChildMappingWorldToLocal[33][7] = 3;
    out.transformChildMappingWorldToLocal[34][0] = 1;
    out.transformChildMappingWorldToLocal[34][1] = 5;
    out.transformChildMappingWorldToLocal[34][2] = 0;
    out.transformChildMappingWorldToLocal[34][3] = 4;
    out.transformChildMappingWorldToLocal[34][4] = 3;
    out.transformChildMappingWorldToLocal[34][5] = 7;
    out.transformChildMappingWorldToLocal[34][6] = 2;
    out.transformChildMappingWorldToLocal[34][7] = 6;
    out.transformChildMappingWorldToLocal[35][0] = 5;
    out.transformChildMappingWorldToLocal[35][1] = 1;
    out.transformChildMappingWorldToLocal[35][2] = 4;
    out.transformChildMappingWorldToLocal[35][3] = 0;
    out.transformChildMappingWorldToLocal[35][4] = 7;
    out.transformChildMappingWorldToLocal[35][5] = 3;
    out.transformChildMappingWorldToLocal[35][6] = 6;
    out.transformChildMappingWorldToLocal[35][7] = 2;
    out.transformChildMappingWorldToLocal[36][0] = 2;
    out.transformChildMappingWorldToLocal[36][1] = 6;
    out.transformChildMappingWorldToLocal[36][2] = 3;
    out.transformChildMappingWorldToLocal[36][3] = 7;
    out.transformChildMappingWorldToLocal[36][4] = 0;
    out.transformChildMappingWorldToLocal[36][5] = 4;
    out.transformChildMappingWorldToLocal[36][6] = 1;
    out.transformChildMappingWorldToLocal[36][7] = 5;
    out.transformChildMappingWorldToLocal[37][0] = 6;
    out.transformChildMappingWorldToLocal[37][1] = 2;
    out.transformChildMappingWorldToLocal[37][2] = 7;
    out.transformChildMappingWorldToLocal[37][3] = 3;
    out.transformChildMappingWorldToLocal[37][4] = 4;
    out.transformChildMappingWorldToLocal[37][5] = 0;
    out.transformChildMappingWorldToLocal[37][6] = 5;
    out.transformChildMappingWorldToLocal[37][7] = 1;
    out.transformChildMappingWorldToLocal[38][0] = 3;
    out.transformChildMappingWorldToLocal[38][1] = 7;
    out.transformChildMappingWorldToLocal[38][2] = 2;
    out.transformChildMappingWorldToLocal[38][3] = 6;
    out.transformChildMappingWorldToLocal[38][4] = 1;
    out.transformChildMappingWorldToLocal[38][5] = 5;
    out.transformChildMappingWorldToLocal[38][6] = 0;
    out.transformChildMappingWorldToLocal[38][7] = 4;
    out.transformChildMappingWorldToLocal[39][0] = 7;
    out.transformChildMappingWorldToLocal[39][1] = 3;
    out.transformChildMappingWorldToLocal[39][2] = 6;
    out.transformChildMappingWorldToLocal[39][3] = 2;
    out.transformChildMappingWorldToLocal[39][4] = 5;
    out.transformChildMappingWorldToLocal[39][5] = 1;
    out.transformChildMappingWorldToLocal[39][6] = 4;
    out.transformChildMappingWorldToLocal[39][7] = 0;
    out.transformChildMappingWorldToLocal[40][0] = 0;
    out.transformChildMappingWorldToLocal[40][1] = 4;
    out.transformChildMappingWorldToLocal[40][2] = 2;
    out.transformChildMappingWorldToLocal[40][3] = 6;
    out.transformChildMappingWorldToLocal[40][4] = 1;
    out.transformChildMappingWorldToLocal[40][5] = 5;
    out.transformChildMappingWorldToLocal[40][6] = 3;
    out.transformChildMappingWorldToLocal[40][7] = 7;
    out.transformChildMappingWorldToLocal[41][0] = 4;
    out.transformChildMappingWorldToLocal[41][1] = 0;
    out.transformChildMappingWorldToLocal[41][2] = 6;
    out.transformChildMappingWorldToLocal[41][3] = 2;
    out.transformChildMappingWorldToLocal[41][4] = 5;
    out.transformChildMappingWorldToLocal[41][5] = 1;
    out.transformChildMappingWorldToLocal[41][6] = 7;
    out.transformChildMappingWorldToLocal[41][7] = 3;
    out.transformChildMappingWorldToLocal[42][0] = 2;
    out.transformChildMappingWorldToLocal[42][1] = 6;
    out.transformChildMappingWorldToLocal[42][2] = 0;
    out.transformChildMappingWorldToLocal[42][3] = 4;
    out.transformChildMappingWorldToLocal[42][4] = 3;
    out.transformChildMappingWorldToLocal[42][5] = 7;
    out.transformChildMappingWorldToLocal[42][6] = 1;
    out.transformChildMappingWorldToLocal[42][7] = 5;
    out.transformChildMappingWorldToLocal[43][0] = 6;
    out.transformChildMappingWorldToLocal[43][1] = 2;
    out.transformChildMappingWorldToLocal[43][2] = 4;
    out.transformChildMappingWorldToLocal[43][3] = 0;
    out.transformChildMappingWorldToLocal[43][4] = 7;
    out.transformChildMappingWorldToLocal[43][5] = 3;
    out.transformChildMappingWorldToLocal[43][6] = 5;
    out.transformChildMappingWorldToLocal[43][7] = 1;
    out.transformChildMappingWorldToLocal[44][0] = 1;
    out.transformChildMappingWorldToLocal[44][1] = 5;
    out.transformChildMappingWorldToLocal[44][2] = 3;
    out.transformChildMappingWorldToLocal[44][3] = 7;
    out.transformChildMappingWorldToLocal[44][4] = 0;
    out.transformChildMappingWorldToLocal[44][5] = 4;
    out.transformChildMappingWorldToLocal[44][6] = 2;
    out.transformChildMappingWorldToLocal[44][7] = 6;
    out.transformChildMappingWorldToLocal[45][0] = 5;
    out.transformChildMappingWorldToLocal[45][1] = 1;
    out.transformChildMappingWorldToLocal[45][2] = 7;
    out.transformChildMappingWorldToLocal[45][3] = 3;
    out.transformChildMappingWorldToLocal[45][4] = 4;
    out.transformChildMappingWorldToLocal[45][5] = 0;
    out.transformChildMappingWorldToLocal[45][6] = 6;
    out.transformChildMappingWorldToLocal[45][7] = 2;
    out.transformChildMappingWorldToLocal[46][0] = 3;
    out.transformChildMappingWorldToLocal[46][1] = 7;
    out.transformChildMappingWorldToLocal[46][2] = 1;
    out.transformChildMappingWorldToLocal[46][3] = 5;
    out.transformChildMappingWorldToLocal[46][4] = 2;
    out.transformChildMappingWorldToLocal[46][5] = 6;
    out.transformChildMappingWorldToLocal[46][6] = 0;
    out.transformChildMappingWorldToLocal[46][7] = 4;
    out.transformChildMappingWorldToLocal[47][0] = 7;
    out.transformChildMappingWorldToLocal[47][1] = 3;
    out.transformChildMappingWorldToLocal[47][2] = 5;
    out.transformChildMappingWorldToLocal[47][3] = 1;
    out.transformChildMappingWorldToLocal[47][4] = 6;
    out.transformChildMappingWorldToLocal[47][5] = 2;
    out.transformChildMappingWorldToLocal[47][6] = 4;
    out.transformChildMappingWorldToLocal[47][7] = 0;
    out.transformMaskMappingLocalToWorld[0][0] = 0;
    out.transformMaskMappingLocalToWorld[0][1] = 1;
    out.transformMaskMappingLocalToWorld[0][2] = 2;
    out.transformMaskMappingLocalToWorld[0][3] = 3;
    out.transformMaskMappingLocalToWorld[0][4] = 4;
    out.transformMaskMappingLocalToWorld[0][5] = 5;
    out.transformMaskMappingLocalToWorld[0][6] = 6;
    out.transformMaskMappingLocalToWorld[0][7] = 7;
    out.transformMaskMappingLocalToWorld[0][8] = 8;
    out.transformMaskMappingLocalToWorld[0][9] = 9;
    out.transformMaskMappingLocalToWorld[0][10] = 10;
    out.transformMaskMappingLocalToWorld[0][11] = 11;
    out.transformMaskMappingLocalToWorld[0][12] = 12;
    out.transformMaskMappingLocalToWorld[0][13] = 13;
    out.transformMaskMappingLocalToWorld[0][14] = 14;
    out.transformMaskMappingLocalToWorld[0][15] = 15;
    out.transformMaskMappingLocalToWorld[0][16] = 16;
    out.transformMaskMappingLocalToWorld[0][17] = 17;
    out.transformMaskMappingLocalToWorld[0][18] = 18;
    out.transformMaskMappingLocalToWorld[0][19] = 19;
    out.transformMaskMappingLocalToWorld[0][20] = 20;
    out.transformMaskMappingLocalToWorld[0][21] = 21;
    out.transformMaskMappingLocalToWorld[0][22] = 22;
    out.transformMaskMappingLocalToWorld[0][23] = 23;
    out.transformMaskMappingLocalToWorld[0][24] = 24;
    out.transformMaskMappingLocalToWorld[0][25] = 25;
    out.transformMaskMappingLocalToWorld[0][26] = 26;
    out.transformMaskMappingLocalToWorld[0][27] = 27;
    out.transformMaskMappingLocalToWorld[0][28] = 28;
    out.transformMaskMappingLocalToWorld[0][29] = 29;
    out.transformMaskMappingLocalToWorld[0][30] = 30;
    out.transformMaskMappingLocalToWorld[0][31] = 31;
    out.transformMaskMappingLocalToWorld[0][32] = 32;
    out.transformMaskMappingLocalToWorld[0][33] = 33;
    out.transformMaskMappingLocalToWorld[0][34] = 34;
    out.transformMaskMappingLocalToWorld[0][35] = 35;
    out.transformMaskMappingLocalToWorld[0][36] = 36;
    out.transformMaskMappingLocalToWorld[0][37] = 37;
    out.transformMaskMappingLocalToWorld[0][38] = 38;
    out.transformMaskMappingLocalToWorld[0][39] = 39;
    out.transformMaskMappingLocalToWorld[0][40] = 40;
    out.transformMaskMappingLocalToWorld[0][41] = 41;
    out.transformMaskMappingLocalToWorld[0][42] = 42;
    out.transformMaskMappingLocalToWorld[0][43] = 43;
    out.transformMaskMappingLocalToWorld[0][44] = 44;
    out.transformMaskMappingLocalToWorld[0][45] = 45;
    out.transformMaskMappingLocalToWorld[0][46] = 46;
    out.transformMaskMappingLocalToWorld[0][47] = 47;
    out.transformMaskMappingLocalToWorld[0][48] = 48;
    out.transformMaskMappingLocalToWorld[0][49] = 49;
    out.transformMaskMappingLocalToWorld[0][50] = 50;
    out.transformMaskMappingLocalToWorld[0][51] = 51;
    out.transformMaskMappingLocalToWorld[0][52] = 52;
    out.transformMaskMappingLocalToWorld[0][53] = 53;
    out.transformMaskMappingLocalToWorld[0][54] = 54;
    out.transformMaskMappingLocalToWorld[0][55] = 55;
    out.transformMaskMappingLocalToWorld[0][56] = 56;
    out.transformMaskMappingLocalToWorld[0][57] = 57;
    out.transformMaskMappingLocalToWorld[0][58] = 58;
    out.transformMaskMappingLocalToWorld[0][59] = 59;
    out.transformMaskMappingLocalToWorld[0][60] = 60;
    out.transformMaskMappingLocalToWorld[0][61] = 61;
    out.transformMaskMappingLocalToWorld[0][62] = 62;
    out.transformMaskMappingLocalToWorld[0][63] = 63;
    out.transformMaskMappingLocalToWorld[0][64] = 64;
    out.transformMaskMappingLocalToWorld[0][65] = 65;
    out.transformMaskMappingLocalToWorld[0][66] = 66;
    out.transformMaskMappingLocalToWorld[0][67] = 67;
    out.transformMaskMappingLocalToWorld[0][68] = 68;
    out.transformMaskMappingLocalToWorld[0][69] = 69;
    out.transformMaskMappingLocalToWorld[0][70] = 70;
    out.transformMaskMappingLocalToWorld[0][71] = 71;
    out.transformMaskMappingLocalToWorld[0][72] = 72;
    out.transformMaskMappingLocalToWorld[0][73] = 73;
    out.transformMaskMappingLocalToWorld[0][74] = 74;
    out.transformMaskMappingLocalToWorld[0][75] = 75;
    out.transformMaskMappingLocalToWorld[0][76] = 76;
    out.transformMaskMappingLocalToWorld[0][77] = 77;
    out.transformMaskMappingLocalToWorld[0][78] = 78;
    out.transformMaskMappingLocalToWorld[0][79] = 79;
    out.transformMaskMappingLocalToWorld[0][80] = 80;
    out.transformMaskMappingLocalToWorld[0][81] = 81;
    out.transformMaskMappingLocalToWorld[0][82] = 82;
    out.transformMaskMappingLocalToWorld[0][83] = 83;
    out.transformMaskMappingLocalToWorld[0][84] = 84;
    out.transformMaskMappingLocalToWorld[0][85] = 85;
    out.transformMaskMappingLocalToWorld[0][86] = 86;
    out.transformMaskMappingLocalToWorld[0][87] = 87;
    out.transformMaskMappingLocalToWorld[0][88] = 88;
    out.transformMaskMappingLocalToWorld[0][89] = 89;
    out.transformMaskMappingLocalToWorld[0][90] = 90;
    out.transformMaskMappingLocalToWorld[0][91] = 91;
    out.transformMaskMappingLocalToWorld[0][92] = 92;
    out.transformMaskMappingLocalToWorld[0][93] = 93;
    out.transformMaskMappingLocalToWorld[0][94] = 94;
    out.transformMaskMappingLocalToWorld[0][95] = 95;
    out.transformMaskMappingLocalToWorld[0][96] = 96;
    out.transformMaskMappingLocalToWorld[0][97] = 97;
    out.transformMaskMappingLocalToWorld[0][98] = 98;
    out.transformMaskMappingLocalToWorld[0][99] = 99;
    out.transformMaskMappingLocalToWorld[0][100] = 100;
    out.transformMaskMappingLocalToWorld[0][101] = 101;
    out.transformMaskMappingLocalToWorld[0][102] = 102;
    out.transformMaskMappingLocalToWorld[0][103] = 103;
    out.transformMaskMappingLocalToWorld[0][104] = 104;
    out.transformMaskMappingLocalToWorld[0][105] = 105;
    out.transformMaskMappingLocalToWorld[0][106] = 106;
    out.transformMaskMappingLocalToWorld[0][107] = 107;
    out.transformMaskMappingLocalToWorld[0][108] = 108;
    out.transformMaskMappingLocalToWorld[0][109] = 109;
    out.transformMaskMappingLocalToWorld[0][110] = 110;
    out.transformMaskMappingLocalToWorld[0][111] = 111;
    out.transformMaskMappingLocalToWorld[0][112] = 112;
    out.transformMaskMappingLocalToWorld[0][113] = 113;
    out.transformMaskMappingLocalToWorld[0][114] = 114;
    out.transformMaskMappingLocalToWorld[0][115] = 115;
    out.transformMaskMappingLocalToWorld[0][116] = 116;
    out.transformMaskMappingLocalToWorld[0][117] = 117;
    out.transformMaskMappingLocalToWorld[0][118] = 118;
    out.transformMaskMappingLocalToWorld[0][119] = 119;
    out.transformMaskMappingLocalToWorld[0][120] = 120;
    out.transformMaskMappingLocalToWorld[0][121] = 121;
    out.transformMaskMappingLocalToWorld[0][122] = 122;
    out.transformMaskMappingLocalToWorld[0][123] = 123;
    out.transformMaskMappingLocalToWorld[0][124] = 124;
    out.transformMaskMappingLocalToWorld[0][125] = 125;
    out.transformMaskMappingLocalToWorld[0][126] = 126;
    out.transformMaskMappingLocalToWorld[0][127] = 127;
    out.transformMaskMappingLocalToWorld[0][128] = 128;
    out.transformMaskMappingLocalToWorld[0][129] = 129;
    out.transformMaskMappingLocalToWorld[0][130] = 130;
    out.transformMaskMappingLocalToWorld[0][131] = 131;
    out.transformMaskMappingLocalToWorld[0][132] = 132;
    out.transformMaskMappingLocalToWorld[0][133] = 133;
    out.transformMaskMappingLocalToWorld[0][134] = 134;
    out.transformMaskMappingLocalToWorld[0][135] = 135;
    out.transformMaskMappingLocalToWorld[0][136] = 136;
    out.transformMaskMappingLocalToWorld[0][137] = 137;
    out.transformMaskMappingLocalToWorld[0][138] = 138;
    out.transformMaskMappingLocalToWorld[0][139] = 139;
    out.transformMaskMappingLocalToWorld[0][140] = 140;
    out.transformMaskMappingLocalToWorld[0][141] = 141;
    out.transformMaskMappingLocalToWorld[0][142] = 142;
    out.transformMaskMappingLocalToWorld[0][143] = 143;
    out.transformMaskMappingLocalToWorld[0][144] = 144;
    out.transformMaskMappingLocalToWorld[0][145] = 145;
    out.transformMaskMappingLocalToWorld[0][146] = 146;
    out.transformMaskMappingLocalToWorld[0][147] = 147;
    out.transformMaskMappingLocalToWorld[0][148] = 148;
    out.transformMaskMappingLocalToWorld[0][149] = 149;
    out.transformMaskMappingLocalToWorld[0][150] = 150;
    out.transformMaskMappingLocalToWorld[0][151] = 151;
    out.transformMaskMappingLocalToWorld[0][152] = 152;
    out.transformMaskMappingLocalToWorld[0][153] = 153;
    out.transformMaskMappingLocalToWorld[0][154] = 154;
    out.transformMaskMappingLocalToWorld[0][155] = 155;
    out.transformMaskMappingLocalToWorld[0][156] = 156;
    out.transformMaskMappingLocalToWorld[0][157] = 157;
    out.transformMaskMappingLocalToWorld[0][158] = 158;
    out.transformMaskMappingLocalToWorld[0][159] = 159;
    out.transformMaskMappingLocalToWorld[0][160] = 160;
    out.transformMaskMappingLocalToWorld[0][161] = 161;
    out.transformMaskMappingLocalToWorld[0][162] = 162;
    out.transformMaskMappingLocalToWorld[0][163] = 163;
    out.transformMaskMappingLocalToWorld[0][164] = 164;
    out.transformMaskMappingLocalToWorld[0][165] = 165;
    out.transformMaskMappingLocalToWorld[0][166] = 166;
    out.transformMaskMappingLocalToWorld[0][167] = 167;
    out.transformMaskMappingLocalToWorld[0][168] = 168;
    out.transformMaskMappingLocalToWorld[0][169] = 169;
    out.transformMaskMappingLocalToWorld[0][170] = 170;
    out.transformMaskMappingLocalToWorld[0][171] = 171;
    out.transformMaskMappingLocalToWorld[0][172] = 172;
    out.transformMaskMappingLocalToWorld[0][173] = 173;
    out.transformMaskMappingLocalToWorld[0][174] = 174;
    out.transformMaskMappingLocalToWorld[0][175] = 175;
    out.transformMaskMappingLocalToWorld[0][176] = 176;
    out.transformMaskMappingLocalToWorld[0][177] = 177;
    out.transformMaskMappingLocalToWorld[0][178] = 178;
    out.transformMaskMappingLocalToWorld[0][179] = 179;
    out.transformMaskMappingLocalToWorld[0][180] = 180;
    out.transformMaskMappingLocalToWorld[0][181] = 181;
    out.transformMaskMappingLocalToWorld[0][182] = 182;
    out.transformMaskMappingLocalToWorld[0][183] = 183;
    out.transformMaskMappingLocalToWorld[0][184] = 184;
    out.transformMaskMappingLocalToWorld[0][185] = 185;
    out.transformMaskMappingLocalToWorld[0][186] = 186;
    out.transformMaskMappingLocalToWorld[0][187] = 187;
    out.transformMaskMappingLocalToWorld[0][188] = 188;
    out.transformMaskMappingLocalToWorld[0][189] = 189;
    out.transformMaskMappingLocalToWorld[0][190] = 190;
    out.transformMaskMappingLocalToWorld[0][191] = 191;
    out.transformMaskMappingLocalToWorld[0][192] = 192;
    out.transformMaskMappingLocalToWorld[0][193] = 193;
    out.transformMaskMappingLocalToWorld[0][194] = 194;
    out.transformMaskMappingLocalToWorld[0][195] = 195;
    out.transformMaskMappingLocalToWorld[0][196] = 196;
    out.transformMaskMappingLocalToWorld[0][197] = 197;
    out.transformMaskMappingLocalToWorld[0][198] = 198;
    out.transformMaskMappingLocalToWorld[0][199] = 199;
    out.transformMaskMappingLocalToWorld[0][200] = 200;
    out.transformMaskMappingLocalToWorld[0][201] = 201;
    out.transformMaskMappingLocalToWorld[0][202] = 202;
    out.transformMaskMappingLocalToWorld[0][203] = 203;
    out.transformMaskMappingLocalToWorld[0][204] = 204;
    out.transformMaskMappingLocalToWorld[0][205] = 205;
    out.transformMaskMappingLocalToWorld[0][206] = 206;
    out.transformMaskMappingLocalToWorld[0][207] = 207;
    out.transformMaskMappingLocalToWorld[0][208] = 208;
    out.transformMaskMappingLocalToWorld[0][209] = 209;
    out.transformMaskMappingLocalToWorld[0][210] = 210;
    out.transformMaskMappingLocalToWorld[0][211] = 211;
    out.transformMaskMappingLocalToWorld[0][212] = 212;
    out.transformMaskMappingLocalToWorld[0][213] = 213;
    out.transformMaskMappingLocalToWorld[0][214] = 214;
    out.transformMaskMappingLocalToWorld[0][215] = 215;
    out.transformMaskMappingLocalToWorld[0][216] = 216;
    out.transformMaskMappingLocalToWorld[0][217] = 217;
    out.transformMaskMappingLocalToWorld[0][218] = 218;
    out.transformMaskMappingLocalToWorld[0][219] = 219;
    out.transformMaskMappingLocalToWorld[0][220] = 220;
    out.transformMaskMappingLocalToWorld[0][221] = 221;
    out.transformMaskMappingLocalToWorld[0][222] = 222;
    out.transformMaskMappingLocalToWorld[0][223] = 223;
    out.transformMaskMappingLocalToWorld[0][224] = 224;
    out.transformMaskMappingLocalToWorld[0][225] = 225;
    out.transformMaskMappingLocalToWorld[0][226] = 226;
    out.transformMaskMappingLocalToWorld[0][227] = 227;
    out.transformMaskMappingLocalToWorld[0][228] = 228;
    out.transformMaskMappingLocalToWorld[0][229] = 229;
    out.transformMaskMappingLocalToWorld[0][230] = 230;
    out.transformMaskMappingLocalToWorld[0][231] = 231;
    out.transformMaskMappingLocalToWorld[0][232] = 232;
    out.transformMaskMappingLocalToWorld[0][233] = 233;
    out.transformMaskMappingLocalToWorld[0][234] = 234;
    out.transformMaskMappingLocalToWorld[0][235] = 235;
    out.transformMaskMappingLocalToWorld[0][236] = 236;
    out.transformMaskMappingLocalToWorld[0][237] = 237;
    out.transformMaskMappingLocalToWorld[0][238] = 238;
    out.transformMaskMappingLocalToWorld[0][239] = 239;
    out.transformMaskMappingLocalToWorld[0][240] = 240;
    out.transformMaskMappingLocalToWorld[0][241] = 241;
    out.transformMaskMappingLocalToWorld[0][242] = 242;
    out.transformMaskMappingLocalToWorld[0][243] = 243;
    out.transformMaskMappingLocalToWorld[0][244] = 244;
    out.transformMaskMappingLocalToWorld[0][245] = 245;
    out.transformMaskMappingLocalToWorld[0][246] = 246;
    out.transformMaskMappingLocalToWorld[0][247] = 247;
    out.transformMaskMappingLocalToWorld[0][248] = 248;
    out.transformMaskMappingLocalToWorld[0][249] = 249;
    out.transformMaskMappingLocalToWorld[0][250] = 250;
    out.transformMaskMappingLocalToWorld[0][251] = 251;
    out.transformMaskMappingLocalToWorld[0][252] = 252;
    out.transformMaskMappingLocalToWorld[0][253] = 253;
    out.transformMaskMappingLocalToWorld[0][254] = 254;
    out.transformMaskMappingLocalToWorld[0][255] = 255;
    out.transformMaskMappingLocalToWorld[1][0] = 0;
    out.transformMaskMappingLocalToWorld[1][1] = 2;
    out.transformMaskMappingLocalToWorld[1][2] = 1;
    out.transformMaskMappingLocalToWorld[1][3] = 3;
    out.transformMaskMappingLocalToWorld[1][4] = 8;
    out.transformMaskMappingLocalToWorld[1][5] = 10;
    out.transformMaskMappingLocalToWorld[1][6] = 9;
    out.transformMaskMappingLocalToWorld[1][7] = 11;
    out.transformMaskMappingLocalToWorld[1][8] = 4;
    out.transformMaskMappingLocalToWorld[1][9] = 6;
    out.transformMaskMappingLocalToWorld[1][10] = 5;
    out.transformMaskMappingLocalToWorld[1][11] = 7;
    out.transformMaskMappingLocalToWorld[1][12] = 12;
    out.transformMaskMappingLocalToWorld[1][13] = 14;
    out.transformMaskMappingLocalToWorld[1][14] = 13;
    out.transformMaskMappingLocalToWorld[1][15] = 15;
    out.transformMaskMappingLocalToWorld[1][16] = 32;
    out.transformMaskMappingLocalToWorld[1][17] = 34;
    out.transformMaskMappingLocalToWorld[1][18] = 33;
    out.transformMaskMappingLocalToWorld[1][19] = 35;
    out.transformMaskMappingLocalToWorld[1][20] = 40;
    out.transformMaskMappingLocalToWorld[1][21] = 42;
    out.transformMaskMappingLocalToWorld[1][22] = 41;
    out.transformMaskMappingLocalToWorld[1][23] = 43;
    out.transformMaskMappingLocalToWorld[1][24] = 36;
    out.transformMaskMappingLocalToWorld[1][25] = 38;
    out.transformMaskMappingLocalToWorld[1][26] = 37;
    out.transformMaskMappingLocalToWorld[1][27] = 39;
    out.transformMaskMappingLocalToWorld[1][28] = 44;
    out.transformMaskMappingLocalToWorld[1][29] = 46;
    out.transformMaskMappingLocalToWorld[1][30] = 45;
    out.transformMaskMappingLocalToWorld[1][31] = 47;
    out.transformMaskMappingLocalToWorld[1][32] = 16;
    out.transformMaskMappingLocalToWorld[1][33] = 18;
    out.transformMaskMappingLocalToWorld[1][34] = 17;
    out.transformMaskMappingLocalToWorld[1][35] = 19;
    out.transformMaskMappingLocalToWorld[1][36] = 24;
    out.transformMaskMappingLocalToWorld[1][37] = 26;
    out.transformMaskMappingLocalToWorld[1][38] = 25;
    out.transformMaskMappingLocalToWorld[1][39] = 27;
    out.transformMaskMappingLocalToWorld[1][40] = 20;
    out.transformMaskMappingLocalToWorld[1][41] = 22;
    out.transformMaskMappingLocalToWorld[1][42] = 21;
    out.transformMaskMappingLocalToWorld[1][43] = 23;
    out.transformMaskMappingLocalToWorld[1][44] = 28;
    out.transformMaskMappingLocalToWorld[1][45] = 30;
    out.transformMaskMappingLocalToWorld[1][46] = 29;
    out.transformMaskMappingLocalToWorld[1][47] = 31;
    out.transformMaskMappingLocalToWorld[1][48] = 48;
    out.transformMaskMappingLocalToWorld[1][49] = 50;
    out.transformMaskMappingLocalToWorld[1][50] = 49;
    out.transformMaskMappingLocalToWorld[1][51] = 51;
    out.transformMaskMappingLocalToWorld[1][52] = 56;
    out.transformMaskMappingLocalToWorld[1][53] = 58;
    out.transformMaskMappingLocalToWorld[1][54] = 57;
    out.transformMaskMappingLocalToWorld[1][55] = 59;
    out.transformMaskMappingLocalToWorld[1][56] = 52;
    out.transformMaskMappingLocalToWorld[1][57] = 54;
    out.transformMaskMappingLocalToWorld[1][58] = 53;
    out.transformMaskMappingLocalToWorld[1][59] = 55;
    out.transformMaskMappingLocalToWorld[1][60] = 60;
    out.transformMaskMappingLocalToWorld[1][61] = 62;
    out.transformMaskMappingLocalToWorld[1][62] = 61;
    out.transformMaskMappingLocalToWorld[1][63] = 63;
    out.transformMaskMappingLocalToWorld[1][64] = 128;
    out.transformMaskMappingLocalToWorld[1][65] = 130;
    out.transformMaskMappingLocalToWorld[1][66] = 129;
    out.transformMaskMappingLocalToWorld[1][67] = 131;
    out.transformMaskMappingLocalToWorld[1][68] = 136;
    out.transformMaskMappingLocalToWorld[1][69] = 138;
    out.transformMaskMappingLocalToWorld[1][70] = 137;
    out.transformMaskMappingLocalToWorld[1][71] = 139;
    out.transformMaskMappingLocalToWorld[1][72] = 132;
    out.transformMaskMappingLocalToWorld[1][73] = 134;
    out.transformMaskMappingLocalToWorld[1][74] = 133;
    out.transformMaskMappingLocalToWorld[1][75] = 135;
    out.transformMaskMappingLocalToWorld[1][76] = 140;
    out.transformMaskMappingLocalToWorld[1][77] = 142;
    out.transformMaskMappingLocalToWorld[1][78] = 141;
    out.transformMaskMappingLocalToWorld[1][79] = 143;
    out.transformMaskMappingLocalToWorld[1][80] = 160;
    out.transformMaskMappingLocalToWorld[1][81] = 162;
    out.transformMaskMappingLocalToWorld[1][82] = 161;
    out.transformMaskMappingLocalToWorld[1][83] = 163;
    out.transformMaskMappingLocalToWorld[1][84] = 168;
    out.transformMaskMappingLocalToWorld[1][85] = 170;
    out.transformMaskMappingLocalToWorld[1][86] = 169;
    out.transformMaskMappingLocalToWorld[1][87] = 171;
    out.transformMaskMappingLocalToWorld[1][88] = 164;
    out.transformMaskMappingLocalToWorld[1][89] = 166;
    out.transformMaskMappingLocalToWorld[1][90] = 165;
    out.transformMaskMappingLocalToWorld[1][91] = 167;
    out.transformMaskMappingLocalToWorld[1][92] = 172;
    out.transformMaskMappingLocalToWorld[1][93] = 174;
    out.transformMaskMappingLocalToWorld[1][94] = 173;
    out.transformMaskMappingLocalToWorld[1][95] = 175;
    out.transformMaskMappingLocalToWorld[1][96] = 144;
    out.transformMaskMappingLocalToWorld[1][97] = 146;
    out.transformMaskMappingLocalToWorld[1][98] = 145;
    out.transformMaskMappingLocalToWorld[1][99] = 147;
    out.transformMaskMappingLocalToWorld[1][100] = 152;
    out.transformMaskMappingLocalToWorld[1][101] = 154;
    out.transformMaskMappingLocalToWorld[1][102] = 153;
    out.transformMaskMappingLocalToWorld[1][103] = 155;
    out.transformMaskMappingLocalToWorld[1][104] = 148;
    out.transformMaskMappingLocalToWorld[1][105] = 150;
    out.transformMaskMappingLocalToWorld[1][106] = 149;
    out.transformMaskMappingLocalToWorld[1][107] = 151;
    out.transformMaskMappingLocalToWorld[1][108] = 156;
    out.transformMaskMappingLocalToWorld[1][109] = 158;
    out.transformMaskMappingLocalToWorld[1][110] = 157;
    out.transformMaskMappingLocalToWorld[1][111] = 159;
    out.transformMaskMappingLocalToWorld[1][112] = 176;
    out.transformMaskMappingLocalToWorld[1][113] = 178;
    out.transformMaskMappingLocalToWorld[1][114] = 177;
    out.transformMaskMappingLocalToWorld[1][115] = 179;
    out.transformMaskMappingLocalToWorld[1][116] = 184;
    out.transformMaskMappingLocalToWorld[1][117] = 186;
    out.transformMaskMappingLocalToWorld[1][118] = 185;
    out.transformMaskMappingLocalToWorld[1][119] = 187;
    out.transformMaskMappingLocalToWorld[1][120] = 180;
    out.transformMaskMappingLocalToWorld[1][121] = 182;
    out.transformMaskMappingLocalToWorld[1][122] = 181;
    out.transformMaskMappingLocalToWorld[1][123] = 183;
    out.transformMaskMappingLocalToWorld[1][124] = 188;
    out.transformMaskMappingLocalToWorld[1][125] = 190;
    out.transformMaskMappingLocalToWorld[1][126] = 189;
    out.transformMaskMappingLocalToWorld[1][127] = 191;
    out.transformMaskMappingLocalToWorld[1][128] = 64;
    out.transformMaskMappingLocalToWorld[1][129] = 66;
    out.transformMaskMappingLocalToWorld[1][130] = 65;
    out.transformMaskMappingLocalToWorld[1][131] = 67;
    out.transformMaskMappingLocalToWorld[1][132] = 72;
    out.transformMaskMappingLocalToWorld[1][133] = 74;
    out.transformMaskMappingLocalToWorld[1][134] = 73;
    out.transformMaskMappingLocalToWorld[1][135] = 75;
    out.transformMaskMappingLocalToWorld[1][136] = 68;
    out.transformMaskMappingLocalToWorld[1][137] = 70;
    out.transformMaskMappingLocalToWorld[1][138] = 69;
    out.transformMaskMappingLocalToWorld[1][139] = 71;
    out.transformMaskMappingLocalToWorld[1][140] = 76;
    out.transformMaskMappingLocalToWorld[1][141] = 78;
    out.transformMaskMappingLocalToWorld[1][142] = 77;
    out.transformMaskMappingLocalToWorld[1][143] = 79;
    out.transformMaskMappingLocalToWorld[1][144] = 96;
    out.transformMaskMappingLocalToWorld[1][145] = 98;
    out.transformMaskMappingLocalToWorld[1][146] = 97;
    out.transformMaskMappingLocalToWorld[1][147] = 99;
    out.transformMaskMappingLocalToWorld[1][148] = 104;
    out.transformMaskMappingLocalToWorld[1][149] = 106;
    out.transformMaskMappingLocalToWorld[1][150] = 105;
    out.transformMaskMappingLocalToWorld[1][151] = 107;
    out.transformMaskMappingLocalToWorld[1][152] = 100;
    out.transformMaskMappingLocalToWorld[1][153] = 102;
    out.transformMaskMappingLocalToWorld[1][154] = 101;
    out.transformMaskMappingLocalToWorld[1][155] = 103;
    out.transformMaskMappingLocalToWorld[1][156] = 108;
    out.transformMaskMappingLocalToWorld[1][157] = 110;
    out.transformMaskMappingLocalToWorld[1][158] = 109;
    out.transformMaskMappingLocalToWorld[1][159] = 111;
    out.transformMaskMappingLocalToWorld[1][160] = 80;
    out.transformMaskMappingLocalToWorld[1][161] = 82;
    out.transformMaskMappingLocalToWorld[1][162] = 81;
    out.transformMaskMappingLocalToWorld[1][163] = 83;
    out.transformMaskMappingLocalToWorld[1][164] = 88;
    out.transformMaskMappingLocalToWorld[1][165] = 90;
    out.transformMaskMappingLocalToWorld[1][166] = 89;
    out.transformMaskMappingLocalToWorld[1][167] = 91;
    out.transformMaskMappingLocalToWorld[1][168] = 84;
    out.transformMaskMappingLocalToWorld[1][169] = 86;
    out.transformMaskMappingLocalToWorld[1][170] = 85;
    out.transformMaskMappingLocalToWorld[1][171] = 87;
    out.transformMaskMappingLocalToWorld[1][172] = 92;
    out.transformMaskMappingLocalToWorld[1][173] = 94;
    out.transformMaskMappingLocalToWorld[1][174] = 93;
    out.transformMaskMappingLocalToWorld[1][175] = 95;
    out.transformMaskMappingLocalToWorld[1][176] = 112;
    out.transformMaskMappingLocalToWorld[1][177] = 114;
    out.transformMaskMappingLocalToWorld[1][178] = 113;
    out.transformMaskMappingLocalToWorld[1][179] = 115;
    out.transformMaskMappingLocalToWorld[1][180] = 120;
    out.transformMaskMappingLocalToWorld[1][181] = 122;
    out.transformMaskMappingLocalToWorld[1][182] = 121;
    out.transformMaskMappingLocalToWorld[1][183] = 123;
    out.transformMaskMappingLocalToWorld[1][184] = 116;
    out.transformMaskMappingLocalToWorld[1][185] = 118;
    out.transformMaskMappingLocalToWorld[1][186] = 117;
    out.transformMaskMappingLocalToWorld[1][187] = 119;
    out.transformMaskMappingLocalToWorld[1][188] = 124;
    out.transformMaskMappingLocalToWorld[1][189] = 126;
    out.transformMaskMappingLocalToWorld[1][190] = 125;
    out.transformMaskMappingLocalToWorld[1][191] = 127;
    out.transformMaskMappingLocalToWorld[1][192] = 192;
    out.transformMaskMappingLocalToWorld[1][193] = 194;
    out.transformMaskMappingLocalToWorld[1][194] = 193;
    out.transformMaskMappingLocalToWorld[1][195] = 195;
    out.transformMaskMappingLocalToWorld[1][196] = 200;
    out.transformMaskMappingLocalToWorld[1][197] = 202;
    out.transformMaskMappingLocalToWorld[1][198] = 201;
    out.transformMaskMappingLocalToWorld[1][199] = 203;
    out.transformMaskMappingLocalToWorld[1][200] = 196;
    out.transformMaskMappingLocalToWorld[1][201] = 198;
    out.transformMaskMappingLocalToWorld[1][202] = 197;
    out.transformMaskMappingLocalToWorld[1][203] = 199;
    out.transformMaskMappingLocalToWorld[1][204] = 204;
    out.transformMaskMappingLocalToWorld[1][205] = 206;
    out.transformMaskMappingLocalToWorld[1][206] = 205;
    out.transformMaskMappingLocalToWorld[1][207] = 207;
    out.transformMaskMappingLocalToWorld[1][208] = 224;
    out.transformMaskMappingLocalToWorld[1][209] = 226;
    out.transformMaskMappingLocalToWorld[1][210] = 225;
    out.transformMaskMappingLocalToWorld[1][211] = 227;
    out.transformMaskMappingLocalToWorld[1][212] = 232;
    out.transformMaskMappingLocalToWorld[1][213] = 234;
    out.transformMaskMappingLocalToWorld[1][214] = 233;
    out.transformMaskMappingLocalToWorld[1][215] = 235;
    out.transformMaskMappingLocalToWorld[1][216] = 228;
    out.transformMaskMappingLocalToWorld[1][217] = 230;
    out.transformMaskMappingLocalToWorld[1][218] = 229;
    out.transformMaskMappingLocalToWorld[1][219] = 231;
    out.transformMaskMappingLocalToWorld[1][220] = 236;
    out.transformMaskMappingLocalToWorld[1][221] = 238;
    out.transformMaskMappingLocalToWorld[1][222] = 237;
    out.transformMaskMappingLocalToWorld[1][223] = 239;
    out.transformMaskMappingLocalToWorld[1][224] = 208;
    out.transformMaskMappingLocalToWorld[1][225] = 210;
    out.transformMaskMappingLocalToWorld[1][226] = 209;
    out.transformMaskMappingLocalToWorld[1][227] = 211;
    out.transformMaskMappingLocalToWorld[1][228] = 216;
    out.transformMaskMappingLocalToWorld[1][229] = 218;
    out.transformMaskMappingLocalToWorld[1][230] = 217;
    out.transformMaskMappingLocalToWorld[1][231] = 219;
    out.transformMaskMappingLocalToWorld[1][232] = 212;
    out.transformMaskMappingLocalToWorld[1][233] = 214;
    out.transformMaskMappingLocalToWorld[1][234] = 213;
    out.transformMaskMappingLocalToWorld[1][235] = 215;
    out.transformMaskMappingLocalToWorld[1][236] = 220;
    out.transformMaskMappingLocalToWorld[1][237] = 222;
    out.transformMaskMappingLocalToWorld[1][238] = 221;
    out.transformMaskMappingLocalToWorld[1][239] = 223;
    out.transformMaskMappingLocalToWorld[1][240] = 240;
    out.transformMaskMappingLocalToWorld[1][241] = 242;
    out.transformMaskMappingLocalToWorld[1][242] = 241;
    out.transformMaskMappingLocalToWorld[1][243] = 243;
    out.transformMaskMappingLocalToWorld[1][244] = 248;
    out.transformMaskMappingLocalToWorld[1][245] = 250;
    out.transformMaskMappingLocalToWorld[1][246] = 249;
    out.transformMaskMappingLocalToWorld[1][247] = 251;
    out.transformMaskMappingLocalToWorld[1][248] = 244;
    out.transformMaskMappingLocalToWorld[1][249] = 246;
    out.transformMaskMappingLocalToWorld[1][250] = 245;
    out.transformMaskMappingLocalToWorld[1][251] = 247;
    out.transformMaskMappingLocalToWorld[1][252] = 252;
    out.transformMaskMappingLocalToWorld[1][253] = 254;
    out.transformMaskMappingLocalToWorld[1][254] = 253;
    out.transformMaskMappingLocalToWorld[1][255] = 255;
    out.transformMaskMappingLocalToWorld[2][0] = 0;
    out.transformMaskMappingLocalToWorld[2][1] = 4;
    out.transformMaskMappingLocalToWorld[2][2] = 8;
    out.transformMaskMappingLocalToWorld[2][3] = 12;
    out.transformMaskMappingLocalToWorld[2][4] = 1;
    out.transformMaskMappingLocalToWorld[2][5] = 5;
    out.transformMaskMappingLocalToWorld[2][6] = 9;
    out.transformMaskMappingLocalToWorld[2][7] = 13;
    out.transformMaskMappingLocalToWorld[2][8] = 2;
    out.transformMaskMappingLocalToWorld[2][9] = 6;
    out.transformMaskMappingLocalToWorld[2][10] = 10;
    out.transformMaskMappingLocalToWorld[2][11] = 14;
    out.transformMaskMappingLocalToWorld[2][12] = 3;
    out.transformMaskMappingLocalToWorld[2][13] = 7;
    out.transformMaskMappingLocalToWorld[2][14] = 11;
    out.transformMaskMappingLocalToWorld[2][15] = 15;
    out.transformMaskMappingLocalToWorld[2][16] = 64;
    out.transformMaskMappingLocalToWorld[2][17] = 68;
    out.transformMaskMappingLocalToWorld[2][18] = 72;
    out.transformMaskMappingLocalToWorld[2][19] = 76;
    out.transformMaskMappingLocalToWorld[2][20] = 65;
    out.transformMaskMappingLocalToWorld[2][21] = 69;
    out.transformMaskMappingLocalToWorld[2][22] = 73;
    out.transformMaskMappingLocalToWorld[2][23] = 77;
    out.transformMaskMappingLocalToWorld[2][24] = 66;
    out.transformMaskMappingLocalToWorld[2][25] = 70;
    out.transformMaskMappingLocalToWorld[2][26] = 74;
    out.transformMaskMappingLocalToWorld[2][27] = 78;
    out.transformMaskMappingLocalToWorld[2][28] = 67;
    out.transformMaskMappingLocalToWorld[2][29] = 71;
    out.transformMaskMappingLocalToWorld[2][30] = 75;
    out.transformMaskMappingLocalToWorld[2][31] = 79;
    out.transformMaskMappingLocalToWorld[2][32] = 128;
    out.transformMaskMappingLocalToWorld[2][33] = 132;
    out.transformMaskMappingLocalToWorld[2][34] = 136;
    out.transformMaskMappingLocalToWorld[2][35] = 140;
    out.transformMaskMappingLocalToWorld[2][36] = 129;
    out.transformMaskMappingLocalToWorld[2][37] = 133;
    out.transformMaskMappingLocalToWorld[2][38] = 137;
    out.transformMaskMappingLocalToWorld[2][39] = 141;
    out.transformMaskMappingLocalToWorld[2][40] = 130;
    out.transformMaskMappingLocalToWorld[2][41] = 134;
    out.transformMaskMappingLocalToWorld[2][42] = 138;
    out.transformMaskMappingLocalToWorld[2][43] = 142;
    out.transformMaskMappingLocalToWorld[2][44] = 131;
    out.transformMaskMappingLocalToWorld[2][45] = 135;
    out.transformMaskMappingLocalToWorld[2][46] = 139;
    out.transformMaskMappingLocalToWorld[2][47] = 143;
    out.transformMaskMappingLocalToWorld[2][48] = 192;
    out.transformMaskMappingLocalToWorld[2][49] = 196;
    out.transformMaskMappingLocalToWorld[2][50] = 200;
    out.transformMaskMappingLocalToWorld[2][51] = 204;
    out.transformMaskMappingLocalToWorld[2][52] = 193;
    out.transformMaskMappingLocalToWorld[2][53] = 197;
    out.transformMaskMappingLocalToWorld[2][54] = 201;
    out.transformMaskMappingLocalToWorld[2][55] = 205;
    out.transformMaskMappingLocalToWorld[2][56] = 194;
    out.transformMaskMappingLocalToWorld[2][57] = 198;
    out.transformMaskMappingLocalToWorld[2][58] = 202;
    out.transformMaskMappingLocalToWorld[2][59] = 206;
    out.transformMaskMappingLocalToWorld[2][60] = 195;
    out.transformMaskMappingLocalToWorld[2][61] = 199;
    out.transformMaskMappingLocalToWorld[2][62] = 203;
    out.transformMaskMappingLocalToWorld[2][63] = 207;
    out.transformMaskMappingLocalToWorld[2][64] = 16;
    out.transformMaskMappingLocalToWorld[2][65] = 20;
    out.transformMaskMappingLocalToWorld[2][66] = 24;
    out.transformMaskMappingLocalToWorld[2][67] = 28;
    out.transformMaskMappingLocalToWorld[2][68] = 17;
    out.transformMaskMappingLocalToWorld[2][69] = 21;
    out.transformMaskMappingLocalToWorld[2][70] = 25;
    out.transformMaskMappingLocalToWorld[2][71] = 29;
    out.transformMaskMappingLocalToWorld[2][72] = 18;
    out.transformMaskMappingLocalToWorld[2][73] = 22;
    out.transformMaskMappingLocalToWorld[2][74] = 26;
    out.transformMaskMappingLocalToWorld[2][75] = 30;
    out.transformMaskMappingLocalToWorld[2][76] = 19;
    out.transformMaskMappingLocalToWorld[2][77] = 23;
    out.transformMaskMappingLocalToWorld[2][78] = 27;
    out.transformMaskMappingLocalToWorld[2][79] = 31;
    out.transformMaskMappingLocalToWorld[2][80] = 80;
    out.transformMaskMappingLocalToWorld[2][81] = 84;
    out.transformMaskMappingLocalToWorld[2][82] = 88;
    out.transformMaskMappingLocalToWorld[2][83] = 92;
    out.transformMaskMappingLocalToWorld[2][84] = 81;
    out.transformMaskMappingLocalToWorld[2][85] = 85;
    out.transformMaskMappingLocalToWorld[2][86] = 89;
    out.transformMaskMappingLocalToWorld[2][87] = 93;
    out.transformMaskMappingLocalToWorld[2][88] = 82;
    out.transformMaskMappingLocalToWorld[2][89] = 86;
    out.transformMaskMappingLocalToWorld[2][90] = 90;
    out.transformMaskMappingLocalToWorld[2][91] = 94;
    out.transformMaskMappingLocalToWorld[2][92] = 83;
    out.transformMaskMappingLocalToWorld[2][93] = 87;
    out.transformMaskMappingLocalToWorld[2][94] = 91;
    out.transformMaskMappingLocalToWorld[2][95] = 95;
    out.transformMaskMappingLocalToWorld[2][96] = 144;
    out.transformMaskMappingLocalToWorld[2][97] = 148;
    out.transformMaskMappingLocalToWorld[2][98] = 152;
    out.transformMaskMappingLocalToWorld[2][99] = 156;
    out.transformMaskMappingLocalToWorld[2][100] = 145;
    out.transformMaskMappingLocalToWorld[2][101] = 149;
    out.transformMaskMappingLocalToWorld[2][102] = 153;
    out.transformMaskMappingLocalToWorld[2][103] = 157;
    out.transformMaskMappingLocalToWorld[2][104] = 146;
    out.transformMaskMappingLocalToWorld[2][105] = 150;
    out.transformMaskMappingLocalToWorld[2][106] = 154;
    out.transformMaskMappingLocalToWorld[2][107] = 158;
    out.transformMaskMappingLocalToWorld[2][108] = 147;
    out.transformMaskMappingLocalToWorld[2][109] = 151;
    out.transformMaskMappingLocalToWorld[2][110] = 155;
    out.transformMaskMappingLocalToWorld[2][111] = 159;
    out.transformMaskMappingLocalToWorld[2][112] = 208;
    out.transformMaskMappingLocalToWorld[2][113] = 212;
    out.transformMaskMappingLocalToWorld[2][114] = 216;
    out.transformMaskMappingLocalToWorld[2][115] = 220;
    out.transformMaskMappingLocalToWorld[2][116] = 209;
    out.transformMaskMappingLocalToWorld[2][117] = 213;
    out.transformMaskMappingLocalToWorld[2][118] = 217;
    out.transformMaskMappingLocalToWorld[2][119] = 221;
    out.transformMaskMappingLocalToWorld[2][120] = 210;
    out.transformMaskMappingLocalToWorld[2][121] = 214;
    out.transformMaskMappingLocalToWorld[2][122] = 218;
    out.transformMaskMappingLocalToWorld[2][123] = 222;
    out.transformMaskMappingLocalToWorld[2][124] = 211;
    out.transformMaskMappingLocalToWorld[2][125] = 215;
    out.transformMaskMappingLocalToWorld[2][126] = 219;
    out.transformMaskMappingLocalToWorld[2][127] = 223;
    out.transformMaskMappingLocalToWorld[2][128] = 32;
    out.transformMaskMappingLocalToWorld[2][129] = 36;
    out.transformMaskMappingLocalToWorld[2][130] = 40;
    out.transformMaskMappingLocalToWorld[2][131] = 44;
    out.transformMaskMappingLocalToWorld[2][132] = 33;
    out.transformMaskMappingLocalToWorld[2][133] = 37;
    out.transformMaskMappingLocalToWorld[2][134] = 41;
    out.transformMaskMappingLocalToWorld[2][135] = 45;
    out.transformMaskMappingLocalToWorld[2][136] = 34;
    out.transformMaskMappingLocalToWorld[2][137] = 38;
    out.transformMaskMappingLocalToWorld[2][138] = 42;
    out.transformMaskMappingLocalToWorld[2][139] = 46;
    out.transformMaskMappingLocalToWorld[2][140] = 35;
    out.transformMaskMappingLocalToWorld[2][141] = 39;
    out.transformMaskMappingLocalToWorld[2][142] = 43;
    out.transformMaskMappingLocalToWorld[2][143] = 47;
    out.transformMaskMappingLocalToWorld[2][144] = 96;
    out.transformMaskMappingLocalToWorld[2][145] = 100;
    out.transformMaskMappingLocalToWorld[2][146] = 104;
    out.transformMaskMappingLocalToWorld[2][147] = 108;
    out.transformMaskMappingLocalToWorld[2][148] = 97;
    out.transformMaskMappingLocalToWorld[2][149] = 101;
    out.transformMaskMappingLocalToWorld[2][150] = 105;
    out.transformMaskMappingLocalToWorld[2][151] = 109;
    out.transformMaskMappingLocalToWorld[2][152] = 98;
    out.transformMaskMappingLocalToWorld[2][153] = 102;
    out.transformMaskMappingLocalToWorld[2][154] = 106;
    out.transformMaskMappingLocalToWorld[2][155] = 110;
    out.transformMaskMappingLocalToWorld[2][156] = 99;
    out.transformMaskMappingLocalToWorld[2][157] = 103;
    out.transformMaskMappingLocalToWorld[2][158] = 107;
    out.transformMaskMappingLocalToWorld[2][159] = 111;
    out.transformMaskMappingLocalToWorld[2][160] = 160;
    out.transformMaskMappingLocalToWorld[2][161] = 164;
    out.transformMaskMappingLocalToWorld[2][162] = 168;
    out.transformMaskMappingLocalToWorld[2][163] = 172;
    out.transformMaskMappingLocalToWorld[2][164] = 161;
    out.transformMaskMappingLocalToWorld[2][165] = 165;
    out.transformMaskMappingLocalToWorld[2][166] = 169;
    out.transformMaskMappingLocalToWorld[2][167] = 173;
    out.transformMaskMappingLocalToWorld[2][168] = 162;
    out.transformMaskMappingLocalToWorld[2][169] = 166;
    out.transformMaskMappingLocalToWorld[2][170] = 170;
    out.transformMaskMappingLocalToWorld[2][171] = 174;
    out.transformMaskMappingLocalToWorld[2][172] = 163;
    out.transformMaskMappingLocalToWorld[2][173] = 167;
    out.transformMaskMappingLocalToWorld[2][174] = 171;
    out.transformMaskMappingLocalToWorld[2][175] = 175;
    out.transformMaskMappingLocalToWorld[2][176] = 224;
    out.transformMaskMappingLocalToWorld[2][177] = 228;
    out.transformMaskMappingLocalToWorld[2][178] = 232;
    out.transformMaskMappingLocalToWorld[2][179] = 236;
    out.transformMaskMappingLocalToWorld[2][180] = 225;
    out.transformMaskMappingLocalToWorld[2][181] = 229;
    out.transformMaskMappingLocalToWorld[2][182] = 233;
    out.transformMaskMappingLocalToWorld[2][183] = 237;
    out.transformMaskMappingLocalToWorld[2][184] = 226;
    out.transformMaskMappingLocalToWorld[2][185] = 230;
    out.transformMaskMappingLocalToWorld[2][186] = 234;
    out.transformMaskMappingLocalToWorld[2][187] = 238;
    out.transformMaskMappingLocalToWorld[2][188] = 227;
    out.transformMaskMappingLocalToWorld[2][189] = 231;
    out.transformMaskMappingLocalToWorld[2][190] = 235;
    out.transformMaskMappingLocalToWorld[2][191] = 239;
    out.transformMaskMappingLocalToWorld[2][192] = 48;
    out.transformMaskMappingLocalToWorld[2][193] = 52;
    out.transformMaskMappingLocalToWorld[2][194] = 56;
    out.transformMaskMappingLocalToWorld[2][195] = 60;
    out.transformMaskMappingLocalToWorld[2][196] = 49;
    out.transformMaskMappingLocalToWorld[2][197] = 53;
    out.transformMaskMappingLocalToWorld[2][198] = 57;
    out.transformMaskMappingLocalToWorld[2][199] = 61;
    out.transformMaskMappingLocalToWorld[2][200] = 50;
    out.transformMaskMappingLocalToWorld[2][201] = 54;
    out.transformMaskMappingLocalToWorld[2][202] = 58;
    out.transformMaskMappingLocalToWorld[2][203] = 62;
    out.transformMaskMappingLocalToWorld[2][204] = 51;
    out.transformMaskMappingLocalToWorld[2][205] = 55;
    out.transformMaskMappingLocalToWorld[2][206] = 59;
    out.transformMaskMappingLocalToWorld[2][207] = 63;
    out.transformMaskMappingLocalToWorld[2][208] = 112;
    out.transformMaskMappingLocalToWorld[2][209] = 116;
    out.transformMaskMappingLocalToWorld[2][210] = 120;
    out.transformMaskMappingLocalToWorld[2][211] = 124;
    out.transformMaskMappingLocalToWorld[2][212] = 113;
    out.transformMaskMappingLocalToWorld[2][213] = 117;
    out.transformMaskMappingLocalToWorld[2][214] = 121;
    out.transformMaskMappingLocalToWorld[2][215] = 125;
    out.transformMaskMappingLocalToWorld[2][216] = 114;
    out.transformMaskMappingLocalToWorld[2][217] = 118;
    out.transformMaskMappingLocalToWorld[2][218] = 122;
    out.transformMaskMappingLocalToWorld[2][219] = 126;
    out.transformMaskMappingLocalToWorld[2][220] = 115;
    out.transformMaskMappingLocalToWorld[2][221] = 119;
    out.transformMaskMappingLocalToWorld[2][222] = 123;
    out.transformMaskMappingLocalToWorld[2][223] = 127;
    out.transformMaskMappingLocalToWorld[2][224] = 176;
    out.transformMaskMappingLocalToWorld[2][225] = 180;
    out.transformMaskMappingLocalToWorld[2][226] = 184;
    out.transformMaskMappingLocalToWorld[2][227] = 188;
    out.transformMaskMappingLocalToWorld[2][228] = 177;
    out.transformMaskMappingLocalToWorld[2][229] = 181;
    out.transformMaskMappingLocalToWorld[2][230] = 185;
    out.transformMaskMappingLocalToWorld[2][231] = 189;
    out.transformMaskMappingLocalToWorld[2][232] = 178;
    out.transformMaskMappingLocalToWorld[2][233] = 182;
    out.transformMaskMappingLocalToWorld[2][234] = 186;
    out.transformMaskMappingLocalToWorld[2][235] = 190;
    out.transformMaskMappingLocalToWorld[2][236] = 179;
    out.transformMaskMappingLocalToWorld[2][237] = 183;
    out.transformMaskMappingLocalToWorld[2][238] = 187;
    out.transformMaskMappingLocalToWorld[2][239] = 191;
    out.transformMaskMappingLocalToWorld[2][240] = 240;
    out.transformMaskMappingLocalToWorld[2][241] = 244;
    out.transformMaskMappingLocalToWorld[2][242] = 248;
    out.transformMaskMappingLocalToWorld[2][243] = 252;
    out.transformMaskMappingLocalToWorld[2][244] = 241;
    out.transformMaskMappingLocalToWorld[2][245] = 245;
    out.transformMaskMappingLocalToWorld[2][246] = 249;
    out.transformMaskMappingLocalToWorld[2][247] = 253;
    out.transformMaskMappingLocalToWorld[2][248] = 242;
    out.transformMaskMappingLocalToWorld[2][249] = 246;
    out.transformMaskMappingLocalToWorld[2][250] = 250;
    out.transformMaskMappingLocalToWorld[2][251] = 254;
    out.transformMaskMappingLocalToWorld[2][252] = 243;
    out.transformMaskMappingLocalToWorld[2][253] = 247;
    out.transformMaskMappingLocalToWorld[2][254] = 251;
    out.transformMaskMappingLocalToWorld[2][255] = 255;
    out.transformMaskMappingLocalToWorld[3][0] = 0;
    out.transformMaskMappingLocalToWorld[3][1] = 8;
    out.transformMaskMappingLocalToWorld[3][2] = 4;
    out.transformMaskMappingLocalToWorld[3][3] = 12;
    out.transformMaskMappingLocalToWorld[3][4] = 2;
    out.transformMaskMappingLocalToWorld[3][5] = 10;
    out.transformMaskMappingLocalToWorld[3][6] = 6;
    out.transformMaskMappingLocalToWorld[3][7] = 14;
    out.transformMaskMappingLocalToWorld[3][8] = 1;
    out.transformMaskMappingLocalToWorld[3][9] = 9;
    out.transformMaskMappingLocalToWorld[3][10] = 5;
    out.transformMaskMappingLocalToWorld[3][11] = 13;
    out.transformMaskMappingLocalToWorld[3][12] = 3;
    out.transformMaskMappingLocalToWorld[3][13] = 11;
    out.transformMaskMappingLocalToWorld[3][14] = 7;
    out.transformMaskMappingLocalToWorld[3][15] = 15;
    out.transformMaskMappingLocalToWorld[3][16] = 128;
    out.transformMaskMappingLocalToWorld[3][17] = 136;
    out.transformMaskMappingLocalToWorld[3][18] = 132;
    out.transformMaskMappingLocalToWorld[3][19] = 140;
    out.transformMaskMappingLocalToWorld[3][20] = 130;
    out.transformMaskMappingLocalToWorld[3][21] = 138;
    out.transformMaskMappingLocalToWorld[3][22] = 134;
    out.transformMaskMappingLocalToWorld[3][23] = 142;
    out.transformMaskMappingLocalToWorld[3][24] = 129;
    out.transformMaskMappingLocalToWorld[3][25] = 137;
    out.transformMaskMappingLocalToWorld[3][26] = 133;
    out.transformMaskMappingLocalToWorld[3][27] = 141;
    out.transformMaskMappingLocalToWorld[3][28] = 131;
    out.transformMaskMappingLocalToWorld[3][29] = 139;
    out.transformMaskMappingLocalToWorld[3][30] = 135;
    out.transformMaskMappingLocalToWorld[3][31] = 143;
    out.transformMaskMappingLocalToWorld[3][32] = 64;
    out.transformMaskMappingLocalToWorld[3][33] = 72;
    out.transformMaskMappingLocalToWorld[3][34] = 68;
    out.transformMaskMappingLocalToWorld[3][35] = 76;
    out.transformMaskMappingLocalToWorld[3][36] = 66;
    out.transformMaskMappingLocalToWorld[3][37] = 74;
    out.transformMaskMappingLocalToWorld[3][38] = 70;
    out.transformMaskMappingLocalToWorld[3][39] = 78;
    out.transformMaskMappingLocalToWorld[3][40] = 65;
    out.transformMaskMappingLocalToWorld[3][41] = 73;
    out.transformMaskMappingLocalToWorld[3][42] = 69;
    out.transformMaskMappingLocalToWorld[3][43] = 77;
    out.transformMaskMappingLocalToWorld[3][44] = 67;
    out.transformMaskMappingLocalToWorld[3][45] = 75;
    out.transformMaskMappingLocalToWorld[3][46] = 71;
    out.transformMaskMappingLocalToWorld[3][47] = 79;
    out.transformMaskMappingLocalToWorld[3][48] = 192;
    out.transformMaskMappingLocalToWorld[3][49] = 200;
    out.transformMaskMappingLocalToWorld[3][50] = 196;
    out.transformMaskMappingLocalToWorld[3][51] = 204;
    out.transformMaskMappingLocalToWorld[3][52] = 194;
    out.transformMaskMappingLocalToWorld[3][53] = 202;
    out.transformMaskMappingLocalToWorld[3][54] = 198;
    out.transformMaskMappingLocalToWorld[3][55] = 206;
    out.transformMaskMappingLocalToWorld[3][56] = 193;
    out.transformMaskMappingLocalToWorld[3][57] = 201;
    out.transformMaskMappingLocalToWorld[3][58] = 197;
    out.transformMaskMappingLocalToWorld[3][59] = 205;
    out.transformMaskMappingLocalToWorld[3][60] = 195;
    out.transformMaskMappingLocalToWorld[3][61] = 203;
    out.transformMaskMappingLocalToWorld[3][62] = 199;
    out.transformMaskMappingLocalToWorld[3][63] = 207;
    out.transformMaskMappingLocalToWorld[3][64] = 32;
    out.transformMaskMappingLocalToWorld[3][65] = 40;
    out.transformMaskMappingLocalToWorld[3][66] = 36;
    out.transformMaskMappingLocalToWorld[3][67] = 44;
    out.transformMaskMappingLocalToWorld[3][68] = 34;
    out.transformMaskMappingLocalToWorld[3][69] = 42;
    out.transformMaskMappingLocalToWorld[3][70] = 38;
    out.transformMaskMappingLocalToWorld[3][71] = 46;
    out.transformMaskMappingLocalToWorld[3][72] = 33;
    out.transformMaskMappingLocalToWorld[3][73] = 41;
    out.transformMaskMappingLocalToWorld[3][74] = 37;
    out.transformMaskMappingLocalToWorld[3][75] = 45;
    out.transformMaskMappingLocalToWorld[3][76] = 35;
    out.transformMaskMappingLocalToWorld[3][77] = 43;
    out.transformMaskMappingLocalToWorld[3][78] = 39;
    out.transformMaskMappingLocalToWorld[3][79] = 47;
    out.transformMaskMappingLocalToWorld[3][80] = 160;
    out.transformMaskMappingLocalToWorld[3][81] = 168;
    out.transformMaskMappingLocalToWorld[3][82] = 164;
    out.transformMaskMappingLocalToWorld[3][83] = 172;
    out.transformMaskMappingLocalToWorld[3][84] = 162;
    out.transformMaskMappingLocalToWorld[3][85] = 170;
    out.transformMaskMappingLocalToWorld[3][86] = 166;
    out.transformMaskMappingLocalToWorld[3][87] = 174;
    out.transformMaskMappingLocalToWorld[3][88] = 161;
    out.transformMaskMappingLocalToWorld[3][89] = 169;
    out.transformMaskMappingLocalToWorld[3][90] = 165;
    out.transformMaskMappingLocalToWorld[3][91] = 173;
    out.transformMaskMappingLocalToWorld[3][92] = 163;
    out.transformMaskMappingLocalToWorld[3][93] = 171;
    out.transformMaskMappingLocalToWorld[3][94] = 167;
    out.transformMaskMappingLocalToWorld[3][95] = 175;
    out.transformMaskMappingLocalToWorld[3][96] = 96;
    out.transformMaskMappingLocalToWorld[3][97] = 104;
    out.transformMaskMappingLocalToWorld[3][98] = 100;
    out.transformMaskMappingLocalToWorld[3][99] = 108;
    out.transformMaskMappingLocalToWorld[3][100] = 98;
    out.transformMaskMappingLocalToWorld[3][101] = 106;
    out.transformMaskMappingLocalToWorld[3][102] = 102;
    out.transformMaskMappingLocalToWorld[3][103] = 110;
    out.transformMaskMappingLocalToWorld[3][104] = 97;
    out.transformMaskMappingLocalToWorld[3][105] = 105;
    out.transformMaskMappingLocalToWorld[3][106] = 101;
    out.transformMaskMappingLocalToWorld[3][107] = 109;
    out.transformMaskMappingLocalToWorld[3][108] = 99;
    out.transformMaskMappingLocalToWorld[3][109] = 107;
    out.transformMaskMappingLocalToWorld[3][110] = 103;
    out.transformMaskMappingLocalToWorld[3][111] = 111;
    out.transformMaskMappingLocalToWorld[3][112] = 224;
    out.transformMaskMappingLocalToWorld[3][113] = 232;
    out.transformMaskMappingLocalToWorld[3][114] = 228;
    out.transformMaskMappingLocalToWorld[3][115] = 236;
    out.transformMaskMappingLocalToWorld[3][116] = 226;
    out.transformMaskMappingLocalToWorld[3][117] = 234;
    out.transformMaskMappingLocalToWorld[3][118] = 230;
    out.transformMaskMappingLocalToWorld[3][119] = 238;
    out.transformMaskMappingLocalToWorld[3][120] = 225;
    out.transformMaskMappingLocalToWorld[3][121] = 233;
    out.transformMaskMappingLocalToWorld[3][122] = 229;
    out.transformMaskMappingLocalToWorld[3][123] = 237;
    out.transformMaskMappingLocalToWorld[3][124] = 227;
    out.transformMaskMappingLocalToWorld[3][125] = 235;
    out.transformMaskMappingLocalToWorld[3][126] = 231;
    out.transformMaskMappingLocalToWorld[3][127] = 239;
    out.transformMaskMappingLocalToWorld[3][128] = 16;
    out.transformMaskMappingLocalToWorld[3][129] = 24;
    out.transformMaskMappingLocalToWorld[3][130] = 20;
    out.transformMaskMappingLocalToWorld[3][131] = 28;
    out.transformMaskMappingLocalToWorld[3][132] = 18;
    out.transformMaskMappingLocalToWorld[3][133] = 26;
    out.transformMaskMappingLocalToWorld[3][134] = 22;
    out.transformMaskMappingLocalToWorld[3][135] = 30;
    out.transformMaskMappingLocalToWorld[3][136] = 17;
    out.transformMaskMappingLocalToWorld[3][137] = 25;
    out.transformMaskMappingLocalToWorld[3][138] = 21;
    out.transformMaskMappingLocalToWorld[3][139] = 29;
    out.transformMaskMappingLocalToWorld[3][140] = 19;
    out.transformMaskMappingLocalToWorld[3][141] = 27;
    out.transformMaskMappingLocalToWorld[3][142] = 23;
    out.transformMaskMappingLocalToWorld[3][143] = 31;
    out.transformMaskMappingLocalToWorld[3][144] = 144;
    out.transformMaskMappingLocalToWorld[3][145] = 152;
    out.transformMaskMappingLocalToWorld[3][146] = 148;
    out.transformMaskMappingLocalToWorld[3][147] = 156;
    out.transformMaskMappingLocalToWorld[3][148] = 146;
    out.transformMaskMappingLocalToWorld[3][149] = 154;
    out.transformMaskMappingLocalToWorld[3][150] = 150;
    out.transformMaskMappingLocalToWorld[3][151] = 158;
    out.transformMaskMappingLocalToWorld[3][152] = 145;
    out.transformMaskMappingLocalToWorld[3][153] = 153;
    out.transformMaskMappingLocalToWorld[3][154] = 149;
    out.transformMaskMappingLocalToWorld[3][155] = 157;
    out.transformMaskMappingLocalToWorld[3][156] = 147;
    out.transformMaskMappingLocalToWorld[3][157] = 155;
    out.transformMaskMappingLocalToWorld[3][158] = 151;
    out.transformMaskMappingLocalToWorld[3][159] = 159;
    out.transformMaskMappingLocalToWorld[3][160] = 80;
    out.transformMaskMappingLocalToWorld[3][161] = 88;
    out.transformMaskMappingLocalToWorld[3][162] = 84;
    out.transformMaskMappingLocalToWorld[3][163] = 92;
    out.transformMaskMappingLocalToWorld[3][164] = 82;
    out.transformMaskMappingLocalToWorld[3][165] = 90;
    out.transformMaskMappingLocalToWorld[3][166] = 86;
    out.transformMaskMappingLocalToWorld[3][167] = 94;
    out.transformMaskMappingLocalToWorld[3][168] = 81;
    out.transformMaskMappingLocalToWorld[3][169] = 89;
    out.transformMaskMappingLocalToWorld[3][170] = 85;
    out.transformMaskMappingLocalToWorld[3][171] = 93;
    out.transformMaskMappingLocalToWorld[3][172] = 83;
    out.transformMaskMappingLocalToWorld[3][173] = 91;
    out.transformMaskMappingLocalToWorld[3][174] = 87;
    out.transformMaskMappingLocalToWorld[3][175] = 95;
    out.transformMaskMappingLocalToWorld[3][176] = 208;
    out.transformMaskMappingLocalToWorld[3][177] = 216;
    out.transformMaskMappingLocalToWorld[3][178] = 212;
    out.transformMaskMappingLocalToWorld[3][179] = 220;
    out.transformMaskMappingLocalToWorld[3][180] = 210;
    out.transformMaskMappingLocalToWorld[3][181] = 218;
    out.transformMaskMappingLocalToWorld[3][182] = 214;
    out.transformMaskMappingLocalToWorld[3][183] = 222;
    out.transformMaskMappingLocalToWorld[3][184] = 209;
    out.transformMaskMappingLocalToWorld[3][185] = 217;
    out.transformMaskMappingLocalToWorld[3][186] = 213;
    out.transformMaskMappingLocalToWorld[3][187] = 221;
    out.transformMaskMappingLocalToWorld[3][188] = 211;
    out.transformMaskMappingLocalToWorld[3][189] = 219;
    out.transformMaskMappingLocalToWorld[3][190] = 215;
    out.transformMaskMappingLocalToWorld[3][191] = 223;
    out.transformMaskMappingLocalToWorld[3][192] = 48;
    out.transformMaskMappingLocalToWorld[3][193] = 56;
    out.transformMaskMappingLocalToWorld[3][194] = 52;
    out.transformMaskMappingLocalToWorld[3][195] = 60;
    out.transformMaskMappingLocalToWorld[3][196] = 50;
    out.transformMaskMappingLocalToWorld[3][197] = 58;
    out.transformMaskMappingLocalToWorld[3][198] = 54;
    out.transformMaskMappingLocalToWorld[3][199] = 62;
    out.transformMaskMappingLocalToWorld[3][200] = 49;
    out.transformMaskMappingLocalToWorld[3][201] = 57;
    out.transformMaskMappingLocalToWorld[3][202] = 53;
    out.transformMaskMappingLocalToWorld[3][203] = 61;
    out.transformMaskMappingLocalToWorld[3][204] = 51;
    out.transformMaskMappingLocalToWorld[3][205] = 59;
    out.transformMaskMappingLocalToWorld[3][206] = 55;
    out.transformMaskMappingLocalToWorld[3][207] = 63;
    out.transformMaskMappingLocalToWorld[3][208] = 176;
    out.transformMaskMappingLocalToWorld[3][209] = 184;
    out.transformMaskMappingLocalToWorld[3][210] = 180;
    out.transformMaskMappingLocalToWorld[3][211] = 188;
    out.transformMaskMappingLocalToWorld[3][212] = 178;
    out.transformMaskMappingLocalToWorld[3][213] = 186;
    out.transformMaskMappingLocalToWorld[3][214] = 182;
    out.transformMaskMappingLocalToWorld[3][215] = 190;
    out.transformMaskMappingLocalToWorld[3][216] = 177;
    out.transformMaskMappingLocalToWorld[3][217] = 185;
    out.transformMaskMappingLocalToWorld[3][218] = 181;
    out.transformMaskMappingLocalToWorld[3][219] = 189;
    out.transformMaskMappingLocalToWorld[3][220] = 179;
    out.transformMaskMappingLocalToWorld[3][221] = 187;
    out.transformMaskMappingLocalToWorld[3][222] = 183;
    out.transformMaskMappingLocalToWorld[3][223] = 191;
    out.transformMaskMappingLocalToWorld[3][224] = 112;
    out.transformMaskMappingLocalToWorld[3][225] = 120;
    out.transformMaskMappingLocalToWorld[3][226] = 116;
    out.transformMaskMappingLocalToWorld[3][227] = 124;
    out.transformMaskMappingLocalToWorld[3][228] = 114;
    out.transformMaskMappingLocalToWorld[3][229] = 122;
    out.transformMaskMappingLocalToWorld[3][230] = 118;
    out.transformMaskMappingLocalToWorld[3][231] = 126;
    out.transformMaskMappingLocalToWorld[3][232] = 113;
    out.transformMaskMappingLocalToWorld[3][233] = 121;
    out.transformMaskMappingLocalToWorld[3][234] = 117;
    out.transformMaskMappingLocalToWorld[3][235] = 125;
    out.transformMaskMappingLocalToWorld[3][236] = 115;
    out.transformMaskMappingLocalToWorld[3][237] = 123;
    out.transformMaskMappingLocalToWorld[3][238] = 119;
    out.transformMaskMappingLocalToWorld[3][239] = 127;
    out.transformMaskMappingLocalToWorld[3][240] = 240;
    out.transformMaskMappingLocalToWorld[3][241] = 248;
    out.transformMaskMappingLocalToWorld[3][242] = 244;
    out.transformMaskMappingLocalToWorld[3][243] = 252;
    out.transformMaskMappingLocalToWorld[3][244] = 242;
    out.transformMaskMappingLocalToWorld[3][245] = 250;
    out.transformMaskMappingLocalToWorld[3][246] = 246;
    out.transformMaskMappingLocalToWorld[3][247] = 254;
    out.transformMaskMappingLocalToWorld[3][248] = 241;
    out.transformMaskMappingLocalToWorld[3][249] = 249;
    out.transformMaskMappingLocalToWorld[3][250] = 245;
    out.transformMaskMappingLocalToWorld[3][251] = 253;
    out.transformMaskMappingLocalToWorld[3][252] = 243;
    out.transformMaskMappingLocalToWorld[3][253] = 251;
    out.transformMaskMappingLocalToWorld[3][254] = 247;
    out.transformMaskMappingLocalToWorld[3][255] = 255;
    out.transformMaskMappingLocalToWorld[4][0] = 0;
    out.transformMaskMappingLocalToWorld[4][1] = 16;
    out.transformMaskMappingLocalToWorld[4][2] = 32;
    out.transformMaskMappingLocalToWorld[4][3] = 48;
    out.transformMaskMappingLocalToWorld[4][4] = 64;
    out.transformMaskMappingLocalToWorld[4][5] = 80;
    out.transformMaskMappingLocalToWorld[4][6] = 96;
    out.transformMaskMappingLocalToWorld[4][7] = 112;
    out.transformMaskMappingLocalToWorld[4][8] = 128;
    out.transformMaskMappingLocalToWorld[4][9] = 144;
    out.transformMaskMappingLocalToWorld[4][10] = 160;
    out.transformMaskMappingLocalToWorld[4][11] = 176;
    out.transformMaskMappingLocalToWorld[4][12] = 192;
    out.transformMaskMappingLocalToWorld[4][13] = 208;
    out.transformMaskMappingLocalToWorld[4][14] = 224;
    out.transformMaskMappingLocalToWorld[4][15] = 240;
    out.transformMaskMappingLocalToWorld[4][16] = 1;
    out.transformMaskMappingLocalToWorld[4][17] = 17;
    out.transformMaskMappingLocalToWorld[4][18] = 33;
    out.transformMaskMappingLocalToWorld[4][19] = 49;
    out.transformMaskMappingLocalToWorld[4][20] = 65;
    out.transformMaskMappingLocalToWorld[4][21] = 81;
    out.transformMaskMappingLocalToWorld[4][22] = 97;
    out.transformMaskMappingLocalToWorld[4][23] = 113;
    out.transformMaskMappingLocalToWorld[4][24] = 129;
    out.transformMaskMappingLocalToWorld[4][25] = 145;
    out.transformMaskMappingLocalToWorld[4][26] = 161;
    out.transformMaskMappingLocalToWorld[4][27] = 177;
    out.transformMaskMappingLocalToWorld[4][28] = 193;
    out.transformMaskMappingLocalToWorld[4][29] = 209;
    out.transformMaskMappingLocalToWorld[4][30] = 225;
    out.transformMaskMappingLocalToWorld[4][31] = 241;
    out.transformMaskMappingLocalToWorld[4][32] = 2;
    out.transformMaskMappingLocalToWorld[4][33] = 18;
    out.transformMaskMappingLocalToWorld[4][34] = 34;
    out.transformMaskMappingLocalToWorld[4][35] = 50;
    out.transformMaskMappingLocalToWorld[4][36] = 66;
    out.transformMaskMappingLocalToWorld[4][37] = 82;
    out.transformMaskMappingLocalToWorld[4][38] = 98;
    out.transformMaskMappingLocalToWorld[4][39] = 114;
    out.transformMaskMappingLocalToWorld[4][40] = 130;
    out.transformMaskMappingLocalToWorld[4][41] = 146;
    out.transformMaskMappingLocalToWorld[4][42] = 162;
    out.transformMaskMappingLocalToWorld[4][43] = 178;
    out.transformMaskMappingLocalToWorld[4][44] = 194;
    out.transformMaskMappingLocalToWorld[4][45] = 210;
    out.transformMaskMappingLocalToWorld[4][46] = 226;
    out.transformMaskMappingLocalToWorld[4][47] = 242;
    out.transformMaskMappingLocalToWorld[4][48] = 3;
    out.transformMaskMappingLocalToWorld[4][49] = 19;
    out.transformMaskMappingLocalToWorld[4][50] = 35;
    out.transformMaskMappingLocalToWorld[4][51] = 51;
    out.transformMaskMappingLocalToWorld[4][52] = 67;
    out.transformMaskMappingLocalToWorld[4][53] = 83;
    out.transformMaskMappingLocalToWorld[4][54] = 99;
    out.transformMaskMappingLocalToWorld[4][55] = 115;
    out.transformMaskMappingLocalToWorld[4][56] = 131;
    out.transformMaskMappingLocalToWorld[4][57] = 147;
    out.transformMaskMappingLocalToWorld[4][58] = 163;
    out.transformMaskMappingLocalToWorld[4][59] = 179;
    out.transformMaskMappingLocalToWorld[4][60] = 195;
    out.transformMaskMappingLocalToWorld[4][61] = 211;
    out.transformMaskMappingLocalToWorld[4][62] = 227;
    out.transformMaskMappingLocalToWorld[4][63] = 243;
    out.transformMaskMappingLocalToWorld[4][64] = 4;
    out.transformMaskMappingLocalToWorld[4][65] = 20;
    out.transformMaskMappingLocalToWorld[4][66] = 36;
    out.transformMaskMappingLocalToWorld[4][67] = 52;
    out.transformMaskMappingLocalToWorld[4][68] = 68;
    out.transformMaskMappingLocalToWorld[4][69] = 84;
    out.transformMaskMappingLocalToWorld[4][70] = 100;
    out.transformMaskMappingLocalToWorld[4][71] = 116;
    out.transformMaskMappingLocalToWorld[4][72] = 132;
    out.transformMaskMappingLocalToWorld[4][73] = 148;
    out.transformMaskMappingLocalToWorld[4][74] = 164;
    out.transformMaskMappingLocalToWorld[4][75] = 180;
    out.transformMaskMappingLocalToWorld[4][76] = 196;
    out.transformMaskMappingLocalToWorld[4][77] = 212;
    out.transformMaskMappingLocalToWorld[4][78] = 228;
    out.transformMaskMappingLocalToWorld[4][79] = 244;
    out.transformMaskMappingLocalToWorld[4][80] = 5;
    out.transformMaskMappingLocalToWorld[4][81] = 21;
    out.transformMaskMappingLocalToWorld[4][82] = 37;
    out.transformMaskMappingLocalToWorld[4][83] = 53;
    out.transformMaskMappingLocalToWorld[4][84] = 69;
    out.transformMaskMappingLocalToWorld[4][85] = 85;
    out.transformMaskMappingLocalToWorld[4][86] = 101;
    out.transformMaskMappingLocalToWorld[4][87] = 117;
    out.transformMaskMappingLocalToWorld[4][88] = 133;
    out.transformMaskMappingLocalToWorld[4][89] = 149;
    out.transformMaskMappingLocalToWorld[4][90] = 165;
    out.transformMaskMappingLocalToWorld[4][91] = 181;
    out.transformMaskMappingLocalToWorld[4][92] = 197;
    out.transformMaskMappingLocalToWorld[4][93] = 213;
    out.transformMaskMappingLocalToWorld[4][94] = 229;
    out.transformMaskMappingLocalToWorld[4][95] = 245;
    out.transformMaskMappingLocalToWorld[4][96] = 6;
    out.transformMaskMappingLocalToWorld[4][97] = 22;
    out.transformMaskMappingLocalToWorld[4][98] = 38;
    out.transformMaskMappingLocalToWorld[4][99] = 54;
    out.transformMaskMappingLocalToWorld[4][100] = 70;
    out.transformMaskMappingLocalToWorld[4][101] = 86;
    out.transformMaskMappingLocalToWorld[4][102] = 102;
    out.transformMaskMappingLocalToWorld[4][103] = 118;
    out.transformMaskMappingLocalToWorld[4][104] = 134;
    out.transformMaskMappingLocalToWorld[4][105] = 150;
    out.transformMaskMappingLocalToWorld[4][106] = 166;
    out.transformMaskMappingLocalToWorld[4][107] = 182;
    out.transformMaskMappingLocalToWorld[4][108] = 198;
    out.transformMaskMappingLocalToWorld[4][109] = 214;
    out.transformMaskMappingLocalToWorld[4][110] = 230;
    out.transformMaskMappingLocalToWorld[4][111] = 246;
    out.transformMaskMappingLocalToWorld[4][112] = 7;
    out.transformMaskMappingLocalToWorld[4][113] = 23;
    out.transformMaskMappingLocalToWorld[4][114] = 39;
    out.transformMaskMappingLocalToWorld[4][115] = 55;
    out.transformMaskMappingLocalToWorld[4][116] = 71;
    out.transformMaskMappingLocalToWorld[4][117] = 87;
    out.transformMaskMappingLocalToWorld[4][118] = 103;
    out.transformMaskMappingLocalToWorld[4][119] = 119;
    out.transformMaskMappingLocalToWorld[4][120] = 135;
    out.transformMaskMappingLocalToWorld[4][121] = 151;
    out.transformMaskMappingLocalToWorld[4][122] = 167;
    out.transformMaskMappingLocalToWorld[4][123] = 183;
    out.transformMaskMappingLocalToWorld[4][124] = 199;
    out.transformMaskMappingLocalToWorld[4][125] = 215;
    out.transformMaskMappingLocalToWorld[4][126] = 231;
    out.transformMaskMappingLocalToWorld[4][127] = 247;
    out.transformMaskMappingLocalToWorld[4][128] = 8;
    out.transformMaskMappingLocalToWorld[4][129] = 24;
    out.transformMaskMappingLocalToWorld[4][130] = 40;
    out.transformMaskMappingLocalToWorld[4][131] = 56;
    out.transformMaskMappingLocalToWorld[4][132] = 72;
    out.transformMaskMappingLocalToWorld[4][133] = 88;
    out.transformMaskMappingLocalToWorld[4][134] = 104;
    out.transformMaskMappingLocalToWorld[4][135] = 120;
    out.transformMaskMappingLocalToWorld[4][136] = 136;
    out.transformMaskMappingLocalToWorld[4][137] = 152;
    out.transformMaskMappingLocalToWorld[4][138] = 168;
    out.transformMaskMappingLocalToWorld[4][139] = 184;
    out.transformMaskMappingLocalToWorld[4][140] = 200;
    out.transformMaskMappingLocalToWorld[4][141] = 216;
    out.transformMaskMappingLocalToWorld[4][142] = 232;
    out.transformMaskMappingLocalToWorld[4][143] = 248;
    out.transformMaskMappingLocalToWorld[4][144] = 9;
    out.transformMaskMappingLocalToWorld[4][145] = 25;
    out.transformMaskMappingLocalToWorld[4][146] = 41;
    out.transformMaskMappingLocalToWorld[4][147] = 57;
    out.transformMaskMappingLocalToWorld[4][148] = 73;
    out.transformMaskMappingLocalToWorld[4][149] = 89;
    out.transformMaskMappingLocalToWorld[4][150] = 105;
    out.transformMaskMappingLocalToWorld[4][151] = 121;
    out.transformMaskMappingLocalToWorld[4][152] = 137;
    out.transformMaskMappingLocalToWorld[4][153] = 153;
    out.transformMaskMappingLocalToWorld[4][154] = 169;
    out.transformMaskMappingLocalToWorld[4][155] = 185;
    out.transformMaskMappingLocalToWorld[4][156] = 201;
    out.transformMaskMappingLocalToWorld[4][157] = 217;
    out.transformMaskMappingLocalToWorld[4][158] = 233;
    out.transformMaskMappingLocalToWorld[4][159] = 249;
    out.transformMaskMappingLocalToWorld[4][160] = 10;
    out.transformMaskMappingLocalToWorld[4][161] = 26;
    out.transformMaskMappingLocalToWorld[4][162] = 42;
    out.transformMaskMappingLocalToWorld[4][163] = 58;
    out.transformMaskMappingLocalToWorld[4][164] = 74;
    out.transformMaskMappingLocalToWorld[4][165] = 90;
    out.transformMaskMappingLocalToWorld[4][166] = 106;
    out.transformMaskMappingLocalToWorld[4][167] = 122;
    out.transformMaskMappingLocalToWorld[4][168] = 138;
    out.transformMaskMappingLocalToWorld[4][169] = 154;
    out.transformMaskMappingLocalToWorld[4][170] = 170;
    out.transformMaskMappingLocalToWorld[4][171] = 186;
    out.transformMaskMappingLocalToWorld[4][172] = 202;
    out.transformMaskMappingLocalToWorld[4][173] = 218;
    out.transformMaskMappingLocalToWorld[4][174] = 234;
    out.transformMaskMappingLocalToWorld[4][175] = 250;
    out.transformMaskMappingLocalToWorld[4][176] = 11;
    out.transformMaskMappingLocalToWorld[4][177] = 27;
    out.transformMaskMappingLocalToWorld[4][178] = 43;
    out.transformMaskMappingLocalToWorld[4][179] = 59;
    out.transformMaskMappingLocalToWorld[4][180] = 75;
    out.transformMaskMappingLocalToWorld[4][181] = 91;
    out.transformMaskMappingLocalToWorld[4][182] = 107;
    out.transformMaskMappingLocalToWorld[4][183] = 123;
    out.transformMaskMappingLocalToWorld[4][184] = 139;
    out.transformMaskMappingLocalToWorld[4][185] = 155;
    out.transformMaskMappingLocalToWorld[4][186] = 171;
    out.transformMaskMappingLocalToWorld[4][187] = 187;
    out.transformMaskMappingLocalToWorld[4][188] = 203;
    out.transformMaskMappingLocalToWorld[4][189] = 219;
    out.transformMaskMappingLocalToWorld[4][190] = 235;
    out.transformMaskMappingLocalToWorld[4][191] = 251;
    out.transformMaskMappingLocalToWorld[4][192] = 12;
    out.transformMaskMappingLocalToWorld[4][193] = 28;
    out.transformMaskMappingLocalToWorld[4][194] = 44;
    out.transformMaskMappingLocalToWorld[4][195] = 60;
    out.transformMaskMappingLocalToWorld[4][196] = 76;
    out.transformMaskMappingLocalToWorld[4][197] = 92;
    out.transformMaskMappingLocalToWorld[4][198] = 108;
    out.transformMaskMappingLocalToWorld[4][199] = 124;
    out.transformMaskMappingLocalToWorld[4][200] = 140;
    out.transformMaskMappingLocalToWorld[4][201] = 156;
    out.transformMaskMappingLocalToWorld[4][202] = 172;
    out.transformMaskMappingLocalToWorld[4][203] = 188;
    out.transformMaskMappingLocalToWorld[4][204] = 204;
    out.transformMaskMappingLocalToWorld[4][205] = 220;
    out.transformMaskMappingLocalToWorld[4][206] = 236;
    out.transformMaskMappingLocalToWorld[4][207] = 252;
    out.transformMaskMappingLocalToWorld[4][208] = 13;
    out.transformMaskMappingLocalToWorld[4][209] = 29;
    out.transformMaskMappingLocalToWorld[4][210] = 45;
    out.transformMaskMappingLocalToWorld[4][211] = 61;
    out.transformMaskMappingLocalToWorld[4][212] = 77;
    out.transformMaskMappingLocalToWorld[4][213] = 93;
    out.transformMaskMappingLocalToWorld[4][214] = 109;
    out.transformMaskMappingLocalToWorld[4][215] = 125;
    out.transformMaskMappingLocalToWorld[4][216] = 141;
    out.transformMaskMappingLocalToWorld[4][217] = 157;
    out.transformMaskMappingLocalToWorld[4][218] = 173;
    out.transformMaskMappingLocalToWorld[4][219] = 189;
    out.transformMaskMappingLocalToWorld[4][220] = 205;
    out.transformMaskMappingLocalToWorld[4][221] = 221;
    out.transformMaskMappingLocalToWorld[4][222] = 237;
    out.transformMaskMappingLocalToWorld[4][223] = 253;
    out.transformMaskMappingLocalToWorld[4][224] = 14;
    out.transformMaskMappingLocalToWorld[4][225] = 30;
    out.transformMaskMappingLocalToWorld[4][226] = 46;
    out.transformMaskMappingLocalToWorld[4][227] = 62;
    out.transformMaskMappingLocalToWorld[4][228] = 78;
    out.transformMaskMappingLocalToWorld[4][229] = 94;
    out.transformMaskMappingLocalToWorld[4][230] = 110;
    out.transformMaskMappingLocalToWorld[4][231] = 126;
    out.transformMaskMappingLocalToWorld[4][232] = 142;
    out.transformMaskMappingLocalToWorld[4][233] = 158;
    out.transformMaskMappingLocalToWorld[4][234] = 174;
    out.transformMaskMappingLocalToWorld[4][235] = 190;
    out.transformMaskMappingLocalToWorld[4][236] = 206;
    out.transformMaskMappingLocalToWorld[4][237] = 222;
    out.transformMaskMappingLocalToWorld[4][238] = 238;
    out.transformMaskMappingLocalToWorld[4][239] = 254;
    out.transformMaskMappingLocalToWorld[4][240] = 15;
    out.transformMaskMappingLocalToWorld[4][241] = 31;
    out.transformMaskMappingLocalToWorld[4][242] = 47;
    out.transformMaskMappingLocalToWorld[4][243] = 63;
    out.transformMaskMappingLocalToWorld[4][244] = 79;
    out.transformMaskMappingLocalToWorld[4][245] = 95;
    out.transformMaskMappingLocalToWorld[4][246] = 111;
    out.transformMaskMappingLocalToWorld[4][247] = 127;
    out.transformMaskMappingLocalToWorld[4][248] = 143;
    out.transformMaskMappingLocalToWorld[4][249] = 159;
    out.transformMaskMappingLocalToWorld[4][250] = 175;
    out.transformMaskMappingLocalToWorld[4][251] = 191;
    out.transformMaskMappingLocalToWorld[4][252] = 207;
    out.transformMaskMappingLocalToWorld[4][253] = 223;
    out.transformMaskMappingLocalToWorld[4][254] = 239;
    out.transformMaskMappingLocalToWorld[4][255] = 255;
    out.transformMaskMappingLocalToWorld[5][0] = 0;
    out.transformMaskMappingLocalToWorld[5][1] = 32;
    out.transformMaskMappingLocalToWorld[5][2] = 16;
    out.transformMaskMappingLocalToWorld[5][3] = 48;
    out.transformMaskMappingLocalToWorld[5][4] = 128;
    out.transformMaskMappingLocalToWorld[5][5] = 160;
    out.transformMaskMappingLocalToWorld[5][6] = 144;
    out.transformMaskMappingLocalToWorld[5][7] = 176;
    out.transformMaskMappingLocalToWorld[5][8] = 64;
    out.transformMaskMappingLocalToWorld[5][9] = 96;
    out.transformMaskMappingLocalToWorld[5][10] = 80;
    out.transformMaskMappingLocalToWorld[5][11] = 112;
    out.transformMaskMappingLocalToWorld[5][12] = 192;
    out.transformMaskMappingLocalToWorld[5][13] = 224;
    out.transformMaskMappingLocalToWorld[5][14] = 208;
    out.transformMaskMappingLocalToWorld[5][15] = 240;
    out.transformMaskMappingLocalToWorld[5][16] = 2;
    out.transformMaskMappingLocalToWorld[5][17] = 34;
    out.transformMaskMappingLocalToWorld[5][18] = 18;
    out.transformMaskMappingLocalToWorld[5][19] = 50;
    out.transformMaskMappingLocalToWorld[5][20] = 130;
    out.transformMaskMappingLocalToWorld[5][21] = 162;
    out.transformMaskMappingLocalToWorld[5][22] = 146;
    out.transformMaskMappingLocalToWorld[5][23] = 178;
    out.transformMaskMappingLocalToWorld[5][24] = 66;
    out.transformMaskMappingLocalToWorld[5][25] = 98;
    out.transformMaskMappingLocalToWorld[5][26] = 82;
    out.transformMaskMappingLocalToWorld[5][27] = 114;
    out.transformMaskMappingLocalToWorld[5][28] = 194;
    out.transformMaskMappingLocalToWorld[5][29] = 226;
    out.transformMaskMappingLocalToWorld[5][30] = 210;
    out.transformMaskMappingLocalToWorld[5][31] = 242;
    out.transformMaskMappingLocalToWorld[5][32] = 1;
    out.transformMaskMappingLocalToWorld[5][33] = 33;
    out.transformMaskMappingLocalToWorld[5][34] = 17;
    out.transformMaskMappingLocalToWorld[5][35] = 49;
    out.transformMaskMappingLocalToWorld[5][36] = 129;
    out.transformMaskMappingLocalToWorld[5][37] = 161;
    out.transformMaskMappingLocalToWorld[5][38] = 145;
    out.transformMaskMappingLocalToWorld[5][39] = 177;
    out.transformMaskMappingLocalToWorld[5][40] = 65;
    out.transformMaskMappingLocalToWorld[5][41] = 97;
    out.transformMaskMappingLocalToWorld[5][42] = 81;
    out.transformMaskMappingLocalToWorld[5][43] = 113;
    out.transformMaskMappingLocalToWorld[5][44] = 193;
    out.transformMaskMappingLocalToWorld[5][45] = 225;
    out.transformMaskMappingLocalToWorld[5][46] = 209;
    out.transformMaskMappingLocalToWorld[5][47] = 241;
    out.transformMaskMappingLocalToWorld[5][48] = 3;
    out.transformMaskMappingLocalToWorld[5][49] = 35;
    out.transformMaskMappingLocalToWorld[5][50] = 19;
    out.transformMaskMappingLocalToWorld[5][51] = 51;
    out.transformMaskMappingLocalToWorld[5][52] = 131;
    out.transformMaskMappingLocalToWorld[5][53] = 163;
    out.transformMaskMappingLocalToWorld[5][54] = 147;
    out.transformMaskMappingLocalToWorld[5][55] = 179;
    out.transformMaskMappingLocalToWorld[5][56] = 67;
    out.transformMaskMappingLocalToWorld[5][57] = 99;
    out.transformMaskMappingLocalToWorld[5][58] = 83;
    out.transformMaskMappingLocalToWorld[5][59] = 115;
    out.transformMaskMappingLocalToWorld[5][60] = 195;
    out.transformMaskMappingLocalToWorld[5][61] = 227;
    out.transformMaskMappingLocalToWorld[5][62] = 211;
    out.transformMaskMappingLocalToWorld[5][63] = 243;
    out.transformMaskMappingLocalToWorld[5][64] = 8;
    out.transformMaskMappingLocalToWorld[5][65] = 40;
    out.transformMaskMappingLocalToWorld[5][66] = 24;
    out.transformMaskMappingLocalToWorld[5][67] = 56;
    out.transformMaskMappingLocalToWorld[5][68] = 136;
    out.transformMaskMappingLocalToWorld[5][69] = 168;
    out.transformMaskMappingLocalToWorld[5][70] = 152;
    out.transformMaskMappingLocalToWorld[5][71] = 184;
    out.transformMaskMappingLocalToWorld[5][72] = 72;
    out.transformMaskMappingLocalToWorld[5][73] = 104;
    out.transformMaskMappingLocalToWorld[5][74] = 88;
    out.transformMaskMappingLocalToWorld[5][75] = 120;
    out.transformMaskMappingLocalToWorld[5][76] = 200;
    out.transformMaskMappingLocalToWorld[5][77] = 232;
    out.transformMaskMappingLocalToWorld[5][78] = 216;
    out.transformMaskMappingLocalToWorld[5][79] = 248;
    out.transformMaskMappingLocalToWorld[5][80] = 10;
    out.transformMaskMappingLocalToWorld[5][81] = 42;
    out.transformMaskMappingLocalToWorld[5][82] = 26;
    out.transformMaskMappingLocalToWorld[5][83] = 58;
    out.transformMaskMappingLocalToWorld[5][84] = 138;
    out.transformMaskMappingLocalToWorld[5][85] = 170;
    out.transformMaskMappingLocalToWorld[5][86] = 154;
    out.transformMaskMappingLocalToWorld[5][87] = 186;
    out.transformMaskMappingLocalToWorld[5][88] = 74;
    out.transformMaskMappingLocalToWorld[5][89] = 106;
    out.transformMaskMappingLocalToWorld[5][90] = 90;
    out.transformMaskMappingLocalToWorld[5][91] = 122;
    out.transformMaskMappingLocalToWorld[5][92] = 202;
    out.transformMaskMappingLocalToWorld[5][93] = 234;
    out.transformMaskMappingLocalToWorld[5][94] = 218;
    out.transformMaskMappingLocalToWorld[5][95] = 250;
    out.transformMaskMappingLocalToWorld[5][96] = 9;
    out.transformMaskMappingLocalToWorld[5][97] = 41;
    out.transformMaskMappingLocalToWorld[5][98] = 25;
    out.transformMaskMappingLocalToWorld[5][99] = 57;
    out.transformMaskMappingLocalToWorld[5][100] = 137;
    out.transformMaskMappingLocalToWorld[5][101] = 169;
    out.transformMaskMappingLocalToWorld[5][102] = 153;
    out.transformMaskMappingLocalToWorld[5][103] = 185;
    out.transformMaskMappingLocalToWorld[5][104] = 73;
    out.transformMaskMappingLocalToWorld[5][105] = 105;
    out.transformMaskMappingLocalToWorld[5][106] = 89;
    out.transformMaskMappingLocalToWorld[5][107] = 121;
    out.transformMaskMappingLocalToWorld[5][108] = 201;
    out.transformMaskMappingLocalToWorld[5][109] = 233;
    out.transformMaskMappingLocalToWorld[5][110] = 217;
    out.transformMaskMappingLocalToWorld[5][111] = 249;
    out.transformMaskMappingLocalToWorld[5][112] = 11;
    out.transformMaskMappingLocalToWorld[5][113] = 43;
    out.transformMaskMappingLocalToWorld[5][114] = 27;
    out.transformMaskMappingLocalToWorld[5][115] = 59;
    out.transformMaskMappingLocalToWorld[5][116] = 139;
    out.transformMaskMappingLocalToWorld[5][117] = 171;
    out.transformMaskMappingLocalToWorld[5][118] = 155;
    out.transformMaskMappingLocalToWorld[5][119] = 187;
    out.transformMaskMappingLocalToWorld[5][120] = 75;
    out.transformMaskMappingLocalToWorld[5][121] = 107;
    out.transformMaskMappingLocalToWorld[5][122] = 91;
    out.transformMaskMappingLocalToWorld[5][123] = 123;
    out.transformMaskMappingLocalToWorld[5][124] = 203;
    out.transformMaskMappingLocalToWorld[5][125] = 235;
    out.transformMaskMappingLocalToWorld[5][126] = 219;
    out.transformMaskMappingLocalToWorld[5][127] = 251;
    out.transformMaskMappingLocalToWorld[5][128] = 4;
    out.transformMaskMappingLocalToWorld[5][129] = 36;
    out.transformMaskMappingLocalToWorld[5][130] = 20;
    out.transformMaskMappingLocalToWorld[5][131] = 52;
    out.transformMaskMappingLocalToWorld[5][132] = 132;
    out.transformMaskMappingLocalToWorld[5][133] = 164;
    out.transformMaskMappingLocalToWorld[5][134] = 148;
    out.transformMaskMappingLocalToWorld[5][135] = 180;
    out.transformMaskMappingLocalToWorld[5][136] = 68;
    out.transformMaskMappingLocalToWorld[5][137] = 100;
    out.transformMaskMappingLocalToWorld[5][138] = 84;
    out.transformMaskMappingLocalToWorld[5][139] = 116;
    out.transformMaskMappingLocalToWorld[5][140] = 196;
    out.transformMaskMappingLocalToWorld[5][141] = 228;
    out.transformMaskMappingLocalToWorld[5][142] = 212;
    out.transformMaskMappingLocalToWorld[5][143] = 244;
    out.transformMaskMappingLocalToWorld[5][144] = 6;
    out.transformMaskMappingLocalToWorld[5][145] = 38;
    out.transformMaskMappingLocalToWorld[5][146] = 22;
    out.transformMaskMappingLocalToWorld[5][147] = 54;
    out.transformMaskMappingLocalToWorld[5][148] = 134;
    out.transformMaskMappingLocalToWorld[5][149] = 166;
    out.transformMaskMappingLocalToWorld[5][150] = 150;
    out.transformMaskMappingLocalToWorld[5][151] = 182;
    out.transformMaskMappingLocalToWorld[5][152] = 70;
    out.transformMaskMappingLocalToWorld[5][153] = 102;
    out.transformMaskMappingLocalToWorld[5][154] = 86;
    out.transformMaskMappingLocalToWorld[5][155] = 118;
    out.transformMaskMappingLocalToWorld[5][156] = 198;
    out.transformMaskMappingLocalToWorld[5][157] = 230;
    out.transformMaskMappingLocalToWorld[5][158] = 214;
    out.transformMaskMappingLocalToWorld[5][159] = 246;
    out.transformMaskMappingLocalToWorld[5][160] = 5;
    out.transformMaskMappingLocalToWorld[5][161] = 37;
    out.transformMaskMappingLocalToWorld[5][162] = 21;
    out.transformMaskMappingLocalToWorld[5][163] = 53;
    out.transformMaskMappingLocalToWorld[5][164] = 133;
    out.transformMaskMappingLocalToWorld[5][165] = 165;
    out.transformMaskMappingLocalToWorld[5][166] = 149;
    out.transformMaskMappingLocalToWorld[5][167] = 181;
    out.transformMaskMappingLocalToWorld[5][168] = 69;
    out.transformMaskMappingLocalToWorld[5][169] = 101;
    out.transformMaskMappingLocalToWorld[5][170] = 85;
    out.transformMaskMappingLocalToWorld[5][171] = 117;
    out.transformMaskMappingLocalToWorld[5][172] = 197;
    out.transformMaskMappingLocalToWorld[5][173] = 229;
    out.transformMaskMappingLocalToWorld[5][174] = 213;
    out.transformMaskMappingLocalToWorld[5][175] = 245;
    out.transformMaskMappingLocalToWorld[5][176] = 7;
    out.transformMaskMappingLocalToWorld[5][177] = 39;
    out.transformMaskMappingLocalToWorld[5][178] = 23;
    out.transformMaskMappingLocalToWorld[5][179] = 55;
    out.transformMaskMappingLocalToWorld[5][180] = 135;
    out.transformMaskMappingLocalToWorld[5][181] = 167;
    out.transformMaskMappingLocalToWorld[5][182] = 151;
    out.transformMaskMappingLocalToWorld[5][183] = 183;
    out.transformMaskMappingLocalToWorld[5][184] = 71;
    out.transformMaskMappingLocalToWorld[5][185] = 103;
    out.transformMaskMappingLocalToWorld[5][186] = 87;
    out.transformMaskMappingLocalToWorld[5][187] = 119;
    out.transformMaskMappingLocalToWorld[5][188] = 199;
    out.transformMaskMappingLocalToWorld[5][189] = 231;
    out.transformMaskMappingLocalToWorld[5][190] = 215;
    out.transformMaskMappingLocalToWorld[5][191] = 247;
    out.transformMaskMappingLocalToWorld[5][192] = 12;
    out.transformMaskMappingLocalToWorld[5][193] = 44;
    out.transformMaskMappingLocalToWorld[5][194] = 28;
    out.transformMaskMappingLocalToWorld[5][195] = 60;
    out.transformMaskMappingLocalToWorld[5][196] = 140;
    out.transformMaskMappingLocalToWorld[5][197] = 172;
    out.transformMaskMappingLocalToWorld[5][198] = 156;
    out.transformMaskMappingLocalToWorld[5][199] = 188;
    out.transformMaskMappingLocalToWorld[5][200] = 76;
    out.transformMaskMappingLocalToWorld[5][201] = 108;
    out.transformMaskMappingLocalToWorld[5][202] = 92;
    out.transformMaskMappingLocalToWorld[5][203] = 124;
    out.transformMaskMappingLocalToWorld[5][204] = 204;
    out.transformMaskMappingLocalToWorld[5][205] = 236;
    out.transformMaskMappingLocalToWorld[5][206] = 220;
    out.transformMaskMappingLocalToWorld[5][207] = 252;
    out.transformMaskMappingLocalToWorld[5][208] = 14;
    out.transformMaskMappingLocalToWorld[5][209] = 46;
    out.transformMaskMappingLocalToWorld[5][210] = 30;
    out.transformMaskMappingLocalToWorld[5][211] = 62;
    out.transformMaskMappingLocalToWorld[5][212] = 142;
    out.transformMaskMappingLocalToWorld[5][213] = 174;
    out.transformMaskMappingLocalToWorld[5][214] = 158;
    out.transformMaskMappingLocalToWorld[5][215] = 190;
    out.transformMaskMappingLocalToWorld[5][216] = 78;
    out.transformMaskMappingLocalToWorld[5][217] = 110;
    out.transformMaskMappingLocalToWorld[5][218] = 94;
    out.transformMaskMappingLocalToWorld[5][219] = 126;
    out.transformMaskMappingLocalToWorld[5][220] = 206;
    out.transformMaskMappingLocalToWorld[5][221] = 238;
    out.transformMaskMappingLocalToWorld[5][222] = 222;
    out.transformMaskMappingLocalToWorld[5][223] = 254;
    out.transformMaskMappingLocalToWorld[5][224] = 13;
    out.transformMaskMappingLocalToWorld[5][225] = 45;
    out.transformMaskMappingLocalToWorld[5][226] = 29;
    out.transformMaskMappingLocalToWorld[5][227] = 61;
    out.transformMaskMappingLocalToWorld[5][228] = 141;
    out.transformMaskMappingLocalToWorld[5][229] = 173;
    out.transformMaskMappingLocalToWorld[5][230] = 157;
    out.transformMaskMappingLocalToWorld[5][231] = 189;
    out.transformMaskMappingLocalToWorld[5][232] = 77;
    out.transformMaskMappingLocalToWorld[5][233] = 109;
    out.transformMaskMappingLocalToWorld[5][234] = 93;
    out.transformMaskMappingLocalToWorld[5][235] = 125;
    out.transformMaskMappingLocalToWorld[5][236] = 205;
    out.transformMaskMappingLocalToWorld[5][237] = 237;
    out.transformMaskMappingLocalToWorld[5][238] = 221;
    out.transformMaskMappingLocalToWorld[5][239] = 253;
    out.transformMaskMappingLocalToWorld[5][240] = 15;
    out.transformMaskMappingLocalToWorld[5][241] = 47;
    out.transformMaskMappingLocalToWorld[5][242] = 31;
    out.transformMaskMappingLocalToWorld[5][243] = 63;
    out.transformMaskMappingLocalToWorld[5][244] = 143;
    out.transformMaskMappingLocalToWorld[5][245] = 175;
    out.transformMaskMappingLocalToWorld[5][246] = 159;
    out.transformMaskMappingLocalToWorld[5][247] = 191;
    out.transformMaskMappingLocalToWorld[5][248] = 79;
    out.transformMaskMappingLocalToWorld[5][249] = 111;
    out.transformMaskMappingLocalToWorld[5][250] = 95;
    out.transformMaskMappingLocalToWorld[5][251] = 127;
    out.transformMaskMappingLocalToWorld[5][252] = 207;
    out.transformMaskMappingLocalToWorld[5][253] = 239;
    out.transformMaskMappingLocalToWorld[5][254] = 223;
    out.transformMaskMappingLocalToWorld[5][255] = 255;
    out.transformMaskMappingLocalToWorld[6][0] = 0;
    out.transformMaskMappingLocalToWorld[6][1] = 64;
    out.transformMaskMappingLocalToWorld[6][2] = 128;
    out.transformMaskMappingLocalToWorld[6][3] = 192;
    out.transformMaskMappingLocalToWorld[6][4] = 16;
    out.transformMaskMappingLocalToWorld[6][5] = 80;
    out.transformMaskMappingLocalToWorld[6][6] = 144;
    out.transformMaskMappingLocalToWorld[6][7] = 208;
    out.transformMaskMappingLocalToWorld[6][8] = 32;
    out.transformMaskMappingLocalToWorld[6][9] = 96;
    out.transformMaskMappingLocalToWorld[6][10] = 160;
    out.transformMaskMappingLocalToWorld[6][11] = 224;
    out.transformMaskMappingLocalToWorld[6][12] = 48;
    out.transformMaskMappingLocalToWorld[6][13] = 112;
    out.transformMaskMappingLocalToWorld[6][14] = 176;
    out.transformMaskMappingLocalToWorld[6][15] = 240;
    out.transformMaskMappingLocalToWorld[6][16] = 4;
    out.transformMaskMappingLocalToWorld[6][17] = 68;
    out.transformMaskMappingLocalToWorld[6][18] = 132;
    out.transformMaskMappingLocalToWorld[6][19] = 196;
    out.transformMaskMappingLocalToWorld[6][20] = 20;
    out.transformMaskMappingLocalToWorld[6][21] = 84;
    out.transformMaskMappingLocalToWorld[6][22] = 148;
    out.transformMaskMappingLocalToWorld[6][23] = 212;
    out.transformMaskMappingLocalToWorld[6][24] = 36;
    out.transformMaskMappingLocalToWorld[6][25] = 100;
    out.transformMaskMappingLocalToWorld[6][26] = 164;
    out.transformMaskMappingLocalToWorld[6][27] = 228;
    out.transformMaskMappingLocalToWorld[6][28] = 52;
    out.transformMaskMappingLocalToWorld[6][29] = 116;
    out.transformMaskMappingLocalToWorld[6][30] = 180;
    out.transformMaskMappingLocalToWorld[6][31] = 244;
    out.transformMaskMappingLocalToWorld[6][32] = 8;
    out.transformMaskMappingLocalToWorld[6][33] = 72;
    out.transformMaskMappingLocalToWorld[6][34] = 136;
    out.transformMaskMappingLocalToWorld[6][35] = 200;
    out.transformMaskMappingLocalToWorld[6][36] = 24;
    out.transformMaskMappingLocalToWorld[6][37] = 88;
    out.transformMaskMappingLocalToWorld[6][38] = 152;
    out.transformMaskMappingLocalToWorld[6][39] = 216;
    out.transformMaskMappingLocalToWorld[6][40] = 40;
    out.transformMaskMappingLocalToWorld[6][41] = 104;
    out.transformMaskMappingLocalToWorld[6][42] = 168;
    out.transformMaskMappingLocalToWorld[6][43] = 232;
    out.transformMaskMappingLocalToWorld[6][44] = 56;
    out.transformMaskMappingLocalToWorld[6][45] = 120;
    out.transformMaskMappingLocalToWorld[6][46] = 184;
    out.transformMaskMappingLocalToWorld[6][47] = 248;
    out.transformMaskMappingLocalToWorld[6][48] = 12;
    out.transformMaskMappingLocalToWorld[6][49] = 76;
    out.transformMaskMappingLocalToWorld[6][50] = 140;
    out.transformMaskMappingLocalToWorld[6][51] = 204;
    out.transformMaskMappingLocalToWorld[6][52] = 28;
    out.transformMaskMappingLocalToWorld[6][53] = 92;
    out.transformMaskMappingLocalToWorld[6][54] = 156;
    out.transformMaskMappingLocalToWorld[6][55] = 220;
    out.transformMaskMappingLocalToWorld[6][56] = 44;
    out.transformMaskMappingLocalToWorld[6][57] = 108;
    out.transformMaskMappingLocalToWorld[6][58] = 172;
    out.transformMaskMappingLocalToWorld[6][59] = 236;
    out.transformMaskMappingLocalToWorld[6][60] = 60;
    out.transformMaskMappingLocalToWorld[6][61] = 124;
    out.transformMaskMappingLocalToWorld[6][62] = 188;
    out.transformMaskMappingLocalToWorld[6][63] = 252;
    out.transformMaskMappingLocalToWorld[6][64] = 1;
    out.transformMaskMappingLocalToWorld[6][65] = 65;
    out.transformMaskMappingLocalToWorld[6][66] = 129;
    out.transformMaskMappingLocalToWorld[6][67] = 193;
    out.transformMaskMappingLocalToWorld[6][68] = 17;
    out.transformMaskMappingLocalToWorld[6][69] = 81;
    out.transformMaskMappingLocalToWorld[6][70] = 145;
    out.transformMaskMappingLocalToWorld[6][71] = 209;
    out.transformMaskMappingLocalToWorld[6][72] = 33;
    out.transformMaskMappingLocalToWorld[6][73] = 97;
    out.transformMaskMappingLocalToWorld[6][74] = 161;
    out.transformMaskMappingLocalToWorld[6][75] = 225;
    out.transformMaskMappingLocalToWorld[6][76] = 49;
    out.transformMaskMappingLocalToWorld[6][77] = 113;
    out.transformMaskMappingLocalToWorld[6][78] = 177;
    out.transformMaskMappingLocalToWorld[6][79] = 241;
    out.transformMaskMappingLocalToWorld[6][80] = 5;
    out.transformMaskMappingLocalToWorld[6][81] = 69;
    out.transformMaskMappingLocalToWorld[6][82] = 133;
    out.transformMaskMappingLocalToWorld[6][83] = 197;
    out.transformMaskMappingLocalToWorld[6][84] = 21;
    out.transformMaskMappingLocalToWorld[6][85] = 85;
    out.transformMaskMappingLocalToWorld[6][86] = 149;
    out.transformMaskMappingLocalToWorld[6][87] = 213;
    out.transformMaskMappingLocalToWorld[6][88] = 37;
    out.transformMaskMappingLocalToWorld[6][89] = 101;
    out.transformMaskMappingLocalToWorld[6][90] = 165;
    out.transformMaskMappingLocalToWorld[6][91] = 229;
    out.transformMaskMappingLocalToWorld[6][92] = 53;
    out.transformMaskMappingLocalToWorld[6][93] = 117;
    out.transformMaskMappingLocalToWorld[6][94] = 181;
    out.transformMaskMappingLocalToWorld[6][95] = 245;
    out.transformMaskMappingLocalToWorld[6][96] = 9;
    out.transformMaskMappingLocalToWorld[6][97] = 73;
    out.transformMaskMappingLocalToWorld[6][98] = 137;
    out.transformMaskMappingLocalToWorld[6][99] = 201;
    out.transformMaskMappingLocalToWorld[6][100] = 25;
    out.transformMaskMappingLocalToWorld[6][101] = 89;
    out.transformMaskMappingLocalToWorld[6][102] = 153;
    out.transformMaskMappingLocalToWorld[6][103] = 217;
    out.transformMaskMappingLocalToWorld[6][104] = 41;
    out.transformMaskMappingLocalToWorld[6][105] = 105;
    out.transformMaskMappingLocalToWorld[6][106] = 169;
    out.transformMaskMappingLocalToWorld[6][107] = 233;
    out.transformMaskMappingLocalToWorld[6][108] = 57;
    out.transformMaskMappingLocalToWorld[6][109] = 121;
    out.transformMaskMappingLocalToWorld[6][110] = 185;
    out.transformMaskMappingLocalToWorld[6][111] = 249;
    out.transformMaskMappingLocalToWorld[6][112] = 13;
    out.transformMaskMappingLocalToWorld[6][113] = 77;
    out.transformMaskMappingLocalToWorld[6][114] = 141;
    out.transformMaskMappingLocalToWorld[6][115] = 205;
    out.transformMaskMappingLocalToWorld[6][116] = 29;
    out.transformMaskMappingLocalToWorld[6][117] = 93;
    out.transformMaskMappingLocalToWorld[6][118] = 157;
    out.transformMaskMappingLocalToWorld[6][119] = 221;
    out.transformMaskMappingLocalToWorld[6][120] = 45;
    out.transformMaskMappingLocalToWorld[6][121] = 109;
    out.transformMaskMappingLocalToWorld[6][122] = 173;
    out.transformMaskMappingLocalToWorld[6][123] = 237;
    out.transformMaskMappingLocalToWorld[6][124] = 61;
    out.transformMaskMappingLocalToWorld[6][125] = 125;
    out.transformMaskMappingLocalToWorld[6][126] = 189;
    out.transformMaskMappingLocalToWorld[6][127] = 253;
    out.transformMaskMappingLocalToWorld[6][128] = 2;
    out.transformMaskMappingLocalToWorld[6][129] = 66;
    out.transformMaskMappingLocalToWorld[6][130] = 130;
    out.transformMaskMappingLocalToWorld[6][131] = 194;
    out.transformMaskMappingLocalToWorld[6][132] = 18;
    out.transformMaskMappingLocalToWorld[6][133] = 82;
    out.transformMaskMappingLocalToWorld[6][134] = 146;
    out.transformMaskMappingLocalToWorld[6][135] = 210;
    out.transformMaskMappingLocalToWorld[6][136] = 34;
    out.transformMaskMappingLocalToWorld[6][137] = 98;
    out.transformMaskMappingLocalToWorld[6][138] = 162;
    out.transformMaskMappingLocalToWorld[6][139] = 226;
    out.transformMaskMappingLocalToWorld[6][140] = 50;
    out.transformMaskMappingLocalToWorld[6][141] = 114;
    out.transformMaskMappingLocalToWorld[6][142] = 178;
    out.transformMaskMappingLocalToWorld[6][143] = 242;
    out.transformMaskMappingLocalToWorld[6][144] = 6;
    out.transformMaskMappingLocalToWorld[6][145] = 70;
    out.transformMaskMappingLocalToWorld[6][146] = 134;
    out.transformMaskMappingLocalToWorld[6][147] = 198;
    out.transformMaskMappingLocalToWorld[6][148] = 22;
    out.transformMaskMappingLocalToWorld[6][149] = 86;
    out.transformMaskMappingLocalToWorld[6][150] = 150;
    out.transformMaskMappingLocalToWorld[6][151] = 214;
    out.transformMaskMappingLocalToWorld[6][152] = 38;
    out.transformMaskMappingLocalToWorld[6][153] = 102;
    out.transformMaskMappingLocalToWorld[6][154] = 166;
    out.transformMaskMappingLocalToWorld[6][155] = 230;
    out.transformMaskMappingLocalToWorld[6][156] = 54;
    out.transformMaskMappingLocalToWorld[6][157] = 118;
    out.transformMaskMappingLocalToWorld[6][158] = 182;
    out.transformMaskMappingLocalToWorld[6][159] = 246;
    out.transformMaskMappingLocalToWorld[6][160] = 10;
    out.transformMaskMappingLocalToWorld[6][161] = 74;
    out.transformMaskMappingLocalToWorld[6][162] = 138;
    out.transformMaskMappingLocalToWorld[6][163] = 202;
    out.transformMaskMappingLocalToWorld[6][164] = 26;
    out.transformMaskMappingLocalToWorld[6][165] = 90;
    out.transformMaskMappingLocalToWorld[6][166] = 154;
    out.transformMaskMappingLocalToWorld[6][167] = 218;
    out.transformMaskMappingLocalToWorld[6][168] = 42;
    out.transformMaskMappingLocalToWorld[6][169] = 106;
    out.transformMaskMappingLocalToWorld[6][170] = 170;
    out.transformMaskMappingLocalToWorld[6][171] = 234;
    out.transformMaskMappingLocalToWorld[6][172] = 58;
    out.transformMaskMappingLocalToWorld[6][173] = 122;
    out.transformMaskMappingLocalToWorld[6][174] = 186;
    out.transformMaskMappingLocalToWorld[6][175] = 250;
    out.transformMaskMappingLocalToWorld[6][176] = 14;
    out.transformMaskMappingLocalToWorld[6][177] = 78;
    out.transformMaskMappingLocalToWorld[6][178] = 142;
    out.transformMaskMappingLocalToWorld[6][179] = 206;
    out.transformMaskMappingLocalToWorld[6][180] = 30;
    out.transformMaskMappingLocalToWorld[6][181] = 94;
    out.transformMaskMappingLocalToWorld[6][182] = 158;
    out.transformMaskMappingLocalToWorld[6][183] = 222;
    out.transformMaskMappingLocalToWorld[6][184] = 46;
    out.transformMaskMappingLocalToWorld[6][185] = 110;
    out.transformMaskMappingLocalToWorld[6][186] = 174;
    out.transformMaskMappingLocalToWorld[6][187] = 238;
    out.transformMaskMappingLocalToWorld[6][188] = 62;
    out.transformMaskMappingLocalToWorld[6][189] = 126;
    out.transformMaskMappingLocalToWorld[6][190] = 190;
    out.transformMaskMappingLocalToWorld[6][191] = 254;
    out.transformMaskMappingLocalToWorld[6][192] = 3;
    out.transformMaskMappingLocalToWorld[6][193] = 67;
    out.transformMaskMappingLocalToWorld[6][194] = 131;
    out.transformMaskMappingLocalToWorld[6][195] = 195;
    out.transformMaskMappingLocalToWorld[6][196] = 19;
    out.transformMaskMappingLocalToWorld[6][197] = 83;
    out.transformMaskMappingLocalToWorld[6][198] = 147;
    out.transformMaskMappingLocalToWorld[6][199] = 211;
    out.transformMaskMappingLocalToWorld[6][200] = 35;
    out.transformMaskMappingLocalToWorld[6][201] = 99;
    out.transformMaskMappingLocalToWorld[6][202] = 163;
    out.transformMaskMappingLocalToWorld[6][203] = 227;
    out.transformMaskMappingLocalToWorld[6][204] = 51;
    out.transformMaskMappingLocalToWorld[6][205] = 115;
    out.transformMaskMappingLocalToWorld[6][206] = 179;
    out.transformMaskMappingLocalToWorld[6][207] = 243;
    out.transformMaskMappingLocalToWorld[6][208] = 7;
    out.transformMaskMappingLocalToWorld[6][209] = 71;
    out.transformMaskMappingLocalToWorld[6][210] = 135;
    out.transformMaskMappingLocalToWorld[6][211] = 199;
    out.transformMaskMappingLocalToWorld[6][212] = 23;
    out.transformMaskMappingLocalToWorld[6][213] = 87;
    out.transformMaskMappingLocalToWorld[6][214] = 151;
    out.transformMaskMappingLocalToWorld[6][215] = 215;
    out.transformMaskMappingLocalToWorld[6][216] = 39;
    out.transformMaskMappingLocalToWorld[6][217] = 103;
    out.transformMaskMappingLocalToWorld[6][218] = 167;
    out.transformMaskMappingLocalToWorld[6][219] = 231;
    out.transformMaskMappingLocalToWorld[6][220] = 55;
    out.transformMaskMappingLocalToWorld[6][221] = 119;
    out.transformMaskMappingLocalToWorld[6][222] = 183;
    out.transformMaskMappingLocalToWorld[6][223] = 247;
    out.transformMaskMappingLocalToWorld[6][224] = 11;
    out.transformMaskMappingLocalToWorld[6][225] = 75;
    out.transformMaskMappingLocalToWorld[6][226] = 139;
    out.transformMaskMappingLocalToWorld[6][227] = 203;
    out.transformMaskMappingLocalToWorld[6][228] = 27;
    out.transformMaskMappingLocalToWorld[6][229] = 91;
    out.transformMaskMappingLocalToWorld[6][230] = 155;
    out.transformMaskMappingLocalToWorld[6][231] = 219;
    out.transformMaskMappingLocalToWorld[6][232] = 43;
    out.transformMaskMappingLocalToWorld[6][233] = 107;
    out.transformMaskMappingLocalToWorld[6][234] = 171;
    out.transformMaskMappingLocalToWorld[6][235] = 235;
    out.transformMaskMappingLocalToWorld[6][236] = 59;
    out.transformMaskMappingLocalToWorld[6][237] = 123;
    out.transformMaskMappingLocalToWorld[6][238] = 187;
    out.transformMaskMappingLocalToWorld[6][239] = 251;
    out.transformMaskMappingLocalToWorld[6][240] = 15;
    out.transformMaskMappingLocalToWorld[6][241] = 79;
    out.transformMaskMappingLocalToWorld[6][242] = 143;
    out.transformMaskMappingLocalToWorld[6][243] = 207;
    out.transformMaskMappingLocalToWorld[6][244] = 31;
    out.transformMaskMappingLocalToWorld[6][245] = 95;
    out.transformMaskMappingLocalToWorld[6][246] = 159;
    out.transformMaskMappingLocalToWorld[6][247] = 223;
    out.transformMaskMappingLocalToWorld[6][248] = 47;
    out.transformMaskMappingLocalToWorld[6][249] = 111;
    out.transformMaskMappingLocalToWorld[6][250] = 175;
    out.transformMaskMappingLocalToWorld[6][251] = 239;
    out.transformMaskMappingLocalToWorld[6][252] = 63;
    out.transformMaskMappingLocalToWorld[6][253] = 127;
    out.transformMaskMappingLocalToWorld[6][254] = 191;
    out.transformMaskMappingLocalToWorld[6][255] = 255;
    out.transformMaskMappingLocalToWorld[7][0] = 0;
    out.transformMaskMappingLocalToWorld[7][1] = 128;
    out.transformMaskMappingLocalToWorld[7][2] = 64;
    out.transformMaskMappingLocalToWorld[7][3] = 192;
    out.transformMaskMappingLocalToWorld[7][4] = 32;
    out.transformMaskMappingLocalToWorld[7][5] = 160;
    out.transformMaskMappingLocalToWorld[7][6] = 96;
    out.transformMaskMappingLocalToWorld[7][7] = 224;
    out.transformMaskMappingLocalToWorld[7][8] = 16;
    out.transformMaskMappingLocalToWorld[7][9] = 144;
    out.transformMaskMappingLocalToWorld[7][10] = 80;
    out.transformMaskMappingLocalToWorld[7][11] = 208;
    out.transformMaskMappingLocalToWorld[7][12] = 48;
    out.transformMaskMappingLocalToWorld[7][13] = 176;
    out.transformMaskMappingLocalToWorld[7][14] = 112;
    out.transformMaskMappingLocalToWorld[7][15] = 240;
    out.transformMaskMappingLocalToWorld[7][16] = 8;
    out.transformMaskMappingLocalToWorld[7][17] = 136;
    out.transformMaskMappingLocalToWorld[7][18] = 72;
    out.transformMaskMappingLocalToWorld[7][19] = 200;
    out.transformMaskMappingLocalToWorld[7][20] = 40;
    out.transformMaskMappingLocalToWorld[7][21] = 168;
    out.transformMaskMappingLocalToWorld[7][22] = 104;
    out.transformMaskMappingLocalToWorld[7][23] = 232;
    out.transformMaskMappingLocalToWorld[7][24] = 24;
    out.transformMaskMappingLocalToWorld[7][25] = 152;
    out.transformMaskMappingLocalToWorld[7][26] = 88;
    out.transformMaskMappingLocalToWorld[7][27] = 216;
    out.transformMaskMappingLocalToWorld[7][28] = 56;
    out.transformMaskMappingLocalToWorld[7][29] = 184;
    out.transformMaskMappingLocalToWorld[7][30] = 120;
    out.transformMaskMappingLocalToWorld[7][31] = 248;
    out.transformMaskMappingLocalToWorld[7][32] = 4;
    out.transformMaskMappingLocalToWorld[7][33] = 132;
    out.transformMaskMappingLocalToWorld[7][34] = 68;
    out.transformMaskMappingLocalToWorld[7][35] = 196;
    out.transformMaskMappingLocalToWorld[7][36] = 36;
    out.transformMaskMappingLocalToWorld[7][37] = 164;
    out.transformMaskMappingLocalToWorld[7][38] = 100;
    out.transformMaskMappingLocalToWorld[7][39] = 228;
    out.transformMaskMappingLocalToWorld[7][40] = 20;
    out.transformMaskMappingLocalToWorld[7][41] = 148;
    out.transformMaskMappingLocalToWorld[7][42] = 84;
    out.transformMaskMappingLocalToWorld[7][43] = 212;
    out.transformMaskMappingLocalToWorld[7][44] = 52;
    out.transformMaskMappingLocalToWorld[7][45] = 180;
    out.transformMaskMappingLocalToWorld[7][46] = 116;
    out.transformMaskMappingLocalToWorld[7][47] = 244;
    out.transformMaskMappingLocalToWorld[7][48] = 12;
    out.transformMaskMappingLocalToWorld[7][49] = 140;
    out.transformMaskMappingLocalToWorld[7][50] = 76;
    out.transformMaskMappingLocalToWorld[7][51] = 204;
    out.transformMaskMappingLocalToWorld[7][52] = 44;
    out.transformMaskMappingLocalToWorld[7][53] = 172;
    out.transformMaskMappingLocalToWorld[7][54] = 108;
    out.transformMaskMappingLocalToWorld[7][55] = 236;
    out.transformMaskMappingLocalToWorld[7][56] = 28;
    out.transformMaskMappingLocalToWorld[7][57] = 156;
    out.transformMaskMappingLocalToWorld[7][58] = 92;
    out.transformMaskMappingLocalToWorld[7][59] = 220;
    out.transformMaskMappingLocalToWorld[7][60] = 60;
    out.transformMaskMappingLocalToWorld[7][61] = 188;
    out.transformMaskMappingLocalToWorld[7][62] = 124;
    out.transformMaskMappingLocalToWorld[7][63] = 252;
    out.transformMaskMappingLocalToWorld[7][64] = 2;
    out.transformMaskMappingLocalToWorld[7][65] = 130;
    out.transformMaskMappingLocalToWorld[7][66] = 66;
    out.transformMaskMappingLocalToWorld[7][67] = 194;
    out.transformMaskMappingLocalToWorld[7][68] = 34;
    out.transformMaskMappingLocalToWorld[7][69] = 162;
    out.transformMaskMappingLocalToWorld[7][70] = 98;
    out.transformMaskMappingLocalToWorld[7][71] = 226;
    out.transformMaskMappingLocalToWorld[7][72] = 18;
    out.transformMaskMappingLocalToWorld[7][73] = 146;
    out.transformMaskMappingLocalToWorld[7][74] = 82;
    out.transformMaskMappingLocalToWorld[7][75] = 210;
    out.transformMaskMappingLocalToWorld[7][76] = 50;
    out.transformMaskMappingLocalToWorld[7][77] = 178;
    out.transformMaskMappingLocalToWorld[7][78] = 114;
    out.transformMaskMappingLocalToWorld[7][79] = 242;
    out.transformMaskMappingLocalToWorld[7][80] = 10;
    out.transformMaskMappingLocalToWorld[7][81] = 138;
    out.transformMaskMappingLocalToWorld[7][82] = 74;
    out.transformMaskMappingLocalToWorld[7][83] = 202;
    out.transformMaskMappingLocalToWorld[7][84] = 42;
    out.transformMaskMappingLocalToWorld[7][85] = 170;
    out.transformMaskMappingLocalToWorld[7][86] = 106;
    out.transformMaskMappingLocalToWorld[7][87] = 234;
    out.transformMaskMappingLocalToWorld[7][88] = 26;
    out.transformMaskMappingLocalToWorld[7][89] = 154;
    out.transformMaskMappingLocalToWorld[7][90] = 90;
    out.transformMaskMappingLocalToWorld[7][91] = 218;
    out.transformMaskMappingLocalToWorld[7][92] = 58;
    out.transformMaskMappingLocalToWorld[7][93] = 186;
    out.transformMaskMappingLocalToWorld[7][94] = 122;
    out.transformMaskMappingLocalToWorld[7][95] = 250;
    out.transformMaskMappingLocalToWorld[7][96] = 6;
    out.transformMaskMappingLocalToWorld[7][97] = 134;
    out.transformMaskMappingLocalToWorld[7][98] = 70;
    out.transformMaskMappingLocalToWorld[7][99] = 198;
    out.transformMaskMappingLocalToWorld[7][100] = 38;
    out.transformMaskMappingLocalToWorld[7][101] = 166;
    out.transformMaskMappingLocalToWorld[7][102] = 102;
    out.transformMaskMappingLocalToWorld[7][103] = 230;
    out.transformMaskMappingLocalToWorld[7][104] = 22;
    out.transformMaskMappingLocalToWorld[7][105] = 150;
    out.transformMaskMappingLocalToWorld[7][106] = 86;
    out.transformMaskMappingLocalToWorld[7][107] = 214;
    out.transformMaskMappingLocalToWorld[7][108] = 54;
    out.transformMaskMappingLocalToWorld[7][109] = 182;
    out.transformMaskMappingLocalToWorld[7][110] = 118;
    out.transformMaskMappingLocalToWorld[7][111] = 246;
    out.transformMaskMappingLocalToWorld[7][112] = 14;
    out.transformMaskMappingLocalToWorld[7][113] = 142;
    out.transformMaskMappingLocalToWorld[7][114] = 78;
    out.transformMaskMappingLocalToWorld[7][115] = 206;
    out.transformMaskMappingLocalToWorld[7][116] = 46;
    out.transformMaskMappingLocalToWorld[7][117] = 174;
    out.transformMaskMappingLocalToWorld[7][118] = 110;
    out.transformMaskMappingLocalToWorld[7][119] = 238;
    out.transformMaskMappingLocalToWorld[7][120] = 30;
    out.transformMaskMappingLocalToWorld[7][121] = 158;
    out.transformMaskMappingLocalToWorld[7][122] = 94;
    out.transformMaskMappingLocalToWorld[7][123] = 222;
    out.transformMaskMappingLocalToWorld[7][124] = 62;
    out.transformMaskMappingLocalToWorld[7][125] = 190;
    out.transformMaskMappingLocalToWorld[7][126] = 126;
    out.transformMaskMappingLocalToWorld[7][127] = 254;
    out.transformMaskMappingLocalToWorld[7][128] = 1;
    out.transformMaskMappingLocalToWorld[7][129] = 129;
    out.transformMaskMappingLocalToWorld[7][130] = 65;
    out.transformMaskMappingLocalToWorld[7][131] = 193;
    out.transformMaskMappingLocalToWorld[7][132] = 33;
    out.transformMaskMappingLocalToWorld[7][133] = 161;
    out.transformMaskMappingLocalToWorld[7][134] = 97;
    out.transformMaskMappingLocalToWorld[7][135] = 225;
    out.transformMaskMappingLocalToWorld[7][136] = 17;
    out.transformMaskMappingLocalToWorld[7][137] = 145;
    out.transformMaskMappingLocalToWorld[7][138] = 81;
    out.transformMaskMappingLocalToWorld[7][139] = 209;
    out.transformMaskMappingLocalToWorld[7][140] = 49;
    out.transformMaskMappingLocalToWorld[7][141] = 177;
    out.transformMaskMappingLocalToWorld[7][142] = 113;
    out.transformMaskMappingLocalToWorld[7][143] = 241;
    out.transformMaskMappingLocalToWorld[7][144] = 9;
    out.transformMaskMappingLocalToWorld[7][145] = 137;
    out.transformMaskMappingLocalToWorld[7][146] = 73;
    out.transformMaskMappingLocalToWorld[7][147] = 201;
    out.transformMaskMappingLocalToWorld[7][148] = 41;
    out.transformMaskMappingLocalToWorld[7][149] = 169;
    out.transformMaskMappingLocalToWorld[7][150] = 105;
    out.transformMaskMappingLocalToWorld[7][151] = 233;
    out.transformMaskMappingLocalToWorld[7][152] = 25;
    out.transformMaskMappingLocalToWorld[7][153] = 153;
    out.transformMaskMappingLocalToWorld[7][154] = 89;
    out.transformMaskMappingLocalToWorld[7][155] = 217;
    out.transformMaskMappingLocalToWorld[7][156] = 57;
    out.transformMaskMappingLocalToWorld[7][157] = 185;
    out.transformMaskMappingLocalToWorld[7][158] = 121;
    out.transformMaskMappingLocalToWorld[7][159] = 249;
    out.transformMaskMappingLocalToWorld[7][160] = 5;
    out.transformMaskMappingLocalToWorld[7][161] = 133;
    out.transformMaskMappingLocalToWorld[7][162] = 69;
    out.transformMaskMappingLocalToWorld[7][163] = 197;
    out.transformMaskMappingLocalToWorld[7][164] = 37;
    out.transformMaskMappingLocalToWorld[7][165] = 165;
    out.transformMaskMappingLocalToWorld[7][166] = 101;
    out.transformMaskMappingLocalToWorld[7][167] = 229;
    out.transformMaskMappingLocalToWorld[7][168] = 21;
    out.transformMaskMappingLocalToWorld[7][169] = 149;
    out.transformMaskMappingLocalToWorld[7][170] = 85;
    out.transformMaskMappingLocalToWorld[7][171] = 213;
    out.transformMaskMappingLocalToWorld[7][172] = 53;
    out.transformMaskMappingLocalToWorld[7][173] = 181;
    out.transformMaskMappingLocalToWorld[7][174] = 117;
    out.transformMaskMappingLocalToWorld[7][175] = 245;
    out.transformMaskMappingLocalToWorld[7][176] = 13;
    out.transformMaskMappingLocalToWorld[7][177] = 141;
    out.transformMaskMappingLocalToWorld[7][178] = 77;
    out.transformMaskMappingLocalToWorld[7][179] = 205;
    out.transformMaskMappingLocalToWorld[7][180] = 45;
    out.transformMaskMappingLocalToWorld[7][181] = 173;
    out.transformMaskMappingLocalToWorld[7][182] = 109;
    out.transformMaskMappingLocalToWorld[7][183] = 237;
    out.transformMaskMappingLocalToWorld[7][184] = 29;
    out.transformMaskMappingLocalToWorld[7][185] = 157;
    out.transformMaskMappingLocalToWorld[7][186] = 93;
    out.transformMaskMappingLocalToWorld[7][187] = 221;
    out.transformMaskMappingLocalToWorld[7][188] = 61;
    out.transformMaskMappingLocalToWorld[7][189] = 189;
    out.transformMaskMappingLocalToWorld[7][190] = 125;
    out.transformMaskMappingLocalToWorld[7][191] = 253;
    out.transformMaskMappingLocalToWorld[7][192] = 3;
    out.transformMaskMappingLocalToWorld[7][193] = 131;
    out.transformMaskMappingLocalToWorld[7][194] = 67;
    out.transformMaskMappingLocalToWorld[7][195] = 195;
    out.transformMaskMappingLocalToWorld[7][196] = 35;
    out.transformMaskMappingLocalToWorld[7][197] = 163;
    out.transformMaskMappingLocalToWorld[7][198] = 99;
    out.transformMaskMappingLocalToWorld[7][199] = 227;
    out.transformMaskMappingLocalToWorld[7][200] = 19;
    out.transformMaskMappingLocalToWorld[7][201] = 147;
    out.transformMaskMappingLocalToWorld[7][202] = 83;
    out.transformMaskMappingLocalToWorld[7][203] = 211;
    out.transformMaskMappingLocalToWorld[7][204] = 51;
    out.transformMaskMappingLocalToWorld[7][205] = 179;
    out.transformMaskMappingLocalToWorld[7][206] = 115;
    out.transformMaskMappingLocalToWorld[7][207] = 243;
    out.transformMaskMappingLocalToWorld[7][208] = 11;
    out.transformMaskMappingLocalToWorld[7][209] = 139;
    out.transformMaskMappingLocalToWorld[7][210] = 75;
    out.transformMaskMappingLocalToWorld[7][211] = 203;
    out.transformMaskMappingLocalToWorld[7][212] = 43;
    out.transformMaskMappingLocalToWorld[7][213] = 171;
    out.transformMaskMappingLocalToWorld[7][214] = 107;
    out.transformMaskMappingLocalToWorld[7][215] = 235;
    out.transformMaskMappingLocalToWorld[7][216] = 27;
    out.transformMaskMappingLocalToWorld[7][217] = 155;
    out.transformMaskMappingLocalToWorld[7][218] = 91;
    out.transformMaskMappingLocalToWorld[7][219] = 219;
    out.transformMaskMappingLocalToWorld[7][220] = 59;
    out.transformMaskMappingLocalToWorld[7][221] = 187;
    out.transformMaskMappingLocalToWorld[7][222] = 123;
    out.transformMaskMappingLocalToWorld[7][223] = 251;
    out.transformMaskMappingLocalToWorld[7][224] = 7;
    out.transformMaskMappingLocalToWorld[7][225] = 135;
    out.transformMaskMappingLocalToWorld[7][226] = 71;
    out.transformMaskMappingLocalToWorld[7][227] = 199;
    out.transformMaskMappingLocalToWorld[7][228] = 39;
    out.transformMaskMappingLocalToWorld[7][229] = 167;
    out.transformMaskMappingLocalToWorld[7][230] = 103;
    out.transformMaskMappingLocalToWorld[7][231] = 231;
    out.transformMaskMappingLocalToWorld[7][232] = 23;
    out.transformMaskMappingLocalToWorld[7][233] = 151;
    out.transformMaskMappingLocalToWorld[7][234] = 87;
    out.transformMaskMappingLocalToWorld[7][235] = 215;
    out.transformMaskMappingLocalToWorld[7][236] = 55;
    out.transformMaskMappingLocalToWorld[7][237] = 183;
    out.transformMaskMappingLocalToWorld[7][238] = 119;
    out.transformMaskMappingLocalToWorld[7][239] = 247;
    out.transformMaskMappingLocalToWorld[7][240] = 15;
    out.transformMaskMappingLocalToWorld[7][241] = 143;
    out.transformMaskMappingLocalToWorld[7][242] = 79;
    out.transformMaskMappingLocalToWorld[7][243] = 207;
    out.transformMaskMappingLocalToWorld[7][244] = 47;
    out.transformMaskMappingLocalToWorld[7][245] = 175;
    out.transformMaskMappingLocalToWorld[7][246] = 111;
    out.transformMaskMappingLocalToWorld[7][247] = 239;
    out.transformMaskMappingLocalToWorld[7][248] = 31;
    out.transformMaskMappingLocalToWorld[7][249] = 159;
    out.transformMaskMappingLocalToWorld[7][250] = 95;
    out.transformMaskMappingLocalToWorld[7][251] = 223;
    out.transformMaskMappingLocalToWorld[7][252] = 63;
    out.transformMaskMappingLocalToWorld[7][253] = 191;
    out.transformMaskMappingLocalToWorld[7][254] = 127;
    out.transformMaskMappingLocalToWorld[7][255] = 255;
    out.transformMaskMappingLocalToWorld[8][0] = 0;
    out.transformMaskMappingLocalToWorld[8][1] = 1;
    out.transformMaskMappingLocalToWorld[8][2] = 2;
    out.transformMaskMappingLocalToWorld[8][3] = 3;
    out.transformMaskMappingLocalToWorld[8][4] = 16;
    out.transformMaskMappingLocalToWorld[8][5] = 17;
    out.transformMaskMappingLocalToWorld[8][6] = 18;
    out.transformMaskMappingLocalToWorld[8][7] = 19;
    out.transformMaskMappingLocalToWorld[8][8] = 32;
    out.transformMaskMappingLocalToWorld[8][9] = 33;
    out.transformMaskMappingLocalToWorld[8][10] = 34;
    out.transformMaskMappingLocalToWorld[8][11] = 35;
    out.transformMaskMappingLocalToWorld[8][12] = 48;
    out.transformMaskMappingLocalToWorld[8][13] = 49;
    out.transformMaskMappingLocalToWorld[8][14] = 50;
    out.transformMaskMappingLocalToWorld[8][15] = 51;
    out.transformMaskMappingLocalToWorld[8][16] = 4;
    out.transformMaskMappingLocalToWorld[8][17] = 5;
    out.transformMaskMappingLocalToWorld[8][18] = 6;
    out.transformMaskMappingLocalToWorld[8][19] = 7;
    out.transformMaskMappingLocalToWorld[8][20] = 20;
    out.transformMaskMappingLocalToWorld[8][21] = 21;
    out.transformMaskMappingLocalToWorld[8][22] = 22;
    out.transformMaskMappingLocalToWorld[8][23] = 23;
    out.transformMaskMappingLocalToWorld[8][24] = 36;
    out.transformMaskMappingLocalToWorld[8][25] = 37;
    out.transformMaskMappingLocalToWorld[8][26] = 38;
    out.transformMaskMappingLocalToWorld[8][27] = 39;
    out.transformMaskMappingLocalToWorld[8][28] = 52;
    out.transformMaskMappingLocalToWorld[8][29] = 53;
    out.transformMaskMappingLocalToWorld[8][30] = 54;
    out.transformMaskMappingLocalToWorld[8][31] = 55;
    out.transformMaskMappingLocalToWorld[8][32] = 8;
    out.transformMaskMappingLocalToWorld[8][33] = 9;
    out.transformMaskMappingLocalToWorld[8][34] = 10;
    out.transformMaskMappingLocalToWorld[8][35] = 11;
    out.transformMaskMappingLocalToWorld[8][36] = 24;
    out.transformMaskMappingLocalToWorld[8][37] = 25;
    out.transformMaskMappingLocalToWorld[8][38] = 26;
    out.transformMaskMappingLocalToWorld[8][39] = 27;
    out.transformMaskMappingLocalToWorld[8][40] = 40;
    out.transformMaskMappingLocalToWorld[8][41] = 41;
    out.transformMaskMappingLocalToWorld[8][42] = 42;
    out.transformMaskMappingLocalToWorld[8][43] = 43;
    out.transformMaskMappingLocalToWorld[8][44] = 56;
    out.transformMaskMappingLocalToWorld[8][45] = 57;
    out.transformMaskMappingLocalToWorld[8][46] = 58;
    out.transformMaskMappingLocalToWorld[8][47] = 59;
    out.transformMaskMappingLocalToWorld[8][48] = 12;
    out.transformMaskMappingLocalToWorld[8][49] = 13;
    out.transformMaskMappingLocalToWorld[8][50] = 14;
    out.transformMaskMappingLocalToWorld[8][51] = 15;
    out.transformMaskMappingLocalToWorld[8][52] = 28;
    out.transformMaskMappingLocalToWorld[8][53] = 29;
    out.transformMaskMappingLocalToWorld[8][54] = 30;
    out.transformMaskMappingLocalToWorld[8][55] = 31;
    out.transformMaskMappingLocalToWorld[8][56] = 44;
    out.transformMaskMappingLocalToWorld[8][57] = 45;
    out.transformMaskMappingLocalToWorld[8][58] = 46;
    out.transformMaskMappingLocalToWorld[8][59] = 47;
    out.transformMaskMappingLocalToWorld[8][60] = 60;
    out.transformMaskMappingLocalToWorld[8][61] = 61;
    out.transformMaskMappingLocalToWorld[8][62] = 62;
    out.transformMaskMappingLocalToWorld[8][63] = 63;
    out.transformMaskMappingLocalToWorld[8][64] = 64;
    out.transformMaskMappingLocalToWorld[8][65] = 65;
    out.transformMaskMappingLocalToWorld[8][66] = 66;
    out.transformMaskMappingLocalToWorld[8][67] = 67;
    out.transformMaskMappingLocalToWorld[8][68] = 80;
    out.transformMaskMappingLocalToWorld[8][69] = 81;
    out.transformMaskMappingLocalToWorld[8][70] = 82;
    out.transformMaskMappingLocalToWorld[8][71] = 83;
    out.transformMaskMappingLocalToWorld[8][72] = 96;
    out.transformMaskMappingLocalToWorld[8][73] = 97;
    out.transformMaskMappingLocalToWorld[8][74] = 98;
    out.transformMaskMappingLocalToWorld[8][75] = 99;
    out.transformMaskMappingLocalToWorld[8][76] = 112;
    out.transformMaskMappingLocalToWorld[8][77] = 113;
    out.transformMaskMappingLocalToWorld[8][78] = 114;
    out.transformMaskMappingLocalToWorld[8][79] = 115;
    out.transformMaskMappingLocalToWorld[8][80] = 68;
    out.transformMaskMappingLocalToWorld[8][81] = 69;
    out.transformMaskMappingLocalToWorld[8][82] = 70;
    out.transformMaskMappingLocalToWorld[8][83] = 71;
    out.transformMaskMappingLocalToWorld[8][84] = 84;
    out.transformMaskMappingLocalToWorld[8][85] = 85;
    out.transformMaskMappingLocalToWorld[8][86] = 86;
    out.transformMaskMappingLocalToWorld[8][87] = 87;
    out.transformMaskMappingLocalToWorld[8][88] = 100;
    out.transformMaskMappingLocalToWorld[8][89] = 101;
    out.transformMaskMappingLocalToWorld[8][90] = 102;
    out.transformMaskMappingLocalToWorld[8][91] = 103;
    out.transformMaskMappingLocalToWorld[8][92] = 116;
    out.transformMaskMappingLocalToWorld[8][93] = 117;
    out.transformMaskMappingLocalToWorld[8][94] = 118;
    out.transformMaskMappingLocalToWorld[8][95] = 119;
    out.transformMaskMappingLocalToWorld[8][96] = 72;
    out.transformMaskMappingLocalToWorld[8][97] = 73;
    out.transformMaskMappingLocalToWorld[8][98] = 74;
    out.transformMaskMappingLocalToWorld[8][99] = 75;
    out.transformMaskMappingLocalToWorld[8][100] = 88;
    out.transformMaskMappingLocalToWorld[8][101] = 89;
    out.transformMaskMappingLocalToWorld[8][102] = 90;
    out.transformMaskMappingLocalToWorld[8][103] = 91;
    out.transformMaskMappingLocalToWorld[8][104] = 104;
    out.transformMaskMappingLocalToWorld[8][105] = 105;
    out.transformMaskMappingLocalToWorld[8][106] = 106;
    out.transformMaskMappingLocalToWorld[8][107] = 107;
    out.transformMaskMappingLocalToWorld[8][108] = 120;
    out.transformMaskMappingLocalToWorld[8][109] = 121;
    out.transformMaskMappingLocalToWorld[8][110] = 122;
    out.transformMaskMappingLocalToWorld[8][111] = 123;
    out.transformMaskMappingLocalToWorld[8][112] = 76;
    out.transformMaskMappingLocalToWorld[8][113] = 77;
    out.transformMaskMappingLocalToWorld[8][114] = 78;
    out.transformMaskMappingLocalToWorld[8][115] = 79;
    out.transformMaskMappingLocalToWorld[8][116] = 92;
    out.transformMaskMappingLocalToWorld[8][117] = 93;
    out.transformMaskMappingLocalToWorld[8][118] = 94;
    out.transformMaskMappingLocalToWorld[8][119] = 95;
    out.transformMaskMappingLocalToWorld[8][120] = 108;
    out.transformMaskMappingLocalToWorld[8][121] = 109;
    out.transformMaskMappingLocalToWorld[8][122] = 110;
    out.transformMaskMappingLocalToWorld[8][123] = 111;
    out.transformMaskMappingLocalToWorld[8][124] = 124;
    out.transformMaskMappingLocalToWorld[8][125] = 125;
    out.transformMaskMappingLocalToWorld[8][126] = 126;
    out.transformMaskMappingLocalToWorld[8][127] = 127;
    out.transformMaskMappingLocalToWorld[8][128] = 128;
    out.transformMaskMappingLocalToWorld[8][129] = 129;
    out.transformMaskMappingLocalToWorld[8][130] = 130;
    out.transformMaskMappingLocalToWorld[8][131] = 131;
    out.transformMaskMappingLocalToWorld[8][132] = 144;
    out.transformMaskMappingLocalToWorld[8][133] = 145;
    out.transformMaskMappingLocalToWorld[8][134] = 146;
    out.transformMaskMappingLocalToWorld[8][135] = 147;
    out.transformMaskMappingLocalToWorld[8][136] = 160;
    out.transformMaskMappingLocalToWorld[8][137] = 161;
    out.transformMaskMappingLocalToWorld[8][138] = 162;
    out.transformMaskMappingLocalToWorld[8][139] = 163;
    out.transformMaskMappingLocalToWorld[8][140] = 176;
    out.transformMaskMappingLocalToWorld[8][141] = 177;
    out.transformMaskMappingLocalToWorld[8][142] = 178;
    out.transformMaskMappingLocalToWorld[8][143] = 179;
    out.transformMaskMappingLocalToWorld[8][144] = 132;
    out.transformMaskMappingLocalToWorld[8][145] = 133;
    out.transformMaskMappingLocalToWorld[8][146] = 134;
    out.transformMaskMappingLocalToWorld[8][147] = 135;
    out.transformMaskMappingLocalToWorld[8][148] = 148;
    out.transformMaskMappingLocalToWorld[8][149] = 149;
    out.transformMaskMappingLocalToWorld[8][150] = 150;
    out.transformMaskMappingLocalToWorld[8][151] = 151;
    out.transformMaskMappingLocalToWorld[8][152] = 164;
    out.transformMaskMappingLocalToWorld[8][153] = 165;
    out.transformMaskMappingLocalToWorld[8][154] = 166;
    out.transformMaskMappingLocalToWorld[8][155] = 167;
    out.transformMaskMappingLocalToWorld[8][156] = 180;
    out.transformMaskMappingLocalToWorld[8][157] = 181;
    out.transformMaskMappingLocalToWorld[8][158] = 182;
    out.transformMaskMappingLocalToWorld[8][159] = 183;
    out.transformMaskMappingLocalToWorld[8][160] = 136;
    out.transformMaskMappingLocalToWorld[8][161] = 137;
    out.transformMaskMappingLocalToWorld[8][162] = 138;
    out.transformMaskMappingLocalToWorld[8][163] = 139;
    out.transformMaskMappingLocalToWorld[8][164] = 152;
    out.transformMaskMappingLocalToWorld[8][165] = 153;
    out.transformMaskMappingLocalToWorld[8][166] = 154;
    out.transformMaskMappingLocalToWorld[8][167] = 155;
    out.transformMaskMappingLocalToWorld[8][168] = 168;
    out.transformMaskMappingLocalToWorld[8][169] = 169;
    out.transformMaskMappingLocalToWorld[8][170] = 170;
    out.transformMaskMappingLocalToWorld[8][171] = 171;
    out.transformMaskMappingLocalToWorld[8][172] = 184;
    out.transformMaskMappingLocalToWorld[8][173] = 185;
    out.transformMaskMappingLocalToWorld[8][174] = 186;
    out.transformMaskMappingLocalToWorld[8][175] = 187;
    out.transformMaskMappingLocalToWorld[8][176] = 140;
    out.transformMaskMappingLocalToWorld[8][177] = 141;
    out.transformMaskMappingLocalToWorld[8][178] = 142;
    out.transformMaskMappingLocalToWorld[8][179] = 143;
    out.transformMaskMappingLocalToWorld[8][180] = 156;
    out.transformMaskMappingLocalToWorld[8][181] = 157;
    out.transformMaskMappingLocalToWorld[8][182] = 158;
    out.transformMaskMappingLocalToWorld[8][183] = 159;
    out.transformMaskMappingLocalToWorld[8][184] = 172;
    out.transformMaskMappingLocalToWorld[8][185] = 173;
    out.transformMaskMappingLocalToWorld[8][186] = 174;
    out.transformMaskMappingLocalToWorld[8][187] = 175;
    out.transformMaskMappingLocalToWorld[8][188] = 188;
    out.transformMaskMappingLocalToWorld[8][189] = 189;
    out.transformMaskMappingLocalToWorld[8][190] = 190;
    out.transformMaskMappingLocalToWorld[8][191] = 191;
    out.transformMaskMappingLocalToWorld[8][192] = 192;
    out.transformMaskMappingLocalToWorld[8][193] = 193;
    out.transformMaskMappingLocalToWorld[8][194] = 194;
    out.transformMaskMappingLocalToWorld[8][195] = 195;
    out.transformMaskMappingLocalToWorld[8][196] = 208;
    out.transformMaskMappingLocalToWorld[8][197] = 209;
    out.transformMaskMappingLocalToWorld[8][198] = 210;
    out.transformMaskMappingLocalToWorld[8][199] = 211;
    out.transformMaskMappingLocalToWorld[8][200] = 224;
    out.transformMaskMappingLocalToWorld[8][201] = 225;
    out.transformMaskMappingLocalToWorld[8][202] = 226;
    out.transformMaskMappingLocalToWorld[8][203] = 227;
    out.transformMaskMappingLocalToWorld[8][204] = 240;
    out.transformMaskMappingLocalToWorld[8][205] = 241;
    out.transformMaskMappingLocalToWorld[8][206] = 242;
    out.transformMaskMappingLocalToWorld[8][207] = 243;
    out.transformMaskMappingLocalToWorld[8][208] = 196;
    out.transformMaskMappingLocalToWorld[8][209] = 197;
    out.transformMaskMappingLocalToWorld[8][210] = 198;
    out.transformMaskMappingLocalToWorld[8][211] = 199;
    out.transformMaskMappingLocalToWorld[8][212] = 212;
    out.transformMaskMappingLocalToWorld[8][213] = 213;
    out.transformMaskMappingLocalToWorld[8][214] = 214;
    out.transformMaskMappingLocalToWorld[8][215] = 215;
    out.transformMaskMappingLocalToWorld[8][216] = 228;
    out.transformMaskMappingLocalToWorld[8][217] = 229;
    out.transformMaskMappingLocalToWorld[8][218] = 230;
    out.transformMaskMappingLocalToWorld[8][219] = 231;
    out.transformMaskMappingLocalToWorld[8][220] = 244;
    out.transformMaskMappingLocalToWorld[8][221] = 245;
    out.transformMaskMappingLocalToWorld[8][222] = 246;
    out.transformMaskMappingLocalToWorld[8][223] = 247;
    out.transformMaskMappingLocalToWorld[8][224] = 200;
    out.transformMaskMappingLocalToWorld[8][225] = 201;
    out.transformMaskMappingLocalToWorld[8][226] = 202;
    out.transformMaskMappingLocalToWorld[8][227] = 203;
    out.transformMaskMappingLocalToWorld[8][228] = 216;
    out.transformMaskMappingLocalToWorld[8][229] = 217;
    out.transformMaskMappingLocalToWorld[8][230] = 218;
    out.transformMaskMappingLocalToWorld[8][231] = 219;
    out.transformMaskMappingLocalToWorld[8][232] = 232;
    out.transformMaskMappingLocalToWorld[8][233] = 233;
    out.transformMaskMappingLocalToWorld[8][234] = 234;
    out.transformMaskMappingLocalToWorld[8][235] = 235;
    out.transformMaskMappingLocalToWorld[8][236] = 248;
    out.transformMaskMappingLocalToWorld[8][237] = 249;
    out.transformMaskMappingLocalToWorld[8][238] = 250;
    out.transformMaskMappingLocalToWorld[8][239] = 251;
    out.transformMaskMappingLocalToWorld[8][240] = 204;
    out.transformMaskMappingLocalToWorld[8][241] = 205;
    out.transformMaskMappingLocalToWorld[8][242] = 206;
    out.transformMaskMappingLocalToWorld[8][243] = 207;
    out.transformMaskMappingLocalToWorld[8][244] = 220;
    out.transformMaskMappingLocalToWorld[8][245] = 221;
    out.transformMaskMappingLocalToWorld[8][246] = 222;
    out.transformMaskMappingLocalToWorld[8][247] = 223;
    out.transformMaskMappingLocalToWorld[8][248] = 236;
    out.transformMaskMappingLocalToWorld[8][249] = 237;
    out.transformMaskMappingLocalToWorld[8][250] = 238;
    out.transformMaskMappingLocalToWorld[8][251] = 239;
    out.transformMaskMappingLocalToWorld[8][252] = 252;
    out.transformMaskMappingLocalToWorld[8][253] = 253;
    out.transformMaskMappingLocalToWorld[8][254] = 254;
    out.transformMaskMappingLocalToWorld[8][255] = 255;
    out.transformMaskMappingLocalToWorld[9][0] = 0;
    out.transformMaskMappingLocalToWorld[9][1] = 2;
    out.transformMaskMappingLocalToWorld[9][2] = 1;
    out.transformMaskMappingLocalToWorld[9][3] = 3;
    out.transformMaskMappingLocalToWorld[9][4] = 32;
    out.transformMaskMappingLocalToWorld[9][5] = 34;
    out.transformMaskMappingLocalToWorld[9][6] = 33;
    out.transformMaskMappingLocalToWorld[9][7] = 35;
    out.transformMaskMappingLocalToWorld[9][8] = 16;
    out.transformMaskMappingLocalToWorld[9][9] = 18;
    out.transformMaskMappingLocalToWorld[9][10] = 17;
    out.transformMaskMappingLocalToWorld[9][11] = 19;
    out.transformMaskMappingLocalToWorld[9][12] = 48;
    out.transformMaskMappingLocalToWorld[9][13] = 50;
    out.transformMaskMappingLocalToWorld[9][14] = 49;
    out.transformMaskMappingLocalToWorld[9][15] = 51;
    out.transformMaskMappingLocalToWorld[9][16] = 8;
    out.transformMaskMappingLocalToWorld[9][17] = 10;
    out.transformMaskMappingLocalToWorld[9][18] = 9;
    out.transformMaskMappingLocalToWorld[9][19] = 11;
    out.transformMaskMappingLocalToWorld[9][20] = 40;
    out.transformMaskMappingLocalToWorld[9][21] = 42;
    out.transformMaskMappingLocalToWorld[9][22] = 41;
    out.transformMaskMappingLocalToWorld[9][23] = 43;
    out.transformMaskMappingLocalToWorld[9][24] = 24;
    out.transformMaskMappingLocalToWorld[9][25] = 26;
    out.transformMaskMappingLocalToWorld[9][26] = 25;
    out.transformMaskMappingLocalToWorld[9][27] = 27;
    out.transformMaskMappingLocalToWorld[9][28] = 56;
    out.transformMaskMappingLocalToWorld[9][29] = 58;
    out.transformMaskMappingLocalToWorld[9][30] = 57;
    out.transformMaskMappingLocalToWorld[9][31] = 59;
    out.transformMaskMappingLocalToWorld[9][32] = 4;
    out.transformMaskMappingLocalToWorld[9][33] = 6;
    out.transformMaskMappingLocalToWorld[9][34] = 5;
    out.transformMaskMappingLocalToWorld[9][35] = 7;
    out.transformMaskMappingLocalToWorld[9][36] = 36;
    out.transformMaskMappingLocalToWorld[9][37] = 38;
    out.transformMaskMappingLocalToWorld[9][38] = 37;
    out.transformMaskMappingLocalToWorld[9][39] = 39;
    out.transformMaskMappingLocalToWorld[9][40] = 20;
    out.transformMaskMappingLocalToWorld[9][41] = 22;
    out.transformMaskMappingLocalToWorld[9][42] = 21;
    out.transformMaskMappingLocalToWorld[9][43] = 23;
    out.transformMaskMappingLocalToWorld[9][44] = 52;
    out.transformMaskMappingLocalToWorld[9][45] = 54;
    out.transformMaskMappingLocalToWorld[9][46] = 53;
    out.transformMaskMappingLocalToWorld[9][47] = 55;
    out.transformMaskMappingLocalToWorld[9][48] = 12;
    out.transformMaskMappingLocalToWorld[9][49] = 14;
    out.transformMaskMappingLocalToWorld[9][50] = 13;
    out.transformMaskMappingLocalToWorld[9][51] = 15;
    out.transformMaskMappingLocalToWorld[9][52] = 44;
    out.transformMaskMappingLocalToWorld[9][53] = 46;
    out.transformMaskMappingLocalToWorld[9][54] = 45;
    out.transformMaskMappingLocalToWorld[9][55] = 47;
    out.transformMaskMappingLocalToWorld[9][56] = 28;
    out.transformMaskMappingLocalToWorld[9][57] = 30;
    out.transformMaskMappingLocalToWorld[9][58] = 29;
    out.transformMaskMappingLocalToWorld[9][59] = 31;
    out.transformMaskMappingLocalToWorld[9][60] = 60;
    out.transformMaskMappingLocalToWorld[9][61] = 62;
    out.transformMaskMappingLocalToWorld[9][62] = 61;
    out.transformMaskMappingLocalToWorld[9][63] = 63;
    out.transformMaskMappingLocalToWorld[9][64] = 128;
    out.transformMaskMappingLocalToWorld[9][65] = 130;
    out.transformMaskMappingLocalToWorld[9][66] = 129;
    out.transformMaskMappingLocalToWorld[9][67] = 131;
    out.transformMaskMappingLocalToWorld[9][68] = 160;
    out.transformMaskMappingLocalToWorld[9][69] = 162;
    out.transformMaskMappingLocalToWorld[9][70] = 161;
    out.transformMaskMappingLocalToWorld[9][71] = 163;
    out.transformMaskMappingLocalToWorld[9][72] = 144;
    out.transformMaskMappingLocalToWorld[9][73] = 146;
    out.transformMaskMappingLocalToWorld[9][74] = 145;
    out.transformMaskMappingLocalToWorld[9][75] = 147;
    out.transformMaskMappingLocalToWorld[9][76] = 176;
    out.transformMaskMappingLocalToWorld[9][77] = 178;
    out.transformMaskMappingLocalToWorld[9][78] = 177;
    out.transformMaskMappingLocalToWorld[9][79] = 179;
    out.transformMaskMappingLocalToWorld[9][80] = 136;
    out.transformMaskMappingLocalToWorld[9][81] = 138;
    out.transformMaskMappingLocalToWorld[9][82] = 137;
    out.transformMaskMappingLocalToWorld[9][83] = 139;
    out.transformMaskMappingLocalToWorld[9][84] = 168;
    out.transformMaskMappingLocalToWorld[9][85] = 170;
    out.transformMaskMappingLocalToWorld[9][86] = 169;
    out.transformMaskMappingLocalToWorld[9][87] = 171;
    out.transformMaskMappingLocalToWorld[9][88] = 152;
    out.transformMaskMappingLocalToWorld[9][89] = 154;
    out.transformMaskMappingLocalToWorld[9][90] = 153;
    out.transformMaskMappingLocalToWorld[9][91] = 155;
    out.transformMaskMappingLocalToWorld[9][92] = 184;
    out.transformMaskMappingLocalToWorld[9][93] = 186;
    out.transformMaskMappingLocalToWorld[9][94] = 185;
    out.transformMaskMappingLocalToWorld[9][95] = 187;
    out.transformMaskMappingLocalToWorld[9][96] = 132;
    out.transformMaskMappingLocalToWorld[9][97] = 134;
    out.transformMaskMappingLocalToWorld[9][98] = 133;
    out.transformMaskMappingLocalToWorld[9][99] = 135;
    out.transformMaskMappingLocalToWorld[9][100] = 164;
    out.transformMaskMappingLocalToWorld[9][101] = 166;
    out.transformMaskMappingLocalToWorld[9][102] = 165;
    out.transformMaskMappingLocalToWorld[9][103] = 167;
    out.transformMaskMappingLocalToWorld[9][104] = 148;
    out.transformMaskMappingLocalToWorld[9][105] = 150;
    out.transformMaskMappingLocalToWorld[9][106] = 149;
    out.transformMaskMappingLocalToWorld[9][107] = 151;
    out.transformMaskMappingLocalToWorld[9][108] = 180;
    out.transformMaskMappingLocalToWorld[9][109] = 182;
    out.transformMaskMappingLocalToWorld[9][110] = 181;
    out.transformMaskMappingLocalToWorld[9][111] = 183;
    out.transformMaskMappingLocalToWorld[9][112] = 140;
    out.transformMaskMappingLocalToWorld[9][113] = 142;
    out.transformMaskMappingLocalToWorld[9][114] = 141;
    out.transformMaskMappingLocalToWorld[9][115] = 143;
    out.transformMaskMappingLocalToWorld[9][116] = 172;
    out.transformMaskMappingLocalToWorld[9][117] = 174;
    out.transformMaskMappingLocalToWorld[9][118] = 173;
    out.transformMaskMappingLocalToWorld[9][119] = 175;
    out.transformMaskMappingLocalToWorld[9][120] = 156;
    out.transformMaskMappingLocalToWorld[9][121] = 158;
    out.transformMaskMappingLocalToWorld[9][122] = 157;
    out.transformMaskMappingLocalToWorld[9][123] = 159;
    out.transformMaskMappingLocalToWorld[9][124] = 188;
    out.transformMaskMappingLocalToWorld[9][125] = 190;
    out.transformMaskMappingLocalToWorld[9][126] = 189;
    out.transformMaskMappingLocalToWorld[9][127] = 191;
    out.transformMaskMappingLocalToWorld[9][128] = 64;
    out.transformMaskMappingLocalToWorld[9][129] = 66;
    out.transformMaskMappingLocalToWorld[9][130] = 65;
    out.transformMaskMappingLocalToWorld[9][131] = 67;
    out.transformMaskMappingLocalToWorld[9][132] = 96;
    out.transformMaskMappingLocalToWorld[9][133] = 98;
    out.transformMaskMappingLocalToWorld[9][134] = 97;
    out.transformMaskMappingLocalToWorld[9][135] = 99;
    out.transformMaskMappingLocalToWorld[9][136] = 80;
    out.transformMaskMappingLocalToWorld[9][137] = 82;
    out.transformMaskMappingLocalToWorld[9][138] = 81;
    out.transformMaskMappingLocalToWorld[9][139] = 83;
    out.transformMaskMappingLocalToWorld[9][140] = 112;
    out.transformMaskMappingLocalToWorld[9][141] = 114;
    out.transformMaskMappingLocalToWorld[9][142] = 113;
    out.transformMaskMappingLocalToWorld[9][143] = 115;
    out.transformMaskMappingLocalToWorld[9][144] = 72;
    out.transformMaskMappingLocalToWorld[9][145] = 74;
    out.transformMaskMappingLocalToWorld[9][146] = 73;
    out.transformMaskMappingLocalToWorld[9][147] = 75;
    out.transformMaskMappingLocalToWorld[9][148] = 104;
    out.transformMaskMappingLocalToWorld[9][149] = 106;
    out.transformMaskMappingLocalToWorld[9][150] = 105;
    out.transformMaskMappingLocalToWorld[9][151] = 107;
    out.transformMaskMappingLocalToWorld[9][152] = 88;
    out.transformMaskMappingLocalToWorld[9][153] = 90;
    out.transformMaskMappingLocalToWorld[9][154] = 89;
    out.transformMaskMappingLocalToWorld[9][155] = 91;
    out.transformMaskMappingLocalToWorld[9][156] = 120;
    out.transformMaskMappingLocalToWorld[9][157] = 122;
    out.transformMaskMappingLocalToWorld[9][158] = 121;
    out.transformMaskMappingLocalToWorld[9][159] = 123;
    out.transformMaskMappingLocalToWorld[9][160] = 68;
    out.transformMaskMappingLocalToWorld[9][161] = 70;
    out.transformMaskMappingLocalToWorld[9][162] = 69;
    out.transformMaskMappingLocalToWorld[9][163] = 71;
    out.transformMaskMappingLocalToWorld[9][164] = 100;
    out.transformMaskMappingLocalToWorld[9][165] = 102;
    out.transformMaskMappingLocalToWorld[9][166] = 101;
    out.transformMaskMappingLocalToWorld[9][167] = 103;
    out.transformMaskMappingLocalToWorld[9][168] = 84;
    out.transformMaskMappingLocalToWorld[9][169] = 86;
    out.transformMaskMappingLocalToWorld[9][170] = 85;
    out.transformMaskMappingLocalToWorld[9][171] = 87;
    out.transformMaskMappingLocalToWorld[9][172] = 116;
    out.transformMaskMappingLocalToWorld[9][173] = 118;
    out.transformMaskMappingLocalToWorld[9][174] = 117;
    out.transformMaskMappingLocalToWorld[9][175] = 119;
    out.transformMaskMappingLocalToWorld[9][176] = 76;
    out.transformMaskMappingLocalToWorld[9][177] = 78;
    out.transformMaskMappingLocalToWorld[9][178] = 77;
    out.transformMaskMappingLocalToWorld[9][179] = 79;
    out.transformMaskMappingLocalToWorld[9][180] = 108;
    out.transformMaskMappingLocalToWorld[9][181] = 110;
    out.transformMaskMappingLocalToWorld[9][182] = 109;
    out.transformMaskMappingLocalToWorld[9][183] = 111;
    out.transformMaskMappingLocalToWorld[9][184] = 92;
    out.transformMaskMappingLocalToWorld[9][185] = 94;
    out.transformMaskMappingLocalToWorld[9][186] = 93;
    out.transformMaskMappingLocalToWorld[9][187] = 95;
    out.transformMaskMappingLocalToWorld[9][188] = 124;
    out.transformMaskMappingLocalToWorld[9][189] = 126;
    out.transformMaskMappingLocalToWorld[9][190] = 125;
    out.transformMaskMappingLocalToWorld[9][191] = 127;
    out.transformMaskMappingLocalToWorld[9][192] = 192;
    out.transformMaskMappingLocalToWorld[9][193] = 194;
    out.transformMaskMappingLocalToWorld[9][194] = 193;
    out.transformMaskMappingLocalToWorld[9][195] = 195;
    out.transformMaskMappingLocalToWorld[9][196] = 224;
    out.transformMaskMappingLocalToWorld[9][197] = 226;
    out.transformMaskMappingLocalToWorld[9][198] = 225;
    out.transformMaskMappingLocalToWorld[9][199] = 227;
    out.transformMaskMappingLocalToWorld[9][200] = 208;
    out.transformMaskMappingLocalToWorld[9][201] = 210;
    out.transformMaskMappingLocalToWorld[9][202] = 209;
    out.transformMaskMappingLocalToWorld[9][203] = 211;
    out.transformMaskMappingLocalToWorld[9][204] = 240;
    out.transformMaskMappingLocalToWorld[9][205] = 242;
    out.transformMaskMappingLocalToWorld[9][206] = 241;
    out.transformMaskMappingLocalToWorld[9][207] = 243;
    out.transformMaskMappingLocalToWorld[9][208] = 200;
    out.transformMaskMappingLocalToWorld[9][209] = 202;
    out.transformMaskMappingLocalToWorld[9][210] = 201;
    out.transformMaskMappingLocalToWorld[9][211] = 203;
    out.transformMaskMappingLocalToWorld[9][212] = 232;
    out.transformMaskMappingLocalToWorld[9][213] = 234;
    out.transformMaskMappingLocalToWorld[9][214] = 233;
    out.transformMaskMappingLocalToWorld[9][215] = 235;
    out.transformMaskMappingLocalToWorld[9][216] = 216;
    out.transformMaskMappingLocalToWorld[9][217] = 218;
    out.transformMaskMappingLocalToWorld[9][218] = 217;
    out.transformMaskMappingLocalToWorld[9][219] = 219;
    out.transformMaskMappingLocalToWorld[9][220] = 248;
    out.transformMaskMappingLocalToWorld[9][221] = 250;
    out.transformMaskMappingLocalToWorld[9][222] = 249;
    out.transformMaskMappingLocalToWorld[9][223] = 251;
    out.transformMaskMappingLocalToWorld[9][224] = 196;
    out.transformMaskMappingLocalToWorld[9][225] = 198;
    out.transformMaskMappingLocalToWorld[9][226] = 197;
    out.transformMaskMappingLocalToWorld[9][227] = 199;
    out.transformMaskMappingLocalToWorld[9][228] = 228;
    out.transformMaskMappingLocalToWorld[9][229] = 230;
    out.transformMaskMappingLocalToWorld[9][230] = 229;
    out.transformMaskMappingLocalToWorld[9][231] = 231;
    out.transformMaskMappingLocalToWorld[9][232] = 212;
    out.transformMaskMappingLocalToWorld[9][233] = 214;
    out.transformMaskMappingLocalToWorld[9][234] = 213;
    out.transformMaskMappingLocalToWorld[9][235] = 215;
    out.transformMaskMappingLocalToWorld[9][236] = 244;
    out.transformMaskMappingLocalToWorld[9][237] = 246;
    out.transformMaskMappingLocalToWorld[9][238] = 245;
    out.transformMaskMappingLocalToWorld[9][239] = 247;
    out.transformMaskMappingLocalToWorld[9][240] = 204;
    out.transformMaskMappingLocalToWorld[9][241] = 206;
    out.transformMaskMappingLocalToWorld[9][242] = 205;
    out.transformMaskMappingLocalToWorld[9][243] = 207;
    out.transformMaskMappingLocalToWorld[9][244] = 236;
    out.transformMaskMappingLocalToWorld[9][245] = 238;
    out.transformMaskMappingLocalToWorld[9][246] = 237;
    out.transformMaskMappingLocalToWorld[9][247] = 239;
    out.transformMaskMappingLocalToWorld[9][248] = 220;
    out.transformMaskMappingLocalToWorld[9][249] = 222;
    out.transformMaskMappingLocalToWorld[9][250] = 221;
    out.transformMaskMappingLocalToWorld[9][251] = 223;
    out.transformMaskMappingLocalToWorld[9][252] = 252;
    out.transformMaskMappingLocalToWorld[9][253] = 254;
    out.transformMaskMappingLocalToWorld[9][254] = 253;
    out.transformMaskMappingLocalToWorld[9][255] = 255;
    out.transformMaskMappingLocalToWorld[10][0] = 0;
    out.transformMaskMappingLocalToWorld[10][1] = 4;
    out.transformMaskMappingLocalToWorld[10][2] = 8;
    out.transformMaskMappingLocalToWorld[10][3] = 12;
    out.transformMaskMappingLocalToWorld[10][4] = 64;
    out.transformMaskMappingLocalToWorld[10][5] = 68;
    out.transformMaskMappingLocalToWorld[10][6] = 72;
    out.transformMaskMappingLocalToWorld[10][7] = 76;
    out.transformMaskMappingLocalToWorld[10][8] = 128;
    out.transformMaskMappingLocalToWorld[10][9] = 132;
    out.transformMaskMappingLocalToWorld[10][10] = 136;
    out.transformMaskMappingLocalToWorld[10][11] = 140;
    out.transformMaskMappingLocalToWorld[10][12] = 192;
    out.transformMaskMappingLocalToWorld[10][13] = 196;
    out.transformMaskMappingLocalToWorld[10][14] = 200;
    out.transformMaskMappingLocalToWorld[10][15] = 204;
    out.transformMaskMappingLocalToWorld[10][16] = 1;
    out.transformMaskMappingLocalToWorld[10][17] = 5;
    out.transformMaskMappingLocalToWorld[10][18] = 9;
    out.transformMaskMappingLocalToWorld[10][19] = 13;
    out.transformMaskMappingLocalToWorld[10][20] = 65;
    out.transformMaskMappingLocalToWorld[10][21] = 69;
    out.transformMaskMappingLocalToWorld[10][22] = 73;
    out.transformMaskMappingLocalToWorld[10][23] = 77;
    out.transformMaskMappingLocalToWorld[10][24] = 129;
    out.transformMaskMappingLocalToWorld[10][25] = 133;
    out.transformMaskMappingLocalToWorld[10][26] = 137;
    out.transformMaskMappingLocalToWorld[10][27] = 141;
    out.transformMaskMappingLocalToWorld[10][28] = 193;
    out.transformMaskMappingLocalToWorld[10][29] = 197;
    out.transformMaskMappingLocalToWorld[10][30] = 201;
    out.transformMaskMappingLocalToWorld[10][31] = 205;
    out.transformMaskMappingLocalToWorld[10][32] = 2;
    out.transformMaskMappingLocalToWorld[10][33] = 6;
    out.transformMaskMappingLocalToWorld[10][34] = 10;
    out.transformMaskMappingLocalToWorld[10][35] = 14;
    out.transformMaskMappingLocalToWorld[10][36] = 66;
    out.transformMaskMappingLocalToWorld[10][37] = 70;
    out.transformMaskMappingLocalToWorld[10][38] = 74;
    out.transformMaskMappingLocalToWorld[10][39] = 78;
    out.transformMaskMappingLocalToWorld[10][40] = 130;
    out.transformMaskMappingLocalToWorld[10][41] = 134;
    out.transformMaskMappingLocalToWorld[10][42] = 138;
    out.transformMaskMappingLocalToWorld[10][43] = 142;
    out.transformMaskMappingLocalToWorld[10][44] = 194;
    out.transformMaskMappingLocalToWorld[10][45] = 198;
    out.transformMaskMappingLocalToWorld[10][46] = 202;
    out.transformMaskMappingLocalToWorld[10][47] = 206;
    out.transformMaskMappingLocalToWorld[10][48] = 3;
    out.transformMaskMappingLocalToWorld[10][49] = 7;
    out.transformMaskMappingLocalToWorld[10][50] = 11;
    out.transformMaskMappingLocalToWorld[10][51] = 15;
    out.transformMaskMappingLocalToWorld[10][52] = 67;
    out.transformMaskMappingLocalToWorld[10][53] = 71;
    out.transformMaskMappingLocalToWorld[10][54] = 75;
    out.transformMaskMappingLocalToWorld[10][55] = 79;
    out.transformMaskMappingLocalToWorld[10][56] = 131;
    out.transformMaskMappingLocalToWorld[10][57] = 135;
    out.transformMaskMappingLocalToWorld[10][58] = 139;
    out.transformMaskMappingLocalToWorld[10][59] = 143;
    out.transformMaskMappingLocalToWorld[10][60] = 195;
    out.transformMaskMappingLocalToWorld[10][61] = 199;
    out.transformMaskMappingLocalToWorld[10][62] = 203;
    out.transformMaskMappingLocalToWorld[10][63] = 207;
    out.transformMaskMappingLocalToWorld[10][64] = 16;
    out.transformMaskMappingLocalToWorld[10][65] = 20;
    out.transformMaskMappingLocalToWorld[10][66] = 24;
    out.transformMaskMappingLocalToWorld[10][67] = 28;
    out.transformMaskMappingLocalToWorld[10][68] = 80;
    out.transformMaskMappingLocalToWorld[10][69] = 84;
    out.transformMaskMappingLocalToWorld[10][70] = 88;
    out.transformMaskMappingLocalToWorld[10][71] = 92;
    out.transformMaskMappingLocalToWorld[10][72] = 144;
    out.transformMaskMappingLocalToWorld[10][73] = 148;
    out.transformMaskMappingLocalToWorld[10][74] = 152;
    out.transformMaskMappingLocalToWorld[10][75] = 156;
    out.transformMaskMappingLocalToWorld[10][76] = 208;
    out.transformMaskMappingLocalToWorld[10][77] = 212;
    out.transformMaskMappingLocalToWorld[10][78] = 216;
    out.transformMaskMappingLocalToWorld[10][79] = 220;
    out.transformMaskMappingLocalToWorld[10][80] = 17;
    out.transformMaskMappingLocalToWorld[10][81] = 21;
    out.transformMaskMappingLocalToWorld[10][82] = 25;
    out.transformMaskMappingLocalToWorld[10][83] = 29;
    out.transformMaskMappingLocalToWorld[10][84] = 81;
    out.transformMaskMappingLocalToWorld[10][85] = 85;
    out.transformMaskMappingLocalToWorld[10][86] = 89;
    out.transformMaskMappingLocalToWorld[10][87] = 93;
    out.transformMaskMappingLocalToWorld[10][88] = 145;
    out.transformMaskMappingLocalToWorld[10][89] = 149;
    out.transformMaskMappingLocalToWorld[10][90] = 153;
    out.transformMaskMappingLocalToWorld[10][91] = 157;
    out.transformMaskMappingLocalToWorld[10][92] = 209;
    out.transformMaskMappingLocalToWorld[10][93] = 213;
    out.transformMaskMappingLocalToWorld[10][94] = 217;
    out.transformMaskMappingLocalToWorld[10][95] = 221;
    out.transformMaskMappingLocalToWorld[10][96] = 18;
    out.transformMaskMappingLocalToWorld[10][97] = 22;
    out.transformMaskMappingLocalToWorld[10][98] = 26;
    out.transformMaskMappingLocalToWorld[10][99] = 30;
    out.transformMaskMappingLocalToWorld[10][100] = 82;
    out.transformMaskMappingLocalToWorld[10][101] = 86;
    out.transformMaskMappingLocalToWorld[10][102] = 90;
    out.transformMaskMappingLocalToWorld[10][103] = 94;
    out.transformMaskMappingLocalToWorld[10][104] = 146;
    out.transformMaskMappingLocalToWorld[10][105] = 150;
    out.transformMaskMappingLocalToWorld[10][106] = 154;
    out.transformMaskMappingLocalToWorld[10][107] = 158;
    out.transformMaskMappingLocalToWorld[10][108] = 210;
    out.transformMaskMappingLocalToWorld[10][109] = 214;
    out.transformMaskMappingLocalToWorld[10][110] = 218;
    out.transformMaskMappingLocalToWorld[10][111] = 222;
    out.transformMaskMappingLocalToWorld[10][112] = 19;
    out.transformMaskMappingLocalToWorld[10][113] = 23;
    out.transformMaskMappingLocalToWorld[10][114] = 27;
    out.transformMaskMappingLocalToWorld[10][115] = 31;
    out.transformMaskMappingLocalToWorld[10][116] = 83;
    out.transformMaskMappingLocalToWorld[10][117] = 87;
    out.transformMaskMappingLocalToWorld[10][118] = 91;
    out.transformMaskMappingLocalToWorld[10][119] = 95;
    out.transformMaskMappingLocalToWorld[10][120] = 147;
    out.transformMaskMappingLocalToWorld[10][121] = 151;
    out.transformMaskMappingLocalToWorld[10][122] = 155;
    out.transformMaskMappingLocalToWorld[10][123] = 159;
    out.transformMaskMappingLocalToWorld[10][124] = 211;
    out.transformMaskMappingLocalToWorld[10][125] = 215;
    out.transformMaskMappingLocalToWorld[10][126] = 219;
    out.transformMaskMappingLocalToWorld[10][127] = 223;
    out.transformMaskMappingLocalToWorld[10][128] = 32;
    out.transformMaskMappingLocalToWorld[10][129] = 36;
    out.transformMaskMappingLocalToWorld[10][130] = 40;
    out.transformMaskMappingLocalToWorld[10][131] = 44;
    out.transformMaskMappingLocalToWorld[10][132] = 96;
    out.transformMaskMappingLocalToWorld[10][133] = 100;
    out.transformMaskMappingLocalToWorld[10][134] = 104;
    out.transformMaskMappingLocalToWorld[10][135] = 108;
    out.transformMaskMappingLocalToWorld[10][136] = 160;
    out.transformMaskMappingLocalToWorld[10][137] = 164;
    out.transformMaskMappingLocalToWorld[10][138] = 168;
    out.transformMaskMappingLocalToWorld[10][139] = 172;
    out.transformMaskMappingLocalToWorld[10][140] = 224;
    out.transformMaskMappingLocalToWorld[10][141] = 228;
    out.transformMaskMappingLocalToWorld[10][142] = 232;
    out.transformMaskMappingLocalToWorld[10][143] = 236;
    out.transformMaskMappingLocalToWorld[10][144] = 33;
    out.transformMaskMappingLocalToWorld[10][145] = 37;
    out.transformMaskMappingLocalToWorld[10][146] = 41;
    out.transformMaskMappingLocalToWorld[10][147] = 45;
    out.transformMaskMappingLocalToWorld[10][148] = 97;
    out.transformMaskMappingLocalToWorld[10][149] = 101;
    out.transformMaskMappingLocalToWorld[10][150] = 105;
    out.transformMaskMappingLocalToWorld[10][151] = 109;
    out.transformMaskMappingLocalToWorld[10][152] = 161;
    out.transformMaskMappingLocalToWorld[10][153] = 165;
    out.transformMaskMappingLocalToWorld[10][154] = 169;
    out.transformMaskMappingLocalToWorld[10][155] = 173;
    out.transformMaskMappingLocalToWorld[10][156] = 225;
    out.transformMaskMappingLocalToWorld[10][157] = 229;
    out.transformMaskMappingLocalToWorld[10][158] = 233;
    out.transformMaskMappingLocalToWorld[10][159] = 237;
    out.transformMaskMappingLocalToWorld[10][160] = 34;
    out.transformMaskMappingLocalToWorld[10][161] = 38;
    out.transformMaskMappingLocalToWorld[10][162] = 42;
    out.transformMaskMappingLocalToWorld[10][163] = 46;
    out.transformMaskMappingLocalToWorld[10][164] = 98;
    out.transformMaskMappingLocalToWorld[10][165] = 102;
    out.transformMaskMappingLocalToWorld[10][166] = 106;
    out.transformMaskMappingLocalToWorld[10][167] = 110;
    out.transformMaskMappingLocalToWorld[10][168] = 162;
    out.transformMaskMappingLocalToWorld[10][169] = 166;
    out.transformMaskMappingLocalToWorld[10][170] = 170;
    out.transformMaskMappingLocalToWorld[10][171] = 174;
    out.transformMaskMappingLocalToWorld[10][172] = 226;
    out.transformMaskMappingLocalToWorld[10][173] = 230;
    out.transformMaskMappingLocalToWorld[10][174] = 234;
    out.transformMaskMappingLocalToWorld[10][175] = 238;
    out.transformMaskMappingLocalToWorld[10][176] = 35;
    out.transformMaskMappingLocalToWorld[10][177] = 39;
    out.transformMaskMappingLocalToWorld[10][178] = 43;
    out.transformMaskMappingLocalToWorld[10][179] = 47;
    out.transformMaskMappingLocalToWorld[10][180] = 99;
    out.transformMaskMappingLocalToWorld[10][181] = 103;
    out.transformMaskMappingLocalToWorld[10][182] = 107;
    out.transformMaskMappingLocalToWorld[10][183] = 111;
    out.transformMaskMappingLocalToWorld[10][184] = 163;
    out.transformMaskMappingLocalToWorld[10][185] = 167;
    out.transformMaskMappingLocalToWorld[10][186] = 171;
    out.transformMaskMappingLocalToWorld[10][187] = 175;
    out.transformMaskMappingLocalToWorld[10][188] = 227;
    out.transformMaskMappingLocalToWorld[10][189] = 231;
    out.transformMaskMappingLocalToWorld[10][190] = 235;
    out.transformMaskMappingLocalToWorld[10][191] = 239;
    out.transformMaskMappingLocalToWorld[10][192] = 48;
    out.transformMaskMappingLocalToWorld[10][193] = 52;
    out.transformMaskMappingLocalToWorld[10][194] = 56;
    out.transformMaskMappingLocalToWorld[10][195] = 60;
    out.transformMaskMappingLocalToWorld[10][196] = 112;
    out.transformMaskMappingLocalToWorld[10][197] = 116;
    out.transformMaskMappingLocalToWorld[10][198] = 120;
    out.transformMaskMappingLocalToWorld[10][199] = 124;
    out.transformMaskMappingLocalToWorld[10][200] = 176;
    out.transformMaskMappingLocalToWorld[10][201] = 180;
    out.transformMaskMappingLocalToWorld[10][202] = 184;
    out.transformMaskMappingLocalToWorld[10][203] = 188;
    out.transformMaskMappingLocalToWorld[10][204] = 240;
    out.transformMaskMappingLocalToWorld[10][205] = 244;
    out.transformMaskMappingLocalToWorld[10][206] = 248;
    out.transformMaskMappingLocalToWorld[10][207] = 252;
    out.transformMaskMappingLocalToWorld[10][208] = 49;
    out.transformMaskMappingLocalToWorld[10][209] = 53;
    out.transformMaskMappingLocalToWorld[10][210] = 57;
    out.transformMaskMappingLocalToWorld[10][211] = 61;
    out.transformMaskMappingLocalToWorld[10][212] = 113;
    out.transformMaskMappingLocalToWorld[10][213] = 117;
    out.transformMaskMappingLocalToWorld[10][214] = 121;
    out.transformMaskMappingLocalToWorld[10][215] = 125;
    out.transformMaskMappingLocalToWorld[10][216] = 177;
    out.transformMaskMappingLocalToWorld[10][217] = 181;
    out.transformMaskMappingLocalToWorld[10][218] = 185;
    out.transformMaskMappingLocalToWorld[10][219] = 189;
    out.transformMaskMappingLocalToWorld[10][220] = 241;
    out.transformMaskMappingLocalToWorld[10][221] = 245;
    out.transformMaskMappingLocalToWorld[10][222] = 249;
    out.transformMaskMappingLocalToWorld[10][223] = 253;
    out.transformMaskMappingLocalToWorld[10][224] = 50;
    out.transformMaskMappingLocalToWorld[10][225] = 54;
    out.transformMaskMappingLocalToWorld[10][226] = 58;
    out.transformMaskMappingLocalToWorld[10][227] = 62;
    out.transformMaskMappingLocalToWorld[10][228] = 114;
    out.transformMaskMappingLocalToWorld[10][229] = 118;
    out.transformMaskMappingLocalToWorld[10][230] = 122;
    out.transformMaskMappingLocalToWorld[10][231] = 126;
    out.transformMaskMappingLocalToWorld[10][232] = 178;
    out.transformMaskMappingLocalToWorld[10][233] = 182;
    out.transformMaskMappingLocalToWorld[10][234] = 186;
    out.transformMaskMappingLocalToWorld[10][235] = 190;
    out.transformMaskMappingLocalToWorld[10][236] = 242;
    out.transformMaskMappingLocalToWorld[10][237] = 246;
    out.transformMaskMappingLocalToWorld[10][238] = 250;
    out.transformMaskMappingLocalToWorld[10][239] = 254;
    out.transformMaskMappingLocalToWorld[10][240] = 51;
    out.transformMaskMappingLocalToWorld[10][241] = 55;
    out.transformMaskMappingLocalToWorld[10][242] = 59;
    out.transformMaskMappingLocalToWorld[10][243] = 63;
    out.transformMaskMappingLocalToWorld[10][244] = 115;
    out.transformMaskMappingLocalToWorld[10][245] = 119;
    out.transformMaskMappingLocalToWorld[10][246] = 123;
    out.transformMaskMappingLocalToWorld[10][247] = 127;
    out.transformMaskMappingLocalToWorld[10][248] = 179;
    out.transformMaskMappingLocalToWorld[10][249] = 183;
    out.transformMaskMappingLocalToWorld[10][250] = 187;
    out.transformMaskMappingLocalToWorld[10][251] = 191;
    out.transformMaskMappingLocalToWorld[10][252] = 243;
    out.transformMaskMappingLocalToWorld[10][253] = 247;
    out.transformMaskMappingLocalToWorld[10][254] = 251;
    out.transformMaskMappingLocalToWorld[10][255] = 255;
    out.transformMaskMappingLocalToWorld[11][0] = 0;
    out.transformMaskMappingLocalToWorld[11][1] = 8;
    out.transformMaskMappingLocalToWorld[11][2] = 4;
    out.transformMaskMappingLocalToWorld[11][3] = 12;
    out.transformMaskMappingLocalToWorld[11][4] = 128;
    out.transformMaskMappingLocalToWorld[11][5] = 136;
    out.transformMaskMappingLocalToWorld[11][6] = 132;
    out.transformMaskMappingLocalToWorld[11][7] = 140;
    out.transformMaskMappingLocalToWorld[11][8] = 64;
    out.transformMaskMappingLocalToWorld[11][9] = 72;
    out.transformMaskMappingLocalToWorld[11][10] = 68;
    out.transformMaskMappingLocalToWorld[11][11] = 76;
    out.transformMaskMappingLocalToWorld[11][12] = 192;
    out.transformMaskMappingLocalToWorld[11][13] = 200;
    out.transformMaskMappingLocalToWorld[11][14] = 196;
    out.transformMaskMappingLocalToWorld[11][15] = 204;
    out.transformMaskMappingLocalToWorld[11][16] = 2;
    out.transformMaskMappingLocalToWorld[11][17] = 10;
    out.transformMaskMappingLocalToWorld[11][18] = 6;
    out.transformMaskMappingLocalToWorld[11][19] = 14;
    out.transformMaskMappingLocalToWorld[11][20] = 130;
    out.transformMaskMappingLocalToWorld[11][21] = 138;
    out.transformMaskMappingLocalToWorld[11][22] = 134;
    out.transformMaskMappingLocalToWorld[11][23] = 142;
    out.transformMaskMappingLocalToWorld[11][24] = 66;
    out.transformMaskMappingLocalToWorld[11][25] = 74;
    out.transformMaskMappingLocalToWorld[11][26] = 70;
    out.transformMaskMappingLocalToWorld[11][27] = 78;
    out.transformMaskMappingLocalToWorld[11][28] = 194;
    out.transformMaskMappingLocalToWorld[11][29] = 202;
    out.transformMaskMappingLocalToWorld[11][30] = 198;
    out.transformMaskMappingLocalToWorld[11][31] = 206;
    out.transformMaskMappingLocalToWorld[11][32] = 1;
    out.transformMaskMappingLocalToWorld[11][33] = 9;
    out.transformMaskMappingLocalToWorld[11][34] = 5;
    out.transformMaskMappingLocalToWorld[11][35] = 13;
    out.transformMaskMappingLocalToWorld[11][36] = 129;
    out.transformMaskMappingLocalToWorld[11][37] = 137;
    out.transformMaskMappingLocalToWorld[11][38] = 133;
    out.transformMaskMappingLocalToWorld[11][39] = 141;
    out.transformMaskMappingLocalToWorld[11][40] = 65;
    out.transformMaskMappingLocalToWorld[11][41] = 73;
    out.transformMaskMappingLocalToWorld[11][42] = 69;
    out.transformMaskMappingLocalToWorld[11][43] = 77;
    out.transformMaskMappingLocalToWorld[11][44] = 193;
    out.transformMaskMappingLocalToWorld[11][45] = 201;
    out.transformMaskMappingLocalToWorld[11][46] = 197;
    out.transformMaskMappingLocalToWorld[11][47] = 205;
    out.transformMaskMappingLocalToWorld[11][48] = 3;
    out.transformMaskMappingLocalToWorld[11][49] = 11;
    out.transformMaskMappingLocalToWorld[11][50] = 7;
    out.transformMaskMappingLocalToWorld[11][51] = 15;
    out.transformMaskMappingLocalToWorld[11][52] = 131;
    out.transformMaskMappingLocalToWorld[11][53] = 139;
    out.transformMaskMappingLocalToWorld[11][54] = 135;
    out.transformMaskMappingLocalToWorld[11][55] = 143;
    out.transformMaskMappingLocalToWorld[11][56] = 67;
    out.transformMaskMappingLocalToWorld[11][57] = 75;
    out.transformMaskMappingLocalToWorld[11][58] = 71;
    out.transformMaskMappingLocalToWorld[11][59] = 79;
    out.transformMaskMappingLocalToWorld[11][60] = 195;
    out.transformMaskMappingLocalToWorld[11][61] = 203;
    out.transformMaskMappingLocalToWorld[11][62] = 199;
    out.transformMaskMappingLocalToWorld[11][63] = 207;
    out.transformMaskMappingLocalToWorld[11][64] = 32;
    out.transformMaskMappingLocalToWorld[11][65] = 40;
    out.transformMaskMappingLocalToWorld[11][66] = 36;
    out.transformMaskMappingLocalToWorld[11][67] = 44;
    out.transformMaskMappingLocalToWorld[11][68] = 160;
    out.transformMaskMappingLocalToWorld[11][69] = 168;
    out.transformMaskMappingLocalToWorld[11][70] = 164;
    out.transformMaskMappingLocalToWorld[11][71] = 172;
    out.transformMaskMappingLocalToWorld[11][72] = 96;
    out.transformMaskMappingLocalToWorld[11][73] = 104;
    out.transformMaskMappingLocalToWorld[11][74] = 100;
    out.transformMaskMappingLocalToWorld[11][75] = 108;
    out.transformMaskMappingLocalToWorld[11][76] = 224;
    out.transformMaskMappingLocalToWorld[11][77] = 232;
    out.transformMaskMappingLocalToWorld[11][78] = 228;
    out.transformMaskMappingLocalToWorld[11][79] = 236;
    out.transformMaskMappingLocalToWorld[11][80] = 34;
    out.transformMaskMappingLocalToWorld[11][81] = 42;
    out.transformMaskMappingLocalToWorld[11][82] = 38;
    out.transformMaskMappingLocalToWorld[11][83] = 46;
    out.transformMaskMappingLocalToWorld[11][84] = 162;
    out.transformMaskMappingLocalToWorld[11][85] = 170;
    out.transformMaskMappingLocalToWorld[11][86] = 166;
    out.transformMaskMappingLocalToWorld[11][87] = 174;
    out.transformMaskMappingLocalToWorld[11][88] = 98;
    out.transformMaskMappingLocalToWorld[11][89] = 106;
    out.transformMaskMappingLocalToWorld[11][90] = 102;
    out.transformMaskMappingLocalToWorld[11][91] = 110;
    out.transformMaskMappingLocalToWorld[11][92] = 226;
    out.transformMaskMappingLocalToWorld[11][93] = 234;
    out.transformMaskMappingLocalToWorld[11][94] = 230;
    out.transformMaskMappingLocalToWorld[11][95] = 238;
    out.transformMaskMappingLocalToWorld[11][96] = 33;
    out.transformMaskMappingLocalToWorld[11][97] = 41;
    out.transformMaskMappingLocalToWorld[11][98] = 37;
    out.transformMaskMappingLocalToWorld[11][99] = 45;
    out.transformMaskMappingLocalToWorld[11][100] = 161;
    out.transformMaskMappingLocalToWorld[11][101] = 169;
    out.transformMaskMappingLocalToWorld[11][102] = 165;
    out.transformMaskMappingLocalToWorld[11][103] = 173;
    out.transformMaskMappingLocalToWorld[11][104] = 97;
    out.transformMaskMappingLocalToWorld[11][105] = 105;
    out.transformMaskMappingLocalToWorld[11][106] = 101;
    out.transformMaskMappingLocalToWorld[11][107] = 109;
    out.transformMaskMappingLocalToWorld[11][108] = 225;
    out.transformMaskMappingLocalToWorld[11][109] = 233;
    out.transformMaskMappingLocalToWorld[11][110] = 229;
    out.transformMaskMappingLocalToWorld[11][111] = 237;
    out.transformMaskMappingLocalToWorld[11][112] = 35;
    out.transformMaskMappingLocalToWorld[11][113] = 43;
    out.transformMaskMappingLocalToWorld[11][114] = 39;
    out.transformMaskMappingLocalToWorld[11][115] = 47;
    out.transformMaskMappingLocalToWorld[11][116] = 163;
    out.transformMaskMappingLocalToWorld[11][117] = 171;
    out.transformMaskMappingLocalToWorld[11][118] = 167;
    out.transformMaskMappingLocalToWorld[11][119] = 175;
    out.transformMaskMappingLocalToWorld[11][120] = 99;
    out.transformMaskMappingLocalToWorld[11][121] = 107;
    out.transformMaskMappingLocalToWorld[11][122] = 103;
    out.transformMaskMappingLocalToWorld[11][123] = 111;
    out.transformMaskMappingLocalToWorld[11][124] = 227;
    out.transformMaskMappingLocalToWorld[11][125] = 235;
    out.transformMaskMappingLocalToWorld[11][126] = 231;
    out.transformMaskMappingLocalToWorld[11][127] = 239;
    out.transformMaskMappingLocalToWorld[11][128] = 16;
    out.transformMaskMappingLocalToWorld[11][129] = 24;
    out.transformMaskMappingLocalToWorld[11][130] = 20;
    out.transformMaskMappingLocalToWorld[11][131] = 28;
    out.transformMaskMappingLocalToWorld[11][132] = 144;
    out.transformMaskMappingLocalToWorld[11][133] = 152;
    out.transformMaskMappingLocalToWorld[11][134] = 148;
    out.transformMaskMappingLocalToWorld[11][135] = 156;
    out.transformMaskMappingLocalToWorld[11][136] = 80;
    out.transformMaskMappingLocalToWorld[11][137] = 88;
    out.transformMaskMappingLocalToWorld[11][138] = 84;
    out.transformMaskMappingLocalToWorld[11][139] = 92;
    out.transformMaskMappingLocalToWorld[11][140] = 208;
    out.transformMaskMappingLocalToWorld[11][141] = 216;
    out.transformMaskMappingLocalToWorld[11][142] = 212;
    out.transformMaskMappingLocalToWorld[11][143] = 220;
    out.transformMaskMappingLocalToWorld[11][144] = 18;
    out.transformMaskMappingLocalToWorld[11][145] = 26;
    out.transformMaskMappingLocalToWorld[11][146] = 22;
    out.transformMaskMappingLocalToWorld[11][147] = 30;
    out.transformMaskMappingLocalToWorld[11][148] = 146;
    out.transformMaskMappingLocalToWorld[11][149] = 154;
    out.transformMaskMappingLocalToWorld[11][150] = 150;
    out.transformMaskMappingLocalToWorld[11][151] = 158;
    out.transformMaskMappingLocalToWorld[11][152] = 82;
    out.transformMaskMappingLocalToWorld[11][153] = 90;
    out.transformMaskMappingLocalToWorld[11][154] = 86;
    out.transformMaskMappingLocalToWorld[11][155] = 94;
    out.transformMaskMappingLocalToWorld[11][156] = 210;
    out.transformMaskMappingLocalToWorld[11][157] = 218;
    out.transformMaskMappingLocalToWorld[11][158] = 214;
    out.transformMaskMappingLocalToWorld[11][159] = 222;
    out.transformMaskMappingLocalToWorld[11][160] = 17;
    out.transformMaskMappingLocalToWorld[11][161] = 25;
    out.transformMaskMappingLocalToWorld[11][162] = 21;
    out.transformMaskMappingLocalToWorld[11][163] = 29;
    out.transformMaskMappingLocalToWorld[11][164] = 145;
    out.transformMaskMappingLocalToWorld[11][165] = 153;
    out.transformMaskMappingLocalToWorld[11][166] = 149;
    out.transformMaskMappingLocalToWorld[11][167] = 157;
    out.transformMaskMappingLocalToWorld[11][168] = 81;
    out.transformMaskMappingLocalToWorld[11][169] = 89;
    out.transformMaskMappingLocalToWorld[11][170] = 85;
    out.transformMaskMappingLocalToWorld[11][171] = 93;
    out.transformMaskMappingLocalToWorld[11][172] = 209;
    out.transformMaskMappingLocalToWorld[11][173] = 217;
    out.transformMaskMappingLocalToWorld[11][174] = 213;
    out.transformMaskMappingLocalToWorld[11][175] = 221;
    out.transformMaskMappingLocalToWorld[11][176] = 19;
    out.transformMaskMappingLocalToWorld[11][177] = 27;
    out.transformMaskMappingLocalToWorld[11][178] = 23;
    out.transformMaskMappingLocalToWorld[11][179] = 31;
    out.transformMaskMappingLocalToWorld[11][180] = 147;
    out.transformMaskMappingLocalToWorld[11][181] = 155;
    out.transformMaskMappingLocalToWorld[11][182] = 151;
    out.transformMaskMappingLocalToWorld[11][183] = 159;
    out.transformMaskMappingLocalToWorld[11][184] = 83;
    out.transformMaskMappingLocalToWorld[11][185] = 91;
    out.transformMaskMappingLocalToWorld[11][186] = 87;
    out.transformMaskMappingLocalToWorld[11][187] = 95;
    out.transformMaskMappingLocalToWorld[11][188] = 211;
    out.transformMaskMappingLocalToWorld[11][189] = 219;
    out.transformMaskMappingLocalToWorld[11][190] = 215;
    out.transformMaskMappingLocalToWorld[11][191] = 223;
    out.transformMaskMappingLocalToWorld[11][192] = 48;
    out.transformMaskMappingLocalToWorld[11][193] = 56;
    out.transformMaskMappingLocalToWorld[11][194] = 52;
    out.transformMaskMappingLocalToWorld[11][195] = 60;
    out.transformMaskMappingLocalToWorld[11][196] = 176;
    out.transformMaskMappingLocalToWorld[11][197] = 184;
    out.transformMaskMappingLocalToWorld[11][198] = 180;
    out.transformMaskMappingLocalToWorld[11][199] = 188;
    out.transformMaskMappingLocalToWorld[11][200] = 112;
    out.transformMaskMappingLocalToWorld[11][201] = 120;
    out.transformMaskMappingLocalToWorld[11][202] = 116;
    out.transformMaskMappingLocalToWorld[11][203] = 124;
    out.transformMaskMappingLocalToWorld[11][204] = 240;
    out.transformMaskMappingLocalToWorld[11][205] = 248;
    out.transformMaskMappingLocalToWorld[11][206] = 244;
    out.transformMaskMappingLocalToWorld[11][207] = 252;
    out.transformMaskMappingLocalToWorld[11][208] = 50;
    out.transformMaskMappingLocalToWorld[11][209] = 58;
    out.transformMaskMappingLocalToWorld[11][210] = 54;
    out.transformMaskMappingLocalToWorld[11][211] = 62;
    out.transformMaskMappingLocalToWorld[11][212] = 178;
    out.transformMaskMappingLocalToWorld[11][213] = 186;
    out.transformMaskMappingLocalToWorld[11][214] = 182;
    out.transformMaskMappingLocalToWorld[11][215] = 190;
    out.transformMaskMappingLocalToWorld[11][216] = 114;
    out.transformMaskMappingLocalToWorld[11][217] = 122;
    out.transformMaskMappingLocalToWorld[11][218] = 118;
    out.transformMaskMappingLocalToWorld[11][219] = 126;
    out.transformMaskMappingLocalToWorld[11][220] = 242;
    out.transformMaskMappingLocalToWorld[11][221] = 250;
    out.transformMaskMappingLocalToWorld[11][222] = 246;
    out.transformMaskMappingLocalToWorld[11][223] = 254;
    out.transformMaskMappingLocalToWorld[11][224] = 49;
    out.transformMaskMappingLocalToWorld[11][225] = 57;
    out.transformMaskMappingLocalToWorld[11][226] = 53;
    out.transformMaskMappingLocalToWorld[11][227] = 61;
    out.transformMaskMappingLocalToWorld[11][228] = 177;
    out.transformMaskMappingLocalToWorld[11][229] = 185;
    out.transformMaskMappingLocalToWorld[11][230] = 181;
    out.transformMaskMappingLocalToWorld[11][231] = 189;
    out.transformMaskMappingLocalToWorld[11][232] = 113;
    out.transformMaskMappingLocalToWorld[11][233] = 121;
    out.transformMaskMappingLocalToWorld[11][234] = 117;
    out.transformMaskMappingLocalToWorld[11][235] = 125;
    out.transformMaskMappingLocalToWorld[11][236] = 241;
    out.transformMaskMappingLocalToWorld[11][237] = 249;
    out.transformMaskMappingLocalToWorld[11][238] = 245;
    out.transformMaskMappingLocalToWorld[11][239] = 253;
    out.transformMaskMappingLocalToWorld[11][240] = 51;
    out.transformMaskMappingLocalToWorld[11][241] = 59;
    out.transformMaskMappingLocalToWorld[11][242] = 55;
    out.transformMaskMappingLocalToWorld[11][243] = 63;
    out.transformMaskMappingLocalToWorld[11][244] = 179;
    out.transformMaskMappingLocalToWorld[11][245] = 187;
    out.transformMaskMappingLocalToWorld[11][246] = 183;
    out.transformMaskMappingLocalToWorld[11][247] = 191;
    out.transformMaskMappingLocalToWorld[11][248] = 115;
    out.transformMaskMappingLocalToWorld[11][249] = 123;
    out.transformMaskMappingLocalToWorld[11][250] = 119;
    out.transformMaskMappingLocalToWorld[11][251] = 127;
    out.transformMaskMappingLocalToWorld[11][252] = 243;
    out.transformMaskMappingLocalToWorld[11][253] = 251;
    out.transformMaskMappingLocalToWorld[11][254] = 247;
    out.transformMaskMappingLocalToWorld[11][255] = 255;
    out.transformMaskMappingLocalToWorld[12][0] = 0;
    out.transformMaskMappingLocalToWorld[12][1] = 16;
    out.transformMaskMappingLocalToWorld[12][2] = 32;
    out.transformMaskMappingLocalToWorld[12][3] = 48;
    out.transformMaskMappingLocalToWorld[12][4] = 1;
    out.transformMaskMappingLocalToWorld[12][5] = 17;
    out.transformMaskMappingLocalToWorld[12][6] = 33;
    out.transformMaskMappingLocalToWorld[12][7] = 49;
    out.transformMaskMappingLocalToWorld[12][8] = 2;
    out.transformMaskMappingLocalToWorld[12][9] = 18;
    out.transformMaskMappingLocalToWorld[12][10] = 34;
    out.transformMaskMappingLocalToWorld[12][11] = 50;
    out.transformMaskMappingLocalToWorld[12][12] = 3;
    out.transformMaskMappingLocalToWorld[12][13] = 19;
    out.transformMaskMappingLocalToWorld[12][14] = 35;
    out.transformMaskMappingLocalToWorld[12][15] = 51;
    out.transformMaskMappingLocalToWorld[12][16] = 64;
    out.transformMaskMappingLocalToWorld[12][17] = 80;
    out.transformMaskMappingLocalToWorld[12][18] = 96;
    out.transformMaskMappingLocalToWorld[12][19] = 112;
    out.transformMaskMappingLocalToWorld[12][20] = 65;
    out.transformMaskMappingLocalToWorld[12][21] = 81;
    out.transformMaskMappingLocalToWorld[12][22] = 97;
    out.transformMaskMappingLocalToWorld[12][23] = 113;
    out.transformMaskMappingLocalToWorld[12][24] = 66;
    out.transformMaskMappingLocalToWorld[12][25] = 82;
    out.transformMaskMappingLocalToWorld[12][26] = 98;
    out.transformMaskMappingLocalToWorld[12][27] = 114;
    out.transformMaskMappingLocalToWorld[12][28] = 67;
    out.transformMaskMappingLocalToWorld[12][29] = 83;
    out.transformMaskMappingLocalToWorld[12][30] = 99;
    out.transformMaskMappingLocalToWorld[12][31] = 115;
    out.transformMaskMappingLocalToWorld[12][32] = 128;
    out.transformMaskMappingLocalToWorld[12][33] = 144;
    out.transformMaskMappingLocalToWorld[12][34] = 160;
    out.transformMaskMappingLocalToWorld[12][35] = 176;
    out.transformMaskMappingLocalToWorld[12][36] = 129;
    out.transformMaskMappingLocalToWorld[12][37] = 145;
    out.transformMaskMappingLocalToWorld[12][38] = 161;
    out.transformMaskMappingLocalToWorld[12][39] = 177;
    out.transformMaskMappingLocalToWorld[12][40] = 130;
    out.transformMaskMappingLocalToWorld[12][41] = 146;
    out.transformMaskMappingLocalToWorld[12][42] = 162;
    out.transformMaskMappingLocalToWorld[12][43] = 178;
    out.transformMaskMappingLocalToWorld[12][44] = 131;
    out.transformMaskMappingLocalToWorld[12][45] = 147;
    out.transformMaskMappingLocalToWorld[12][46] = 163;
    out.transformMaskMappingLocalToWorld[12][47] = 179;
    out.transformMaskMappingLocalToWorld[12][48] = 192;
    out.transformMaskMappingLocalToWorld[12][49] = 208;
    out.transformMaskMappingLocalToWorld[12][50] = 224;
    out.transformMaskMappingLocalToWorld[12][51] = 240;
    out.transformMaskMappingLocalToWorld[12][52] = 193;
    out.transformMaskMappingLocalToWorld[12][53] = 209;
    out.transformMaskMappingLocalToWorld[12][54] = 225;
    out.transformMaskMappingLocalToWorld[12][55] = 241;
    out.transformMaskMappingLocalToWorld[12][56] = 194;
    out.transformMaskMappingLocalToWorld[12][57] = 210;
    out.transformMaskMappingLocalToWorld[12][58] = 226;
    out.transformMaskMappingLocalToWorld[12][59] = 242;
    out.transformMaskMappingLocalToWorld[12][60] = 195;
    out.transformMaskMappingLocalToWorld[12][61] = 211;
    out.transformMaskMappingLocalToWorld[12][62] = 227;
    out.transformMaskMappingLocalToWorld[12][63] = 243;
    out.transformMaskMappingLocalToWorld[12][64] = 4;
    out.transformMaskMappingLocalToWorld[12][65] = 20;
    out.transformMaskMappingLocalToWorld[12][66] = 36;
    out.transformMaskMappingLocalToWorld[12][67] = 52;
    out.transformMaskMappingLocalToWorld[12][68] = 5;
    out.transformMaskMappingLocalToWorld[12][69] = 21;
    out.transformMaskMappingLocalToWorld[12][70] = 37;
    out.transformMaskMappingLocalToWorld[12][71] = 53;
    out.transformMaskMappingLocalToWorld[12][72] = 6;
    out.transformMaskMappingLocalToWorld[12][73] = 22;
    out.transformMaskMappingLocalToWorld[12][74] = 38;
    out.transformMaskMappingLocalToWorld[12][75] = 54;
    out.transformMaskMappingLocalToWorld[12][76] = 7;
    out.transformMaskMappingLocalToWorld[12][77] = 23;
    out.transformMaskMappingLocalToWorld[12][78] = 39;
    out.transformMaskMappingLocalToWorld[12][79] = 55;
    out.transformMaskMappingLocalToWorld[12][80] = 68;
    out.transformMaskMappingLocalToWorld[12][81] = 84;
    out.transformMaskMappingLocalToWorld[12][82] = 100;
    out.transformMaskMappingLocalToWorld[12][83] = 116;
    out.transformMaskMappingLocalToWorld[12][84] = 69;
    out.transformMaskMappingLocalToWorld[12][85] = 85;
    out.transformMaskMappingLocalToWorld[12][86] = 101;
    out.transformMaskMappingLocalToWorld[12][87] = 117;
    out.transformMaskMappingLocalToWorld[12][88] = 70;
    out.transformMaskMappingLocalToWorld[12][89] = 86;
    out.transformMaskMappingLocalToWorld[12][90] = 102;
    out.transformMaskMappingLocalToWorld[12][91] = 118;
    out.transformMaskMappingLocalToWorld[12][92] = 71;
    out.transformMaskMappingLocalToWorld[12][93] = 87;
    out.transformMaskMappingLocalToWorld[12][94] = 103;
    out.transformMaskMappingLocalToWorld[12][95] = 119;
    out.transformMaskMappingLocalToWorld[12][96] = 132;
    out.transformMaskMappingLocalToWorld[12][97] = 148;
    out.transformMaskMappingLocalToWorld[12][98] = 164;
    out.transformMaskMappingLocalToWorld[12][99] = 180;
    out.transformMaskMappingLocalToWorld[12][100] = 133;
    out.transformMaskMappingLocalToWorld[12][101] = 149;
    out.transformMaskMappingLocalToWorld[12][102] = 165;
    out.transformMaskMappingLocalToWorld[12][103] = 181;
    out.transformMaskMappingLocalToWorld[12][104] = 134;
    out.transformMaskMappingLocalToWorld[12][105] = 150;
    out.transformMaskMappingLocalToWorld[12][106] = 166;
    out.transformMaskMappingLocalToWorld[12][107] = 182;
    out.transformMaskMappingLocalToWorld[12][108] = 135;
    out.transformMaskMappingLocalToWorld[12][109] = 151;
    out.transformMaskMappingLocalToWorld[12][110] = 167;
    out.transformMaskMappingLocalToWorld[12][111] = 183;
    out.transformMaskMappingLocalToWorld[12][112] = 196;
    out.transformMaskMappingLocalToWorld[12][113] = 212;
    out.transformMaskMappingLocalToWorld[12][114] = 228;
    out.transformMaskMappingLocalToWorld[12][115] = 244;
    out.transformMaskMappingLocalToWorld[12][116] = 197;
    out.transformMaskMappingLocalToWorld[12][117] = 213;
    out.transformMaskMappingLocalToWorld[12][118] = 229;
    out.transformMaskMappingLocalToWorld[12][119] = 245;
    out.transformMaskMappingLocalToWorld[12][120] = 198;
    out.transformMaskMappingLocalToWorld[12][121] = 214;
    out.transformMaskMappingLocalToWorld[12][122] = 230;
    out.transformMaskMappingLocalToWorld[12][123] = 246;
    out.transformMaskMappingLocalToWorld[12][124] = 199;
    out.transformMaskMappingLocalToWorld[12][125] = 215;
    out.transformMaskMappingLocalToWorld[12][126] = 231;
    out.transformMaskMappingLocalToWorld[12][127] = 247;
    out.transformMaskMappingLocalToWorld[12][128] = 8;
    out.transformMaskMappingLocalToWorld[12][129] = 24;
    out.transformMaskMappingLocalToWorld[12][130] = 40;
    out.transformMaskMappingLocalToWorld[12][131] = 56;
    out.transformMaskMappingLocalToWorld[12][132] = 9;
    out.transformMaskMappingLocalToWorld[12][133] = 25;
    out.transformMaskMappingLocalToWorld[12][134] = 41;
    out.transformMaskMappingLocalToWorld[12][135] = 57;
    out.transformMaskMappingLocalToWorld[12][136] = 10;
    out.transformMaskMappingLocalToWorld[12][137] = 26;
    out.transformMaskMappingLocalToWorld[12][138] = 42;
    out.transformMaskMappingLocalToWorld[12][139] = 58;
    out.transformMaskMappingLocalToWorld[12][140] = 11;
    out.transformMaskMappingLocalToWorld[12][141] = 27;
    out.transformMaskMappingLocalToWorld[12][142] = 43;
    out.transformMaskMappingLocalToWorld[12][143] = 59;
    out.transformMaskMappingLocalToWorld[12][144] = 72;
    out.transformMaskMappingLocalToWorld[12][145] = 88;
    out.transformMaskMappingLocalToWorld[12][146] = 104;
    out.transformMaskMappingLocalToWorld[12][147] = 120;
    out.transformMaskMappingLocalToWorld[12][148] = 73;
    out.transformMaskMappingLocalToWorld[12][149] = 89;
    out.transformMaskMappingLocalToWorld[12][150] = 105;
    out.transformMaskMappingLocalToWorld[12][151] = 121;
    out.transformMaskMappingLocalToWorld[12][152] = 74;
    out.transformMaskMappingLocalToWorld[12][153] = 90;
    out.transformMaskMappingLocalToWorld[12][154] = 106;
    out.transformMaskMappingLocalToWorld[12][155] = 122;
    out.transformMaskMappingLocalToWorld[12][156] = 75;
    out.transformMaskMappingLocalToWorld[12][157] = 91;
    out.transformMaskMappingLocalToWorld[12][158] = 107;
    out.transformMaskMappingLocalToWorld[12][159] = 123;
    out.transformMaskMappingLocalToWorld[12][160] = 136;
    out.transformMaskMappingLocalToWorld[12][161] = 152;
    out.transformMaskMappingLocalToWorld[12][162] = 168;
    out.transformMaskMappingLocalToWorld[12][163] = 184;
    out.transformMaskMappingLocalToWorld[12][164] = 137;
    out.transformMaskMappingLocalToWorld[12][165] = 153;
    out.transformMaskMappingLocalToWorld[12][166] = 169;
    out.transformMaskMappingLocalToWorld[12][167] = 185;
    out.transformMaskMappingLocalToWorld[12][168] = 138;
    out.transformMaskMappingLocalToWorld[12][169] = 154;
    out.transformMaskMappingLocalToWorld[12][170] = 170;
    out.transformMaskMappingLocalToWorld[12][171] = 186;
    out.transformMaskMappingLocalToWorld[12][172] = 139;
    out.transformMaskMappingLocalToWorld[12][173] = 155;
    out.transformMaskMappingLocalToWorld[12][174] = 171;
    out.transformMaskMappingLocalToWorld[12][175] = 187;
    out.transformMaskMappingLocalToWorld[12][176] = 200;
    out.transformMaskMappingLocalToWorld[12][177] = 216;
    out.transformMaskMappingLocalToWorld[12][178] = 232;
    out.transformMaskMappingLocalToWorld[12][179] = 248;
    out.transformMaskMappingLocalToWorld[12][180] = 201;
    out.transformMaskMappingLocalToWorld[12][181] = 217;
    out.transformMaskMappingLocalToWorld[12][182] = 233;
    out.transformMaskMappingLocalToWorld[12][183] = 249;
    out.transformMaskMappingLocalToWorld[12][184] = 202;
    out.transformMaskMappingLocalToWorld[12][185] = 218;
    out.transformMaskMappingLocalToWorld[12][186] = 234;
    out.transformMaskMappingLocalToWorld[12][187] = 250;
    out.transformMaskMappingLocalToWorld[12][188] = 203;
    out.transformMaskMappingLocalToWorld[12][189] = 219;
    out.transformMaskMappingLocalToWorld[12][190] = 235;
    out.transformMaskMappingLocalToWorld[12][191] = 251;
    out.transformMaskMappingLocalToWorld[12][192] = 12;
    out.transformMaskMappingLocalToWorld[12][193] = 28;
    out.transformMaskMappingLocalToWorld[12][194] = 44;
    out.transformMaskMappingLocalToWorld[12][195] = 60;
    out.transformMaskMappingLocalToWorld[12][196] = 13;
    out.transformMaskMappingLocalToWorld[12][197] = 29;
    out.transformMaskMappingLocalToWorld[12][198] = 45;
    out.transformMaskMappingLocalToWorld[12][199] = 61;
    out.transformMaskMappingLocalToWorld[12][200] = 14;
    out.transformMaskMappingLocalToWorld[12][201] = 30;
    out.transformMaskMappingLocalToWorld[12][202] = 46;
    out.transformMaskMappingLocalToWorld[12][203] = 62;
    out.transformMaskMappingLocalToWorld[12][204] = 15;
    out.transformMaskMappingLocalToWorld[12][205] = 31;
    out.transformMaskMappingLocalToWorld[12][206] = 47;
    out.transformMaskMappingLocalToWorld[12][207] = 63;
    out.transformMaskMappingLocalToWorld[12][208] = 76;
    out.transformMaskMappingLocalToWorld[12][209] = 92;
    out.transformMaskMappingLocalToWorld[12][210] = 108;
    out.transformMaskMappingLocalToWorld[12][211] = 124;
    out.transformMaskMappingLocalToWorld[12][212] = 77;
    out.transformMaskMappingLocalToWorld[12][213] = 93;
    out.transformMaskMappingLocalToWorld[12][214] = 109;
    out.transformMaskMappingLocalToWorld[12][215] = 125;
    out.transformMaskMappingLocalToWorld[12][216] = 78;
    out.transformMaskMappingLocalToWorld[12][217] = 94;
    out.transformMaskMappingLocalToWorld[12][218] = 110;
    out.transformMaskMappingLocalToWorld[12][219] = 126;
    out.transformMaskMappingLocalToWorld[12][220] = 79;
    out.transformMaskMappingLocalToWorld[12][221] = 95;
    out.transformMaskMappingLocalToWorld[12][222] = 111;
    out.transformMaskMappingLocalToWorld[12][223] = 127;
    out.transformMaskMappingLocalToWorld[12][224] = 140;
    out.transformMaskMappingLocalToWorld[12][225] = 156;
    out.transformMaskMappingLocalToWorld[12][226] = 172;
    out.transformMaskMappingLocalToWorld[12][227] = 188;
    out.transformMaskMappingLocalToWorld[12][228] = 141;
    out.transformMaskMappingLocalToWorld[12][229] = 157;
    out.transformMaskMappingLocalToWorld[12][230] = 173;
    out.transformMaskMappingLocalToWorld[12][231] = 189;
    out.transformMaskMappingLocalToWorld[12][232] = 142;
    out.transformMaskMappingLocalToWorld[12][233] = 158;
    out.transformMaskMappingLocalToWorld[12][234] = 174;
    out.transformMaskMappingLocalToWorld[12][235] = 190;
    out.transformMaskMappingLocalToWorld[12][236] = 143;
    out.transformMaskMappingLocalToWorld[12][237] = 159;
    out.transformMaskMappingLocalToWorld[12][238] = 175;
    out.transformMaskMappingLocalToWorld[12][239] = 191;
    out.transformMaskMappingLocalToWorld[12][240] = 204;
    out.transformMaskMappingLocalToWorld[12][241] = 220;
    out.transformMaskMappingLocalToWorld[12][242] = 236;
    out.transformMaskMappingLocalToWorld[12][243] = 252;
    out.transformMaskMappingLocalToWorld[12][244] = 205;
    out.transformMaskMappingLocalToWorld[12][245] = 221;
    out.transformMaskMappingLocalToWorld[12][246] = 237;
    out.transformMaskMappingLocalToWorld[12][247] = 253;
    out.transformMaskMappingLocalToWorld[12][248] = 206;
    out.transformMaskMappingLocalToWorld[12][249] = 222;
    out.transformMaskMappingLocalToWorld[12][250] = 238;
    out.transformMaskMappingLocalToWorld[12][251] = 254;
    out.transformMaskMappingLocalToWorld[12][252] = 207;
    out.transformMaskMappingLocalToWorld[12][253] = 223;
    out.transformMaskMappingLocalToWorld[12][254] = 239;
    out.transformMaskMappingLocalToWorld[12][255] = 255;
    out.transformMaskMappingLocalToWorld[13][0] = 0;
    out.transformMaskMappingLocalToWorld[13][1] = 32;
    out.transformMaskMappingLocalToWorld[13][2] = 16;
    out.transformMaskMappingLocalToWorld[13][3] = 48;
    out.transformMaskMappingLocalToWorld[13][4] = 2;
    out.transformMaskMappingLocalToWorld[13][5] = 34;
    out.transformMaskMappingLocalToWorld[13][6] = 18;
    out.transformMaskMappingLocalToWorld[13][7] = 50;
    out.transformMaskMappingLocalToWorld[13][8] = 1;
    out.transformMaskMappingLocalToWorld[13][9] = 33;
    out.transformMaskMappingLocalToWorld[13][10] = 17;
    out.transformMaskMappingLocalToWorld[13][11] = 49;
    out.transformMaskMappingLocalToWorld[13][12] = 3;
    out.transformMaskMappingLocalToWorld[13][13] = 35;
    out.transformMaskMappingLocalToWorld[13][14] = 19;
    out.transformMaskMappingLocalToWorld[13][15] = 51;
    out.transformMaskMappingLocalToWorld[13][16] = 128;
    out.transformMaskMappingLocalToWorld[13][17] = 160;
    out.transformMaskMappingLocalToWorld[13][18] = 144;
    out.transformMaskMappingLocalToWorld[13][19] = 176;
    out.transformMaskMappingLocalToWorld[13][20] = 130;
    out.transformMaskMappingLocalToWorld[13][21] = 162;
    out.transformMaskMappingLocalToWorld[13][22] = 146;
    out.transformMaskMappingLocalToWorld[13][23] = 178;
    out.transformMaskMappingLocalToWorld[13][24] = 129;
    out.transformMaskMappingLocalToWorld[13][25] = 161;
    out.transformMaskMappingLocalToWorld[13][26] = 145;
    out.transformMaskMappingLocalToWorld[13][27] = 177;
    out.transformMaskMappingLocalToWorld[13][28] = 131;
    out.transformMaskMappingLocalToWorld[13][29] = 163;
    out.transformMaskMappingLocalToWorld[13][30] = 147;
    out.transformMaskMappingLocalToWorld[13][31] = 179;
    out.transformMaskMappingLocalToWorld[13][32] = 64;
    out.transformMaskMappingLocalToWorld[13][33] = 96;
    out.transformMaskMappingLocalToWorld[13][34] = 80;
    out.transformMaskMappingLocalToWorld[13][35] = 112;
    out.transformMaskMappingLocalToWorld[13][36] = 66;
    out.transformMaskMappingLocalToWorld[13][37] = 98;
    out.transformMaskMappingLocalToWorld[13][38] = 82;
    out.transformMaskMappingLocalToWorld[13][39] = 114;
    out.transformMaskMappingLocalToWorld[13][40] = 65;
    out.transformMaskMappingLocalToWorld[13][41] = 97;
    out.transformMaskMappingLocalToWorld[13][42] = 81;
    out.transformMaskMappingLocalToWorld[13][43] = 113;
    out.transformMaskMappingLocalToWorld[13][44] = 67;
    out.transformMaskMappingLocalToWorld[13][45] = 99;
    out.transformMaskMappingLocalToWorld[13][46] = 83;
    out.transformMaskMappingLocalToWorld[13][47] = 115;
    out.transformMaskMappingLocalToWorld[13][48] = 192;
    out.transformMaskMappingLocalToWorld[13][49] = 224;
    out.transformMaskMappingLocalToWorld[13][50] = 208;
    out.transformMaskMappingLocalToWorld[13][51] = 240;
    out.transformMaskMappingLocalToWorld[13][52] = 194;
    out.transformMaskMappingLocalToWorld[13][53] = 226;
    out.transformMaskMappingLocalToWorld[13][54] = 210;
    out.transformMaskMappingLocalToWorld[13][55] = 242;
    out.transformMaskMappingLocalToWorld[13][56] = 193;
    out.transformMaskMappingLocalToWorld[13][57] = 225;
    out.transformMaskMappingLocalToWorld[13][58] = 209;
    out.transformMaskMappingLocalToWorld[13][59] = 241;
    out.transformMaskMappingLocalToWorld[13][60] = 195;
    out.transformMaskMappingLocalToWorld[13][61] = 227;
    out.transformMaskMappingLocalToWorld[13][62] = 211;
    out.transformMaskMappingLocalToWorld[13][63] = 243;
    out.transformMaskMappingLocalToWorld[13][64] = 8;
    out.transformMaskMappingLocalToWorld[13][65] = 40;
    out.transformMaskMappingLocalToWorld[13][66] = 24;
    out.transformMaskMappingLocalToWorld[13][67] = 56;
    out.transformMaskMappingLocalToWorld[13][68] = 10;
    out.transformMaskMappingLocalToWorld[13][69] = 42;
    out.transformMaskMappingLocalToWorld[13][70] = 26;
    out.transformMaskMappingLocalToWorld[13][71] = 58;
    out.transformMaskMappingLocalToWorld[13][72] = 9;
    out.transformMaskMappingLocalToWorld[13][73] = 41;
    out.transformMaskMappingLocalToWorld[13][74] = 25;
    out.transformMaskMappingLocalToWorld[13][75] = 57;
    out.transformMaskMappingLocalToWorld[13][76] = 11;
    out.transformMaskMappingLocalToWorld[13][77] = 43;
    out.transformMaskMappingLocalToWorld[13][78] = 27;
    out.transformMaskMappingLocalToWorld[13][79] = 59;
    out.transformMaskMappingLocalToWorld[13][80] = 136;
    out.transformMaskMappingLocalToWorld[13][81] = 168;
    out.transformMaskMappingLocalToWorld[13][82] = 152;
    out.transformMaskMappingLocalToWorld[13][83] = 184;
    out.transformMaskMappingLocalToWorld[13][84] = 138;
    out.transformMaskMappingLocalToWorld[13][85] = 170;
    out.transformMaskMappingLocalToWorld[13][86] = 154;
    out.transformMaskMappingLocalToWorld[13][87] = 186;
    out.transformMaskMappingLocalToWorld[13][88] = 137;
    out.transformMaskMappingLocalToWorld[13][89] = 169;
    out.transformMaskMappingLocalToWorld[13][90] = 153;
    out.transformMaskMappingLocalToWorld[13][91] = 185;
    out.transformMaskMappingLocalToWorld[13][92] = 139;
    out.transformMaskMappingLocalToWorld[13][93] = 171;
    out.transformMaskMappingLocalToWorld[13][94] = 155;
    out.transformMaskMappingLocalToWorld[13][95] = 187;
    out.transformMaskMappingLocalToWorld[13][96] = 72;
    out.transformMaskMappingLocalToWorld[13][97] = 104;
    out.transformMaskMappingLocalToWorld[13][98] = 88;
    out.transformMaskMappingLocalToWorld[13][99] = 120;
    out.transformMaskMappingLocalToWorld[13][100] = 74;
    out.transformMaskMappingLocalToWorld[13][101] = 106;
    out.transformMaskMappingLocalToWorld[13][102] = 90;
    out.transformMaskMappingLocalToWorld[13][103] = 122;
    out.transformMaskMappingLocalToWorld[13][104] = 73;
    out.transformMaskMappingLocalToWorld[13][105] = 105;
    out.transformMaskMappingLocalToWorld[13][106] = 89;
    out.transformMaskMappingLocalToWorld[13][107] = 121;
    out.transformMaskMappingLocalToWorld[13][108] = 75;
    out.transformMaskMappingLocalToWorld[13][109] = 107;
    out.transformMaskMappingLocalToWorld[13][110] = 91;
    out.transformMaskMappingLocalToWorld[13][111] = 123;
    out.transformMaskMappingLocalToWorld[13][112] = 200;
    out.transformMaskMappingLocalToWorld[13][113] = 232;
    out.transformMaskMappingLocalToWorld[13][114] = 216;
    out.transformMaskMappingLocalToWorld[13][115] = 248;
    out.transformMaskMappingLocalToWorld[13][116] = 202;
    out.transformMaskMappingLocalToWorld[13][117] = 234;
    out.transformMaskMappingLocalToWorld[13][118] = 218;
    out.transformMaskMappingLocalToWorld[13][119] = 250;
    out.transformMaskMappingLocalToWorld[13][120] = 201;
    out.transformMaskMappingLocalToWorld[13][121] = 233;
    out.transformMaskMappingLocalToWorld[13][122] = 217;
    out.transformMaskMappingLocalToWorld[13][123] = 249;
    out.transformMaskMappingLocalToWorld[13][124] = 203;
    out.transformMaskMappingLocalToWorld[13][125] = 235;
    out.transformMaskMappingLocalToWorld[13][126] = 219;
    out.transformMaskMappingLocalToWorld[13][127] = 251;
    out.transformMaskMappingLocalToWorld[13][128] = 4;
    out.transformMaskMappingLocalToWorld[13][129] = 36;
    out.transformMaskMappingLocalToWorld[13][130] = 20;
    out.transformMaskMappingLocalToWorld[13][131] = 52;
    out.transformMaskMappingLocalToWorld[13][132] = 6;
    out.transformMaskMappingLocalToWorld[13][133] = 38;
    out.transformMaskMappingLocalToWorld[13][134] = 22;
    out.transformMaskMappingLocalToWorld[13][135] = 54;
    out.transformMaskMappingLocalToWorld[13][136] = 5;
    out.transformMaskMappingLocalToWorld[13][137] = 37;
    out.transformMaskMappingLocalToWorld[13][138] = 21;
    out.transformMaskMappingLocalToWorld[13][139] = 53;
    out.transformMaskMappingLocalToWorld[13][140] = 7;
    out.transformMaskMappingLocalToWorld[13][141] = 39;
    out.transformMaskMappingLocalToWorld[13][142] = 23;
    out.transformMaskMappingLocalToWorld[13][143] = 55;
    out.transformMaskMappingLocalToWorld[13][144] = 132;
    out.transformMaskMappingLocalToWorld[13][145] = 164;
    out.transformMaskMappingLocalToWorld[13][146] = 148;
    out.transformMaskMappingLocalToWorld[13][147] = 180;
    out.transformMaskMappingLocalToWorld[13][148] = 134;
    out.transformMaskMappingLocalToWorld[13][149] = 166;
    out.transformMaskMappingLocalToWorld[13][150] = 150;
    out.transformMaskMappingLocalToWorld[13][151] = 182;
    out.transformMaskMappingLocalToWorld[13][152] = 133;
    out.transformMaskMappingLocalToWorld[13][153] = 165;
    out.transformMaskMappingLocalToWorld[13][154] = 149;
    out.transformMaskMappingLocalToWorld[13][155] = 181;
    out.transformMaskMappingLocalToWorld[13][156] = 135;
    out.transformMaskMappingLocalToWorld[13][157] = 167;
    out.transformMaskMappingLocalToWorld[13][158] = 151;
    out.transformMaskMappingLocalToWorld[13][159] = 183;
    out.transformMaskMappingLocalToWorld[13][160] = 68;
    out.transformMaskMappingLocalToWorld[13][161] = 100;
    out.transformMaskMappingLocalToWorld[13][162] = 84;
    out.transformMaskMappingLocalToWorld[13][163] = 116;
    out.transformMaskMappingLocalToWorld[13][164] = 70;
    out.transformMaskMappingLocalToWorld[13][165] = 102;
    out.transformMaskMappingLocalToWorld[13][166] = 86;
    out.transformMaskMappingLocalToWorld[13][167] = 118;
    out.transformMaskMappingLocalToWorld[13][168] = 69;
    out.transformMaskMappingLocalToWorld[13][169] = 101;
    out.transformMaskMappingLocalToWorld[13][170] = 85;
    out.transformMaskMappingLocalToWorld[13][171] = 117;
    out.transformMaskMappingLocalToWorld[13][172] = 71;
    out.transformMaskMappingLocalToWorld[13][173] = 103;
    out.transformMaskMappingLocalToWorld[13][174] = 87;
    out.transformMaskMappingLocalToWorld[13][175] = 119;
    out.transformMaskMappingLocalToWorld[13][176] = 196;
    out.transformMaskMappingLocalToWorld[13][177] = 228;
    out.transformMaskMappingLocalToWorld[13][178] = 212;
    out.transformMaskMappingLocalToWorld[13][179] = 244;
    out.transformMaskMappingLocalToWorld[13][180] = 198;
    out.transformMaskMappingLocalToWorld[13][181] = 230;
    out.transformMaskMappingLocalToWorld[13][182] = 214;
    out.transformMaskMappingLocalToWorld[13][183] = 246;
    out.transformMaskMappingLocalToWorld[13][184] = 197;
    out.transformMaskMappingLocalToWorld[13][185] = 229;
    out.transformMaskMappingLocalToWorld[13][186] = 213;
    out.transformMaskMappingLocalToWorld[13][187] = 245;
    out.transformMaskMappingLocalToWorld[13][188] = 199;
    out.transformMaskMappingLocalToWorld[13][189] = 231;
    out.transformMaskMappingLocalToWorld[13][190] = 215;
    out.transformMaskMappingLocalToWorld[13][191] = 247;
    out.transformMaskMappingLocalToWorld[13][192] = 12;
    out.transformMaskMappingLocalToWorld[13][193] = 44;
    out.transformMaskMappingLocalToWorld[13][194] = 28;
    out.transformMaskMappingLocalToWorld[13][195] = 60;
    out.transformMaskMappingLocalToWorld[13][196] = 14;
    out.transformMaskMappingLocalToWorld[13][197] = 46;
    out.transformMaskMappingLocalToWorld[13][198] = 30;
    out.transformMaskMappingLocalToWorld[13][199] = 62;
    out.transformMaskMappingLocalToWorld[13][200] = 13;
    out.transformMaskMappingLocalToWorld[13][201] = 45;
    out.transformMaskMappingLocalToWorld[13][202] = 29;
    out.transformMaskMappingLocalToWorld[13][203] = 61;
    out.transformMaskMappingLocalToWorld[13][204] = 15;
    out.transformMaskMappingLocalToWorld[13][205] = 47;
    out.transformMaskMappingLocalToWorld[13][206] = 31;
    out.transformMaskMappingLocalToWorld[13][207] = 63;
    out.transformMaskMappingLocalToWorld[13][208] = 140;
    out.transformMaskMappingLocalToWorld[13][209] = 172;
    out.transformMaskMappingLocalToWorld[13][210] = 156;
    out.transformMaskMappingLocalToWorld[13][211] = 188;
    out.transformMaskMappingLocalToWorld[13][212] = 142;
    out.transformMaskMappingLocalToWorld[13][213] = 174;
    out.transformMaskMappingLocalToWorld[13][214] = 158;
    out.transformMaskMappingLocalToWorld[13][215] = 190;
    out.transformMaskMappingLocalToWorld[13][216] = 141;
    out.transformMaskMappingLocalToWorld[13][217] = 173;
    out.transformMaskMappingLocalToWorld[13][218] = 157;
    out.transformMaskMappingLocalToWorld[13][219] = 189;
    out.transformMaskMappingLocalToWorld[13][220] = 143;
    out.transformMaskMappingLocalToWorld[13][221] = 175;
    out.transformMaskMappingLocalToWorld[13][222] = 159;
    out.transformMaskMappingLocalToWorld[13][223] = 191;
    out.transformMaskMappingLocalToWorld[13][224] = 76;
    out.transformMaskMappingLocalToWorld[13][225] = 108;
    out.transformMaskMappingLocalToWorld[13][226] = 92;
    out.transformMaskMappingLocalToWorld[13][227] = 124;
    out.transformMaskMappingLocalToWorld[13][228] = 78;
    out.transformMaskMappingLocalToWorld[13][229] = 110;
    out.transformMaskMappingLocalToWorld[13][230] = 94;
    out.transformMaskMappingLocalToWorld[13][231] = 126;
    out.transformMaskMappingLocalToWorld[13][232] = 77;
    out.transformMaskMappingLocalToWorld[13][233] = 109;
    out.transformMaskMappingLocalToWorld[13][234] = 93;
    out.transformMaskMappingLocalToWorld[13][235] = 125;
    out.transformMaskMappingLocalToWorld[13][236] = 79;
    out.transformMaskMappingLocalToWorld[13][237] = 111;
    out.transformMaskMappingLocalToWorld[13][238] = 95;
    out.transformMaskMappingLocalToWorld[13][239] = 127;
    out.transformMaskMappingLocalToWorld[13][240] = 204;
    out.transformMaskMappingLocalToWorld[13][241] = 236;
    out.transformMaskMappingLocalToWorld[13][242] = 220;
    out.transformMaskMappingLocalToWorld[13][243] = 252;
    out.transformMaskMappingLocalToWorld[13][244] = 206;
    out.transformMaskMappingLocalToWorld[13][245] = 238;
    out.transformMaskMappingLocalToWorld[13][246] = 222;
    out.transformMaskMappingLocalToWorld[13][247] = 254;
    out.transformMaskMappingLocalToWorld[13][248] = 205;
    out.transformMaskMappingLocalToWorld[13][249] = 237;
    out.transformMaskMappingLocalToWorld[13][250] = 221;
    out.transformMaskMappingLocalToWorld[13][251] = 253;
    out.transformMaskMappingLocalToWorld[13][252] = 207;
    out.transformMaskMappingLocalToWorld[13][253] = 239;
    out.transformMaskMappingLocalToWorld[13][254] = 223;
    out.transformMaskMappingLocalToWorld[13][255] = 255;
    out.transformMaskMappingLocalToWorld[14][0] = 0;
    out.transformMaskMappingLocalToWorld[14][1] = 64;
    out.transformMaskMappingLocalToWorld[14][2] = 128;
    out.transformMaskMappingLocalToWorld[14][3] = 192;
    out.transformMaskMappingLocalToWorld[14][4] = 4;
    out.transformMaskMappingLocalToWorld[14][5] = 68;
    out.transformMaskMappingLocalToWorld[14][6] = 132;
    out.transformMaskMappingLocalToWorld[14][7] = 196;
    out.transformMaskMappingLocalToWorld[14][8] = 8;
    out.transformMaskMappingLocalToWorld[14][9] = 72;
    out.transformMaskMappingLocalToWorld[14][10] = 136;
    out.transformMaskMappingLocalToWorld[14][11] = 200;
    out.transformMaskMappingLocalToWorld[14][12] = 12;
    out.transformMaskMappingLocalToWorld[14][13] = 76;
    out.transformMaskMappingLocalToWorld[14][14] = 140;
    out.transformMaskMappingLocalToWorld[14][15] = 204;
    out.transformMaskMappingLocalToWorld[14][16] = 16;
    out.transformMaskMappingLocalToWorld[14][17] = 80;
    out.transformMaskMappingLocalToWorld[14][18] = 144;
    out.transformMaskMappingLocalToWorld[14][19] = 208;
    out.transformMaskMappingLocalToWorld[14][20] = 20;
    out.transformMaskMappingLocalToWorld[14][21] = 84;
    out.transformMaskMappingLocalToWorld[14][22] = 148;
    out.transformMaskMappingLocalToWorld[14][23] = 212;
    out.transformMaskMappingLocalToWorld[14][24] = 24;
    out.transformMaskMappingLocalToWorld[14][25] = 88;
    out.transformMaskMappingLocalToWorld[14][26] = 152;
    out.transformMaskMappingLocalToWorld[14][27] = 216;
    out.transformMaskMappingLocalToWorld[14][28] = 28;
    out.transformMaskMappingLocalToWorld[14][29] = 92;
    out.transformMaskMappingLocalToWorld[14][30] = 156;
    out.transformMaskMappingLocalToWorld[14][31] = 220;
    out.transformMaskMappingLocalToWorld[14][32] = 32;
    out.transformMaskMappingLocalToWorld[14][33] = 96;
    out.transformMaskMappingLocalToWorld[14][34] = 160;
    out.transformMaskMappingLocalToWorld[14][35] = 224;
    out.transformMaskMappingLocalToWorld[14][36] = 36;
    out.transformMaskMappingLocalToWorld[14][37] = 100;
    out.transformMaskMappingLocalToWorld[14][38] = 164;
    out.transformMaskMappingLocalToWorld[14][39] = 228;
    out.transformMaskMappingLocalToWorld[14][40] = 40;
    out.transformMaskMappingLocalToWorld[14][41] = 104;
    out.transformMaskMappingLocalToWorld[14][42] = 168;
    out.transformMaskMappingLocalToWorld[14][43] = 232;
    out.transformMaskMappingLocalToWorld[14][44] = 44;
    out.transformMaskMappingLocalToWorld[14][45] = 108;
    out.transformMaskMappingLocalToWorld[14][46] = 172;
    out.transformMaskMappingLocalToWorld[14][47] = 236;
    out.transformMaskMappingLocalToWorld[14][48] = 48;
    out.transformMaskMappingLocalToWorld[14][49] = 112;
    out.transformMaskMappingLocalToWorld[14][50] = 176;
    out.transformMaskMappingLocalToWorld[14][51] = 240;
    out.transformMaskMappingLocalToWorld[14][52] = 52;
    out.transformMaskMappingLocalToWorld[14][53] = 116;
    out.transformMaskMappingLocalToWorld[14][54] = 180;
    out.transformMaskMappingLocalToWorld[14][55] = 244;
    out.transformMaskMappingLocalToWorld[14][56] = 56;
    out.transformMaskMappingLocalToWorld[14][57] = 120;
    out.transformMaskMappingLocalToWorld[14][58] = 184;
    out.transformMaskMappingLocalToWorld[14][59] = 248;
    out.transformMaskMappingLocalToWorld[14][60] = 60;
    out.transformMaskMappingLocalToWorld[14][61] = 124;
    out.transformMaskMappingLocalToWorld[14][62] = 188;
    out.transformMaskMappingLocalToWorld[14][63] = 252;
    out.transformMaskMappingLocalToWorld[14][64] = 1;
    out.transformMaskMappingLocalToWorld[14][65] = 65;
    out.transformMaskMappingLocalToWorld[14][66] = 129;
    out.transformMaskMappingLocalToWorld[14][67] = 193;
    out.transformMaskMappingLocalToWorld[14][68] = 5;
    out.transformMaskMappingLocalToWorld[14][69] = 69;
    out.transformMaskMappingLocalToWorld[14][70] = 133;
    out.transformMaskMappingLocalToWorld[14][71] = 197;
    out.transformMaskMappingLocalToWorld[14][72] = 9;
    out.transformMaskMappingLocalToWorld[14][73] = 73;
    out.transformMaskMappingLocalToWorld[14][74] = 137;
    out.transformMaskMappingLocalToWorld[14][75] = 201;
    out.transformMaskMappingLocalToWorld[14][76] = 13;
    out.transformMaskMappingLocalToWorld[14][77] = 77;
    out.transformMaskMappingLocalToWorld[14][78] = 141;
    out.transformMaskMappingLocalToWorld[14][79] = 205;
    out.transformMaskMappingLocalToWorld[14][80] = 17;
    out.transformMaskMappingLocalToWorld[14][81] = 81;
    out.transformMaskMappingLocalToWorld[14][82] = 145;
    out.transformMaskMappingLocalToWorld[14][83] = 209;
    out.transformMaskMappingLocalToWorld[14][84] = 21;
    out.transformMaskMappingLocalToWorld[14][85] = 85;
    out.transformMaskMappingLocalToWorld[14][86] = 149;
    out.transformMaskMappingLocalToWorld[14][87] = 213;
    out.transformMaskMappingLocalToWorld[14][88] = 25;
    out.transformMaskMappingLocalToWorld[14][89] = 89;
    out.transformMaskMappingLocalToWorld[14][90] = 153;
    out.transformMaskMappingLocalToWorld[14][91] = 217;
    out.transformMaskMappingLocalToWorld[14][92] = 29;
    out.transformMaskMappingLocalToWorld[14][93] = 93;
    out.transformMaskMappingLocalToWorld[14][94] = 157;
    out.transformMaskMappingLocalToWorld[14][95] = 221;
    out.transformMaskMappingLocalToWorld[14][96] = 33;
    out.transformMaskMappingLocalToWorld[14][97] = 97;
    out.transformMaskMappingLocalToWorld[14][98] = 161;
    out.transformMaskMappingLocalToWorld[14][99] = 225;
    out.transformMaskMappingLocalToWorld[14][100] = 37;
    out.transformMaskMappingLocalToWorld[14][101] = 101;
    out.transformMaskMappingLocalToWorld[14][102] = 165;
    out.transformMaskMappingLocalToWorld[14][103] = 229;
    out.transformMaskMappingLocalToWorld[14][104] = 41;
    out.transformMaskMappingLocalToWorld[14][105] = 105;
    out.transformMaskMappingLocalToWorld[14][106] = 169;
    out.transformMaskMappingLocalToWorld[14][107] = 233;
    out.transformMaskMappingLocalToWorld[14][108] = 45;
    out.transformMaskMappingLocalToWorld[14][109] = 109;
    out.transformMaskMappingLocalToWorld[14][110] = 173;
    out.transformMaskMappingLocalToWorld[14][111] = 237;
    out.transformMaskMappingLocalToWorld[14][112] = 49;
    out.transformMaskMappingLocalToWorld[14][113] = 113;
    out.transformMaskMappingLocalToWorld[14][114] = 177;
    out.transformMaskMappingLocalToWorld[14][115] = 241;
    out.transformMaskMappingLocalToWorld[14][116] = 53;
    out.transformMaskMappingLocalToWorld[14][117] = 117;
    out.transformMaskMappingLocalToWorld[14][118] = 181;
    out.transformMaskMappingLocalToWorld[14][119] = 245;
    out.transformMaskMappingLocalToWorld[14][120] = 57;
    out.transformMaskMappingLocalToWorld[14][121] = 121;
    out.transformMaskMappingLocalToWorld[14][122] = 185;
    out.transformMaskMappingLocalToWorld[14][123] = 249;
    out.transformMaskMappingLocalToWorld[14][124] = 61;
    out.transformMaskMappingLocalToWorld[14][125] = 125;
    out.transformMaskMappingLocalToWorld[14][126] = 189;
    out.transformMaskMappingLocalToWorld[14][127] = 253;
    out.transformMaskMappingLocalToWorld[14][128] = 2;
    out.transformMaskMappingLocalToWorld[14][129] = 66;
    out.transformMaskMappingLocalToWorld[14][130] = 130;
    out.transformMaskMappingLocalToWorld[14][131] = 194;
    out.transformMaskMappingLocalToWorld[14][132] = 6;
    out.transformMaskMappingLocalToWorld[14][133] = 70;
    out.transformMaskMappingLocalToWorld[14][134] = 134;
    out.transformMaskMappingLocalToWorld[14][135] = 198;
    out.transformMaskMappingLocalToWorld[14][136] = 10;
    out.transformMaskMappingLocalToWorld[14][137] = 74;
    out.transformMaskMappingLocalToWorld[14][138] = 138;
    out.transformMaskMappingLocalToWorld[14][139] = 202;
    out.transformMaskMappingLocalToWorld[14][140] = 14;
    out.transformMaskMappingLocalToWorld[14][141] = 78;
    out.transformMaskMappingLocalToWorld[14][142] = 142;
    out.transformMaskMappingLocalToWorld[14][143] = 206;
    out.transformMaskMappingLocalToWorld[14][144] = 18;
    out.transformMaskMappingLocalToWorld[14][145] = 82;
    out.transformMaskMappingLocalToWorld[14][146] = 146;
    out.transformMaskMappingLocalToWorld[14][147] = 210;
    out.transformMaskMappingLocalToWorld[14][148] = 22;
    out.transformMaskMappingLocalToWorld[14][149] = 86;
    out.transformMaskMappingLocalToWorld[14][150] = 150;
    out.transformMaskMappingLocalToWorld[14][151] = 214;
    out.transformMaskMappingLocalToWorld[14][152] = 26;
    out.transformMaskMappingLocalToWorld[14][153] = 90;
    out.transformMaskMappingLocalToWorld[14][154] = 154;
    out.transformMaskMappingLocalToWorld[14][155] = 218;
    out.transformMaskMappingLocalToWorld[14][156] = 30;
    out.transformMaskMappingLocalToWorld[14][157] = 94;
    out.transformMaskMappingLocalToWorld[14][158] = 158;
    out.transformMaskMappingLocalToWorld[14][159] = 222;
    out.transformMaskMappingLocalToWorld[14][160] = 34;
    out.transformMaskMappingLocalToWorld[14][161] = 98;
    out.transformMaskMappingLocalToWorld[14][162] = 162;
    out.transformMaskMappingLocalToWorld[14][163] = 226;
    out.transformMaskMappingLocalToWorld[14][164] = 38;
    out.transformMaskMappingLocalToWorld[14][165] = 102;
    out.transformMaskMappingLocalToWorld[14][166] = 166;
    out.transformMaskMappingLocalToWorld[14][167] = 230;
    out.transformMaskMappingLocalToWorld[14][168] = 42;
    out.transformMaskMappingLocalToWorld[14][169] = 106;
    out.transformMaskMappingLocalToWorld[14][170] = 170;
    out.transformMaskMappingLocalToWorld[14][171] = 234;
    out.transformMaskMappingLocalToWorld[14][172] = 46;
    out.transformMaskMappingLocalToWorld[14][173] = 110;
    out.transformMaskMappingLocalToWorld[14][174] = 174;
    out.transformMaskMappingLocalToWorld[14][175] = 238;
    out.transformMaskMappingLocalToWorld[14][176] = 50;
    out.transformMaskMappingLocalToWorld[14][177] = 114;
    out.transformMaskMappingLocalToWorld[14][178] = 178;
    out.transformMaskMappingLocalToWorld[14][179] = 242;
    out.transformMaskMappingLocalToWorld[14][180] = 54;
    out.transformMaskMappingLocalToWorld[14][181] = 118;
    out.transformMaskMappingLocalToWorld[14][182] = 182;
    out.transformMaskMappingLocalToWorld[14][183] = 246;
    out.transformMaskMappingLocalToWorld[14][184] = 58;
    out.transformMaskMappingLocalToWorld[14][185] = 122;
    out.transformMaskMappingLocalToWorld[14][186] = 186;
    out.transformMaskMappingLocalToWorld[14][187] = 250;
    out.transformMaskMappingLocalToWorld[14][188] = 62;
    out.transformMaskMappingLocalToWorld[14][189] = 126;
    out.transformMaskMappingLocalToWorld[14][190] = 190;
    out.transformMaskMappingLocalToWorld[14][191] = 254;
    out.transformMaskMappingLocalToWorld[14][192] = 3;
    out.transformMaskMappingLocalToWorld[14][193] = 67;
    out.transformMaskMappingLocalToWorld[14][194] = 131;
    out.transformMaskMappingLocalToWorld[14][195] = 195;
    out.transformMaskMappingLocalToWorld[14][196] = 7;
    out.transformMaskMappingLocalToWorld[14][197] = 71;
    out.transformMaskMappingLocalToWorld[14][198] = 135;
    out.transformMaskMappingLocalToWorld[14][199] = 199;
    out.transformMaskMappingLocalToWorld[14][200] = 11;
    out.transformMaskMappingLocalToWorld[14][201] = 75;
    out.transformMaskMappingLocalToWorld[14][202] = 139;
    out.transformMaskMappingLocalToWorld[14][203] = 203;
    out.transformMaskMappingLocalToWorld[14][204] = 15;
    out.transformMaskMappingLocalToWorld[14][205] = 79;
    out.transformMaskMappingLocalToWorld[14][206] = 143;
    out.transformMaskMappingLocalToWorld[14][207] = 207;
    out.transformMaskMappingLocalToWorld[14][208] = 19;
    out.transformMaskMappingLocalToWorld[14][209] = 83;
    out.transformMaskMappingLocalToWorld[14][210] = 147;
    out.transformMaskMappingLocalToWorld[14][211] = 211;
    out.transformMaskMappingLocalToWorld[14][212] = 23;
    out.transformMaskMappingLocalToWorld[14][213] = 87;
    out.transformMaskMappingLocalToWorld[14][214] = 151;
    out.transformMaskMappingLocalToWorld[14][215] = 215;
    out.transformMaskMappingLocalToWorld[14][216] = 27;
    out.transformMaskMappingLocalToWorld[14][217] = 91;
    out.transformMaskMappingLocalToWorld[14][218] = 155;
    out.transformMaskMappingLocalToWorld[14][219] = 219;
    out.transformMaskMappingLocalToWorld[14][220] = 31;
    out.transformMaskMappingLocalToWorld[14][221] = 95;
    out.transformMaskMappingLocalToWorld[14][222] = 159;
    out.transformMaskMappingLocalToWorld[14][223] = 223;
    out.transformMaskMappingLocalToWorld[14][224] = 35;
    out.transformMaskMappingLocalToWorld[14][225] = 99;
    out.transformMaskMappingLocalToWorld[14][226] = 163;
    out.transformMaskMappingLocalToWorld[14][227] = 227;
    out.transformMaskMappingLocalToWorld[14][228] = 39;
    out.transformMaskMappingLocalToWorld[14][229] = 103;
    out.transformMaskMappingLocalToWorld[14][230] = 167;
    out.transformMaskMappingLocalToWorld[14][231] = 231;
    out.transformMaskMappingLocalToWorld[14][232] = 43;
    out.transformMaskMappingLocalToWorld[14][233] = 107;
    out.transformMaskMappingLocalToWorld[14][234] = 171;
    out.transformMaskMappingLocalToWorld[14][235] = 235;
    out.transformMaskMappingLocalToWorld[14][236] = 47;
    out.transformMaskMappingLocalToWorld[14][237] = 111;
    out.transformMaskMappingLocalToWorld[14][238] = 175;
    out.transformMaskMappingLocalToWorld[14][239] = 239;
    out.transformMaskMappingLocalToWorld[14][240] = 51;
    out.transformMaskMappingLocalToWorld[14][241] = 115;
    out.transformMaskMappingLocalToWorld[14][242] = 179;
    out.transformMaskMappingLocalToWorld[14][243] = 243;
    out.transformMaskMappingLocalToWorld[14][244] = 55;
    out.transformMaskMappingLocalToWorld[14][245] = 119;
    out.transformMaskMappingLocalToWorld[14][246] = 183;
    out.transformMaskMappingLocalToWorld[14][247] = 247;
    out.transformMaskMappingLocalToWorld[14][248] = 59;
    out.transformMaskMappingLocalToWorld[14][249] = 123;
    out.transformMaskMappingLocalToWorld[14][250] = 187;
    out.transformMaskMappingLocalToWorld[14][251] = 251;
    out.transformMaskMappingLocalToWorld[14][252] = 63;
    out.transformMaskMappingLocalToWorld[14][253] = 127;
    out.transformMaskMappingLocalToWorld[14][254] = 191;
    out.transformMaskMappingLocalToWorld[14][255] = 255;
    out.transformMaskMappingLocalToWorld[15][0] = 0;
    out.transformMaskMappingLocalToWorld[15][1] = 128;
    out.transformMaskMappingLocalToWorld[15][2] = 64;
    out.transformMaskMappingLocalToWorld[15][3] = 192;
    out.transformMaskMappingLocalToWorld[15][4] = 8;
    out.transformMaskMappingLocalToWorld[15][5] = 136;
    out.transformMaskMappingLocalToWorld[15][6] = 72;
    out.transformMaskMappingLocalToWorld[15][7] = 200;
    out.transformMaskMappingLocalToWorld[15][8] = 4;
    out.transformMaskMappingLocalToWorld[15][9] = 132;
    out.transformMaskMappingLocalToWorld[15][10] = 68;
    out.transformMaskMappingLocalToWorld[15][11] = 196;
    out.transformMaskMappingLocalToWorld[15][12] = 12;
    out.transformMaskMappingLocalToWorld[15][13] = 140;
    out.transformMaskMappingLocalToWorld[15][14] = 76;
    out.transformMaskMappingLocalToWorld[15][15] = 204;
    out.transformMaskMappingLocalToWorld[15][16] = 32;
    out.transformMaskMappingLocalToWorld[15][17] = 160;
    out.transformMaskMappingLocalToWorld[15][18] = 96;
    out.transformMaskMappingLocalToWorld[15][19] = 224;
    out.transformMaskMappingLocalToWorld[15][20] = 40;
    out.transformMaskMappingLocalToWorld[15][21] = 168;
    out.transformMaskMappingLocalToWorld[15][22] = 104;
    out.transformMaskMappingLocalToWorld[15][23] = 232;
    out.transformMaskMappingLocalToWorld[15][24] = 36;
    out.transformMaskMappingLocalToWorld[15][25] = 164;
    out.transformMaskMappingLocalToWorld[15][26] = 100;
    out.transformMaskMappingLocalToWorld[15][27] = 228;
    out.transformMaskMappingLocalToWorld[15][28] = 44;
    out.transformMaskMappingLocalToWorld[15][29] = 172;
    out.transformMaskMappingLocalToWorld[15][30] = 108;
    out.transformMaskMappingLocalToWorld[15][31] = 236;
    out.transformMaskMappingLocalToWorld[15][32] = 16;
    out.transformMaskMappingLocalToWorld[15][33] = 144;
    out.transformMaskMappingLocalToWorld[15][34] = 80;
    out.transformMaskMappingLocalToWorld[15][35] = 208;
    out.transformMaskMappingLocalToWorld[15][36] = 24;
    out.transformMaskMappingLocalToWorld[15][37] = 152;
    out.transformMaskMappingLocalToWorld[15][38] = 88;
    out.transformMaskMappingLocalToWorld[15][39] = 216;
    out.transformMaskMappingLocalToWorld[15][40] = 20;
    out.transformMaskMappingLocalToWorld[15][41] = 148;
    out.transformMaskMappingLocalToWorld[15][42] = 84;
    out.transformMaskMappingLocalToWorld[15][43] = 212;
    out.transformMaskMappingLocalToWorld[15][44] = 28;
    out.transformMaskMappingLocalToWorld[15][45] = 156;
    out.transformMaskMappingLocalToWorld[15][46] = 92;
    out.transformMaskMappingLocalToWorld[15][47] = 220;
    out.transformMaskMappingLocalToWorld[15][48] = 48;
    out.transformMaskMappingLocalToWorld[15][49] = 176;
    out.transformMaskMappingLocalToWorld[15][50] = 112;
    out.transformMaskMappingLocalToWorld[15][51] = 240;
    out.transformMaskMappingLocalToWorld[15][52] = 56;
    out.transformMaskMappingLocalToWorld[15][53] = 184;
    out.transformMaskMappingLocalToWorld[15][54] = 120;
    out.transformMaskMappingLocalToWorld[15][55] = 248;
    out.transformMaskMappingLocalToWorld[15][56] = 52;
    out.transformMaskMappingLocalToWorld[15][57] = 180;
    out.transformMaskMappingLocalToWorld[15][58] = 116;
    out.transformMaskMappingLocalToWorld[15][59] = 244;
    out.transformMaskMappingLocalToWorld[15][60] = 60;
    out.transformMaskMappingLocalToWorld[15][61] = 188;
    out.transformMaskMappingLocalToWorld[15][62] = 124;
    out.transformMaskMappingLocalToWorld[15][63] = 252;
    out.transformMaskMappingLocalToWorld[15][64] = 2;
    out.transformMaskMappingLocalToWorld[15][65] = 130;
    out.transformMaskMappingLocalToWorld[15][66] = 66;
    out.transformMaskMappingLocalToWorld[15][67] = 194;
    out.transformMaskMappingLocalToWorld[15][68] = 10;
    out.transformMaskMappingLocalToWorld[15][69] = 138;
    out.transformMaskMappingLocalToWorld[15][70] = 74;
    out.transformMaskMappingLocalToWorld[15][71] = 202;
    out.transformMaskMappingLocalToWorld[15][72] = 6;
    out.transformMaskMappingLocalToWorld[15][73] = 134;
    out.transformMaskMappingLocalToWorld[15][74] = 70;
    out.transformMaskMappingLocalToWorld[15][75] = 198;
    out.transformMaskMappingLocalToWorld[15][76] = 14;
    out.transformMaskMappingLocalToWorld[15][77] = 142;
    out.transformMaskMappingLocalToWorld[15][78] = 78;
    out.transformMaskMappingLocalToWorld[15][79] = 206;
    out.transformMaskMappingLocalToWorld[15][80] = 34;
    out.transformMaskMappingLocalToWorld[15][81] = 162;
    out.transformMaskMappingLocalToWorld[15][82] = 98;
    out.transformMaskMappingLocalToWorld[15][83] = 226;
    out.transformMaskMappingLocalToWorld[15][84] = 42;
    out.transformMaskMappingLocalToWorld[15][85] = 170;
    out.transformMaskMappingLocalToWorld[15][86] = 106;
    out.transformMaskMappingLocalToWorld[15][87] = 234;
    out.transformMaskMappingLocalToWorld[15][88] = 38;
    out.transformMaskMappingLocalToWorld[15][89] = 166;
    out.transformMaskMappingLocalToWorld[15][90] = 102;
    out.transformMaskMappingLocalToWorld[15][91] = 230;
    out.transformMaskMappingLocalToWorld[15][92] = 46;
    out.transformMaskMappingLocalToWorld[15][93] = 174;
    out.transformMaskMappingLocalToWorld[15][94] = 110;
    out.transformMaskMappingLocalToWorld[15][95] = 238;
    out.transformMaskMappingLocalToWorld[15][96] = 18;
    out.transformMaskMappingLocalToWorld[15][97] = 146;
    out.transformMaskMappingLocalToWorld[15][98] = 82;
    out.transformMaskMappingLocalToWorld[15][99] = 210;
    out.transformMaskMappingLocalToWorld[15][100] = 26;
    out.transformMaskMappingLocalToWorld[15][101] = 154;
    out.transformMaskMappingLocalToWorld[15][102] = 90;
    out.transformMaskMappingLocalToWorld[15][103] = 218;
    out.transformMaskMappingLocalToWorld[15][104] = 22;
    out.transformMaskMappingLocalToWorld[15][105] = 150;
    out.transformMaskMappingLocalToWorld[15][106] = 86;
    out.transformMaskMappingLocalToWorld[15][107] = 214;
    out.transformMaskMappingLocalToWorld[15][108] = 30;
    out.transformMaskMappingLocalToWorld[15][109] = 158;
    out.transformMaskMappingLocalToWorld[15][110] = 94;
    out.transformMaskMappingLocalToWorld[15][111] = 222;
    out.transformMaskMappingLocalToWorld[15][112] = 50;
    out.transformMaskMappingLocalToWorld[15][113] = 178;
    out.transformMaskMappingLocalToWorld[15][114] = 114;
    out.transformMaskMappingLocalToWorld[15][115] = 242;
    out.transformMaskMappingLocalToWorld[15][116] = 58;
    out.transformMaskMappingLocalToWorld[15][117] = 186;
    out.transformMaskMappingLocalToWorld[15][118] = 122;
    out.transformMaskMappingLocalToWorld[15][119] = 250;
    out.transformMaskMappingLocalToWorld[15][120] = 54;
    out.transformMaskMappingLocalToWorld[15][121] = 182;
    out.transformMaskMappingLocalToWorld[15][122] = 118;
    out.transformMaskMappingLocalToWorld[15][123] = 246;
    out.transformMaskMappingLocalToWorld[15][124] = 62;
    out.transformMaskMappingLocalToWorld[15][125] = 190;
    out.transformMaskMappingLocalToWorld[15][126] = 126;
    out.transformMaskMappingLocalToWorld[15][127] = 254;
    out.transformMaskMappingLocalToWorld[15][128] = 1;
    out.transformMaskMappingLocalToWorld[15][129] = 129;
    out.transformMaskMappingLocalToWorld[15][130] = 65;
    out.transformMaskMappingLocalToWorld[15][131] = 193;
    out.transformMaskMappingLocalToWorld[15][132] = 9;
    out.transformMaskMappingLocalToWorld[15][133] = 137;
    out.transformMaskMappingLocalToWorld[15][134] = 73;
    out.transformMaskMappingLocalToWorld[15][135] = 201;
    out.transformMaskMappingLocalToWorld[15][136] = 5;
    out.transformMaskMappingLocalToWorld[15][137] = 133;
    out.transformMaskMappingLocalToWorld[15][138] = 69;
    out.transformMaskMappingLocalToWorld[15][139] = 197;
    out.transformMaskMappingLocalToWorld[15][140] = 13;
    out.transformMaskMappingLocalToWorld[15][141] = 141;
    out.transformMaskMappingLocalToWorld[15][142] = 77;
    out.transformMaskMappingLocalToWorld[15][143] = 205;
    out.transformMaskMappingLocalToWorld[15][144] = 33;
    out.transformMaskMappingLocalToWorld[15][145] = 161;
    out.transformMaskMappingLocalToWorld[15][146] = 97;
    out.transformMaskMappingLocalToWorld[15][147] = 225;
    out.transformMaskMappingLocalToWorld[15][148] = 41;
    out.transformMaskMappingLocalToWorld[15][149] = 169;
    out.transformMaskMappingLocalToWorld[15][150] = 105;
    out.transformMaskMappingLocalToWorld[15][151] = 233;
    out.transformMaskMappingLocalToWorld[15][152] = 37;
    out.transformMaskMappingLocalToWorld[15][153] = 165;
    out.transformMaskMappingLocalToWorld[15][154] = 101;
    out.transformMaskMappingLocalToWorld[15][155] = 229;
    out.transformMaskMappingLocalToWorld[15][156] = 45;
    out.transformMaskMappingLocalToWorld[15][157] = 173;
    out.transformMaskMappingLocalToWorld[15][158] = 109;
    out.transformMaskMappingLocalToWorld[15][159] = 237;
    out.transformMaskMappingLocalToWorld[15][160] = 17;
    out.transformMaskMappingLocalToWorld[15][161] = 145;
    out.transformMaskMappingLocalToWorld[15][162] = 81;
    out.transformMaskMappingLocalToWorld[15][163] = 209;
    out.transformMaskMappingLocalToWorld[15][164] = 25;
    out.transformMaskMappingLocalToWorld[15][165] = 153;
    out.transformMaskMappingLocalToWorld[15][166] = 89;
    out.transformMaskMappingLocalToWorld[15][167] = 217;
    out.transformMaskMappingLocalToWorld[15][168] = 21;
    out.transformMaskMappingLocalToWorld[15][169] = 149;
    out.transformMaskMappingLocalToWorld[15][170] = 85;
    out.transformMaskMappingLocalToWorld[15][171] = 213;
    out.transformMaskMappingLocalToWorld[15][172] = 29;
    out.transformMaskMappingLocalToWorld[15][173] = 157;
    out.transformMaskMappingLocalToWorld[15][174] = 93;
    out.transformMaskMappingLocalToWorld[15][175] = 221;
    out.transformMaskMappingLocalToWorld[15][176] = 49;
    out.transformMaskMappingLocalToWorld[15][177] = 177;
    out.transformMaskMappingLocalToWorld[15][178] = 113;
    out.transformMaskMappingLocalToWorld[15][179] = 241;
    out.transformMaskMappingLocalToWorld[15][180] = 57;
    out.transformMaskMappingLocalToWorld[15][181] = 185;
    out.transformMaskMappingLocalToWorld[15][182] = 121;
    out.transformMaskMappingLocalToWorld[15][183] = 249;
    out.transformMaskMappingLocalToWorld[15][184] = 53;
    out.transformMaskMappingLocalToWorld[15][185] = 181;
    out.transformMaskMappingLocalToWorld[15][186] = 117;
    out.transformMaskMappingLocalToWorld[15][187] = 245;
    out.transformMaskMappingLocalToWorld[15][188] = 61;
    out.transformMaskMappingLocalToWorld[15][189] = 189;
    out.transformMaskMappingLocalToWorld[15][190] = 125;
    out.transformMaskMappingLocalToWorld[15][191] = 253;
    out.transformMaskMappingLocalToWorld[15][192] = 3;
    out.transformMaskMappingLocalToWorld[15][193] = 131;
    out.transformMaskMappingLocalToWorld[15][194] = 67;
    out.transformMaskMappingLocalToWorld[15][195] = 195;
    out.transformMaskMappingLocalToWorld[15][196] = 11;
    out.transformMaskMappingLocalToWorld[15][197] = 139;
    out.transformMaskMappingLocalToWorld[15][198] = 75;
    out.transformMaskMappingLocalToWorld[15][199] = 203;
    out.transformMaskMappingLocalToWorld[15][200] = 7;
    out.transformMaskMappingLocalToWorld[15][201] = 135;
    out.transformMaskMappingLocalToWorld[15][202] = 71;
    out.transformMaskMappingLocalToWorld[15][203] = 199;
    out.transformMaskMappingLocalToWorld[15][204] = 15;
    out.transformMaskMappingLocalToWorld[15][205] = 143;
    out.transformMaskMappingLocalToWorld[15][206] = 79;
    out.transformMaskMappingLocalToWorld[15][207] = 207;
    out.transformMaskMappingLocalToWorld[15][208] = 35;
    out.transformMaskMappingLocalToWorld[15][209] = 163;
    out.transformMaskMappingLocalToWorld[15][210] = 99;
    out.transformMaskMappingLocalToWorld[15][211] = 227;
    out.transformMaskMappingLocalToWorld[15][212] = 43;
    out.transformMaskMappingLocalToWorld[15][213] = 171;
    out.transformMaskMappingLocalToWorld[15][214] = 107;
    out.transformMaskMappingLocalToWorld[15][215] = 235;
    out.transformMaskMappingLocalToWorld[15][216] = 39;
    out.transformMaskMappingLocalToWorld[15][217] = 167;
    out.transformMaskMappingLocalToWorld[15][218] = 103;
    out.transformMaskMappingLocalToWorld[15][219] = 231;
    out.transformMaskMappingLocalToWorld[15][220] = 47;
    out.transformMaskMappingLocalToWorld[15][221] = 175;
    out.transformMaskMappingLocalToWorld[15][222] = 111;
    out.transformMaskMappingLocalToWorld[15][223] = 239;
    out.transformMaskMappingLocalToWorld[15][224] = 19;
    out.transformMaskMappingLocalToWorld[15][225] = 147;
    out.transformMaskMappingLocalToWorld[15][226] = 83;
    out.transformMaskMappingLocalToWorld[15][227] = 211;
    out.transformMaskMappingLocalToWorld[15][228] = 27;
    out.transformMaskMappingLocalToWorld[15][229] = 155;
    out.transformMaskMappingLocalToWorld[15][230] = 91;
    out.transformMaskMappingLocalToWorld[15][231] = 219;
    out.transformMaskMappingLocalToWorld[15][232] = 23;
    out.transformMaskMappingLocalToWorld[15][233] = 151;
    out.transformMaskMappingLocalToWorld[15][234] = 87;
    out.transformMaskMappingLocalToWorld[15][235] = 215;
    out.transformMaskMappingLocalToWorld[15][236] = 31;
    out.transformMaskMappingLocalToWorld[15][237] = 159;
    out.transformMaskMappingLocalToWorld[15][238] = 95;
    out.transformMaskMappingLocalToWorld[15][239] = 223;
    out.transformMaskMappingLocalToWorld[15][240] = 51;
    out.transformMaskMappingLocalToWorld[15][241] = 179;
    out.transformMaskMappingLocalToWorld[15][242] = 115;
    out.transformMaskMappingLocalToWorld[15][243] = 243;
    out.transformMaskMappingLocalToWorld[15][244] = 59;
    out.transformMaskMappingLocalToWorld[15][245] = 187;
    out.transformMaskMappingLocalToWorld[15][246] = 123;
    out.transformMaskMappingLocalToWorld[15][247] = 251;
    out.transformMaskMappingLocalToWorld[15][248] = 55;
    out.transformMaskMappingLocalToWorld[15][249] = 183;
    out.transformMaskMappingLocalToWorld[15][250] = 119;
    out.transformMaskMappingLocalToWorld[15][251] = 247;
    out.transformMaskMappingLocalToWorld[15][252] = 63;
    out.transformMaskMappingLocalToWorld[15][253] = 191;
    out.transformMaskMappingLocalToWorld[15][254] = 127;
    out.transformMaskMappingLocalToWorld[15][255] = 255;
    out.transformMaskMappingLocalToWorld[16][0] = 0;
    out.transformMaskMappingLocalToWorld[16][1] = 1;
    out.transformMaskMappingLocalToWorld[16][2] = 4;
    out.transformMaskMappingLocalToWorld[16][3] = 5;
    out.transformMaskMappingLocalToWorld[16][4] = 2;
    out.transformMaskMappingLocalToWorld[16][5] = 3;
    out.transformMaskMappingLocalToWorld[16][6] = 6;
    out.transformMaskMappingLocalToWorld[16][7] = 7;
    out.transformMaskMappingLocalToWorld[16][8] = 8;
    out.transformMaskMappingLocalToWorld[16][9] = 9;
    out.transformMaskMappingLocalToWorld[16][10] = 12;
    out.transformMaskMappingLocalToWorld[16][11] = 13;
    out.transformMaskMappingLocalToWorld[16][12] = 10;
    out.transformMaskMappingLocalToWorld[16][13] = 11;
    out.transformMaskMappingLocalToWorld[16][14] = 14;
    out.transformMaskMappingLocalToWorld[16][15] = 15;
    out.transformMaskMappingLocalToWorld[16][16] = 16;
    out.transformMaskMappingLocalToWorld[16][17] = 17;
    out.transformMaskMappingLocalToWorld[16][18] = 20;
    out.transformMaskMappingLocalToWorld[16][19] = 21;
    out.transformMaskMappingLocalToWorld[16][20] = 18;
    out.transformMaskMappingLocalToWorld[16][21] = 19;
    out.transformMaskMappingLocalToWorld[16][22] = 22;
    out.transformMaskMappingLocalToWorld[16][23] = 23;
    out.transformMaskMappingLocalToWorld[16][24] = 24;
    out.transformMaskMappingLocalToWorld[16][25] = 25;
    out.transformMaskMappingLocalToWorld[16][26] = 28;
    out.transformMaskMappingLocalToWorld[16][27] = 29;
    out.transformMaskMappingLocalToWorld[16][28] = 26;
    out.transformMaskMappingLocalToWorld[16][29] = 27;
    out.transformMaskMappingLocalToWorld[16][30] = 30;
    out.transformMaskMappingLocalToWorld[16][31] = 31;
    out.transformMaskMappingLocalToWorld[16][32] = 64;
    out.transformMaskMappingLocalToWorld[16][33] = 65;
    out.transformMaskMappingLocalToWorld[16][34] = 68;
    out.transformMaskMappingLocalToWorld[16][35] = 69;
    out.transformMaskMappingLocalToWorld[16][36] = 66;
    out.transformMaskMappingLocalToWorld[16][37] = 67;
    out.transformMaskMappingLocalToWorld[16][38] = 70;
    out.transformMaskMappingLocalToWorld[16][39] = 71;
    out.transformMaskMappingLocalToWorld[16][40] = 72;
    out.transformMaskMappingLocalToWorld[16][41] = 73;
    out.transformMaskMappingLocalToWorld[16][42] = 76;
    out.transformMaskMappingLocalToWorld[16][43] = 77;
    out.transformMaskMappingLocalToWorld[16][44] = 74;
    out.transformMaskMappingLocalToWorld[16][45] = 75;
    out.transformMaskMappingLocalToWorld[16][46] = 78;
    out.transformMaskMappingLocalToWorld[16][47] = 79;
    out.transformMaskMappingLocalToWorld[16][48] = 80;
    out.transformMaskMappingLocalToWorld[16][49] = 81;
    out.transformMaskMappingLocalToWorld[16][50] = 84;
    out.transformMaskMappingLocalToWorld[16][51] = 85;
    out.transformMaskMappingLocalToWorld[16][52] = 82;
    out.transformMaskMappingLocalToWorld[16][53] = 83;
    out.transformMaskMappingLocalToWorld[16][54] = 86;
    out.transformMaskMappingLocalToWorld[16][55] = 87;
    out.transformMaskMappingLocalToWorld[16][56] = 88;
    out.transformMaskMappingLocalToWorld[16][57] = 89;
    out.transformMaskMappingLocalToWorld[16][58] = 92;
    out.transformMaskMappingLocalToWorld[16][59] = 93;
    out.transformMaskMappingLocalToWorld[16][60] = 90;
    out.transformMaskMappingLocalToWorld[16][61] = 91;
    out.transformMaskMappingLocalToWorld[16][62] = 94;
    out.transformMaskMappingLocalToWorld[16][63] = 95;
    out.transformMaskMappingLocalToWorld[16][64] = 32;
    out.transformMaskMappingLocalToWorld[16][65] = 33;
    out.transformMaskMappingLocalToWorld[16][66] = 36;
    out.transformMaskMappingLocalToWorld[16][67] = 37;
    out.transformMaskMappingLocalToWorld[16][68] = 34;
    out.transformMaskMappingLocalToWorld[16][69] = 35;
    out.transformMaskMappingLocalToWorld[16][70] = 38;
    out.transformMaskMappingLocalToWorld[16][71] = 39;
    out.transformMaskMappingLocalToWorld[16][72] = 40;
    out.transformMaskMappingLocalToWorld[16][73] = 41;
    out.transformMaskMappingLocalToWorld[16][74] = 44;
    out.transformMaskMappingLocalToWorld[16][75] = 45;
    out.transformMaskMappingLocalToWorld[16][76] = 42;
    out.transformMaskMappingLocalToWorld[16][77] = 43;
    out.transformMaskMappingLocalToWorld[16][78] = 46;
    out.transformMaskMappingLocalToWorld[16][79] = 47;
    out.transformMaskMappingLocalToWorld[16][80] = 48;
    out.transformMaskMappingLocalToWorld[16][81] = 49;
    out.transformMaskMappingLocalToWorld[16][82] = 52;
    out.transformMaskMappingLocalToWorld[16][83] = 53;
    out.transformMaskMappingLocalToWorld[16][84] = 50;
    out.transformMaskMappingLocalToWorld[16][85] = 51;
    out.transformMaskMappingLocalToWorld[16][86] = 54;
    out.transformMaskMappingLocalToWorld[16][87] = 55;
    out.transformMaskMappingLocalToWorld[16][88] = 56;
    out.transformMaskMappingLocalToWorld[16][89] = 57;
    out.transformMaskMappingLocalToWorld[16][90] = 60;
    out.transformMaskMappingLocalToWorld[16][91] = 61;
    out.transformMaskMappingLocalToWorld[16][92] = 58;
    out.transformMaskMappingLocalToWorld[16][93] = 59;
    out.transformMaskMappingLocalToWorld[16][94] = 62;
    out.transformMaskMappingLocalToWorld[16][95] = 63;
    out.transformMaskMappingLocalToWorld[16][96] = 96;
    out.transformMaskMappingLocalToWorld[16][97] = 97;
    out.transformMaskMappingLocalToWorld[16][98] = 100;
    out.transformMaskMappingLocalToWorld[16][99] = 101;
    out.transformMaskMappingLocalToWorld[16][100] = 98;
    out.transformMaskMappingLocalToWorld[16][101] = 99;
    out.transformMaskMappingLocalToWorld[16][102] = 102;
    out.transformMaskMappingLocalToWorld[16][103] = 103;
    out.transformMaskMappingLocalToWorld[16][104] = 104;
    out.transformMaskMappingLocalToWorld[16][105] = 105;
    out.transformMaskMappingLocalToWorld[16][106] = 108;
    out.transformMaskMappingLocalToWorld[16][107] = 109;
    out.transformMaskMappingLocalToWorld[16][108] = 106;
    out.transformMaskMappingLocalToWorld[16][109] = 107;
    out.transformMaskMappingLocalToWorld[16][110] = 110;
    out.transformMaskMappingLocalToWorld[16][111] = 111;
    out.transformMaskMappingLocalToWorld[16][112] = 112;
    out.transformMaskMappingLocalToWorld[16][113] = 113;
    out.transformMaskMappingLocalToWorld[16][114] = 116;
    out.transformMaskMappingLocalToWorld[16][115] = 117;
    out.transformMaskMappingLocalToWorld[16][116] = 114;
    out.transformMaskMappingLocalToWorld[16][117] = 115;
    out.transformMaskMappingLocalToWorld[16][118] = 118;
    out.transformMaskMappingLocalToWorld[16][119] = 119;
    out.transformMaskMappingLocalToWorld[16][120] = 120;
    out.transformMaskMappingLocalToWorld[16][121] = 121;
    out.transformMaskMappingLocalToWorld[16][122] = 124;
    out.transformMaskMappingLocalToWorld[16][123] = 125;
    out.transformMaskMappingLocalToWorld[16][124] = 122;
    out.transformMaskMappingLocalToWorld[16][125] = 123;
    out.transformMaskMappingLocalToWorld[16][126] = 126;
    out.transformMaskMappingLocalToWorld[16][127] = 127;
    out.transformMaskMappingLocalToWorld[16][128] = 128;
    out.transformMaskMappingLocalToWorld[16][129] = 129;
    out.transformMaskMappingLocalToWorld[16][130] = 132;
    out.transformMaskMappingLocalToWorld[16][131] = 133;
    out.transformMaskMappingLocalToWorld[16][132] = 130;
    out.transformMaskMappingLocalToWorld[16][133] = 131;
    out.transformMaskMappingLocalToWorld[16][134] = 134;
    out.transformMaskMappingLocalToWorld[16][135] = 135;
    out.transformMaskMappingLocalToWorld[16][136] = 136;
    out.transformMaskMappingLocalToWorld[16][137] = 137;
    out.transformMaskMappingLocalToWorld[16][138] = 140;
    out.transformMaskMappingLocalToWorld[16][139] = 141;
    out.transformMaskMappingLocalToWorld[16][140] = 138;
    out.transformMaskMappingLocalToWorld[16][141] = 139;
    out.transformMaskMappingLocalToWorld[16][142] = 142;
    out.transformMaskMappingLocalToWorld[16][143] = 143;
    out.transformMaskMappingLocalToWorld[16][144] = 144;
    out.transformMaskMappingLocalToWorld[16][145] = 145;
    out.transformMaskMappingLocalToWorld[16][146] = 148;
    out.transformMaskMappingLocalToWorld[16][147] = 149;
    out.transformMaskMappingLocalToWorld[16][148] = 146;
    out.transformMaskMappingLocalToWorld[16][149] = 147;
    out.transformMaskMappingLocalToWorld[16][150] = 150;
    out.transformMaskMappingLocalToWorld[16][151] = 151;
    out.transformMaskMappingLocalToWorld[16][152] = 152;
    out.transformMaskMappingLocalToWorld[16][153] = 153;
    out.transformMaskMappingLocalToWorld[16][154] = 156;
    out.transformMaskMappingLocalToWorld[16][155] = 157;
    out.transformMaskMappingLocalToWorld[16][156] = 154;
    out.transformMaskMappingLocalToWorld[16][157] = 155;
    out.transformMaskMappingLocalToWorld[16][158] = 158;
    out.transformMaskMappingLocalToWorld[16][159] = 159;
    out.transformMaskMappingLocalToWorld[16][160] = 192;
    out.transformMaskMappingLocalToWorld[16][161] = 193;
    out.transformMaskMappingLocalToWorld[16][162] = 196;
    out.transformMaskMappingLocalToWorld[16][163] = 197;
    out.transformMaskMappingLocalToWorld[16][164] = 194;
    out.transformMaskMappingLocalToWorld[16][165] = 195;
    out.transformMaskMappingLocalToWorld[16][166] = 198;
    out.transformMaskMappingLocalToWorld[16][167] = 199;
    out.transformMaskMappingLocalToWorld[16][168] = 200;
    out.transformMaskMappingLocalToWorld[16][169] = 201;
    out.transformMaskMappingLocalToWorld[16][170] = 204;
    out.transformMaskMappingLocalToWorld[16][171] = 205;
    out.transformMaskMappingLocalToWorld[16][172] = 202;
    out.transformMaskMappingLocalToWorld[16][173] = 203;
    out.transformMaskMappingLocalToWorld[16][174] = 206;
    out.transformMaskMappingLocalToWorld[16][175] = 207;
    out.transformMaskMappingLocalToWorld[16][176] = 208;
    out.transformMaskMappingLocalToWorld[16][177] = 209;
    out.transformMaskMappingLocalToWorld[16][178] = 212;
    out.transformMaskMappingLocalToWorld[16][179] = 213;
    out.transformMaskMappingLocalToWorld[16][180] = 210;
    out.transformMaskMappingLocalToWorld[16][181] = 211;
    out.transformMaskMappingLocalToWorld[16][182] = 214;
    out.transformMaskMappingLocalToWorld[16][183] = 215;
    out.transformMaskMappingLocalToWorld[16][184] = 216;
    out.transformMaskMappingLocalToWorld[16][185] = 217;
    out.transformMaskMappingLocalToWorld[16][186] = 220;
    out.transformMaskMappingLocalToWorld[16][187] = 221;
    out.transformMaskMappingLocalToWorld[16][188] = 218;
    out.transformMaskMappingLocalToWorld[16][189] = 219;
    out.transformMaskMappingLocalToWorld[16][190] = 222;
    out.transformMaskMappingLocalToWorld[16][191] = 223;
    out.transformMaskMappingLocalToWorld[16][192] = 160;
    out.transformMaskMappingLocalToWorld[16][193] = 161;
    out.transformMaskMappingLocalToWorld[16][194] = 164;
    out.transformMaskMappingLocalToWorld[16][195] = 165;
    out.transformMaskMappingLocalToWorld[16][196] = 162;
    out.transformMaskMappingLocalToWorld[16][197] = 163;
    out.transformMaskMappingLocalToWorld[16][198] = 166;
    out.transformMaskMappingLocalToWorld[16][199] = 167;
    out.transformMaskMappingLocalToWorld[16][200] = 168;
    out.transformMaskMappingLocalToWorld[16][201] = 169;
    out.transformMaskMappingLocalToWorld[16][202] = 172;
    out.transformMaskMappingLocalToWorld[16][203] = 173;
    out.transformMaskMappingLocalToWorld[16][204] = 170;
    out.transformMaskMappingLocalToWorld[16][205] = 171;
    out.transformMaskMappingLocalToWorld[16][206] = 174;
    out.transformMaskMappingLocalToWorld[16][207] = 175;
    out.transformMaskMappingLocalToWorld[16][208] = 176;
    out.transformMaskMappingLocalToWorld[16][209] = 177;
    out.transformMaskMappingLocalToWorld[16][210] = 180;
    out.transformMaskMappingLocalToWorld[16][211] = 181;
    out.transformMaskMappingLocalToWorld[16][212] = 178;
    out.transformMaskMappingLocalToWorld[16][213] = 179;
    out.transformMaskMappingLocalToWorld[16][214] = 182;
    out.transformMaskMappingLocalToWorld[16][215] = 183;
    out.transformMaskMappingLocalToWorld[16][216] = 184;
    out.transformMaskMappingLocalToWorld[16][217] = 185;
    out.transformMaskMappingLocalToWorld[16][218] = 188;
    out.transformMaskMappingLocalToWorld[16][219] = 189;
    out.transformMaskMappingLocalToWorld[16][220] = 186;
    out.transformMaskMappingLocalToWorld[16][221] = 187;
    out.transformMaskMappingLocalToWorld[16][222] = 190;
    out.transformMaskMappingLocalToWorld[16][223] = 191;
    out.transformMaskMappingLocalToWorld[16][224] = 224;
    out.transformMaskMappingLocalToWorld[16][225] = 225;
    out.transformMaskMappingLocalToWorld[16][226] = 228;
    out.transformMaskMappingLocalToWorld[16][227] = 229;
    out.transformMaskMappingLocalToWorld[16][228] = 226;
    out.transformMaskMappingLocalToWorld[16][229] = 227;
    out.transformMaskMappingLocalToWorld[16][230] = 230;
    out.transformMaskMappingLocalToWorld[16][231] = 231;
    out.transformMaskMappingLocalToWorld[16][232] = 232;
    out.transformMaskMappingLocalToWorld[16][233] = 233;
    out.transformMaskMappingLocalToWorld[16][234] = 236;
    out.transformMaskMappingLocalToWorld[16][235] = 237;
    out.transformMaskMappingLocalToWorld[16][236] = 234;
    out.transformMaskMappingLocalToWorld[16][237] = 235;
    out.transformMaskMappingLocalToWorld[16][238] = 238;
    out.transformMaskMappingLocalToWorld[16][239] = 239;
    out.transformMaskMappingLocalToWorld[16][240] = 240;
    out.transformMaskMappingLocalToWorld[16][241] = 241;
    out.transformMaskMappingLocalToWorld[16][242] = 244;
    out.transformMaskMappingLocalToWorld[16][243] = 245;
    out.transformMaskMappingLocalToWorld[16][244] = 242;
    out.transformMaskMappingLocalToWorld[16][245] = 243;
    out.transformMaskMappingLocalToWorld[16][246] = 246;
    out.transformMaskMappingLocalToWorld[16][247] = 247;
    out.transformMaskMappingLocalToWorld[16][248] = 248;
    out.transformMaskMappingLocalToWorld[16][249] = 249;
    out.transformMaskMappingLocalToWorld[16][250] = 252;
    out.transformMaskMappingLocalToWorld[16][251] = 253;
    out.transformMaskMappingLocalToWorld[16][252] = 250;
    out.transformMaskMappingLocalToWorld[16][253] = 251;
    out.transformMaskMappingLocalToWorld[16][254] = 254;
    out.transformMaskMappingLocalToWorld[16][255] = 255;
    out.transformMaskMappingLocalToWorld[17][0] = 0;
    out.transformMaskMappingLocalToWorld[17][1] = 2;
    out.transformMaskMappingLocalToWorld[17][2] = 8;
    out.transformMaskMappingLocalToWorld[17][3] = 10;
    out.transformMaskMappingLocalToWorld[17][4] = 1;
    out.transformMaskMappingLocalToWorld[17][5] = 3;
    out.transformMaskMappingLocalToWorld[17][6] = 9;
    out.transformMaskMappingLocalToWorld[17][7] = 11;
    out.transformMaskMappingLocalToWorld[17][8] = 4;
    out.transformMaskMappingLocalToWorld[17][9] = 6;
    out.transformMaskMappingLocalToWorld[17][10] = 12;
    out.transformMaskMappingLocalToWorld[17][11] = 14;
    out.transformMaskMappingLocalToWorld[17][12] = 5;
    out.transformMaskMappingLocalToWorld[17][13] = 7;
    out.transformMaskMappingLocalToWorld[17][14] = 13;
    out.transformMaskMappingLocalToWorld[17][15] = 15;
    out.transformMaskMappingLocalToWorld[17][16] = 32;
    out.transformMaskMappingLocalToWorld[17][17] = 34;
    out.transformMaskMappingLocalToWorld[17][18] = 40;
    out.transformMaskMappingLocalToWorld[17][19] = 42;
    out.transformMaskMappingLocalToWorld[17][20] = 33;
    out.transformMaskMappingLocalToWorld[17][21] = 35;
    out.transformMaskMappingLocalToWorld[17][22] = 41;
    out.transformMaskMappingLocalToWorld[17][23] = 43;
    out.transformMaskMappingLocalToWorld[17][24] = 36;
    out.transformMaskMappingLocalToWorld[17][25] = 38;
    out.transformMaskMappingLocalToWorld[17][26] = 44;
    out.transformMaskMappingLocalToWorld[17][27] = 46;
    out.transformMaskMappingLocalToWorld[17][28] = 37;
    out.transformMaskMappingLocalToWorld[17][29] = 39;
    out.transformMaskMappingLocalToWorld[17][30] = 45;
    out.transformMaskMappingLocalToWorld[17][31] = 47;
    out.transformMaskMappingLocalToWorld[17][32] = 128;
    out.transformMaskMappingLocalToWorld[17][33] = 130;
    out.transformMaskMappingLocalToWorld[17][34] = 136;
    out.transformMaskMappingLocalToWorld[17][35] = 138;
    out.transformMaskMappingLocalToWorld[17][36] = 129;
    out.transformMaskMappingLocalToWorld[17][37] = 131;
    out.transformMaskMappingLocalToWorld[17][38] = 137;
    out.transformMaskMappingLocalToWorld[17][39] = 139;
    out.transformMaskMappingLocalToWorld[17][40] = 132;
    out.transformMaskMappingLocalToWorld[17][41] = 134;
    out.transformMaskMappingLocalToWorld[17][42] = 140;
    out.transformMaskMappingLocalToWorld[17][43] = 142;
    out.transformMaskMappingLocalToWorld[17][44] = 133;
    out.transformMaskMappingLocalToWorld[17][45] = 135;
    out.transformMaskMappingLocalToWorld[17][46] = 141;
    out.transformMaskMappingLocalToWorld[17][47] = 143;
    out.transformMaskMappingLocalToWorld[17][48] = 160;
    out.transformMaskMappingLocalToWorld[17][49] = 162;
    out.transformMaskMappingLocalToWorld[17][50] = 168;
    out.transformMaskMappingLocalToWorld[17][51] = 170;
    out.transformMaskMappingLocalToWorld[17][52] = 161;
    out.transformMaskMappingLocalToWorld[17][53] = 163;
    out.transformMaskMappingLocalToWorld[17][54] = 169;
    out.transformMaskMappingLocalToWorld[17][55] = 171;
    out.transformMaskMappingLocalToWorld[17][56] = 164;
    out.transformMaskMappingLocalToWorld[17][57] = 166;
    out.transformMaskMappingLocalToWorld[17][58] = 172;
    out.transformMaskMappingLocalToWorld[17][59] = 174;
    out.transformMaskMappingLocalToWorld[17][60] = 165;
    out.transformMaskMappingLocalToWorld[17][61] = 167;
    out.transformMaskMappingLocalToWorld[17][62] = 173;
    out.transformMaskMappingLocalToWorld[17][63] = 175;
    out.transformMaskMappingLocalToWorld[17][64] = 16;
    out.transformMaskMappingLocalToWorld[17][65] = 18;
    out.transformMaskMappingLocalToWorld[17][66] = 24;
    out.transformMaskMappingLocalToWorld[17][67] = 26;
    out.transformMaskMappingLocalToWorld[17][68] = 17;
    out.transformMaskMappingLocalToWorld[17][69] = 19;
    out.transformMaskMappingLocalToWorld[17][70] = 25;
    out.transformMaskMappingLocalToWorld[17][71] = 27;
    out.transformMaskMappingLocalToWorld[17][72] = 20;
    out.transformMaskMappingLocalToWorld[17][73] = 22;
    out.transformMaskMappingLocalToWorld[17][74] = 28;
    out.transformMaskMappingLocalToWorld[17][75] = 30;
    out.transformMaskMappingLocalToWorld[17][76] = 21;
    out.transformMaskMappingLocalToWorld[17][77] = 23;
    out.transformMaskMappingLocalToWorld[17][78] = 29;
    out.transformMaskMappingLocalToWorld[17][79] = 31;
    out.transformMaskMappingLocalToWorld[17][80] = 48;
    out.transformMaskMappingLocalToWorld[17][81] = 50;
    out.transformMaskMappingLocalToWorld[17][82] = 56;
    out.transformMaskMappingLocalToWorld[17][83] = 58;
    out.transformMaskMappingLocalToWorld[17][84] = 49;
    out.transformMaskMappingLocalToWorld[17][85] = 51;
    out.transformMaskMappingLocalToWorld[17][86] = 57;
    out.transformMaskMappingLocalToWorld[17][87] = 59;
    out.transformMaskMappingLocalToWorld[17][88] = 52;
    out.transformMaskMappingLocalToWorld[17][89] = 54;
    out.transformMaskMappingLocalToWorld[17][90] = 60;
    out.transformMaskMappingLocalToWorld[17][91] = 62;
    out.transformMaskMappingLocalToWorld[17][92] = 53;
    out.transformMaskMappingLocalToWorld[17][93] = 55;
    out.transformMaskMappingLocalToWorld[17][94] = 61;
    out.transformMaskMappingLocalToWorld[17][95] = 63;
    out.transformMaskMappingLocalToWorld[17][96] = 144;
    out.transformMaskMappingLocalToWorld[17][97] = 146;
    out.transformMaskMappingLocalToWorld[17][98] = 152;
    out.transformMaskMappingLocalToWorld[17][99] = 154;
    out.transformMaskMappingLocalToWorld[17][100] = 145;
    out.transformMaskMappingLocalToWorld[17][101] = 147;
    out.transformMaskMappingLocalToWorld[17][102] = 153;
    out.transformMaskMappingLocalToWorld[17][103] = 155;
    out.transformMaskMappingLocalToWorld[17][104] = 148;
    out.transformMaskMappingLocalToWorld[17][105] = 150;
    out.transformMaskMappingLocalToWorld[17][106] = 156;
    out.transformMaskMappingLocalToWorld[17][107] = 158;
    out.transformMaskMappingLocalToWorld[17][108] = 149;
    out.transformMaskMappingLocalToWorld[17][109] = 151;
    out.transformMaskMappingLocalToWorld[17][110] = 157;
    out.transformMaskMappingLocalToWorld[17][111] = 159;
    out.transformMaskMappingLocalToWorld[17][112] = 176;
    out.transformMaskMappingLocalToWorld[17][113] = 178;
    out.transformMaskMappingLocalToWorld[17][114] = 184;
    out.transformMaskMappingLocalToWorld[17][115] = 186;
    out.transformMaskMappingLocalToWorld[17][116] = 177;
    out.transformMaskMappingLocalToWorld[17][117] = 179;
    out.transformMaskMappingLocalToWorld[17][118] = 185;
    out.transformMaskMappingLocalToWorld[17][119] = 187;
    out.transformMaskMappingLocalToWorld[17][120] = 180;
    out.transformMaskMappingLocalToWorld[17][121] = 182;
    out.transformMaskMappingLocalToWorld[17][122] = 188;
    out.transformMaskMappingLocalToWorld[17][123] = 190;
    out.transformMaskMappingLocalToWorld[17][124] = 181;
    out.transformMaskMappingLocalToWorld[17][125] = 183;
    out.transformMaskMappingLocalToWorld[17][126] = 189;
    out.transformMaskMappingLocalToWorld[17][127] = 191;
    out.transformMaskMappingLocalToWorld[17][128] = 64;
    out.transformMaskMappingLocalToWorld[17][129] = 66;
    out.transformMaskMappingLocalToWorld[17][130] = 72;
    out.transformMaskMappingLocalToWorld[17][131] = 74;
    out.transformMaskMappingLocalToWorld[17][132] = 65;
    out.transformMaskMappingLocalToWorld[17][133] = 67;
    out.transformMaskMappingLocalToWorld[17][134] = 73;
    out.transformMaskMappingLocalToWorld[17][135] = 75;
    out.transformMaskMappingLocalToWorld[17][136] = 68;
    out.transformMaskMappingLocalToWorld[17][137] = 70;
    out.transformMaskMappingLocalToWorld[17][138] = 76;
    out.transformMaskMappingLocalToWorld[17][139] = 78;
    out.transformMaskMappingLocalToWorld[17][140] = 69;
    out.transformMaskMappingLocalToWorld[17][141] = 71;
    out.transformMaskMappingLocalToWorld[17][142] = 77;
    out.transformMaskMappingLocalToWorld[17][143] = 79;
    out.transformMaskMappingLocalToWorld[17][144] = 96;
    out.transformMaskMappingLocalToWorld[17][145] = 98;
    out.transformMaskMappingLocalToWorld[17][146] = 104;
    out.transformMaskMappingLocalToWorld[17][147] = 106;
    out.transformMaskMappingLocalToWorld[17][148] = 97;
    out.transformMaskMappingLocalToWorld[17][149] = 99;
    out.transformMaskMappingLocalToWorld[17][150] = 105;
    out.transformMaskMappingLocalToWorld[17][151] = 107;
    out.transformMaskMappingLocalToWorld[17][152] = 100;
    out.transformMaskMappingLocalToWorld[17][153] = 102;
    out.transformMaskMappingLocalToWorld[17][154] = 108;
    out.transformMaskMappingLocalToWorld[17][155] = 110;
    out.transformMaskMappingLocalToWorld[17][156] = 101;
    out.transformMaskMappingLocalToWorld[17][157] = 103;
    out.transformMaskMappingLocalToWorld[17][158] = 109;
    out.transformMaskMappingLocalToWorld[17][159] = 111;
    out.transformMaskMappingLocalToWorld[17][160] = 192;
    out.transformMaskMappingLocalToWorld[17][161] = 194;
    out.transformMaskMappingLocalToWorld[17][162] = 200;
    out.transformMaskMappingLocalToWorld[17][163] = 202;
    out.transformMaskMappingLocalToWorld[17][164] = 193;
    out.transformMaskMappingLocalToWorld[17][165] = 195;
    out.transformMaskMappingLocalToWorld[17][166] = 201;
    out.transformMaskMappingLocalToWorld[17][167] = 203;
    out.transformMaskMappingLocalToWorld[17][168] = 196;
    out.transformMaskMappingLocalToWorld[17][169] = 198;
    out.transformMaskMappingLocalToWorld[17][170] = 204;
    out.transformMaskMappingLocalToWorld[17][171] = 206;
    out.transformMaskMappingLocalToWorld[17][172] = 197;
    out.transformMaskMappingLocalToWorld[17][173] = 199;
    out.transformMaskMappingLocalToWorld[17][174] = 205;
    out.transformMaskMappingLocalToWorld[17][175] = 207;
    out.transformMaskMappingLocalToWorld[17][176] = 224;
    out.transformMaskMappingLocalToWorld[17][177] = 226;
    out.transformMaskMappingLocalToWorld[17][178] = 232;
    out.transformMaskMappingLocalToWorld[17][179] = 234;
    out.transformMaskMappingLocalToWorld[17][180] = 225;
    out.transformMaskMappingLocalToWorld[17][181] = 227;
    out.transformMaskMappingLocalToWorld[17][182] = 233;
    out.transformMaskMappingLocalToWorld[17][183] = 235;
    out.transformMaskMappingLocalToWorld[17][184] = 228;
    out.transformMaskMappingLocalToWorld[17][185] = 230;
    out.transformMaskMappingLocalToWorld[17][186] = 236;
    out.transformMaskMappingLocalToWorld[17][187] = 238;
    out.transformMaskMappingLocalToWorld[17][188] = 229;
    out.transformMaskMappingLocalToWorld[17][189] = 231;
    out.transformMaskMappingLocalToWorld[17][190] = 237;
    out.transformMaskMappingLocalToWorld[17][191] = 239;
    out.transformMaskMappingLocalToWorld[17][192] = 80;
    out.transformMaskMappingLocalToWorld[17][193] = 82;
    out.transformMaskMappingLocalToWorld[17][194] = 88;
    out.transformMaskMappingLocalToWorld[17][195] = 90;
    out.transformMaskMappingLocalToWorld[17][196] = 81;
    out.transformMaskMappingLocalToWorld[17][197] = 83;
    out.transformMaskMappingLocalToWorld[17][198] = 89;
    out.transformMaskMappingLocalToWorld[17][199] = 91;
    out.transformMaskMappingLocalToWorld[17][200] = 84;
    out.transformMaskMappingLocalToWorld[17][201] = 86;
    out.transformMaskMappingLocalToWorld[17][202] = 92;
    out.transformMaskMappingLocalToWorld[17][203] = 94;
    out.transformMaskMappingLocalToWorld[17][204] = 85;
    out.transformMaskMappingLocalToWorld[17][205] = 87;
    out.transformMaskMappingLocalToWorld[17][206] = 93;
    out.transformMaskMappingLocalToWorld[17][207] = 95;
    out.transformMaskMappingLocalToWorld[17][208] = 112;
    out.transformMaskMappingLocalToWorld[17][209] = 114;
    out.transformMaskMappingLocalToWorld[17][210] = 120;
    out.transformMaskMappingLocalToWorld[17][211] = 122;
    out.transformMaskMappingLocalToWorld[17][212] = 113;
    out.transformMaskMappingLocalToWorld[17][213] = 115;
    out.transformMaskMappingLocalToWorld[17][214] = 121;
    out.transformMaskMappingLocalToWorld[17][215] = 123;
    out.transformMaskMappingLocalToWorld[17][216] = 116;
    out.transformMaskMappingLocalToWorld[17][217] = 118;
    out.transformMaskMappingLocalToWorld[17][218] = 124;
    out.transformMaskMappingLocalToWorld[17][219] = 126;
    out.transformMaskMappingLocalToWorld[17][220] = 117;
    out.transformMaskMappingLocalToWorld[17][221] = 119;
    out.transformMaskMappingLocalToWorld[17][222] = 125;
    out.transformMaskMappingLocalToWorld[17][223] = 127;
    out.transformMaskMappingLocalToWorld[17][224] = 208;
    out.transformMaskMappingLocalToWorld[17][225] = 210;
    out.transformMaskMappingLocalToWorld[17][226] = 216;
    out.transformMaskMappingLocalToWorld[17][227] = 218;
    out.transformMaskMappingLocalToWorld[17][228] = 209;
    out.transformMaskMappingLocalToWorld[17][229] = 211;
    out.transformMaskMappingLocalToWorld[17][230] = 217;
    out.transformMaskMappingLocalToWorld[17][231] = 219;
    out.transformMaskMappingLocalToWorld[17][232] = 212;
    out.transformMaskMappingLocalToWorld[17][233] = 214;
    out.transformMaskMappingLocalToWorld[17][234] = 220;
    out.transformMaskMappingLocalToWorld[17][235] = 222;
    out.transformMaskMappingLocalToWorld[17][236] = 213;
    out.transformMaskMappingLocalToWorld[17][237] = 215;
    out.transformMaskMappingLocalToWorld[17][238] = 221;
    out.transformMaskMappingLocalToWorld[17][239] = 223;
    out.transformMaskMappingLocalToWorld[17][240] = 240;
    out.transformMaskMappingLocalToWorld[17][241] = 242;
    out.transformMaskMappingLocalToWorld[17][242] = 248;
    out.transformMaskMappingLocalToWorld[17][243] = 250;
    out.transformMaskMappingLocalToWorld[17][244] = 241;
    out.transformMaskMappingLocalToWorld[17][245] = 243;
    out.transformMaskMappingLocalToWorld[17][246] = 249;
    out.transformMaskMappingLocalToWorld[17][247] = 251;
    out.transformMaskMappingLocalToWorld[17][248] = 244;
    out.transformMaskMappingLocalToWorld[17][249] = 246;
    out.transformMaskMappingLocalToWorld[17][250] = 252;
    out.transformMaskMappingLocalToWorld[17][251] = 254;
    out.transformMaskMappingLocalToWorld[17][252] = 245;
    out.transformMaskMappingLocalToWorld[17][253] = 247;
    out.transformMaskMappingLocalToWorld[17][254] = 253;
    out.transformMaskMappingLocalToWorld[17][255] = 255;
    out.transformMaskMappingLocalToWorld[18][0] = 0;
    out.transformMaskMappingLocalToWorld[18][1] = 4;
    out.transformMaskMappingLocalToWorld[18][2] = 1;
    out.transformMaskMappingLocalToWorld[18][3] = 5;
    out.transformMaskMappingLocalToWorld[18][4] = 8;
    out.transformMaskMappingLocalToWorld[18][5] = 12;
    out.transformMaskMappingLocalToWorld[18][6] = 9;
    out.transformMaskMappingLocalToWorld[18][7] = 13;
    out.transformMaskMappingLocalToWorld[18][8] = 2;
    out.transformMaskMappingLocalToWorld[18][9] = 6;
    out.transformMaskMappingLocalToWorld[18][10] = 3;
    out.transformMaskMappingLocalToWorld[18][11] = 7;
    out.transformMaskMappingLocalToWorld[18][12] = 10;
    out.transformMaskMappingLocalToWorld[18][13] = 14;
    out.transformMaskMappingLocalToWorld[18][14] = 11;
    out.transformMaskMappingLocalToWorld[18][15] = 15;
    out.transformMaskMappingLocalToWorld[18][16] = 64;
    out.transformMaskMappingLocalToWorld[18][17] = 68;
    out.transformMaskMappingLocalToWorld[18][18] = 65;
    out.transformMaskMappingLocalToWorld[18][19] = 69;
    out.transformMaskMappingLocalToWorld[18][20] = 72;
    out.transformMaskMappingLocalToWorld[18][21] = 76;
    out.transformMaskMappingLocalToWorld[18][22] = 73;
    out.transformMaskMappingLocalToWorld[18][23] = 77;
    out.transformMaskMappingLocalToWorld[18][24] = 66;
    out.transformMaskMappingLocalToWorld[18][25] = 70;
    out.transformMaskMappingLocalToWorld[18][26] = 67;
    out.transformMaskMappingLocalToWorld[18][27] = 71;
    out.transformMaskMappingLocalToWorld[18][28] = 74;
    out.transformMaskMappingLocalToWorld[18][29] = 78;
    out.transformMaskMappingLocalToWorld[18][30] = 75;
    out.transformMaskMappingLocalToWorld[18][31] = 79;
    out.transformMaskMappingLocalToWorld[18][32] = 16;
    out.transformMaskMappingLocalToWorld[18][33] = 20;
    out.transformMaskMappingLocalToWorld[18][34] = 17;
    out.transformMaskMappingLocalToWorld[18][35] = 21;
    out.transformMaskMappingLocalToWorld[18][36] = 24;
    out.transformMaskMappingLocalToWorld[18][37] = 28;
    out.transformMaskMappingLocalToWorld[18][38] = 25;
    out.transformMaskMappingLocalToWorld[18][39] = 29;
    out.transformMaskMappingLocalToWorld[18][40] = 18;
    out.transformMaskMappingLocalToWorld[18][41] = 22;
    out.transformMaskMappingLocalToWorld[18][42] = 19;
    out.transformMaskMappingLocalToWorld[18][43] = 23;
    out.transformMaskMappingLocalToWorld[18][44] = 26;
    out.transformMaskMappingLocalToWorld[18][45] = 30;
    out.transformMaskMappingLocalToWorld[18][46] = 27;
    out.transformMaskMappingLocalToWorld[18][47] = 31;
    out.transformMaskMappingLocalToWorld[18][48] = 80;
    out.transformMaskMappingLocalToWorld[18][49] = 84;
    out.transformMaskMappingLocalToWorld[18][50] = 81;
    out.transformMaskMappingLocalToWorld[18][51] = 85;
    out.transformMaskMappingLocalToWorld[18][52] = 88;
    out.transformMaskMappingLocalToWorld[18][53] = 92;
    out.transformMaskMappingLocalToWorld[18][54] = 89;
    out.transformMaskMappingLocalToWorld[18][55] = 93;
    out.transformMaskMappingLocalToWorld[18][56] = 82;
    out.transformMaskMappingLocalToWorld[18][57] = 86;
    out.transformMaskMappingLocalToWorld[18][58] = 83;
    out.transformMaskMappingLocalToWorld[18][59] = 87;
    out.transformMaskMappingLocalToWorld[18][60] = 90;
    out.transformMaskMappingLocalToWorld[18][61] = 94;
    out.transformMaskMappingLocalToWorld[18][62] = 91;
    out.transformMaskMappingLocalToWorld[18][63] = 95;
    out.transformMaskMappingLocalToWorld[18][64] = 128;
    out.transformMaskMappingLocalToWorld[18][65] = 132;
    out.transformMaskMappingLocalToWorld[18][66] = 129;
    out.transformMaskMappingLocalToWorld[18][67] = 133;
    out.transformMaskMappingLocalToWorld[18][68] = 136;
    out.transformMaskMappingLocalToWorld[18][69] = 140;
    out.transformMaskMappingLocalToWorld[18][70] = 137;
    out.transformMaskMappingLocalToWorld[18][71] = 141;
    out.transformMaskMappingLocalToWorld[18][72] = 130;
    out.transformMaskMappingLocalToWorld[18][73] = 134;
    out.transformMaskMappingLocalToWorld[18][74] = 131;
    out.transformMaskMappingLocalToWorld[18][75] = 135;
    out.transformMaskMappingLocalToWorld[18][76] = 138;
    out.transformMaskMappingLocalToWorld[18][77] = 142;
    out.transformMaskMappingLocalToWorld[18][78] = 139;
    out.transformMaskMappingLocalToWorld[18][79] = 143;
    out.transformMaskMappingLocalToWorld[18][80] = 192;
    out.transformMaskMappingLocalToWorld[18][81] = 196;
    out.transformMaskMappingLocalToWorld[18][82] = 193;
    out.transformMaskMappingLocalToWorld[18][83] = 197;
    out.transformMaskMappingLocalToWorld[18][84] = 200;
    out.transformMaskMappingLocalToWorld[18][85] = 204;
    out.transformMaskMappingLocalToWorld[18][86] = 201;
    out.transformMaskMappingLocalToWorld[18][87] = 205;
    out.transformMaskMappingLocalToWorld[18][88] = 194;
    out.transformMaskMappingLocalToWorld[18][89] = 198;
    out.transformMaskMappingLocalToWorld[18][90] = 195;
    out.transformMaskMappingLocalToWorld[18][91] = 199;
    out.transformMaskMappingLocalToWorld[18][92] = 202;
    out.transformMaskMappingLocalToWorld[18][93] = 206;
    out.transformMaskMappingLocalToWorld[18][94] = 203;
    out.transformMaskMappingLocalToWorld[18][95] = 207;
    out.transformMaskMappingLocalToWorld[18][96] = 144;
    out.transformMaskMappingLocalToWorld[18][97] = 148;
    out.transformMaskMappingLocalToWorld[18][98] = 145;
    out.transformMaskMappingLocalToWorld[18][99] = 149;
    out.transformMaskMappingLocalToWorld[18][100] = 152;
    out.transformMaskMappingLocalToWorld[18][101] = 156;
    out.transformMaskMappingLocalToWorld[18][102] = 153;
    out.transformMaskMappingLocalToWorld[18][103] = 157;
    out.transformMaskMappingLocalToWorld[18][104] = 146;
    out.transformMaskMappingLocalToWorld[18][105] = 150;
    out.transformMaskMappingLocalToWorld[18][106] = 147;
    out.transformMaskMappingLocalToWorld[18][107] = 151;
    out.transformMaskMappingLocalToWorld[18][108] = 154;
    out.transformMaskMappingLocalToWorld[18][109] = 158;
    out.transformMaskMappingLocalToWorld[18][110] = 155;
    out.transformMaskMappingLocalToWorld[18][111] = 159;
    out.transformMaskMappingLocalToWorld[18][112] = 208;
    out.transformMaskMappingLocalToWorld[18][113] = 212;
    out.transformMaskMappingLocalToWorld[18][114] = 209;
    out.transformMaskMappingLocalToWorld[18][115] = 213;
    out.transformMaskMappingLocalToWorld[18][116] = 216;
    out.transformMaskMappingLocalToWorld[18][117] = 220;
    out.transformMaskMappingLocalToWorld[18][118] = 217;
    out.transformMaskMappingLocalToWorld[18][119] = 221;
    out.transformMaskMappingLocalToWorld[18][120] = 210;
    out.transformMaskMappingLocalToWorld[18][121] = 214;
    out.transformMaskMappingLocalToWorld[18][122] = 211;
    out.transformMaskMappingLocalToWorld[18][123] = 215;
    out.transformMaskMappingLocalToWorld[18][124] = 218;
    out.transformMaskMappingLocalToWorld[18][125] = 222;
    out.transformMaskMappingLocalToWorld[18][126] = 219;
    out.transformMaskMappingLocalToWorld[18][127] = 223;
    out.transformMaskMappingLocalToWorld[18][128] = 32;
    out.transformMaskMappingLocalToWorld[18][129] = 36;
    out.transformMaskMappingLocalToWorld[18][130] = 33;
    out.transformMaskMappingLocalToWorld[18][131] = 37;
    out.transformMaskMappingLocalToWorld[18][132] = 40;
    out.transformMaskMappingLocalToWorld[18][133] = 44;
    out.transformMaskMappingLocalToWorld[18][134] = 41;
    out.transformMaskMappingLocalToWorld[18][135] = 45;
    out.transformMaskMappingLocalToWorld[18][136] = 34;
    out.transformMaskMappingLocalToWorld[18][137] = 38;
    out.transformMaskMappingLocalToWorld[18][138] = 35;
    out.transformMaskMappingLocalToWorld[18][139] = 39;
    out.transformMaskMappingLocalToWorld[18][140] = 42;
    out.transformMaskMappingLocalToWorld[18][141] = 46;
    out.transformMaskMappingLocalToWorld[18][142] = 43;
    out.transformMaskMappingLocalToWorld[18][143] = 47;
    out.transformMaskMappingLocalToWorld[18][144] = 96;
    out.transformMaskMappingLocalToWorld[18][145] = 100;
    out.transformMaskMappingLocalToWorld[18][146] = 97;
    out.transformMaskMappingLocalToWorld[18][147] = 101;
    out.transformMaskMappingLocalToWorld[18][148] = 104;
    out.transformMaskMappingLocalToWorld[18][149] = 108;
    out.transformMaskMappingLocalToWorld[18][150] = 105;
    out.transformMaskMappingLocalToWorld[18][151] = 109;
    out.transformMaskMappingLocalToWorld[18][152] = 98;
    out.transformMaskMappingLocalToWorld[18][153] = 102;
    out.transformMaskMappingLocalToWorld[18][154] = 99;
    out.transformMaskMappingLocalToWorld[18][155] = 103;
    out.transformMaskMappingLocalToWorld[18][156] = 106;
    out.transformMaskMappingLocalToWorld[18][157] = 110;
    out.transformMaskMappingLocalToWorld[18][158] = 107;
    out.transformMaskMappingLocalToWorld[18][159] = 111;
    out.transformMaskMappingLocalToWorld[18][160] = 48;
    out.transformMaskMappingLocalToWorld[18][161] = 52;
    out.transformMaskMappingLocalToWorld[18][162] = 49;
    out.transformMaskMappingLocalToWorld[18][163] = 53;
    out.transformMaskMappingLocalToWorld[18][164] = 56;
    out.transformMaskMappingLocalToWorld[18][165] = 60;
    out.transformMaskMappingLocalToWorld[18][166] = 57;
    out.transformMaskMappingLocalToWorld[18][167] = 61;
    out.transformMaskMappingLocalToWorld[18][168] = 50;
    out.transformMaskMappingLocalToWorld[18][169] = 54;
    out.transformMaskMappingLocalToWorld[18][170] = 51;
    out.transformMaskMappingLocalToWorld[18][171] = 55;
    out.transformMaskMappingLocalToWorld[18][172] = 58;
    out.transformMaskMappingLocalToWorld[18][173] = 62;
    out.transformMaskMappingLocalToWorld[18][174] = 59;
    out.transformMaskMappingLocalToWorld[18][175] = 63;
    out.transformMaskMappingLocalToWorld[18][176] = 112;
    out.transformMaskMappingLocalToWorld[18][177] = 116;
    out.transformMaskMappingLocalToWorld[18][178] = 113;
    out.transformMaskMappingLocalToWorld[18][179] = 117;
    out.transformMaskMappingLocalToWorld[18][180] = 120;
    out.transformMaskMappingLocalToWorld[18][181] = 124;
    out.transformMaskMappingLocalToWorld[18][182] = 121;
    out.transformMaskMappingLocalToWorld[18][183] = 125;
    out.transformMaskMappingLocalToWorld[18][184] = 114;
    out.transformMaskMappingLocalToWorld[18][185] = 118;
    out.transformMaskMappingLocalToWorld[18][186] = 115;
    out.transformMaskMappingLocalToWorld[18][187] = 119;
    out.transformMaskMappingLocalToWorld[18][188] = 122;
    out.transformMaskMappingLocalToWorld[18][189] = 126;
    out.transformMaskMappingLocalToWorld[18][190] = 123;
    out.transformMaskMappingLocalToWorld[18][191] = 127;
    out.transformMaskMappingLocalToWorld[18][192] = 160;
    out.transformMaskMappingLocalToWorld[18][193] = 164;
    out.transformMaskMappingLocalToWorld[18][194] = 161;
    out.transformMaskMappingLocalToWorld[18][195] = 165;
    out.transformMaskMappingLocalToWorld[18][196] = 168;
    out.transformMaskMappingLocalToWorld[18][197] = 172;
    out.transformMaskMappingLocalToWorld[18][198] = 169;
    out.transformMaskMappingLocalToWorld[18][199] = 173;
    out.transformMaskMappingLocalToWorld[18][200] = 162;
    out.transformMaskMappingLocalToWorld[18][201] = 166;
    out.transformMaskMappingLocalToWorld[18][202] = 163;
    out.transformMaskMappingLocalToWorld[18][203] = 167;
    out.transformMaskMappingLocalToWorld[18][204] = 170;
    out.transformMaskMappingLocalToWorld[18][205] = 174;
    out.transformMaskMappingLocalToWorld[18][206] = 171;
    out.transformMaskMappingLocalToWorld[18][207] = 175;
    out.transformMaskMappingLocalToWorld[18][208] = 224;
    out.transformMaskMappingLocalToWorld[18][209] = 228;
    out.transformMaskMappingLocalToWorld[18][210] = 225;
    out.transformMaskMappingLocalToWorld[18][211] = 229;
    out.transformMaskMappingLocalToWorld[18][212] = 232;
    out.transformMaskMappingLocalToWorld[18][213] = 236;
    out.transformMaskMappingLocalToWorld[18][214] = 233;
    out.transformMaskMappingLocalToWorld[18][215] = 237;
    out.transformMaskMappingLocalToWorld[18][216] = 226;
    out.transformMaskMappingLocalToWorld[18][217] = 230;
    out.transformMaskMappingLocalToWorld[18][218] = 227;
    out.transformMaskMappingLocalToWorld[18][219] = 231;
    out.transformMaskMappingLocalToWorld[18][220] = 234;
    out.transformMaskMappingLocalToWorld[18][221] = 238;
    out.transformMaskMappingLocalToWorld[18][222] = 235;
    out.transformMaskMappingLocalToWorld[18][223] = 239;
    out.transformMaskMappingLocalToWorld[18][224] = 176;
    out.transformMaskMappingLocalToWorld[18][225] = 180;
    out.transformMaskMappingLocalToWorld[18][226] = 177;
    out.transformMaskMappingLocalToWorld[18][227] = 181;
    out.transformMaskMappingLocalToWorld[18][228] = 184;
    out.transformMaskMappingLocalToWorld[18][229] = 188;
    out.transformMaskMappingLocalToWorld[18][230] = 185;
    out.transformMaskMappingLocalToWorld[18][231] = 189;
    out.transformMaskMappingLocalToWorld[18][232] = 178;
    out.transformMaskMappingLocalToWorld[18][233] = 182;
    out.transformMaskMappingLocalToWorld[18][234] = 179;
    out.transformMaskMappingLocalToWorld[18][235] = 183;
    out.transformMaskMappingLocalToWorld[18][236] = 186;
    out.transformMaskMappingLocalToWorld[18][237] = 190;
    out.transformMaskMappingLocalToWorld[18][238] = 187;
    out.transformMaskMappingLocalToWorld[18][239] = 191;
    out.transformMaskMappingLocalToWorld[18][240] = 240;
    out.transformMaskMappingLocalToWorld[18][241] = 244;
    out.transformMaskMappingLocalToWorld[18][242] = 241;
    out.transformMaskMappingLocalToWorld[18][243] = 245;
    out.transformMaskMappingLocalToWorld[18][244] = 248;
    out.transformMaskMappingLocalToWorld[18][245] = 252;
    out.transformMaskMappingLocalToWorld[18][246] = 249;
    out.transformMaskMappingLocalToWorld[18][247] = 253;
    out.transformMaskMappingLocalToWorld[18][248] = 242;
    out.transformMaskMappingLocalToWorld[18][249] = 246;
    out.transformMaskMappingLocalToWorld[18][250] = 243;
    out.transformMaskMappingLocalToWorld[18][251] = 247;
    out.transformMaskMappingLocalToWorld[18][252] = 250;
    out.transformMaskMappingLocalToWorld[18][253] = 254;
    out.transformMaskMappingLocalToWorld[18][254] = 251;
    out.transformMaskMappingLocalToWorld[18][255] = 255;
    out.transformMaskMappingLocalToWorld[19][0] = 0;
    out.transformMaskMappingLocalToWorld[19][1] = 8;
    out.transformMaskMappingLocalToWorld[19][2] = 2;
    out.transformMaskMappingLocalToWorld[19][3] = 10;
    out.transformMaskMappingLocalToWorld[19][4] = 4;
    out.transformMaskMappingLocalToWorld[19][5] = 12;
    out.transformMaskMappingLocalToWorld[19][6] = 6;
    out.transformMaskMappingLocalToWorld[19][7] = 14;
    out.transformMaskMappingLocalToWorld[19][8] = 1;
    out.transformMaskMappingLocalToWorld[19][9] = 9;
    out.transformMaskMappingLocalToWorld[19][10] = 3;
    out.transformMaskMappingLocalToWorld[19][11] = 11;
    out.transformMaskMappingLocalToWorld[19][12] = 5;
    out.transformMaskMappingLocalToWorld[19][13] = 13;
    out.transformMaskMappingLocalToWorld[19][14] = 7;
    out.transformMaskMappingLocalToWorld[19][15] = 15;
    out.transformMaskMappingLocalToWorld[19][16] = 128;
    out.transformMaskMappingLocalToWorld[19][17] = 136;
    out.transformMaskMappingLocalToWorld[19][18] = 130;
    out.transformMaskMappingLocalToWorld[19][19] = 138;
    out.transformMaskMappingLocalToWorld[19][20] = 132;
    out.transformMaskMappingLocalToWorld[19][21] = 140;
    out.transformMaskMappingLocalToWorld[19][22] = 134;
    out.transformMaskMappingLocalToWorld[19][23] = 142;
    out.transformMaskMappingLocalToWorld[19][24] = 129;
    out.transformMaskMappingLocalToWorld[19][25] = 137;
    out.transformMaskMappingLocalToWorld[19][26] = 131;
    out.transformMaskMappingLocalToWorld[19][27] = 139;
    out.transformMaskMappingLocalToWorld[19][28] = 133;
    out.transformMaskMappingLocalToWorld[19][29] = 141;
    out.transformMaskMappingLocalToWorld[19][30] = 135;
    out.transformMaskMappingLocalToWorld[19][31] = 143;
    out.transformMaskMappingLocalToWorld[19][32] = 32;
    out.transformMaskMappingLocalToWorld[19][33] = 40;
    out.transformMaskMappingLocalToWorld[19][34] = 34;
    out.transformMaskMappingLocalToWorld[19][35] = 42;
    out.transformMaskMappingLocalToWorld[19][36] = 36;
    out.transformMaskMappingLocalToWorld[19][37] = 44;
    out.transformMaskMappingLocalToWorld[19][38] = 38;
    out.transformMaskMappingLocalToWorld[19][39] = 46;
    out.transformMaskMappingLocalToWorld[19][40] = 33;
    out.transformMaskMappingLocalToWorld[19][41] = 41;
    out.transformMaskMappingLocalToWorld[19][42] = 35;
    out.transformMaskMappingLocalToWorld[19][43] = 43;
    out.transformMaskMappingLocalToWorld[19][44] = 37;
    out.transformMaskMappingLocalToWorld[19][45] = 45;
    out.transformMaskMappingLocalToWorld[19][46] = 39;
    out.transformMaskMappingLocalToWorld[19][47] = 47;
    out.transformMaskMappingLocalToWorld[19][48] = 160;
    out.transformMaskMappingLocalToWorld[19][49] = 168;
    out.transformMaskMappingLocalToWorld[19][50] = 162;
    out.transformMaskMappingLocalToWorld[19][51] = 170;
    out.transformMaskMappingLocalToWorld[19][52] = 164;
    out.transformMaskMappingLocalToWorld[19][53] = 172;
    out.transformMaskMappingLocalToWorld[19][54] = 166;
    out.transformMaskMappingLocalToWorld[19][55] = 174;
    out.transformMaskMappingLocalToWorld[19][56] = 161;
    out.transformMaskMappingLocalToWorld[19][57] = 169;
    out.transformMaskMappingLocalToWorld[19][58] = 163;
    out.transformMaskMappingLocalToWorld[19][59] = 171;
    out.transformMaskMappingLocalToWorld[19][60] = 165;
    out.transformMaskMappingLocalToWorld[19][61] = 173;
    out.transformMaskMappingLocalToWorld[19][62] = 167;
    out.transformMaskMappingLocalToWorld[19][63] = 175;
    out.transformMaskMappingLocalToWorld[19][64] = 64;
    out.transformMaskMappingLocalToWorld[19][65] = 72;
    out.transformMaskMappingLocalToWorld[19][66] = 66;
    out.transformMaskMappingLocalToWorld[19][67] = 74;
    out.transformMaskMappingLocalToWorld[19][68] = 68;
    out.transformMaskMappingLocalToWorld[19][69] = 76;
    out.transformMaskMappingLocalToWorld[19][70] = 70;
    out.transformMaskMappingLocalToWorld[19][71] = 78;
    out.transformMaskMappingLocalToWorld[19][72] = 65;
    out.transformMaskMappingLocalToWorld[19][73] = 73;
    out.transformMaskMappingLocalToWorld[19][74] = 67;
    out.transformMaskMappingLocalToWorld[19][75] = 75;
    out.transformMaskMappingLocalToWorld[19][76] = 69;
    out.transformMaskMappingLocalToWorld[19][77] = 77;
    out.transformMaskMappingLocalToWorld[19][78] = 71;
    out.transformMaskMappingLocalToWorld[19][79] = 79;
    out.transformMaskMappingLocalToWorld[19][80] = 192;
    out.transformMaskMappingLocalToWorld[19][81] = 200;
    out.transformMaskMappingLocalToWorld[19][82] = 194;
    out.transformMaskMappingLocalToWorld[19][83] = 202;
    out.transformMaskMappingLocalToWorld[19][84] = 196;
    out.transformMaskMappingLocalToWorld[19][85] = 204;
    out.transformMaskMappingLocalToWorld[19][86] = 198;
    out.transformMaskMappingLocalToWorld[19][87] = 206;
    out.transformMaskMappingLocalToWorld[19][88] = 193;
    out.transformMaskMappingLocalToWorld[19][89] = 201;
    out.transformMaskMappingLocalToWorld[19][90] = 195;
    out.transformMaskMappingLocalToWorld[19][91] = 203;
    out.transformMaskMappingLocalToWorld[19][92] = 197;
    out.transformMaskMappingLocalToWorld[19][93] = 205;
    out.transformMaskMappingLocalToWorld[19][94] = 199;
    out.transformMaskMappingLocalToWorld[19][95] = 207;
    out.transformMaskMappingLocalToWorld[19][96] = 96;
    out.transformMaskMappingLocalToWorld[19][97] = 104;
    out.transformMaskMappingLocalToWorld[19][98] = 98;
    out.transformMaskMappingLocalToWorld[19][99] = 106;
    out.transformMaskMappingLocalToWorld[19][100] = 100;
    out.transformMaskMappingLocalToWorld[19][101] = 108;
    out.transformMaskMappingLocalToWorld[19][102] = 102;
    out.transformMaskMappingLocalToWorld[19][103] = 110;
    out.transformMaskMappingLocalToWorld[19][104] = 97;
    out.transformMaskMappingLocalToWorld[19][105] = 105;
    out.transformMaskMappingLocalToWorld[19][106] = 99;
    out.transformMaskMappingLocalToWorld[19][107] = 107;
    out.transformMaskMappingLocalToWorld[19][108] = 101;
    out.transformMaskMappingLocalToWorld[19][109] = 109;
    out.transformMaskMappingLocalToWorld[19][110] = 103;
    out.transformMaskMappingLocalToWorld[19][111] = 111;
    out.transformMaskMappingLocalToWorld[19][112] = 224;
    out.transformMaskMappingLocalToWorld[19][113] = 232;
    out.transformMaskMappingLocalToWorld[19][114] = 226;
    out.transformMaskMappingLocalToWorld[19][115] = 234;
    out.transformMaskMappingLocalToWorld[19][116] = 228;
    out.transformMaskMappingLocalToWorld[19][117] = 236;
    out.transformMaskMappingLocalToWorld[19][118] = 230;
    out.transformMaskMappingLocalToWorld[19][119] = 238;
    out.transformMaskMappingLocalToWorld[19][120] = 225;
    out.transformMaskMappingLocalToWorld[19][121] = 233;
    out.transformMaskMappingLocalToWorld[19][122] = 227;
    out.transformMaskMappingLocalToWorld[19][123] = 235;
    out.transformMaskMappingLocalToWorld[19][124] = 229;
    out.transformMaskMappingLocalToWorld[19][125] = 237;
    out.transformMaskMappingLocalToWorld[19][126] = 231;
    out.transformMaskMappingLocalToWorld[19][127] = 239;
    out.transformMaskMappingLocalToWorld[19][128] = 16;
    out.transformMaskMappingLocalToWorld[19][129] = 24;
    out.transformMaskMappingLocalToWorld[19][130] = 18;
    out.transformMaskMappingLocalToWorld[19][131] = 26;
    out.transformMaskMappingLocalToWorld[19][132] = 20;
    out.transformMaskMappingLocalToWorld[19][133] = 28;
    out.transformMaskMappingLocalToWorld[19][134] = 22;
    out.transformMaskMappingLocalToWorld[19][135] = 30;
    out.transformMaskMappingLocalToWorld[19][136] = 17;
    out.transformMaskMappingLocalToWorld[19][137] = 25;
    out.transformMaskMappingLocalToWorld[19][138] = 19;
    out.transformMaskMappingLocalToWorld[19][139] = 27;
    out.transformMaskMappingLocalToWorld[19][140] = 21;
    out.transformMaskMappingLocalToWorld[19][141] = 29;
    out.transformMaskMappingLocalToWorld[19][142] = 23;
    out.transformMaskMappingLocalToWorld[19][143] = 31;
    out.transformMaskMappingLocalToWorld[19][144] = 144;
    out.transformMaskMappingLocalToWorld[19][145] = 152;
    out.transformMaskMappingLocalToWorld[19][146] = 146;
    out.transformMaskMappingLocalToWorld[19][147] = 154;
    out.transformMaskMappingLocalToWorld[19][148] = 148;
    out.transformMaskMappingLocalToWorld[19][149] = 156;
    out.transformMaskMappingLocalToWorld[19][150] = 150;
    out.transformMaskMappingLocalToWorld[19][151] = 158;
    out.transformMaskMappingLocalToWorld[19][152] = 145;
    out.transformMaskMappingLocalToWorld[19][153] = 153;
    out.transformMaskMappingLocalToWorld[19][154] = 147;
    out.transformMaskMappingLocalToWorld[19][155] = 155;
    out.transformMaskMappingLocalToWorld[19][156] = 149;
    out.transformMaskMappingLocalToWorld[19][157] = 157;
    out.transformMaskMappingLocalToWorld[19][158] = 151;
    out.transformMaskMappingLocalToWorld[19][159] = 159;
    out.transformMaskMappingLocalToWorld[19][160] = 48;
    out.transformMaskMappingLocalToWorld[19][161] = 56;
    out.transformMaskMappingLocalToWorld[19][162] = 50;
    out.transformMaskMappingLocalToWorld[19][163] = 58;
    out.transformMaskMappingLocalToWorld[19][164] = 52;
    out.transformMaskMappingLocalToWorld[19][165] = 60;
    out.transformMaskMappingLocalToWorld[19][166] = 54;
    out.transformMaskMappingLocalToWorld[19][167] = 62;
    out.transformMaskMappingLocalToWorld[19][168] = 49;
    out.transformMaskMappingLocalToWorld[19][169] = 57;
    out.transformMaskMappingLocalToWorld[19][170] = 51;
    out.transformMaskMappingLocalToWorld[19][171] = 59;
    out.transformMaskMappingLocalToWorld[19][172] = 53;
    out.transformMaskMappingLocalToWorld[19][173] = 61;
    out.transformMaskMappingLocalToWorld[19][174] = 55;
    out.transformMaskMappingLocalToWorld[19][175] = 63;
    out.transformMaskMappingLocalToWorld[19][176] = 176;
    out.transformMaskMappingLocalToWorld[19][177] = 184;
    out.transformMaskMappingLocalToWorld[19][178] = 178;
    out.transformMaskMappingLocalToWorld[19][179] = 186;
    out.transformMaskMappingLocalToWorld[19][180] = 180;
    out.transformMaskMappingLocalToWorld[19][181] = 188;
    out.transformMaskMappingLocalToWorld[19][182] = 182;
    out.transformMaskMappingLocalToWorld[19][183] = 190;
    out.transformMaskMappingLocalToWorld[19][184] = 177;
    out.transformMaskMappingLocalToWorld[19][185] = 185;
    out.transformMaskMappingLocalToWorld[19][186] = 179;
    out.transformMaskMappingLocalToWorld[19][187] = 187;
    out.transformMaskMappingLocalToWorld[19][188] = 181;
    out.transformMaskMappingLocalToWorld[19][189] = 189;
    out.transformMaskMappingLocalToWorld[19][190] = 183;
    out.transformMaskMappingLocalToWorld[19][191] = 191;
    out.transformMaskMappingLocalToWorld[19][192] = 80;
    out.transformMaskMappingLocalToWorld[19][193] = 88;
    out.transformMaskMappingLocalToWorld[19][194] = 82;
    out.transformMaskMappingLocalToWorld[19][195] = 90;
    out.transformMaskMappingLocalToWorld[19][196] = 84;
    out.transformMaskMappingLocalToWorld[19][197] = 92;
    out.transformMaskMappingLocalToWorld[19][198] = 86;
    out.transformMaskMappingLocalToWorld[19][199] = 94;
    out.transformMaskMappingLocalToWorld[19][200] = 81;
    out.transformMaskMappingLocalToWorld[19][201] = 89;
    out.transformMaskMappingLocalToWorld[19][202] = 83;
    out.transformMaskMappingLocalToWorld[19][203] = 91;
    out.transformMaskMappingLocalToWorld[19][204] = 85;
    out.transformMaskMappingLocalToWorld[19][205] = 93;
    out.transformMaskMappingLocalToWorld[19][206] = 87;
    out.transformMaskMappingLocalToWorld[19][207] = 95;
    out.transformMaskMappingLocalToWorld[19][208] = 208;
    out.transformMaskMappingLocalToWorld[19][209] = 216;
    out.transformMaskMappingLocalToWorld[19][210] = 210;
    out.transformMaskMappingLocalToWorld[19][211] = 218;
    out.transformMaskMappingLocalToWorld[19][212] = 212;
    out.transformMaskMappingLocalToWorld[19][213] = 220;
    out.transformMaskMappingLocalToWorld[19][214] = 214;
    out.transformMaskMappingLocalToWorld[19][215] = 222;
    out.transformMaskMappingLocalToWorld[19][216] = 209;
    out.transformMaskMappingLocalToWorld[19][217] = 217;
    out.transformMaskMappingLocalToWorld[19][218] = 211;
    out.transformMaskMappingLocalToWorld[19][219] = 219;
    out.transformMaskMappingLocalToWorld[19][220] = 213;
    out.transformMaskMappingLocalToWorld[19][221] = 221;
    out.transformMaskMappingLocalToWorld[19][222] = 215;
    out.transformMaskMappingLocalToWorld[19][223] = 223;
    out.transformMaskMappingLocalToWorld[19][224] = 112;
    out.transformMaskMappingLocalToWorld[19][225] = 120;
    out.transformMaskMappingLocalToWorld[19][226] = 114;
    out.transformMaskMappingLocalToWorld[19][227] = 122;
    out.transformMaskMappingLocalToWorld[19][228] = 116;
    out.transformMaskMappingLocalToWorld[19][229] = 124;
    out.transformMaskMappingLocalToWorld[19][230] = 118;
    out.transformMaskMappingLocalToWorld[19][231] = 126;
    out.transformMaskMappingLocalToWorld[19][232] = 113;
    out.transformMaskMappingLocalToWorld[19][233] = 121;
    out.transformMaskMappingLocalToWorld[19][234] = 115;
    out.transformMaskMappingLocalToWorld[19][235] = 123;
    out.transformMaskMappingLocalToWorld[19][236] = 117;
    out.transformMaskMappingLocalToWorld[19][237] = 125;
    out.transformMaskMappingLocalToWorld[19][238] = 119;
    out.transformMaskMappingLocalToWorld[19][239] = 127;
    out.transformMaskMappingLocalToWorld[19][240] = 240;
    out.transformMaskMappingLocalToWorld[19][241] = 248;
    out.transformMaskMappingLocalToWorld[19][242] = 242;
    out.transformMaskMappingLocalToWorld[19][243] = 250;
    out.transformMaskMappingLocalToWorld[19][244] = 244;
    out.transformMaskMappingLocalToWorld[19][245] = 252;
    out.transformMaskMappingLocalToWorld[19][246] = 246;
    out.transformMaskMappingLocalToWorld[19][247] = 254;
    out.transformMaskMappingLocalToWorld[19][248] = 241;
    out.transformMaskMappingLocalToWorld[19][249] = 249;
    out.transformMaskMappingLocalToWorld[19][250] = 243;
    out.transformMaskMappingLocalToWorld[19][251] = 251;
    out.transformMaskMappingLocalToWorld[19][252] = 245;
    out.transformMaskMappingLocalToWorld[19][253] = 253;
    out.transformMaskMappingLocalToWorld[19][254] = 247;
    out.transformMaskMappingLocalToWorld[19][255] = 255;
    out.transformMaskMappingLocalToWorld[20][0] = 0;
    out.transformMaskMappingLocalToWorld[20][1] = 16;
    out.transformMaskMappingLocalToWorld[20][2] = 64;
    out.transformMaskMappingLocalToWorld[20][3] = 80;
    out.transformMaskMappingLocalToWorld[20][4] = 32;
    out.transformMaskMappingLocalToWorld[20][5] = 48;
    out.transformMaskMappingLocalToWorld[20][6] = 96;
    out.transformMaskMappingLocalToWorld[20][7] = 112;
    out.transformMaskMappingLocalToWorld[20][8] = 128;
    out.transformMaskMappingLocalToWorld[20][9] = 144;
    out.transformMaskMappingLocalToWorld[20][10] = 192;
    out.transformMaskMappingLocalToWorld[20][11] = 208;
    out.transformMaskMappingLocalToWorld[20][12] = 160;
    out.transformMaskMappingLocalToWorld[20][13] = 176;
    out.transformMaskMappingLocalToWorld[20][14] = 224;
    out.transformMaskMappingLocalToWorld[20][15] = 240;
    out.transformMaskMappingLocalToWorld[20][16] = 1;
    out.transformMaskMappingLocalToWorld[20][17] = 17;
    out.transformMaskMappingLocalToWorld[20][18] = 65;
    out.transformMaskMappingLocalToWorld[20][19] = 81;
    out.transformMaskMappingLocalToWorld[20][20] = 33;
    out.transformMaskMappingLocalToWorld[20][21] = 49;
    out.transformMaskMappingLocalToWorld[20][22] = 97;
    out.transformMaskMappingLocalToWorld[20][23] = 113;
    out.transformMaskMappingLocalToWorld[20][24] = 129;
    out.transformMaskMappingLocalToWorld[20][25] = 145;
    out.transformMaskMappingLocalToWorld[20][26] = 193;
    out.transformMaskMappingLocalToWorld[20][27] = 209;
    out.transformMaskMappingLocalToWorld[20][28] = 161;
    out.transformMaskMappingLocalToWorld[20][29] = 177;
    out.transformMaskMappingLocalToWorld[20][30] = 225;
    out.transformMaskMappingLocalToWorld[20][31] = 241;
    out.transformMaskMappingLocalToWorld[20][32] = 4;
    out.transformMaskMappingLocalToWorld[20][33] = 20;
    out.transformMaskMappingLocalToWorld[20][34] = 68;
    out.transformMaskMappingLocalToWorld[20][35] = 84;
    out.transformMaskMappingLocalToWorld[20][36] = 36;
    out.transformMaskMappingLocalToWorld[20][37] = 52;
    out.transformMaskMappingLocalToWorld[20][38] = 100;
    out.transformMaskMappingLocalToWorld[20][39] = 116;
    out.transformMaskMappingLocalToWorld[20][40] = 132;
    out.transformMaskMappingLocalToWorld[20][41] = 148;
    out.transformMaskMappingLocalToWorld[20][42] = 196;
    out.transformMaskMappingLocalToWorld[20][43] = 212;
    out.transformMaskMappingLocalToWorld[20][44] = 164;
    out.transformMaskMappingLocalToWorld[20][45] = 180;
    out.transformMaskMappingLocalToWorld[20][46] = 228;
    out.transformMaskMappingLocalToWorld[20][47] = 244;
    out.transformMaskMappingLocalToWorld[20][48] = 5;
    out.transformMaskMappingLocalToWorld[20][49] = 21;
    out.transformMaskMappingLocalToWorld[20][50] = 69;
    out.transformMaskMappingLocalToWorld[20][51] = 85;
    out.transformMaskMappingLocalToWorld[20][52] = 37;
    out.transformMaskMappingLocalToWorld[20][53] = 53;
    out.transformMaskMappingLocalToWorld[20][54] = 101;
    out.transformMaskMappingLocalToWorld[20][55] = 117;
    out.transformMaskMappingLocalToWorld[20][56] = 133;
    out.transformMaskMappingLocalToWorld[20][57] = 149;
    out.transformMaskMappingLocalToWorld[20][58] = 197;
    out.transformMaskMappingLocalToWorld[20][59] = 213;
    out.transformMaskMappingLocalToWorld[20][60] = 165;
    out.transformMaskMappingLocalToWorld[20][61] = 181;
    out.transformMaskMappingLocalToWorld[20][62] = 229;
    out.transformMaskMappingLocalToWorld[20][63] = 245;
    out.transformMaskMappingLocalToWorld[20][64] = 2;
    out.transformMaskMappingLocalToWorld[20][65] = 18;
    out.transformMaskMappingLocalToWorld[20][66] = 66;
    out.transformMaskMappingLocalToWorld[20][67] = 82;
    out.transformMaskMappingLocalToWorld[20][68] = 34;
    out.transformMaskMappingLocalToWorld[20][69] = 50;
    out.transformMaskMappingLocalToWorld[20][70] = 98;
    out.transformMaskMappingLocalToWorld[20][71] = 114;
    out.transformMaskMappingLocalToWorld[20][72] = 130;
    out.transformMaskMappingLocalToWorld[20][73] = 146;
    out.transformMaskMappingLocalToWorld[20][74] = 194;
    out.transformMaskMappingLocalToWorld[20][75] = 210;
    out.transformMaskMappingLocalToWorld[20][76] = 162;
    out.transformMaskMappingLocalToWorld[20][77] = 178;
    out.transformMaskMappingLocalToWorld[20][78] = 226;
    out.transformMaskMappingLocalToWorld[20][79] = 242;
    out.transformMaskMappingLocalToWorld[20][80] = 3;
    out.transformMaskMappingLocalToWorld[20][81] = 19;
    out.transformMaskMappingLocalToWorld[20][82] = 67;
    out.transformMaskMappingLocalToWorld[20][83] = 83;
    out.transformMaskMappingLocalToWorld[20][84] = 35;
    out.transformMaskMappingLocalToWorld[20][85] = 51;
    out.transformMaskMappingLocalToWorld[20][86] = 99;
    out.transformMaskMappingLocalToWorld[20][87] = 115;
    out.transformMaskMappingLocalToWorld[20][88] = 131;
    out.transformMaskMappingLocalToWorld[20][89] = 147;
    out.transformMaskMappingLocalToWorld[20][90] = 195;
    out.transformMaskMappingLocalToWorld[20][91] = 211;
    out.transformMaskMappingLocalToWorld[20][92] = 163;
    out.transformMaskMappingLocalToWorld[20][93] = 179;
    out.transformMaskMappingLocalToWorld[20][94] = 227;
    out.transformMaskMappingLocalToWorld[20][95] = 243;
    out.transformMaskMappingLocalToWorld[20][96] = 6;
    out.transformMaskMappingLocalToWorld[20][97] = 22;
    out.transformMaskMappingLocalToWorld[20][98] = 70;
    out.transformMaskMappingLocalToWorld[20][99] = 86;
    out.transformMaskMappingLocalToWorld[20][100] = 38;
    out.transformMaskMappingLocalToWorld[20][101] = 54;
    out.transformMaskMappingLocalToWorld[20][102] = 102;
    out.transformMaskMappingLocalToWorld[20][103] = 118;
    out.transformMaskMappingLocalToWorld[20][104] = 134;
    out.transformMaskMappingLocalToWorld[20][105] = 150;
    out.transformMaskMappingLocalToWorld[20][106] = 198;
    out.transformMaskMappingLocalToWorld[20][107] = 214;
    out.transformMaskMappingLocalToWorld[20][108] = 166;
    out.transformMaskMappingLocalToWorld[20][109] = 182;
    out.transformMaskMappingLocalToWorld[20][110] = 230;
    out.transformMaskMappingLocalToWorld[20][111] = 246;
    out.transformMaskMappingLocalToWorld[20][112] = 7;
    out.transformMaskMappingLocalToWorld[20][113] = 23;
    out.transformMaskMappingLocalToWorld[20][114] = 71;
    out.transformMaskMappingLocalToWorld[20][115] = 87;
    out.transformMaskMappingLocalToWorld[20][116] = 39;
    out.transformMaskMappingLocalToWorld[20][117] = 55;
    out.transformMaskMappingLocalToWorld[20][118] = 103;
    out.transformMaskMappingLocalToWorld[20][119] = 119;
    out.transformMaskMappingLocalToWorld[20][120] = 135;
    out.transformMaskMappingLocalToWorld[20][121] = 151;
    out.transformMaskMappingLocalToWorld[20][122] = 199;
    out.transformMaskMappingLocalToWorld[20][123] = 215;
    out.transformMaskMappingLocalToWorld[20][124] = 167;
    out.transformMaskMappingLocalToWorld[20][125] = 183;
    out.transformMaskMappingLocalToWorld[20][126] = 231;
    out.transformMaskMappingLocalToWorld[20][127] = 247;
    out.transformMaskMappingLocalToWorld[20][128] = 8;
    out.transformMaskMappingLocalToWorld[20][129] = 24;
    out.transformMaskMappingLocalToWorld[20][130] = 72;
    out.transformMaskMappingLocalToWorld[20][131] = 88;
    out.transformMaskMappingLocalToWorld[20][132] = 40;
    out.transformMaskMappingLocalToWorld[20][133] = 56;
    out.transformMaskMappingLocalToWorld[20][134] = 104;
    out.transformMaskMappingLocalToWorld[20][135] = 120;
    out.transformMaskMappingLocalToWorld[20][136] = 136;
    out.transformMaskMappingLocalToWorld[20][137] = 152;
    out.transformMaskMappingLocalToWorld[20][138] = 200;
    out.transformMaskMappingLocalToWorld[20][139] = 216;
    out.transformMaskMappingLocalToWorld[20][140] = 168;
    out.transformMaskMappingLocalToWorld[20][141] = 184;
    out.transformMaskMappingLocalToWorld[20][142] = 232;
    out.transformMaskMappingLocalToWorld[20][143] = 248;
    out.transformMaskMappingLocalToWorld[20][144] = 9;
    out.transformMaskMappingLocalToWorld[20][145] = 25;
    out.transformMaskMappingLocalToWorld[20][146] = 73;
    out.transformMaskMappingLocalToWorld[20][147] = 89;
    out.transformMaskMappingLocalToWorld[20][148] = 41;
    out.transformMaskMappingLocalToWorld[20][149] = 57;
    out.transformMaskMappingLocalToWorld[20][150] = 105;
    out.transformMaskMappingLocalToWorld[20][151] = 121;
    out.transformMaskMappingLocalToWorld[20][152] = 137;
    out.transformMaskMappingLocalToWorld[20][153] = 153;
    out.transformMaskMappingLocalToWorld[20][154] = 201;
    out.transformMaskMappingLocalToWorld[20][155] = 217;
    out.transformMaskMappingLocalToWorld[20][156] = 169;
    out.transformMaskMappingLocalToWorld[20][157] = 185;
    out.transformMaskMappingLocalToWorld[20][158] = 233;
    out.transformMaskMappingLocalToWorld[20][159] = 249;
    out.transformMaskMappingLocalToWorld[20][160] = 12;
    out.transformMaskMappingLocalToWorld[20][161] = 28;
    out.transformMaskMappingLocalToWorld[20][162] = 76;
    out.transformMaskMappingLocalToWorld[20][163] = 92;
    out.transformMaskMappingLocalToWorld[20][164] = 44;
    out.transformMaskMappingLocalToWorld[20][165] = 60;
    out.transformMaskMappingLocalToWorld[20][166] = 108;
    out.transformMaskMappingLocalToWorld[20][167] = 124;
    out.transformMaskMappingLocalToWorld[20][168] = 140;
    out.transformMaskMappingLocalToWorld[20][169] = 156;
    out.transformMaskMappingLocalToWorld[20][170] = 204;
    out.transformMaskMappingLocalToWorld[20][171] = 220;
    out.transformMaskMappingLocalToWorld[20][172] = 172;
    out.transformMaskMappingLocalToWorld[20][173] = 188;
    out.transformMaskMappingLocalToWorld[20][174] = 236;
    out.transformMaskMappingLocalToWorld[20][175] = 252;
    out.transformMaskMappingLocalToWorld[20][176] = 13;
    out.transformMaskMappingLocalToWorld[20][177] = 29;
    out.transformMaskMappingLocalToWorld[20][178] = 77;
    out.transformMaskMappingLocalToWorld[20][179] = 93;
    out.transformMaskMappingLocalToWorld[20][180] = 45;
    out.transformMaskMappingLocalToWorld[20][181] = 61;
    out.transformMaskMappingLocalToWorld[20][182] = 109;
    out.transformMaskMappingLocalToWorld[20][183] = 125;
    out.transformMaskMappingLocalToWorld[20][184] = 141;
    out.transformMaskMappingLocalToWorld[20][185] = 157;
    out.transformMaskMappingLocalToWorld[20][186] = 205;
    out.transformMaskMappingLocalToWorld[20][187] = 221;
    out.transformMaskMappingLocalToWorld[20][188] = 173;
    out.transformMaskMappingLocalToWorld[20][189] = 189;
    out.transformMaskMappingLocalToWorld[20][190] = 237;
    out.transformMaskMappingLocalToWorld[20][191] = 253;
    out.transformMaskMappingLocalToWorld[20][192] = 10;
    out.transformMaskMappingLocalToWorld[20][193] = 26;
    out.transformMaskMappingLocalToWorld[20][194] = 74;
    out.transformMaskMappingLocalToWorld[20][195] = 90;
    out.transformMaskMappingLocalToWorld[20][196] = 42;
    out.transformMaskMappingLocalToWorld[20][197] = 58;
    out.transformMaskMappingLocalToWorld[20][198] = 106;
    out.transformMaskMappingLocalToWorld[20][199] = 122;
    out.transformMaskMappingLocalToWorld[20][200] = 138;
    out.transformMaskMappingLocalToWorld[20][201] = 154;
    out.transformMaskMappingLocalToWorld[20][202] = 202;
    out.transformMaskMappingLocalToWorld[20][203] = 218;
    out.transformMaskMappingLocalToWorld[20][204] = 170;
    out.transformMaskMappingLocalToWorld[20][205] = 186;
    out.transformMaskMappingLocalToWorld[20][206] = 234;
    out.transformMaskMappingLocalToWorld[20][207] = 250;
    out.transformMaskMappingLocalToWorld[20][208] = 11;
    out.transformMaskMappingLocalToWorld[20][209] = 27;
    out.transformMaskMappingLocalToWorld[20][210] = 75;
    out.transformMaskMappingLocalToWorld[20][211] = 91;
    out.transformMaskMappingLocalToWorld[20][212] = 43;
    out.transformMaskMappingLocalToWorld[20][213] = 59;
    out.transformMaskMappingLocalToWorld[20][214] = 107;
    out.transformMaskMappingLocalToWorld[20][215] = 123;
    out.transformMaskMappingLocalToWorld[20][216] = 139;
    out.transformMaskMappingLocalToWorld[20][217] = 155;
    out.transformMaskMappingLocalToWorld[20][218] = 203;
    out.transformMaskMappingLocalToWorld[20][219] = 219;
    out.transformMaskMappingLocalToWorld[20][220] = 171;
    out.transformMaskMappingLocalToWorld[20][221] = 187;
    out.transformMaskMappingLocalToWorld[20][222] = 235;
    out.transformMaskMappingLocalToWorld[20][223] = 251;
    out.transformMaskMappingLocalToWorld[20][224] = 14;
    out.transformMaskMappingLocalToWorld[20][225] = 30;
    out.transformMaskMappingLocalToWorld[20][226] = 78;
    out.transformMaskMappingLocalToWorld[20][227] = 94;
    out.transformMaskMappingLocalToWorld[20][228] = 46;
    out.transformMaskMappingLocalToWorld[20][229] = 62;
    out.transformMaskMappingLocalToWorld[20][230] = 110;
    out.transformMaskMappingLocalToWorld[20][231] = 126;
    out.transformMaskMappingLocalToWorld[20][232] = 142;
    out.transformMaskMappingLocalToWorld[20][233] = 158;
    out.transformMaskMappingLocalToWorld[20][234] = 206;
    out.transformMaskMappingLocalToWorld[20][235] = 222;
    out.transformMaskMappingLocalToWorld[20][236] = 174;
    out.transformMaskMappingLocalToWorld[20][237] = 190;
    out.transformMaskMappingLocalToWorld[20][238] = 238;
    out.transformMaskMappingLocalToWorld[20][239] = 254;
    out.transformMaskMappingLocalToWorld[20][240] = 15;
    out.transformMaskMappingLocalToWorld[20][241] = 31;
    out.transformMaskMappingLocalToWorld[20][242] = 79;
    out.transformMaskMappingLocalToWorld[20][243] = 95;
    out.transformMaskMappingLocalToWorld[20][244] = 47;
    out.transformMaskMappingLocalToWorld[20][245] = 63;
    out.transformMaskMappingLocalToWorld[20][246] = 111;
    out.transformMaskMappingLocalToWorld[20][247] = 127;
    out.transformMaskMappingLocalToWorld[20][248] = 143;
    out.transformMaskMappingLocalToWorld[20][249] = 159;
    out.transformMaskMappingLocalToWorld[20][250] = 207;
    out.transformMaskMappingLocalToWorld[20][251] = 223;
    out.transformMaskMappingLocalToWorld[20][252] = 175;
    out.transformMaskMappingLocalToWorld[20][253] = 191;
    out.transformMaskMappingLocalToWorld[20][254] = 239;
    out.transformMaskMappingLocalToWorld[20][255] = 255;
    out.transformMaskMappingLocalToWorld[21][0] = 0;
    out.transformMaskMappingLocalToWorld[21][1] = 32;
    out.transformMaskMappingLocalToWorld[21][2] = 128;
    out.transformMaskMappingLocalToWorld[21][3] = 160;
    out.transformMaskMappingLocalToWorld[21][4] = 16;
    out.transformMaskMappingLocalToWorld[21][5] = 48;
    out.transformMaskMappingLocalToWorld[21][6] = 144;
    out.transformMaskMappingLocalToWorld[21][7] = 176;
    out.transformMaskMappingLocalToWorld[21][8] = 64;
    out.transformMaskMappingLocalToWorld[21][9] = 96;
    out.transformMaskMappingLocalToWorld[21][10] = 192;
    out.transformMaskMappingLocalToWorld[21][11] = 224;
    out.transformMaskMappingLocalToWorld[21][12] = 80;
    out.transformMaskMappingLocalToWorld[21][13] = 112;
    out.transformMaskMappingLocalToWorld[21][14] = 208;
    out.transformMaskMappingLocalToWorld[21][15] = 240;
    out.transformMaskMappingLocalToWorld[21][16] = 2;
    out.transformMaskMappingLocalToWorld[21][17] = 34;
    out.transformMaskMappingLocalToWorld[21][18] = 130;
    out.transformMaskMappingLocalToWorld[21][19] = 162;
    out.transformMaskMappingLocalToWorld[21][20] = 18;
    out.transformMaskMappingLocalToWorld[21][21] = 50;
    out.transformMaskMappingLocalToWorld[21][22] = 146;
    out.transformMaskMappingLocalToWorld[21][23] = 178;
    out.transformMaskMappingLocalToWorld[21][24] = 66;
    out.transformMaskMappingLocalToWorld[21][25] = 98;
    out.transformMaskMappingLocalToWorld[21][26] = 194;
    out.transformMaskMappingLocalToWorld[21][27] = 226;
    out.transformMaskMappingLocalToWorld[21][28] = 82;
    out.transformMaskMappingLocalToWorld[21][29] = 114;
    out.transformMaskMappingLocalToWorld[21][30] = 210;
    out.transformMaskMappingLocalToWorld[21][31] = 242;
    out.transformMaskMappingLocalToWorld[21][32] = 8;
    out.transformMaskMappingLocalToWorld[21][33] = 40;
    out.transformMaskMappingLocalToWorld[21][34] = 136;
    out.transformMaskMappingLocalToWorld[21][35] = 168;
    out.transformMaskMappingLocalToWorld[21][36] = 24;
    out.transformMaskMappingLocalToWorld[21][37] = 56;
    out.transformMaskMappingLocalToWorld[21][38] = 152;
    out.transformMaskMappingLocalToWorld[21][39] = 184;
    out.transformMaskMappingLocalToWorld[21][40] = 72;
    out.transformMaskMappingLocalToWorld[21][41] = 104;
    out.transformMaskMappingLocalToWorld[21][42] = 200;
    out.transformMaskMappingLocalToWorld[21][43] = 232;
    out.transformMaskMappingLocalToWorld[21][44] = 88;
    out.transformMaskMappingLocalToWorld[21][45] = 120;
    out.transformMaskMappingLocalToWorld[21][46] = 216;
    out.transformMaskMappingLocalToWorld[21][47] = 248;
    out.transformMaskMappingLocalToWorld[21][48] = 10;
    out.transformMaskMappingLocalToWorld[21][49] = 42;
    out.transformMaskMappingLocalToWorld[21][50] = 138;
    out.transformMaskMappingLocalToWorld[21][51] = 170;
    out.transformMaskMappingLocalToWorld[21][52] = 26;
    out.transformMaskMappingLocalToWorld[21][53] = 58;
    out.transformMaskMappingLocalToWorld[21][54] = 154;
    out.transformMaskMappingLocalToWorld[21][55] = 186;
    out.transformMaskMappingLocalToWorld[21][56] = 74;
    out.transformMaskMappingLocalToWorld[21][57] = 106;
    out.transformMaskMappingLocalToWorld[21][58] = 202;
    out.transformMaskMappingLocalToWorld[21][59] = 234;
    out.transformMaskMappingLocalToWorld[21][60] = 90;
    out.transformMaskMappingLocalToWorld[21][61] = 122;
    out.transformMaskMappingLocalToWorld[21][62] = 218;
    out.transformMaskMappingLocalToWorld[21][63] = 250;
    out.transformMaskMappingLocalToWorld[21][64] = 1;
    out.transformMaskMappingLocalToWorld[21][65] = 33;
    out.transformMaskMappingLocalToWorld[21][66] = 129;
    out.transformMaskMappingLocalToWorld[21][67] = 161;
    out.transformMaskMappingLocalToWorld[21][68] = 17;
    out.transformMaskMappingLocalToWorld[21][69] = 49;
    out.transformMaskMappingLocalToWorld[21][70] = 145;
    out.transformMaskMappingLocalToWorld[21][71] = 177;
    out.transformMaskMappingLocalToWorld[21][72] = 65;
    out.transformMaskMappingLocalToWorld[21][73] = 97;
    out.transformMaskMappingLocalToWorld[21][74] = 193;
    out.transformMaskMappingLocalToWorld[21][75] = 225;
    out.transformMaskMappingLocalToWorld[21][76] = 81;
    out.transformMaskMappingLocalToWorld[21][77] = 113;
    out.transformMaskMappingLocalToWorld[21][78] = 209;
    out.transformMaskMappingLocalToWorld[21][79] = 241;
    out.transformMaskMappingLocalToWorld[21][80] = 3;
    out.transformMaskMappingLocalToWorld[21][81] = 35;
    out.transformMaskMappingLocalToWorld[21][82] = 131;
    out.transformMaskMappingLocalToWorld[21][83] = 163;
    out.transformMaskMappingLocalToWorld[21][84] = 19;
    out.transformMaskMappingLocalToWorld[21][85] = 51;
    out.transformMaskMappingLocalToWorld[21][86] = 147;
    out.transformMaskMappingLocalToWorld[21][87] = 179;
    out.transformMaskMappingLocalToWorld[21][88] = 67;
    out.transformMaskMappingLocalToWorld[21][89] = 99;
    out.transformMaskMappingLocalToWorld[21][90] = 195;
    out.transformMaskMappingLocalToWorld[21][91] = 227;
    out.transformMaskMappingLocalToWorld[21][92] = 83;
    out.transformMaskMappingLocalToWorld[21][93] = 115;
    out.transformMaskMappingLocalToWorld[21][94] = 211;
    out.transformMaskMappingLocalToWorld[21][95] = 243;
    out.transformMaskMappingLocalToWorld[21][96] = 9;
    out.transformMaskMappingLocalToWorld[21][97] = 41;
    out.transformMaskMappingLocalToWorld[21][98] = 137;
    out.transformMaskMappingLocalToWorld[21][99] = 169;
    out.transformMaskMappingLocalToWorld[21][100] = 25;
    out.transformMaskMappingLocalToWorld[21][101] = 57;
    out.transformMaskMappingLocalToWorld[21][102] = 153;
    out.transformMaskMappingLocalToWorld[21][103] = 185;
    out.transformMaskMappingLocalToWorld[21][104] = 73;
    out.transformMaskMappingLocalToWorld[21][105] = 105;
    out.transformMaskMappingLocalToWorld[21][106] = 201;
    out.transformMaskMappingLocalToWorld[21][107] = 233;
    out.transformMaskMappingLocalToWorld[21][108] = 89;
    out.transformMaskMappingLocalToWorld[21][109] = 121;
    out.transformMaskMappingLocalToWorld[21][110] = 217;
    out.transformMaskMappingLocalToWorld[21][111] = 249;
    out.transformMaskMappingLocalToWorld[21][112] = 11;
    out.transformMaskMappingLocalToWorld[21][113] = 43;
    out.transformMaskMappingLocalToWorld[21][114] = 139;
    out.transformMaskMappingLocalToWorld[21][115] = 171;
    out.transformMaskMappingLocalToWorld[21][116] = 27;
    out.transformMaskMappingLocalToWorld[21][117] = 59;
    out.transformMaskMappingLocalToWorld[21][118] = 155;
    out.transformMaskMappingLocalToWorld[21][119] = 187;
    out.transformMaskMappingLocalToWorld[21][120] = 75;
    out.transformMaskMappingLocalToWorld[21][121] = 107;
    out.transformMaskMappingLocalToWorld[21][122] = 203;
    out.transformMaskMappingLocalToWorld[21][123] = 235;
    out.transformMaskMappingLocalToWorld[21][124] = 91;
    out.transformMaskMappingLocalToWorld[21][125] = 123;
    out.transformMaskMappingLocalToWorld[21][126] = 219;
    out.transformMaskMappingLocalToWorld[21][127] = 251;
    out.transformMaskMappingLocalToWorld[21][128] = 4;
    out.transformMaskMappingLocalToWorld[21][129] = 36;
    out.transformMaskMappingLocalToWorld[21][130] = 132;
    out.transformMaskMappingLocalToWorld[21][131] = 164;
    out.transformMaskMappingLocalToWorld[21][132] = 20;
    out.transformMaskMappingLocalToWorld[21][133] = 52;
    out.transformMaskMappingLocalToWorld[21][134] = 148;
    out.transformMaskMappingLocalToWorld[21][135] = 180;
    out.transformMaskMappingLocalToWorld[21][136] = 68;
    out.transformMaskMappingLocalToWorld[21][137] = 100;
    out.transformMaskMappingLocalToWorld[21][138] = 196;
    out.transformMaskMappingLocalToWorld[21][139] = 228;
    out.transformMaskMappingLocalToWorld[21][140] = 84;
    out.transformMaskMappingLocalToWorld[21][141] = 116;
    out.transformMaskMappingLocalToWorld[21][142] = 212;
    out.transformMaskMappingLocalToWorld[21][143] = 244;
    out.transformMaskMappingLocalToWorld[21][144] = 6;
    out.transformMaskMappingLocalToWorld[21][145] = 38;
    out.transformMaskMappingLocalToWorld[21][146] = 134;
    out.transformMaskMappingLocalToWorld[21][147] = 166;
    out.transformMaskMappingLocalToWorld[21][148] = 22;
    out.transformMaskMappingLocalToWorld[21][149] = 54;
    out.transformMaskMappingLocalToWorld[21][150] = 150;
    out.transformMaskMappingLocalToWorld[21][151] = 182;
    out.transformMaskMappingLocalToWorld[21][152] = 70;
    out.transformMaskMappingLocalToWorld[21][153] = 102;
    out.transformMaskMappingLocalToWorld[21][154] = 198;
    out.transformMaskMappingLocalToWorld[21][155] = 230;
    out.transformMaskMappingLocalToWorld[21][156] = 86;
    out.transformMaskMappingLocalToWorld[21][157] = 118;
    out.transformMaskMappingLocalToWorld[21][158] = 214;
    out.transformMaskMappingLocalToWorld[21][159] = 246;
    out.transformMaskMappingLocalToWorld[21][160] = 12;
    out.transformMaskMappingLocalToWorld[21][161] = 44;
    out.transformMaskMappingLocalToWorld[21][162] = 140;
    out.transformMaskMappingLocalToWorld[21][163] = 172;
    out.transformMaskMappingLocalToWorld[21][164] = 28;
    out.transformMaskMappingLocalToWorld[21][165] = 60;
    out.transformMaskMappingLocalToWorld[21][166] = 156;
    out.transformMaskMappingLocalToWorld[21][167] = 188;
    out.transformMaskMappingLocalToWorld[21][168] = 76;
    out.transformMaskMappingLocalToWorld[21][169] = 108;
    out.transformMaskMappingLocalToWorld[21][170] = 204;
    out.transformMaskMappingLocalToWorld[21][171] = 236;
    out.transformMaskMappingLocalToWorld[21][172] = 92;
    out.transformMaskMappingLocalToWorld[21][173] = 124;
    out.transformMaskMappingLocalToWorld[21][174] = 220;
    out.transformMaskMappingLocalToWorld[21][175] = 252;
    out.transformMaskMappingLocalToWorld[21][176] = 14;
    out.transformMaskMappingLocalToWorld[21][177] = 46;
    out.transformMaskMappingLocalToWorld[21][178] = 142;
    out.transformMaskMappingLocalToWorld[21][179] = 174;
    out.transformMaskMappingLocalToWorld[21][180] = 30;
    out.transformMaskMappingLocalToWorld[21][181] = 62;
    out.transformMaskMappingLocalToWorld[21][182] = 158;
    out.transformMaskMappingLocalToWorld[21][183] = 190;
    out.transformMaskMappingLocalToWorld[21][184] = 78;
    out.transformMaskMappingLocalToWorld[21][185] = 110;
    out.transformMaskMappingLocalToWorld[21][186] = 206;
    out.transformMaskMappingLocalToWorld[21][187] = 238;
    out.transformMaskMappingLocalToWorld[21][188] = 94;
    out.transformMaskMappingLocalToWorld[21][189] = 126;
    out.transformMaskMappingLocalToWorld[21][190] = 222;
    out.transformMaskMappingLocalToWorld[21][191] = 254;
    out.transformMaskMappingLocalToWorld[21][192] = 5;
    out.transformMaskMappingLocalToWorld[21][193] = 37;
    out.transformMaskMappingLocalToWorld[21][194] = 133;
    out.transformMaskMappingLocalToWorld[21][195] = 165;
    out.transformMaskMappingLocalToWorld[21][196] = 21;
    out.transformMaskMappingLocalToWorld[21][197] = 53;
    out.transformMaskMappingLocalToWorld[21][198] = 149;
    out.transformMaskMappingLocalToWorld[21][199] = 181;
    out.transformMaskMappingLocalToWorld[21][200] = 69;
    out.transformMaskMappingLocalToWorld[21][201] = 101;
    out.transformMaskMappingLocalToWorld[21][202] = 197;
    out.transformMaskMappingLocalToWorld[21][203] = 229;
    out.transformMaskMappingLocalToWorld[21][204] = 85;
    out.transformMaskMappingLocalToWorld[21][205] = 117;
    out.transformMaskMappingLocalToWorld[21][206] = 213;
    out.transformMaskMappingLocalToWorld[21][207] = 245;
    out.transformMaskMappingLocalToWorld[21][208] = 7;
    out.transformMaskMappingLocalToWorld[21][209] = 39;
    out.transformMaskMappingLocalToWorld[21][210] = 135;
    out.transformMaskMappingLocalToWorld[21][211] = 167;
    out.transformMaskMappingLocalToWorld[21][212] = 23;
    out.transformMaskMappingLocalToWorld[21][213] = 55;
    out.transformMaskMappingLocalToWorld[21][214] = 151;
    out.transformMaskMappingLocalToWorld[21][215] = 183;
    out.transformMaskMappingLocalToWorld[21][216] = 71;
    out.transformMaskMappingLocalToWorld[21][217] = 103;
    out.transformMaskMappingLocalToWorld[21][218] = 199;
    out.transformMaskMappingLocalToWorld[21][219] = 231;
    out.transformMaskMappingLocalToWorld[21][220] = 87;
    out.transformMaskMappingLocalToWorld[21][221] = 119;
    out.transformMaskMappingLocalToWorld[21][222] = 215;
    out.transformMaskMappingLocalToWorld[21][223] = 247;
    out.transformMaskMappingLocalToWorld[21][224] = 13;
    out.transformMaskMappingLocalToWorld[21][225] = 45;
    out.transformMaskMappingLocalToWorld[21][226] = 141;
    out.transformMaskMappingLocalToWorld[21][227] = 173;
    out.transformMaskMappingLocalToWorld[21][228] = 29;
    out.transformMaskMappingLocalToWorld[21][229] = 61;
    out.transformMaskMappingLocalToWorld[21][230] = 157;
    out.transformMaskMappingLocalToWorld[21][231] = 189;
    out.transformMaskMappingLocalToWorld[21][232] = 77;
    out.transformMaskMappingLocalToWorld[21][233] = 109;
    out.transformMaskMappingLocalToWorld[21][234] = 205;
    out.transformMaskMappingLocalToWorld[21][235] = 237;
    out.transformMaskMappingLocalToWorld[21][236] = 93;
    out.transformMaskMappingLocalToWorld[21][237] = 125;
    out.transformMaskMappingLocalToWorld[21][238] = 221;
    out.transformMaskMappingLocalToWorld[21][239] = 253;
    out.transformMaskMappingLocalToWorld[21][240] = 15;
    out.transformMaskMappingLocalToWorld[21][241] = 47;
    out.transformMaskMappingLocalToWorld[21][242] = 143;
    out.transformMaskMappingLocalToWorld[21][243] = 175;
    out.transformMaskMappingLocalToWorld[21][244] = 31;
    out.transformMaskMappingLocalToWorld[21][245] = 63;
    out.transformMaskMappingLocalToWorld[21][246] = 159;
    out.transformMaskMappingLocalToWorld[21][247] = 191;
    out.transformMaskMappingLocalToWorld[21][248] = 79;
    out.transformMaskMappingLocalToWorld[21][249] = 111;
    out.transformMaskMappingLocalToWorld[21][250] = 207;
    out.transformMaskMappingLocalToWorld[21][251] = 239;
    out.transformMaskMappingLocalToWorld[21][252] = 95;
    out.transformMaskMappingLocalToWorld[21][253] = 127;
    out.transformMaskMappingLocalToWorld[21][254] = 223;
    out.transformMaskMappingLocalToWorld[21][255] = 255;
    out.transformMaskMappingLocalToWorld[22][0] = 0;
    out.transformMaskMappingLocalToWorld[22][1] = 64;
    out.transformMaskMappingLocalToWorld[22][2] = 16;
    out.transformMaskMappingLocalToWorld[22][3] = 80;
    out.transformMaskMappingLocalToWorld[22][4] = 128;
    out.transformMaskMappingLocalToWorld[22][5] = 192;
    out.transformMaskMappingLocalToWorld[22][6] = 144;
    out.transformMaskMappingLocalToWorld[22][7] = 208;
    out.transformMaskMappingLocalToWorld[22][8] = 32;
    out.transformMaskMappingLocalToWorld[22][9] = 96;
    out.transformMaskMappingLocalToWorld[22][10] = 48;
    out.transformMaskMappingLocalToWorld[22][11] = 112;
    out.transformMaskMappingLocalToWorld[22][12] = 160;
    out.transformMaskMappingLocalToWorld[22][13] = 224;
    out.transformMaskMappingLocalToWorld[22][14] = 176;
    out.transformMaskMappingLocalToWorld[22][15] = 240;
    out.transformMaskMappingLocalToWorld[22][16] = 4;
    out.transformMaskMappingLocalToWorld[22][17] = 68;
    out.transformMaskMappingLocalToWorld[22][18] = 20;
    out.transformMaskMappingLocalToWorld[22][19] = 84;
    out.transformMaskMappingLocalToWorld[22][20] = 132;
    out.transformMaskMappingLocalToWorld[22][21] = 196;
    out.transformMaskMappingLocalToWorld[22][22] = 148;
    out.transformMaskMappingLocalToWorld[22][23] = 212;
    out.transformMaskMappingLocalToWorld[22][24] = 36;
    out.transformMaskMappingLocalToWorld[22][25] = 100;
    out.transformMaskMappingLocalToWorld[22][26] = 52;
    out.transformMaskMappingLocalToWorld[22][27] = 116;
    out.transformMaskMappingLocalToWorld[22][28] = 164;
    out.transformMaskMappingLocalToWorld[22][29] = 228;
    out.transformMaskMappingLocalToWorld[22][30] = 180;
    out.transformMaskMappingLocalToWorld[22][31] = 244;
    out.transformMaskMappingLocalToWorld[22][32] = 1;
    out.transformMaskMappingLocalToWorld[22][33] = 65;
    out.transformMaskMappingLocalToWorld[22][34] = 17;
    out.transformMaskMappingLocalToWorld[22][35] = 81;
    out.transformMaskMappingLocalToWorld[22][36] = 129;
    out.transformMaskMappingLocalToWorld[22][37] = 193;
    out.transformMaskMappingLocalToWorld[22][38] = 145;
    out.transformMaskMappingLocalToWorld[22][39] = 209;
    out.transformMaskMappingLocalToWorld[22][40] = 33;
    out.transformMaskMappingLocalToWorld[22][41] = 97;
    out.transformMaskMappingLocalToWorld[22][42] = 49;
    out.transformMaskMappingLocalToWorld[22][43] = 113;
    out.transformMaskMappingLocalToWorld[22][44] = 161;
    out.transformMaskMappingLocalToWorld[22][45] = 225;
    out.transformMaskMappingLocalToWorld[22][46] = 177;
    out.transformMaskMappingLocalToWorld[22][47] = 241;
    out.transformMaskMappingLocalToWorld[22][48] = 5;
    out.transformMaskMappingLocalToWorld[22][49] = 69;
    out.transformMaskMappingLocalToWorld[22][50] = 21;
    out.transformMaskMappingLocalToWorld[22][51] = 85;
    out.transformMaskMappingLocalToWorld[22][52] = 133;
    out.transformMaskMappingLocalToWorld[22][53] = 197;
    out.transformMaskMappingLocalToWorld[22][54] = 149;
    out.transformMaskMappingLocalToWorld[22][55] = 213;
    out.transformMaskMappingLocalToWorld[22][56] = 37;
    out.transformMaskMappingLocalToWorld[22][57] = 101;
    out.transformMaskMappingLocalToWorld[22][58] = 53;
    out.transformMaskMappingLocalToWorld[22][59] = 117;
    out.transformMaskMappingLocalToWorld[22][60] = 165;
    out.transformMaskMappingLocalToWorld[22][61] = 229;
    out.transformMaskMappingLocalToWorld[22][62] = 181;
    out.transformMaskMappingLocalToWorld[22][63] = 245;
    out.transformMaskMappingLocalToWorld[22][64] = 8;
    out.transformMaskMappingLocalToWorld[22][65] = 72;
    out.transformMaskMappingLocalToWorld[22][66] = 24;
    out.transformMaskMappingLocalToWorld[22][67] = 88;
    out.transformMaskMappingLocalToWorld[22][68] = 136;
    out.transformMaskMappingLocalToWorld[22][69] = 200;
    out.transformMaskMappingLocalToWorld[22][70] = 152;
    out.transformMaskMappingLocalToWorld[22][71] = 216;
    out.transformMaskMappingLocalToWorld[22][72] = 40;
    out.transformMaskMappingLocalToWorld[22][73] = 104;
    out.transformMaskMappingLocalToWorld[22][74] = 56;
    out.transformMaskMappingLocalToWorld[22][75] = 120;
    out.transformMaskMappingLocalToWorld[22][76] = 168;
    out.transformMaskMappingLocalToWorld[22][77] = 232;
    out.transformMaskMappingLocalToWorld[22][78] = 184;
    out.transformMaskMappingLocalToWorld[22][79] = 248;
    out.transformMaskMappingLocalToWorld[22][80] = 12;
    out.transformMaskMappingLocalToWorld[22][81] = 76;
    out.transformMaskMappingLocalToWorld[22][82] = 28;
    out.transformMaskMappingLocalToWorld[22][83] = 92;
    out.transformMaskMappingLocalToWorld[22][84] = 140;
    out.transformMaskMappingLocalToWorld[22][85] = 204;
    out.transformMaskMappingLocalToWorld[22][86] = 156;
    out.transformMaskMappingLocalToWorld[22][87] = 220;
    out.transformMaskMappingLocalToWorld[22][88] = 44;
    out.transformMaskMappingLocalToWorld[22][89] = 108;
    out.transformMaskMappingLocalToWorld[22][90] = 60;
    out.transformMaskMappingLocalToWorld[22][91] = 124;
    out.transformMaskMappingLocalToWorld[22][92] = 172;
    out.transformMaskMappingLocalToWorld[22][93] = 236;
    out.transformMaskMappingLocalToWorld[22][94] = 188;
    out.transformMaskMappingLocalToWorld[22][95] = 252;
    out.transformMaskMappingLocalToWorld[22][96] = 9;
    out.transformMaskMappingLocalToWorld[22][97] = 73;
    out.transformMaskMappingLocalToWorld[22][98] = 25;
    out.transformMaskMappingLocalToWorld[22][99] = 89;
    out.transformMaskMappingLocalToWorld[22][100] = 137;
    out.transformMaskMappingLocalToWorld[22][101] = 201;
    out.transformMaskMappingLocalToWorld[22][102] = 153;
    out.transformMaskMappingLocalToWorld[22][103] = 217;
    out.transformMaskMappingLocalToWorld[22][104] = 41;
    out.transformMaskMappingLocalToWorld[22][105] = 105;
    out.transformMaskMappingLocalToWorld[22][106] = 57;
    out.transformMaskMappingLocalToWorld[22][107] = 121;
    out.transformMaskMappingLocalToWorld[22][108] = 169;
    out.transformMaskMappingLocalToWorld[22][109] = 233;
    out.transformMaskMappingLocalToWorld[22][110] = 185;
    out.transformMaskMappingLocalToWorld[22][111] = 249;
    out.transformMaskMappingLocalToWorld[22][112] = 13;
    out.transformMaskMappingLocalToWorld[22][113] = 77;
    out.transformMaskMappingLocalToWorld[22][114] = 29;
    out.transformMaskMappingLocalToWorld[22][115] = 93;
    out.transformMaskMappingLocalToWorld[22][116] = 141;
    out.transformMaskMappingLocalToWorld[22][117] = 205;
    out.transformMaskMappingLocalToWorld[22][118] = 157;
    out.transformMaskMappingLocalToWorld[22][119] = 221;
    out.transformMaskMappingLocalToWorld[22][120] = 45;
    out.transformMaskMappingLocalToWorld[22][121] = 109;
    out.transformMaskMappingLocalToWorld[22][122] = 61;
    out.transformMaskMappingLocalToWorld[22][123] = 125;
    out.transformMaskMappingLocalToWorld[22][124] = 173;
    out.transformMaskMappingLocalToWorld[22][125] = 237;
    out.transformMaskMappingLocalToWorld[22][126] = 189;
    out.transformMaskMappingLocalToWorld[22][127] = 253;
    out.transformMaskMappingLocalToWorld[22][128] = 2;
    out.transformMaskMappingLocalToWorld[22][129] = 66;
    out.transformMaskMappingLocalToWorld[22][130] = 18;
    out.transformMaskMappingLocalToWorld[22][131] = 82;
    out.transformMaskMappingLocalToWorld[22][132] = 130;
    out.transformMaskMappingLocalToWorld[22][133] = 194;
    out.transformMaskMappingLocalToWorld[22][134] = 146;
    out.transformMaskMappingLocalToWorld[22][135] = 210;
    out.transformMaskMappingLocalToWorld[22][136] = 34;
    out.transformMaskMappingLocalToWorld[22][137] = 98;
    out.transformMaskMappingLocalToWorld[22][138] = 50;
    out.transformMaskMappingLocalToWorld[22][139] = 114;
    out.transformMaskMappingLocalToWorld[22][140] = 162;
    out.transformMaskMappingLocalToWorld[22][141] = 226;
    out.transformMaskMappingLocalToWorld[22][142] = 178;
    out.transformMaskMappingLocalToWorld[22][143] = 242;
    out.transformMaskMappingLocalToWorld[22][144] = 6;
    out.transformMaskMappingLocalToWorld[22][145] = 70;
    out.transformMaskMappingLocalToWorld[22][146] = 22;
    out.transformMaskMappingLocalToWorld[22][147] = 86;
    out.transformMaskMappingLocalToWorld[22][148] = 134;
    out.transformMaskMappingLocalToWorld[22][149] = 198;
    out.transformMaskMappingLocalToWorld[22][150] = 150;
    out.transformMaskMappingLocalToWorld[22][151] = 214;
    out.transformMaskMappingLocalToWorld[22][152] = 38;
    out.transformMaskMappingLocalToWorld[22][153] = 102;
    out.transformMaskMappingLocalToWorld[22][154] = 54;
    out.transformMaskMappingLocalToWorld[22][155] = 118;
    out.transformMaskMappingLocalToWorld[22][156] = 166;
    out.transformMaskMappingLocalToWorld[22][157] = 230;
    out.transformMaskMappingLocalToWorld[22][158] = 182;
    out.transformMaskMappingLocalToWorld[22][159] = 246;
    out.transformMaskMappingLocalToWorld[22][160] = 3;
    out.transformMaskMappingLocalToWorld[22][161] = 67;
    out.transformMaskMappingLocalToWorld[22][162] = 19;
    out.transformMaskMappingLocalToWorld[22][163] = 83;
    out.transformMaskMappingLocalToWorld[22][164] = 131;
    out.transformMaskMappingLocalToWorld[22][165] = 195;
    out.transformMaskMappingLocalToWorld[22][166] = 147;
    out.transformMaskMappingLocalToWorld[22][167] = 211;
    out.transformMaskMappingLocalToWorld[22][168] = 35;
    out.transformMaskMappingLocalToWorld[22][169] = 99;
    out.transformMaskMappingLocalToWorld[22][170] = 51;
    out.transformMaskMappingLocalToWorld[22][171] = 115;
    out.transformMaskMappingLocalToWorld[22][172] = 163;
    out.transformMaskMappingLocalToWorld[22][173] = 227;
    out.transformMaskMappingLocalToWorld[22][174] = 179;
    out.transformMaskMappingLocalToWorld[22][175] = 243;
    out.transformMaskMappingLocalToWorld[22][176] = 7;
    out.transformMaskMappingLocalToWorld[22][177] = 71;
    out.transformMaskMappingLocalToWorld[22][178] = 23;
    out.transformMaskMappingLocalToWorld[22][179] = 87;
    out.transformMaskMappingLocalToWorld[22][180] = 135;
    out.transformMaskMappingLocalToWorld[22][181] = 199;
    out.transformMaskMappingLocalToWorld[22][182] = 151;
    out.transformMaskMappingLocalToWorld[22][183] = 215;
    out.transformMaskMappingLocalToWorld[22][184] = 39;
    out.transformMaskMappingLocalToWorld[22][185] = 103;
    out.transformMaskMappingLocalToWorld[22][186] = 55;
    out.transformMaskMappingLocalToWorld[22][187] = 119;
    out.transformMaskMappingLocalToWorld[22][188] = 167;
    out.transformMaskMappingLocalToWorld[22][189] = 231;
    out.transformMaskMappingLocalToWorld[22][190] = 183;
    out.transformMaskMappingLocalToWorld[22][191] = 247;
    out.transformMaskMappingLocalToWorld[22][192] = 10;
    out.transformMaskMappingLocalToWorld[22][193] = 74;
    out.transformMaskMappingLocalToWorld[22][194] = 26;
    out.transformMaskMappingLocalToWorld[22][195] = 90;
    out.transformMaskMappingLocalToWorld[22][196] = 138;
    out.transformMaskMappingLocalToWorld[22][197] = 202;
    out.transformMaskMappingLocalToWorld[22][198] = 154;
    out.transformMaskMappingLocalToWorld[22][199] = 218;
    out.transformMaskMappingLocalToWorld[22][200] = 42;
    out.transformMaskMappingLocalToWorld[22][201] = 106;
    out.transformMaskMappingLocalToWorld[22][202] = 58;
    out.transformMaskMappingLocalToWorld[22][203] = 122;
    out.transformMaskMappingLocalToWorld[22][204] = 170;
    out.transformMaskMappingLocalToWorld[22][205] = 234;
    out.transformMaskMappingLocalToWorld[22][206] = 186;
    out.transformMaskMappingLocalToWorld[22][207] = 250;
    out.transformMaskMappingLocalToWorld[22][208] = 14;
    out.transformMaskMappingLocalToWorld[22][209] = 78;
    out.transformMaskMappingLocalToWorld[22][210] = 30;
    out.transformMaskMappingLocalToWorld[22][211] = 94;
    out.transformMaskMappingLocalToWorld[22][212] = 142;
    out.transformMaskMappingLocalToWorld[22][213] = 206;
    out.transformMaskMappingLocalToWorld[22][214] = 158;
    out.transformMaskMappingLocalToWorld[22][215] = 222;
    out.transformMaskMappingLocalToWorld[22][216] = 46;
    out.transformMaskMappingLocalToWorld[22][217] = 110;
    out.transformMaskMappingLocalToWorld[22][218] = 62;
    out.transformMaskMappingLocalToWorld[22][219] = 126;
    out.transformMaskMappingLocalToWorld[22][220] = 174;
    out.transformMaskMappingLocalToWorld[22][221] = 238;
    out.transformMaskMappingLocalToWorld[22][222] = 190;
    out.transformMaskMappingLocalToWorld[22][223] = 254;
    out.transformMaskMappingLocalToWorld[22][224] = 11;
    out.transformMaskMappingLocalToWorld[22][225] = 75;
    out.transformMaskMappingLocalToWorld[22][226] = 27;
    out.transformMaskMappingLocalToWorld[22][227] = 91;
    out.transformMaskMappingLocalToWorld[22][228] = 139;
    out.transformMaskMappingLocalToWorld[22][229] = 203;
    out.transformMaskMappingLocalToWorld[22][230] = 155;
    out.transformMaskMappingLocalToWorld[22][231] = 219;
    out.transformMaskMappingLocalToWorld[22][232] = 43;
    out.transformMaskMappingLocalToWorld[22][233] = 107;
    out.transformMaskMappingLocalToWorld[22][234] = 59;
    out.transformMaskMappingLocalToWorld[22][235] = 123;
    out.transformMaskMappingLocalToWorld[22][236] = 171;
    out.transformMaskMappingLocalToWorld[22][237] = 235;
    out.transformMaskMappingLocalToWorld[22][238] = 187;
    out.transformMaskMappingLocalToWorld[22][239] = 251;
    out.transformMaskMappingLocalToWorld[22][240] = 15;
    out.transformMaskMappingLocalToWorld[22][241] = 79;
    out.transformMaskMappingLocalToWorld[22][242] = 31;
    out.transformMaskMappingLocalToWorld[22][243] = 95;
    out.transformMaskMappingLocalToWorld[22][244] = 143;
    out.transformMaskMappingLocalToWorld[22][245] = 207;
    out.transformMaskMappingLocalToWorld[22][246] = 159;
    out.transformMaskMappingLocalToWorld[22][247] = 223;
    out.transformMaskMappingLocalToWorld[22][248] = 47;
    out.transformMaskMappingLocalToWorld[22][249] = 111;
    out.transformMaskMappingLocalToWorld[22][250] = 63;
    out.transformMaskMappingLocalToWorld[22][251] = 127;
    out.transformMaskMappingLocalToWorld[22][252] = 175;
    out.transformMaskMappingLocalToWorld[22][253] = 239;
    out.transformMaskMappingLocalToWorld[22][254] = 191;
    out.transformMaskMappingLocalToWorld[22][255] = 255;
    out.transformMaskMappingLocalToWorld[23][0] = 0;
    out.transformMaskMappingLocalToWorld[23][1] = 128;
    out.transformMaskMappingLocalToWorld[23][2] = 32;
    out.transformMaskMappingLocalToWorld[23][3] = 160;
    out.transformMaskMappingLocalToWorld[23][4] = 64;
    out.transformMaskMappingLocalToWorld[23][5] = 192;
    out.transformMaskMappingLocalToWorld[23][6] = 96;
    out.transformMaskMappingLocalToWorld[23][7] = 224;
    out.transformMaskMappingLocalToWorld[23][8] = 16;
    out.transformMaskMappingLocalToWorld[23][9] = 144;
    out.transformMaskMappingLocalToWorld[23][10] = 48;
    out.transformMaskMappingLocalToWorld[23][11] = 176;
    out.transformMaskMappingLocalToWorld[23][12] = 80;
    out.transformMaskMappingLocalToWorld[23][13] = 208;
    out.transformMaskMappingLocalToWorld[23][14] = 112;
    out.transformMaskMappingLocalToWorld[23][15] = 240;
    out.transformMaskMappingLocalToWorld[23][16] = 8;
    out.transformMaskMappingLocalToWorld[23][17] = 136;
    out.transformMaskMappingLocalToWorld[23][18] = 40;
    out.transformMaskMappingLocalToWorld[23][19] = 168;
    out.transformMaskMappingLocalToWorld[23][20] = 72;
    out.transformMaskMappingLocalToWorld[23][21] = 200;
    out.transformMaskMappingLocalToWorld[23][22] = 104;
    out.transformMaskMappingLocalToWorld[23][23] = 232;
    out.transformMaskMappingLocalToWorld[23][24] = 24;
    out.transformMaskMappingLocalToWorld[23][25] = 152;
    out.transformMaskMappingLocalToWorld[23][26] = 56;
    out.transformMaskMappingLocalToWorld[23][27] = 184;
    out.transformMaskMappingLocalToWorld[23][28] = 88;
    out.transformMaskMappingLocalToWorld[23][29] = 216;
    out.transformMaskMappingLocalToWorld[23][30] = 120;
    out.transformMaskMappingLocalToWorld[23][31] = 248;
    out.transformMaskMappingLocalToWorld[23][32] = 2;
    out.transformMaskMappingLocalToWorld[23][33] = 130;
    out.transformMaskMappingLocalToWorld[23][34] = 34;
    out.transformMaskMappingLocalToWorld[23][35] = 162;
    out.transformMaskMappingLocalToWorld[23][36] = 66;
    out.transformMaskMappingLocalToWorld[23][37] = 194;
    out.transformMaskMappingLocalToWorld[23][38] = 98;
    out.transformMaskMappingLocalToWorld[23][39] = 226;
    out.transformMaskMappingLocalToWorld[23][40] = 18;
    out.transformMaskMappingLocalToWorld[23][41] = 146;
    out.transformMaskMappingLocalToWorld[23][42] = 50;
    out.transformMaskMappingLocalToWorld[23][43] = 178;
    out.transformMaskMappingLocalToWorld[23][44] = 82;
    out.transformMaskMappingLocalToWorld[23][45] = 210;
    out.transformMaskMappingLocalToWorld[23][46] = 114;
    out.transformMaskMappingLocalToWorld[23][47] = 242;
    out.transformMaskMappingLocalToWorld[23][48] = 10;
    out.transformMaskMappingLocalToWorld[23][49] = 138;
    out.transformMaskMappingLocalToWorld[23][50] = 42;
    out.transformMaskMappingLocalToWorld[23][51] = 170;
    out.transformMaskMappingLocalToWorld[23][52] = 74;
    out.transformMaskMappingLocalToWorld[23][53] = 202;
    out.transformMaskMappingLocalToWorld[23][54] = 106;
    out.transformMaskMappingLocalToWorld[23][55] = 234;
    out.transformMaskMappingLocalToWorld[23][56] = 26;
    out.transformMaskMappingLocalToWorld[23][57] = 154;
    out.transformMaskMappingLocalToWorld[23][58] = 58;
    out.transformMaskMappingLocalToWorld[23][59] = 186;
    out.transformMaskMappingLocalToWorld[23][60] = 90;
    out.transformMaskMappingLocalToWorld[23][61] = 218;
    out.transformMaskMappingLocalToWorld[23][62] = 122;
    out.transformMaskMappingLocalToWorld[23][63] = 250;
    out.transformMaskMappingLocalToWorld[23][64] = 4;
    out.transformMaskMappingLocalToWorld[23][65] = 132;
    out.transformMaskMappingLocalToWorld[23][66] = 36;
    out.transformMaskMappingLocalToWorld[23][67] = 164;
    out.transformMaskMappingLocalToWorld[23][68] = 68;
    out.transformMaskMappingLocalToWorld[23][69] = 196;
    out.transformMaskMappingLocalToWorld[23][70] = 100;
    out.transformMaskMappingLocalToWorld[23][71] = 228;
    out.transformMaskMappingLocalToWorld[23][72] = 20;
    out.transformMaskMappingLocalToWorld[23][73] = 148;
    out.transformMaskMappingLocalToWorld[23][74] = 52;
    out.transformMaskMappingLocalToWorld[23][75] = 180;
    out.transformMaskMappingLocalToWorld[23][76] = 84;
    out.transformMaskMappingLocalToWorld[23][77] = 212;
    out.transformMaskMappingLocalToWorld[23][78] = 116;
    out.transformMaskMappingLocalToWorld[23][79] = 244;
    out.transformMaskMappingLocalToWorld[23][80] = 12;
    out.transformMaskMappingLocalToWorld[23][81] = 140;
    out.transformMaskMappingLocalToWorld[23][82] = 44;
    out.transformMaskMappingLocalToWorld[23][83] = 172;
    out.transformMaskMappingLocalToWorld[23][84] = 76;
    out.transformMaskMappingLocalToWorld[23][85] = 204;
    out.transformMaskMappingLocalToWorld[23][86] = 108;
    out.transformMaskMappingLocalToWorld[23][87] = 236;
    out.transformMaskMappingLocalToWorld[23][88] = 28;
    out.transformMaskMappingLocalToWorld[23][89] = 156;
    out.transformMaskMappingLocalToWorld[23][90] = 60;
    out.transformMaskMappingLocalToWorld[23][91] = 188;
    out.transformMaskMappingLocalToWorld[23][92] = 92;
    out.transformMaskMappingLocalToWorld[23][93] = 220;
    out.transformMaskMappingLocalToWorld[23][94] = 124;
    out.transformMaskMappingLocalToWorld[23][95] = 252;
    out.transformMaskMappingLocalToWorld[23][96] = 6;
    out.transformMaskMappingLocalToWorld[23][97] = 134;
    out.transformMaskMappingLocalToWorld[23][98] = 38;
    out.transformMaskMappingLocalToWorld[23][99] = 166;
    out.transformMaskMappingLocalToWorld[23][100] = 70;
    out.transformMaskMappingLocalToWorld[23][101] = 198;
    out.transformMaskMappingLocalToWorld[23][102] = 102;
    out.transformMaskMappingLocalToWorld[23][103] = 230;
    out.transformMaskMappingLocalToWorld[23][104] = 22;
    out.transformMaskMappingLocalToWorld[23][105] = 150;
    out.transformMaskMappingLocalToWorld[23][106] = 54;
    out.transformMaskMappingLocalToWorld[23][107] = 182;
    out.transformMaskMappingLocalToWorld[23][108] = 86;
    out.transformMaskMappingLocalToWorld[23][109] = 214;
    out.transformMaskMappingLocalToWorld[23][110] = 118;
    out.transformMaskMappingLocalToWorld[23][111] = 246;
    out.transformMaskMappingLocalToWorld[23][112] = 14;
    out.transformMaskMappingLocalToWorld[23][113] = 142;
    out.transformMaskMappingLocalToWorld[23][114] = 46;
    out.transformMaskMappingLocalToWorld[23][115] = 174;
    out.transformMaskMappingLocalToWorld[23][116] = 78;
    out.transformMaskMappingLocalToWorld[23][117] = 206;
    out.transformMaskMappingLocalToWorld[23][118] = 110;
    out.transformMaskMappingLocalToWorld[23][119] = 238;
    out.transformMaskMappingLocalToWorld[23][120] = 30;
    out.transformMaskMappingLocalToWorld[23][121] = 158;
    out.transformMaskMappingLocalToWorld[23][122] = 62;
    out.transformMaskMappingLocalToWorld[23][123] = 190;
    out.transformMaskMappingLocalToWorld[23][124] = 94;
    out.transformMaskMappingLocalToWorld[23][125] = 222;
    out.transformMaskMappingLocalToWorld[23][126] = 126;
    out.transformMaskMappingLocalToWorld[23][127] = 254;
    out.transformMaskMappingLocalToWorld[23][128] = 1;
    out.transformMaskMappingLocalToWorld[23][129] = 129;
    out.transformMaskMappingLocalToWorld[23][130] = 33;
    out.transformMaskMappingLocalToWorld[23][131] = 161;
    out.transformMaskMappingLocalToWorld[23][132] = 65;
    out.transformMaskMappingLocalToWorld[23][133] = 193;
    out.transformMaskMappingLocalToWorld[23][134] = 97;
    out.transformMaskMappingLocalToWorld[23][135] = 225;
    out.transformMaskMappingLocalToWorld[23][136] = 17;
    out.transformMaskMappingLocalToWorld[23][137] = 145;
    out.transformMaskMappingLocalToWorld[23][138] = 49;
    out.transformMaskMappingLocalToWorld[23][139] = 177;
    out.transformMaskMappingLocalToWorld[23][140] = 81;
    out.transformMaskMappingLocalToWorld[23][141] = 209;
    out.transformMaskMappingLocalToWorld[23][142] = 113;
    out.transformMaskMappingLocalToWorld[23][143] = 241;
    out.transformMaskMappingLocalToWorld[23][144] = 9;
    out.transformMaskMappingLocalToWorld[23][145] = 137;
    out.transformMaskMappingLocalToWorld[23][146] = 41;
    out.transformMaskMappingLocalToWorld[23][147] = 169;
    out.transformMaskMappingLocalToWorld[23][148] = 73;
    out.transformMaskMappingLocalToWorld[23][149] = 201;
    out.transformMaskMappingLocalToWorld[23][150] = 105;
    out.transformMaskMappingLocalToWorld[23][151] = 233;
    out.transformMaskMappingLocalToWorld[23][152] = 25;
    out.transformMaskMappingLocalToWorld[23][153] = 153;
    out.transformMaskMappingLocalToWorld[23][154] = 57;
    out.transformMaskMappingLocalToWorld[23][155] = 185;
    out.transformMaskMappingLocalToWorld[23][156] = 89;
    out.transformMaskMappingLocalToWorld[23][157] = 217;
    out.transformMaskMappingLocalToWorld[23][158] = 121;
    out.transformMaskMappingLocalToWorld[23][159] = 249;
    out.transformMaskMappingLocalToWorld[23][160] = 3;
    out.transformMaskMappingLocalToWorld[23][161] = 131;
    out.transformMaskMappingLocalToWorld[23][162] = 35;
    out.transformMaskMappingLocalToWorld[23][163] = 163;
    out.transformMaskMappingLocalToWorld[23][164] = 67;
    out.transformMaskMappingLocalToWorld[23][165] = 195;
    out.transformMaskMappingLocalToWorld[23][166] = 99;
    out.transformMaskMappingLocalToWorld[23][167] = 227;
    out.transformMaskMappingLocalToWorld[23][168] = 19;
    out.transformMaskMappingLocalToWorld[23][169] = 147;
    out.transformMaskMappingLocalToWorld[23][170] = 51;
    out.transformMaskMappingLocalToWorld[23][171] = 179;
    out.transformMaskMappingLocalToWorld[23][172] = 83;
    out.transformMaskMappingLocalToWorld[23][173] = 211;
    out.transformMaskMappingLocalToWorld[23][174] = 115;
    out.transformMaskMappingLocalToWorld[23][175] = 243;
    out.transformMaskMappingLocalToWorld[23][176] = 11;
    out.transformMaskMappingLocalToWorld[23][177] = 139;
    out.transformMaskMappingLocalToWorld[23][178] = 43;
    out.transformMaskMappingLocalToWorld[23][179] = 171;
    out.transformMaskMappingLocalToWorld[23][180] = 75;
    out.transformMaskMappingLocalToWorld[23][181] = 203;
    out.transformMaskMappingLocalToWorld[23][182] = 107;
    out.transformMaskMappingLocalToWorld[23][183] = 235;
    out.transformMaskMappingLocalToWorld[23][184] = 27;
    out.transformMaskMappingLocalToWorld[23][185] = 155;
    out.transformMaskMappingLocalToWorld[23][186] = 59;
    out.transformMaskMappingLocalToWorld[23][187] = 187;
    out.transformMaskMappingLocalToWorld[23][188] = 91;
    out.transformMaskMappingLocalToWorld[23][189] = 219;
    out.transformMaskMappingLocalToWorld[23][190] = 123;
    out.transformMaskMappingLocalToWorld[23][191] = 251;
    out.transformMaskMappingLocalToWorld[23][192] = 5;
    out.transformMaskMappingLocalToWorld[23][193] = 133;
    out.transformMaskMappingLocalToWorld[23][194] = 37;
    out.transformMaskMappingLocalToWorld[23][195] = 165;
    out.transformMaskMappingLocalToWorld[23][196] = 69;
    out.transformMaskMappingLocalToWorld[23][197] = 197;
    out.transformMaskMappingLocalToWorld[23][198] = 101;
    out.transformMaskMappingLocalToWorld[23][199] = 229;
    out.transformMaskMappingLocalToWorld[23][200] = 21;
    out.transformMaskMappingLocalToWorld[23][201] = 149;
    out.transformMaskMappingLocalToWorld[23][202] = 53;
    out.transformMaskMappingLocalToWorld[23][203] = 181;
    out.transformMaskMappingLocalToWorld[23][204] = 85;
    out.transformMaskMappingLocalToWorld[23][205] = 213;
    out.transformMaskMappingLocalToWorld[23][206] = 117;
    out.transformMaskMappingLocalToWorld[23][207] = 245;
    out.transformMaskMappingLocalToWorld[23][208] = 13;
    out.transformMaskMappingLocalToWorld[23][209] = 141;
    out.transformMaskMappingLocalToWorld[23][210] = 45;
    out.transformMaskMappingLocalToWorld[23][211] = 173;
    out.transformMaskMappingLocalToWorld[23][212] = 77;
    out.transformMaskMappingLocalToWorld[23][213] = 205;
    out.transformMaskMappingLocalToWorld[23][214] = 109;
    out.transformMaskMappingLocalToWorld[23][215] = 237;
    out.transformMaskMappingLocalToWorld[23][216] = 29;
    out.transformMaskMappingLocalToWorld[23][217] = 157;
    out.transformMaskMappingLocalToWorld[23][218] = 61;
    out.transformMaskMappingLocalToWorld[23][219] = 189;
    out.transformMaskMappingLocalToWorld[23][220] = 93;
    out.transformMaskMappingLocalToWorld[23][221] = 221;
    out.transformMaskMappingLocalToWorld[23][222] = 125;
    out.transformMaskMappingLocalToWorld[23][223] = 253;
    out.transformMaskMappingLocalToWorld[23][224] = 7;
    out.transformMaskMappingLocalToWorld[23][225] = 135;
    out.transformMaskMappingLocalToWorld[23][226] = 39;
    out.transformMaskMappingLocalToWorld[23][227] = 167;
    out.transformMaskMappingLocalToWorld[23][228] = 71;
    out.transformMaskMappingLocalToWorld[23][229] = 199;
    out.transformMaskMappingLocalToWorld[23][230] = 103;
    out.transformMaskMappingLocalToWorld[23][231] = 231;
    out.transformMaskMappingLocalToWorld[23][232] = 23;
    out.transformMaskMappingLocalToWorld[23][233] = 151;
    out.transformMaskMappingLocalToWorld[23][234] = 55;
    out.transformMaskMappingLocalToWorld[23][235] = 183;
    out.transformMaskMappingLocalToWorld[23][236] = 87;
    out.transformMaskMappingLocalToWorld[23][237] = 215;
    out.transformMaskMappingLocalToWorld[23][238] = 119;
    out.transformMaskMappingLocalToWorld[23][239] = 247;
    out.transformMaskMappingLocalToWorld[23][240] = 15;
    out.transformMaskMappingLocalToWorld[23][241] = 143;
    out.transformMaskMappingLocalToWorld[23][242] = 47;
    out.transformMaskMappingLocalToWorld[23][243] = 175;
    out.transformMaskMappingLocalToWorld[23][244] = 79;
    out.transformMaskMappingLocalToWorld[23][245] = 207;
    out.transformMaskMappingLocalToWorld[23][246] = 111;
    out.transformMaskMappingLocalToWorld[23][247] = 239;
    out.transformMaskMappingLocalToWorld[23][248] = 31;
    out.transformMaskMappingLocalToWorld[23][249] = 159;
    out.transformMaskMappingLocalToWorld[23][250] = 63;
    out.transformMaskMappingLocalToWorld[23][251] = 191;
    out.transformMaskMappingLocalToWorld[23][252] = 95;
    out.transformMaskMappingLocalToWorld[23][253] = 223;
    out.transformMaskMappingLocalToWorld[23][254] = 127;
    out.transformMaskMappingLocalToWorld[23][255] = 255;
    out.transformMaskMappingLocalToWorld[24][0] = 0;
    out.transformMaskMappingLocalToWorld[24][1] = 1;
    out.transformMaskMappingLocalToWorld[24][2] = 16;
    out.transformMaskMappingLocalToWorld[24][3] = 17;
    out.transformMaskMappingLocalToWorld[24][4] = 2;
    out.transformMaskMappingLocalToWorld[24][5] = 3;
    out.transformMaskMappingLocalToWorld[24][6] = 18;
    out.transformMaskMappingLocalToWorld[24][7] = 19;
    out.transformMaskMappingLocalToWorld[24][8] = 32;
    out.transformMaskMappingLocalToWorld[24][9] = 33;
    out.transformMaskMappingLocalToWorld[24][10] = 48;
    out.transformMaskMappingLocalToWorld[24][11] = 49;
    out.transformMaskMappingLocalToWorld[24][12] = 34;
    out.transformMaskMappingLocalToWorld[24][13] = 35;
    out.transformMaskMappingLocalToWorld[24][14] = 50;
    out.transformMaskMappingLocalToWorld[24][15] = 51;
    out.transformMaskMappingLocalToWorld[24][16] = 4;
    out.transformMaskMappingLocalToWorld[24][17] = 5;
    out.transformMaskMappingLocalToWorld[24][18] = 20;
    out.transformMaskMappingLocalToWorld[24][19] = 21;
    out.transformMaskMappingLocalToWorld[24][20] = 6;
    out.transformMaskMappingLocalToWorld[24][21] = 7;
    out.transformMaskMappingLocalToWorld[24][22] = 22;
    out.transformMaskMappingLocalToWorld[24][23] = 23;
    out.transformMaskMappingLocalToWorld[24][24] = 36;
    out.transformMaskMappingLocalToWorld[24][25] = 37;
    out.transformMaskMappingLocalToWorld[24][26] = 52;
    out.transformMaskMappingLocalToWorld[24][27] = 53;
    out.transformMaskMappingLocalToWorld[24][28] = 38;
    out.transformMaskMappingLocalToWorld[24][29] = 39;
    out.transformMaskMappingLocalToWorld[24][30] = 54;
    out.transformMaskMappingLocalToWorld[24][31] = 55;
    out.transformMaskMappingLocalToWorld[24][32] = 64;
    out.transformMaskMappingLocalToWorld[24][33] = 65;
    out.transformMaskMappingLocalToWorld[24][34] = 80;
    out.transformMaskMappingLocalToWorld[24][35] = 81;
    out.transformMaskMappingLocalToWorld[24][36] = 66;
    out.transformMaskMappingLocalToWorld[24][37] = 67;
    out.transformMaskMappingLocalToWorld[24][38] = 82;
    out.transformMaskMappingLocalToWorld[24][39] = 83;
    out.transformMaskMappingLocalToWorld[24][40] = 96;
    out.transformMaskMappingLocalToWorld[24][41] = 97;
    out.transformMaskMappingLocalToWorld[24][42] = 112;
    out.transformMaskMappingLocalToWorld[24][43] = 113;
    out.transformMaskMappingLocalToWorld[24][44] = 98;
    out.transformMaskMappingLocalToWorld[24][45] = 99;
    out.transformMaskMappingLocalToWorld[24][46] = 114;
    out.transformMaskMappingLocalToWorld[24][47] = 115;
    out.transformMaskMappingLocalToWorld[24][48] = 68;
    out.transformMaskMappingLocalToWorld[24][49] = 69;
    out.transformMaskMappingLocalToWorld[24][50] = 84;
    out.transformMaskMappingLocalToWorld[24][51] = 85;
    out.transformMaskMappingLocalToWorld[24][52] = 70;
    out.transformMaskMappingLocalToWorld[24][53] = 71;
    out.transformMaskMappingLocalToWorld[24][54] = 86;
    out.transformMaskMappingLocalToWorld[24][55] = 87;
    out.transformMaskMappingLocalToWorld[24][56] = 100;
    out.transformMaskMappingLocalToWorld[24][57] = 101;
    out.transformMaskMappingLocalToWorld[24][58] = 116;
    out.transformMaskMappingLocalToWorld[24][59] = 117;
    out.transformMaskMappingLocalToWorld[24][60] = 102;
    out.transformMaskMappingLocalToWorld[24][61] = 103;
    out.transformMaskMappingLocalToWorld[24][62] = 118;
    out.transformMaskMappingLocalToWorld[24][63] = 119;
    out.transformMaskMappingLocalToWorld[24][64] = 8;
    out.transformMaskMappingLocalToWorld[24][65] = 9;
    out.transformMaskMappingLocalToWorld[24][66] = 24;
    out.transformMaskMappingLocalToWorld[24][67] = 25;
    out.transformMaskMappingLocalToWorld[24][68] = 10;
    out.transformMaskMappingLocalToWorld[24][69] = 11;
    out.transformMaskMappingLocalToWorld[24][70] = 26;
    out.transformMaskMappingLocalToWorld[24][71] = 27;
    out.transformMaskMappingLocalToWorld[24][72] = 40;
    out.transformMaskMappingLocalToWorld[24][73] = 41;
    out.transformMaskMappingLocalToWorld[24][74] = 56;
    out.transformMaskMappingLocalToWorld[24][75] = 57;
    out.transformMaskMappingLocalToWorld[24][76] = 42;
    out.transformMaskMappingLocalToWorld[24][77] = 43;
    out.transformMaskMappingLocalToWorld[24][78] = 58;
    out.transformMaskMappingLocalToWorld[24][79] = 59;
    out.transformMaskMappingLocalToWorld[24][80] = 12;
    out.transformMaskMappingLocalToWorld[24][81] = 13;
    out.transformMaskMappingLocalToWorld[24][82] = 28;
    out.transformMaskMappingLocalToWorld[24][83] = 29;
    out.transformMaskMappingLocalToWorld[24][84] = 14;
    out.transformMaskMappingLocalToWorld[24][85] = 15;
    out.transformMaskMappingLocalToWorld[24][86] = 30;
    out.transformMaskMappingLocalToWorld[24][87] = 31;
    out.transformMaskMappingLocalToWorld[24][88] = 44;
    out.transformMaskMappingLocalToWorld[24][89] = 45;
    out.transformMaskMappingLocalToWorld[24][90] = 60;
    out.transformMaskMappingLocalToWorld[24][91] = 61;
    out.transformMaskMappingLocalToWorld[24][92] = 46;
    out.transformMaskMappingLocalToWorld[24][93] = 47;
    out.transformMaskMappingLocalToWorld[24][94] = 62;
    out.transformMaskMappingLocalToWorld[24][95] = 63;
    out.transformMaskMappingLocalToWorld[24][96] = 72;
    out.transformMaskMappingLocalToWorld[24][97] = 73;
    out.transformMaskMappingLocalToWorld[24][98] = 88;
    out.transformMaskMappingLocalToWorld[24][99] = 89;
    out.transformMaskMappingLocalToWorld[24][100] = 74;
    out.transformMaskMappingLocalToWorld[24][101] = 75;
    out.transformMaskMappingLocalToWorld[24][102] = 90;
    out.transformMaskMappingLocalToWorld[24][103] = 91;
    out.transformMaskMappingLocalToWorld[24][104] = 104;
    out.transformMaskMappingLocalToWorld[24][105] = 105;
    out.transformMaskMappingLocalToWorld[24][106] = 120;
    out.transformMaskMappingLocalToWorld[24][107] = 121;
    out.transformMaskMappingLocalToWorld[24][108] = 106;
    out.transformMaskMappingLocalToWorld[24][109] = 107;
    out.transformMaskMappingLocalToWorld[24][110] = 122;
    out.transformMaskMappingLocalToWorld[24][111] = 123;
    out.transformMaskMappingLocalToWorld[24][112] = 76;
    out.transformMaskMappingLocalToWorld[24][113] = 77;
    out.transformMaskMappingLocalToWorld[24][114] = 92;
    out.transformMaskMappingLocalToWorld[24][115] = 93;
    out.transformMaskMappingLocalToWorld[24][116] = 78;
    out.transformMaskMappingLocalToWorld[24][117] = 79;
    out.transformMaskMappingLocalToWorld[24][118] = 94;
    out.transformMaskMappingLocalToWorld[24][119] = 95;
    out.transformMaskMappingLocalToWorld[24][120] = 108;
    out.transformMaskMappingLocalToWorld[24][121] = 109;
    out.transformMaskMappingLocalToWorld[24][122] = 124;
    out.transformMaskMappingLocalToWorld[24][123] = 125;
    out.transformMaskMappingLocalToWorld[24][124] = 110;
    out.transformMaskMappingLocalToWorld[24][125] = 111;
    out.transformMaskMappingLocalToWorld[24][126] = 126;
    out.transformMaskMappingLocalToWorld[24][127] = 127;
    out.transformMaskMappingLocalToWorld[24][128] = 128;
    out.transformMaskMappingLocalToWorld[24][129] = 129;
    out.transformMaskMappingLocalToWorld[24][130] = 144;
    out.transformMaskMappingLocalToWorld[24][131] = 145;
    out.transformMaskMappingLocalToWorld[24][132] = 130;
    out.transformMaskMappingLocalToWorld[24][133] = 131;
    out.transformMaskMappingLocalToWorld[24][134] = 146;
    out.transformMaskMappingLocalToWorld[24][135] = 147;
    out.transformMaskMappingLocalToWorld[24][136] = 160;
    out.transformMaskMappingLocalToWorld[24][137] = 161;
    out.transformMaskMappingLocalToWorld[24][138] = 176;
    out.transformMaskMappingLocalToWorld[24][139] = 177;
    out.transformMaskMappingLocalToWorld[24][140] = 162;
    out.transformMaskMappingLocalToWorld[24][141] = 163;
    out.transformMaskMappingLocalToWorld[24][142] = 178;
    out.transformMaskMappingLocalToWorld[24][143] = 179;
    out.transformMaskMappingLocalToWorld[24][144] = 132;
    out.transformMaskMappingLocalToWorld[24][145] = 133;
    out.transformMaskMappingLocalToWorld[24][146] = 148;
    out.transformMaskMappingLocalToWorld[24][147] = 149;
    out.transformMaskMappingLocalToWorld[24][148] = 134;
    out.transformMaskMappingLocalToWorld[24][149] = 135;
    out.transformMaskMappingLocalToWorld[24][150] = 150;
    out.transformMaskMappingLocalToWorld[24][151] = 151;
    out.transformMaskMappingLocalToWorld[24][152] = 164;
    out.transformMaskMappingLocalToWorld[24][153] = 165;
    out.transformMaskMappingLocalToWorld[24][154] = 180;
    out.transformMaskMappingLocalToWorld[24][155] = 181;
    out.transformMaskMappingLocalToWorld[24][156] = 166;
    out.transformMaskMappingLocalToWorld[24][157] = 167;
    out.transformMaskMappingLocalToWorld[24][158] = 182;
    out.transformMaskMappingLocalToWorld[24][159] = 183;
    out.transformMaskMappingLocalToWorld[24][160] = 192;
    out.transformMaskMappingLocalToWorld[24][161] = 193;
    out.transformMaskMappingLocalToWorld[24][162] = 208;
    out.transformMaskMappingLocalToWorld[24][163] = 209;
    out.transformMaskMappingLocalToWorld[24][164] = 194;
    out.transformMaskMappingLocalToWorld[24][165] = 195;
    out.transformMaskMappingLocalToWorld[24][166] = 210;
    out.transformMaskMappingLocalToWorld[24][167] = 211;
    out.transformMaskMappingLocalToWorld[24][168] = 224;
    out.transformMaskMappingLocalToWorld[24][169] = 225;
    out.transformMaskMappingLocalToWorld[24][170] = 240;
    out.transformMaskMappingLocalToWorld[24][171] = 241;
    out.transformMaskMappingLocalToWorld[24][172] = 226;
    out.transformMaskMappingLocalToWorld[24][173] = 227;
    out.transformMaskMappingLocalToWorld[24][174] = 242;
    out.transformMaskMappingLocalToWorld[24][175] = 243;
    out.transformMaskMappingLocalToWorld[24][176] = 196;
    out.transformMaskMappingLocalToWorld[24][177] = 197;
    out.transformMaskMappingLocalToWorld[24][178] = 212;
    out.transformMaskMappingLocalToWorld[24][179] = 213;
    out.transformMaskMappingLocalToWorld[24][180] = 198;
    out.transformMaskMappingLocalToWorld[24][181] = 199;
    out.transformMaskMappingLocalToWorld[24][182] = 214;
    out.transformMaskMappingLocalToWorld[24][183] = 215;
    out.transformMaskMappingLocalToWorld[24][184] = 228;
    out.transformMaskMappingLocalToWorld[24][185] = 229;
    out.transformMaskMappingLocalToWorld[24][186] = 244;
    out.transformMaskMappingLocalToWorld[24][187] = 245;
    out.transformMaskMappingLocalToWorld[24][188] = 230;
    out.transformMaskMappingLocalToWorld[24][189] = 231;
    out.transformMaskMappingLocalToWorld[24][190] = 246;
    out.transformMaskMappingLocalToWorld[24][191] = 247;
    out.transformMaskMappingLocalToWorld[24][192] = 136;
    out.transformMaskMappingLocalToWorld[24][193] = 137;
    out.transformMaskMappingLocalToWorld[24][194] = 152;
    out.transformMaskMappingLocalToWorld[24][195] = 153;
    out.transformMaskMappingLocalToWorld[24][196] = 138;
    out.transformMaskMappingLocalToWorld[24][197] = 139;
    out.transformMaskMappingLocalToWorld[24][198] = 154;
    out.transformMaskMappingLocalToWorld[24][199] = 155;
    out.transformMaskMappingLocalToWorld[24][200] = 168;
    out.transformMaskMappingLocalToWorld[24][201] = 169;
    out.transformMaskMappingLocalToWorld[24][202] = 184;
    out.transformMaskMappingLocalToWorld[24][203] = 185;
    out.transformMaskMappingLocalToWorld[24][204] = 170;
    out.transformMaskMappingLocalToWorld[24][205] = 171;
    out.transformMaskMappingLocalToWorld[24][206] = 186;
    out.transformMaskMappingLocalToWorld[24][207] = 187;
    out.transformMaskMappingLocalToWorld[24][208] = 140;
    out.transformMaskMappingLocalToWorld[24][209] = 141;
    out.transformMaskMappingLocalToWorld[24][210] = 156;
    out.transformMaskMappingLocalToWorld[24][211] = 157;
    out.transformMaskMappingLocalToWorld[24][212] = 142;
    out.transformMaskMappingLocalToWorld[24][213] = 143;
    out.transformMaskMappingLocalToWorld[24][214] = 158;
    out.transformMaskMappingLocalToWorld[24][215] = 159;
    out.transformMaskMappingLocalToWorld[24][216] = 172;
    out.transformMaskMappingLocalToWorld[24][217] = 173;
    out.transformMaskMappingLocalToWorld[24][218] = 188;
    out.transformMaskMappingLocalToWorld[24][219] = 189;
    out.transformMaskMappingLocalToWorld[24][220] = 174;
    out.transformMaskMappingLocalToWorld[24][221] = 175;
    out.transformMaskMappingLocalToWorld[24][222] = 190;
    out.transformMaskMappingLocalToWorld[24][223] = 191;
    out.transformMaskMappingLocalToWorld[24][224] = 200;
    out.transformMaskMappingLocalToWorld[24][225] = 201;
    out.transformMaskMappingLocalToWorld[24][226] = 216;
    out.transformMaskMappingLocalToWorld[24][227] = 217;
    out.transformMaskMappingLocalToWorld[24][228] = 202;
    out.transformMaskMappingLocalToWorld[24][229] = 203;
    out.transformMaskMappingLocalToWorld[24][230] = 218;
    out.transformMaskMappingLocalToWorld[24][231] = 219;
    out.transformMaskMappingLocalToWorld[24][232] = 232;
    out.transformMaskMappingLocalToWorld[24][233] = 233;
    out.transformMaskMappingLocalToWorld[24][234] = 248;
    out.transformMaskMappingLocalToWorld[24][235] = 249;
    out.transformMaskMappingLocalToWorld[24][236] = 234;
    out.transformMaskMappingLocalToWorld[24][237] = 235;
    out.transformMaskMappingLocalToWorld[24][238] = 250;
    out.transformMaskMappingLocalToWorld[24][239] = 251;
    out.transformMaskMappingLocalToWorld[24][240] = 204;
    out.transformMaskMappingLocalToWorld[24][241] = 205;
    out.transformMaskMappingLocalToWorld[24][242] = 220;
    out.transformMaskMappingLocalToWorld[24][243] = 221;
    out.transformMaskMappingLocalToWorld[24][244] = 206;
    out.transformMaskMappingLocalToWorld[24][245] = 207;
    out.transformMaskMappingLocalToWorld[24][246] = 222;
    out.transformMaskMappingLocalToWorld[24][247] = 223;
    out.transformMaskMappingLocalToWorld[24][248] = 236;
    out.transformMaskMappingLocalToWorld[24][249] = 237;
    out.transformMaskMappingLocalToWorld[24][250] = 252;
    out.transformMaskMappingLocalToWorld[24][251] = 253;
    out.transformMaskMappingLocalToWorld[24][252] = 238;
    out.transformMaskMappingLocalToWorld[24][253] = 239;
    out.transformMaskMappingLocalToWorld[24][254] = 254;
    out.transformMaskMappingLocalToWorld[24][255] = 255;
    out.transformMaskMappingLocalToWorld[25][0] = 0;
    out.transformMaskMappingLocalToWorld[25][1] = 2;
    out.transformMaskMappingLocalToWorld[25][2] = 32;
    out.transformMaskMappingLocalToWorld[25][3] = 34;
    out.transformMaskMappingLocalToWorld[25][4] = 1;
    out.transformMaskMappingLocalToWorld[25][5] = 3;
    out.transformMaskMappingLocalToWorld[25][6] = 33;
    out.transformMaskMappingLocalToWorld[25][7] = 35;
    out.transformMaskMappingLocalToWorld[25][8] = 16;
    out.transformMaskMappingLocalToWorld[25][9] = 18;
    out.transformMaskMappingLocalToWorld[25][10] = 48;
    out.transformMaskMappingLocalToWorld[25][11] = 50;
    out.transformMaskMappingLocalToWorld[25][12] = 17;
    out.transformMaskMappingLocalToWorld[25][13] = 19;
    out.transformMaskMappingLocalToWorld[25][14] = 49;
    out.transformMaskMappingLocalToWorld[25][15] = 51;
    out.transformMaskMappingLocalToWorld[25][16] = 8;
    out.transformMaskMappingLocalToWorld[25][17] = 10;
    out.transformMaskMappingLocalToWorld[25][18] = 40;
    out.transformMaskMappingLocalToWorld[25][19] = 42;
    out.transformMaskMappingLocalToWorld[25][20] = 9;
    out.transformMaskMappingLocalToWorld[25][21] = 11;
    out.transformMaskMappingLocalToWorld[25][22] = 41;
    out.transformMaskMappingLocalToWorld[25][23] = 43;
    out.transformMaskMappingLocalToWorld[25][24] = 24;
    out.transformMaskMappingLocalToWorld[25][25] = 26;
    out.transformMaskMappingLocalToWorld[25][26] = 56;
    out.transformMaskMappingLocalToWorld[25][27] = 58;
    out.transformMaskMappingLocalToWorld[25][28] = 25;
    out.transformMaskMappingLocalToWorld[25][29] = 27;
    out.transformMaskMappingLocalToWorld[25][30] = 57;
    out.transformMaskMappingLocalToWorld[25][31] = 59;
    out.transformMaskMappingLocalToWorld[25][32] = 128;
    out.transformMaskMappingLocalToWorld[25][33] = 130;
    out.transformMaskMappingLocalToWorld[25][34] = 160;
    out.transformMaskMappingLocalToWorld[25][35] = 162;
    out.transformMaskMappingLocalToWorld[25][36] = 129;
    out.transformMaskMappingLocalToWorld[25][37] = 131;
    out.transformMaskMappingLocalToWorld[25][38] = 161;
    out.transformMaskMappingLocalToWorld[25][39] = 163;
    out.transformMaskMappingLocalToWorld[25][40] = 144;
    out.transformMaskMappingLocalToWorld[25][41] = 146;
    out.transformMaskMappingLocalToWorld[25][42] = 176;
    out.transformMaskMappingLocalToWorld[25][43] = 178;
    out.transformMaskMappingLocalToWorld[25][44] = 145;
    out.transformMaskMappingLocalToWorld[25][45] = 147;
    out.transformMaskMappingLocalToWorld[25][46] = 177;
    out.transformMaskMappingLocalToWorld[25][47] = 179;
    out.transformMaskMappingLocalToWorld[25][48] = 136;
    out.transformMaskMappingLocalToWorld[25][49] = 138;
    out.transformMaskMappingLocalToWorld[25][50] = 168;
    out.transformMaskMappingLocalToWorld[25][51] = 170;
    out.transformMaskMappingLocalToWorld[25][52] = 137;
    out.transformMaskMappingLocalToWorld[25][53] = 139;
    out.transformMaskMappingLocalToWorld[25][54] = 169;
    out.transformMaskMappingLocalToWorld[25][55] = 171;
    out.transformMaskMappingLocalToWorld[25][56] = 152;
    out.transformMaskMappingLocalToWorld[25][57] = 154;
    out.transformMaskMappingLocalToWorld[25][58] = 184;
    out.transformMaskMappingLocalToWorld[25][59] = 186;
    out.transformMaskMappingLocalToWorld[25][60] = 153;
    out.transformMaskMappingLocalToWorld[25][61] = 155;
    out.transformMaskMappingLocalToWorld[25][62] = 185;
    out.transformMaskMappingLocalToWorld[25][63] = 187;
    out.transformMaskMappingLocalToWorld[25][64] = 4;
    out.transformMaskMappingLocalToWorld[25][65] = 6;
    out.transformMaskMappingLocalToWorld[25][66] = 36;
    out.transformMaskMappingLocalToWorld[25][67] = 38;
    out.transformMaskMappingLocalToWorld[25][68] = 5;
    out.transformMaskMappingLocalToWorld[25][69] = 7;
    out.transformMaskMappingLocalToWorld[25][70] = 37;
    out.transformMaskMappingLocalToWorld[25][71] = 39;
    out.transformMaskMappingLocalToWorld[25][72] = 20;
    out.transformMaskMappingLocalToWorld[25][73] = 22;
    out.transformMaskMappingLocalToWorld[25][74] = 52;
    out.transformMaskMappingLocalToWorld[25][75] = 54;
    out.transformMaskMappingLocalToWorld[25][76] = 21;
    out.transformMaskMappingLocalToWorld[25][77] = 23;
    out.transformMaskMappingLocalToWorld[25][78] = 53;
    out.transformMaskMappingLocalToWorld[25][79] = 55;
    out.transformMaskMappingLocalToWorld[25][80] = 12;
    out.transformMaskMappingLocalToWorld[25][81] = 14;
    out.transformMaskMappingLocalToWorld[25][82] = 44;
    out.transformMaskMappingLocalToWorld[25][83] = 46;
    out.transformMaskMappingLocalToWorld[25][84] = 13;
    out.transformMaskMappingLocalToWorld[25][85] = 15;
    out.transformMaskMappingLocalToWorld[25][86] = 45;
    out.transformMaskMappingLocalToWorld[25][87] = 47;
    out.transformMaskMappingLocalToWorld[25][88] = 28;
    out.transformMaskMappingLocalToWorld[25][89] = 30;
    out.transformMaskMappingLocalToWorld[25][90] = 60;
    out.transformMaskMappingLocalToWorld[25][91] = 62;
    out.transformMaskMappingLocalToWorld[25][92] = 29;
    out.transformMaskMappingLocalToWorld[25][93] = 31;
    out.transformMaskMappingLocalToWorld[25][94] = 61;
    out.transformMaskMappingLocalToWorld[25][95] = 63;
    out.transformMaskMappingLocalToWorld[25][96] = 132;
    out.transformMaskMappingLocalToWorld[25][97] = 134;
    out.transformMaskMappingLocalToWorld[25][98] = 164;
    out.transformMaskMappingLocalToWorld[25][99] = 166;
    out.transformMaskMappingLocalToWorld[25][100] = 133;
    out.transformMaskMappingLocalToWorld[25][101] = 135;
    out.transformMaskMappingLocalToWorld[25][102] = 165;
    out.transformMaskMappingLocalToWorld[25][103] = 167;
    out.transformMaskMappingLocalToWorld[25][104] = 148;
    out.transformMaskMappingLocalToWorld[25][105] = 150;
    out.transformMaskMappingLocalToWorld[25][106] = 180;
    out.transformMaskMappingLocalToWorld[25][107] = 182;
    out.transformMaskMappingLocalToWorld[25][108] = 149;
    out.transformMaskMappingLocalToWorld[25][109] = 151;
    out.transformMaskMappingLocalToWorld[25][110] = 181;
    out.transformMaskMappingLocalToWorld[25][111] = 183;
    out.transformMaskMappingLocalToWorld[25][112] = 140;
    out.transformMaskMappingLocalToWorld[25][113] = 142;
    out.transformMaskMappingLocalToWorld[25][114] = 172;
    out.transformMaskMappingLocalToWorld[25][115] = 174;
    out.transformMaskMappingLocalToWorld[25][116] = 141;
    out.transformMaskMappingLocalToWorld[25][117] = 143;
    out.transformMaskMappingLocalToWorld[25][118] = 173;
    out.transformMaskMappingLocalToWorld[25][119] = 175;
    out.transformMaskMappingLocalToWorld[25][120] = 156;
    out.transformMaskMappingLocalToWorld[25][121] = 158;
    out.transformMaskMappingLocalToWorld[25][122] = 188;
    out.transformMaskMappingLocalToWorld[25][123] = 190;
    out.transformMaskMappingLocalToWorld[25][124] = 157;
    out.transformMaskMappingLocalToWorld[25][125] = 159;
    out.transformMaskMappingLocalToWorld[25][126] = 189;
    out.transformMaskMappingLocalToWorld[25][127] = 191;
    out.transformMaskMappingLocalToWorld[25][128] = 64;
    out.transformMaskMappingLocalToWorld[25][129] = 66;
    out.transformMaskMappingLocalToWorld[25][130] = 96;
    out.transformMaskMappingLocalToWorld[25][131] = 98;
    out.transformMaskMappingLocalToWorld[25][132] = 65;
    out.transformMaskMappingLocalToWorld[25][133] = 67;
    out.transformMaskMappingLocalToWorld[25][134] = 97;
    out.transformMaskMappingLocalToWorld[25][135] = 99;
    out.transformMaskMappingLocalToWorld[25][136] = 80;
    out.transformMaskMappingLocalToWorld[25][137] = 82;
    out.transformMaskMappingLocalToWorld[25][138] = 112;
    out.transformMaskMappingLocalToWorld[25][139] = 114;
    out.transformMaskMappingLocalToWorld[25][140] = 81;
    out.transformMaskMappingLocalToWorld[25][141] = 83;
    out.transformMaskMappingLocalToWorld[25][142] = 113;
    out.transformMaskMappingLocalToWorld[25][143] = 115;
    out.transformMaskMappingLocalToWorld[25][144] = 72;
    out.transformMaskMappingLocalToWorld[25][145] = 74;
    out.transformMaskMappingLocalToWorld[25][146] = 104;
    out.transformMaskMappingLocalToWorld[25][147] = 106;
    out.transformMaskMappingLocalToWorld[25][148] = 73;
    out.transformMaskMappingLocalToWorld[25][149] = 75;
    out.transformMaskMappingLocalToWorld[25][150] = 105;
    out.transformMaskMappingLocalToWorld[25][151] = 107;
    out.transformMaskMappingLocalToWorld[25][152] = 88;
    out.transformMaskMappingLocalToWorld[25][153] = 90;
    out.transformMaskMappingLocalToWorld[25][154] = 120;
    out.transformMaskMappingLocalToWorld[25][155] = 122;
    out.transformMaskMappingLocalToWorld[25][156] = 89;
    out.transformMaskMappingLocalToWorld[25][157] = 91;
    out.transformMaskMappingLocalToWorld[25][158] = 121;
    out.transformMaskMappingLocalToWorld[25][159] = 123;
    out.transformMaskMappingLocalToWorld[25][160] = 192;
    out.transformMaskMappingLocalToWorld[25][161] = 194;
    out.transformMaskMappingLocalToWorld[25][162] = 224;
    out.transformMaskMappingLocalToWorld[25][163] = 226;
    out.transformMaskMappingLocalToWorld[25][164] = 193;
    out.transformMaskMappingLocalToWorld[25][165] = 195;
    out.transformMaskMappingLocalToWorld[25][166] = 225;
    out.transformMaskMappingLocalToWorld[25][167] = 227;
    out.transformMaskMappingLocalToWorld[25][168] = 208;
    out.transformMaskMappingLocalToWorld[25][169] = 210;
    out.transformMaskMappingLocalToWorld[25][170] = 240;
    out.transformMaskMappingLocalToWorld[25][171] = 242;
    out.transformMaskMappingLocalToWorld[25][172] = 209;
    out.transformMaskMappingLocalToWorld[25][173] = 211;
    out.transformMaskMappingLocalToWorld[25][174] = 241;
    out.transformMaskMappingLocalToWorld[25][175] = 243;
    out.transformMaskMappingLocalToWorld[25][176] = 200;
    out.transformMaskMappingLocalToWorld[25][177] = 202;
    out.transformMaskMappingLocalToWorld[25][178] = 232;
    out.transformMaskMappingLocalToWorld[25][179] = 234;
    out.transformMaskMappingLocalToWorld[25][180] = 201;
    out.transformMaskMappingLocalToWorld[25][181] = 203;
    out.transformMaskMappingLocalToWorld[25][182] = 233;
    out.transformMaskMappingLocalToWorld[25][183] = 235;
    out.transformMaskMappingLocalToWorld[25][184] = 216;
    out.transformMaskMappingLocalToWorld[25][185] = 218;
    out.transformMaskMappingLocalToWorld[25][186] = 248;
    out.transformMaskMappingLocalToWorld[25][187] = 250;
    out.transformMaskMappingLocalToWorld[25][188] = 217;
    out.transformMaskMappingLocalToWorld[25][189] = 219;
    out.transformMaskMappingLocalToWorld[25][190] = 249;
    out.transformMaskMappingLocalToWorld[25][191] = 251;
    out.transformMaskMappingLocalToWorld[25][192] = 68;
    out.transformMaskMappingLocalToWorld[25][193] = 70;
    out.transformMaskMappingLocalToWorld[25][194] = 100;
    out.transformMaskMappingLocalToWorld[25][195] = 102;
    out.transformMaskMappingLocalToWorld[25][196] = 69;
    out.transformMaskMappingLocalToWorld[25][197] = 71;
    out.transformMaskMappingLocalToWorld[25][198] = 101;
    out.transformMaskMappingLocalToWorld[25][199] = 103;
    out.transformMaskMappingLocalToWorld[25][200] = 84;
    out.transformMaskMappingLocalToWorld[25][201] = 86;
    out.transformMaskMappingLocalToWorld[25][202] = 116;
    out.transformMaskMappingLocalToWorld[25][203] = 118;
    out.transformMaskMappingLocalToWorld[25][204] = 85;
    out.transformMaskMappingLocalToWorld[25][205] = 87;
    out.transformMaskMappingLocalToWorld[25][206] = 117;
    out.transformMaskMappingLocalToWorld[25][207] = 119;
    out.transformMaskMappingLocalToWorld[25][208] = 76;
    out.transformMaskMappingLocalToWorld[25][209] = 78;
    out.transformMaskMappingLocalToWorld[25][210] = 108;
    out.transformMaskMappingLocalToWorld[25][211] = 110;
    out.transformMaskMappingLocalToWorld[25][212] = 77;
    out.transformMaskMappingLocalToWorld[25][213] = 79;
    out.transformMaskMappingLocalToWorld[25][214] = 109;
    out.transformMaskMappingLocalToWorld[25][215] = 111;
    out.transformMaskMappingLocalToWorld[25][216] = 92;
    out.transformMaskMappingLocalToWorld[25][217] = 94;
    out.transformMaskMappingLocalToWorld[25][218] = 124;
    out.transformMaskMappingLocalToWorld[25][219] = 126;
    out.transformMaskMappingLocalToWorld[25][220] = 93;
    out.transformMaskMappingLocalToWorld[25][221] = 95;
    out.transformMaskMappingLocalToWorld[25][222] = 125;
    out.transformMaskMappingLocalToWorld[25][223] = 127;
    out.transformMaskMappingLocalToWorld[25][224] = 196;
    out.transformMaskMappingLocalToWorld[25][225] = 198;
    out.transformMaskMappingLocalToWorld[25][226] = 228;
    out.transformMaskMappingLocalToWorld[25][227] = 230;
    out.transformMaskMappingLocalToWorld[25][228] = 197;
    out.transformMaskMappingLocalToWorld[25][229] = 199;
    out.transformMaskMappingLocalToWorld[25][230] = 229;
    out.transformMaskMappingLocalToWorld[25][231] = 231;
    out.transformMaskMappingLocalToWorld[25][232] = 212;
    out.transformMaskMappingLocalToWorld[25][233] = 214;
    out.transformMaskMappingLocalToWorld[25][234] = 244;
    out.transformMaskMappingLocalToWorld[25][235] = 246;
    out.transformMaskMappingLocalToWorld[25][236] = 213;
    out.transformMaskMappingLocalToWorld[25][237] = 215;
    out.transformMaskMappingLocalToWorld[25][238] = 245;
    out.transformMaskMappingLocalToWorld[25][239] = 247;
    out.transformMaskMappingLocalToWorld[25][240] = 204;
    out.transformMaskMappingLocalToWorld[25][241] = 206;
    out.transformMaskMappingLocalToWorld[25][242] = 236;
    out.transformMaskMappingLocalToWorld[25][243] = 238;
    out.transformMaskMappingLocalToWorld[25][244] = 205;
    out.transformMaskMappingLocalToWorld[25][245] = 207;
    out.transformMaskMappingLocalToWorld[25][246] = 237;
    out.transformMaskMappingLocalToWorld[25][247] = 239;
    out.transformMaskMappingLocalToWorld[25][248] = 220;
    out.transformMaskMappingLocalToWorld[25][249] = 222;
    out.transformMaskMappingLocalToWorld[25][250] = 252;
    out.transformMaskMappingLocalToWorld[25][251] = 254;
    out.transformMaskMappingLocalToWorld[25][252] = 221;
    out.transformMaskMappingLocalToWorld[25][253] = 223;
    out.transformMaskMappingLocalToWorld[25][254] = 253;
    out.transformMaskMappingLocalToWorld[25][255] = 255;
    out.transformMaskMappingLocalToWorld[26][0] = 0;
    out.transformMaskMappingLocalToWorld[26][1] = 4;
    out.transformMaskMappingLocalToWorld[26][2] = 64;
    out.transformMaskMappingLocalToWorld[26][3] = 68;
    out.transformMaskMappingLocalToWorld[26][4] = 8;
    out.transformMaskMappingLocalToWorld[26][5] = 12;
    out.transformMaskMappingLocalToWorld[26][6] = 72;
    out.transformMaskMappingLocalToWorld[26][7] = 76;
    out.transformMaskMappingLocalToWorld[26][8] = 128;
    out.transformMaskMappingLocalToWorld[26][9] = 132;
    out.transformMaskMappingLocalToWorld[26][10] = 192;
    out.transformMaskMappingLocalToWorld[26][11] = 196;
    out.transformMaskMappingLocalToWorld[26][12] = 136;
    out.transformMaskMappingLocalToWorld[26][13] = 140;
    out.transformMaskMappingLocalToWorld[26][14] = 200;
    out.transformMaskMappingLocalToWorld[26][15] = 204;
    out.transformMaskMappingLocalToWorld[26][16] = 1;
    out.transformMaskMappingLocalToWorld[26][17] = 5;
    out.transformMaskMappingLocalToWorld[26][18] = 65;
    out.transformMaskMappingLocalToWorld[26][19] = 69;
    out.transformMaskMappingLocalToWorld[26][20] = 9;
    out.transformMaskMappingLocalToWorld[26][21] = 13;
    out.transformMaskMappingLocalToWorld[26][22] = 73;
    out.transformMaskMappingLocalToWorld[26][23] = 77;
    out.transformMaskMappingLocalToWorld[26][24] = 129;
    out.transformMaskMappingLocalToWorld[26][25] = 133;
    out.transformMaskMappingLocalToWorld[26][26] = 193;
    out.transformMaskMappingLocalToWorld[26][27] = 197;
    out.transformMaskMappingLocalToWorld[26][28] = 137;
    out.transformMaskMappingLocalToWorld[26][29] = 141;
    out.transformMaskMappingLocalToWorld[26][30] = 201;
    out.transformMaskMappingLocalToWorld[26][31] = 205;
    out.transformMaskMappingLocalToWorld[26][32] = 16;
    out.transformMaskMappingLocalToWorld[26][33] = 20;
    out.transformMaskMappingLocalToWorld[26][34] = 80;
    out.transformMaskMappingLocalToWorld[26][35] = 84;
    out.transformMaskMappingLocalToWorld[26][36] = 24;
    out.transformMaskMappingLocalToWorld[26][37] = 28;
    out.transformMaskMappingLocalToWorld[26][38] = 88;
    out.transformMaskMappingLocalToWorld[26][39] = 92;
    out.transformMaskMappingLocalToWorld[26][40] = 144;
    out.transformMaskMappingLocalToWorld[26][41] = 148;
    out.transformMaskMappingLocalToWorld[26][42] = 208;
    out.transformMaskMappingLocalToWorld[26][43] = 212;
    out.transformMaskMappingLocalToWorld[26][44] = 152;
    out.transformMaskMappingLocalToWorld[26][45] = 156;
    out.transformMaskMappingLocalToWorld[26][46] = 216;
    out.transformMaskMappingLocalToWorld[26][47] = 220;
    out.transformMaskMappingLocalToWorld[26][48] = 17;
    out.transformMaskMappingLocalToWorld[26][49] = 21;
    out.transformMaskMappingLocalToWorld[26][50] = 81;
    out.transformMaskMappingLocalToWorld[26][51] = 85;
    out.transformMaskMappingLocalToWorld[26][52] = 25;
    out.transformMaskMappingLocalToWorld[26][53] = 29;
    out.transformMaskMappingLocalToWorld[26][54] = 89;
    out.transformMaskMappingLocalToWorld[26][55] = 93;
    out.transformMaskMappingLocalToWorld[26][56] = 145;
    out.transformMaskMappingLocalToWorld[26][57] = 149;
    out.transformMaskMappingLocalToWorld[26][58] = 209;
    out.transformMaskMappingLocalToWorld[26][59] = 213;
    out.transformMaskMappingLocalToWorld[26][60] = 153;
    out.transformMaskMappingLocalToWorld[26][61] = 157;
    out.transformMaskMappingLocalToWorld[26][62] = 217;
    out.transformMaskMappingLocalToWorld[26][63] = 221;
    out.transformMaskMappingLocalToWorld[26][64] = 2;
    out.transformMaskMappingLocalToWorld[26][65] = 6;
    out.transformMaskMappingLocalToWorld[26][66] = 66;
    out.transformMaskMappingLocalToWorld[26][67] = 70;
    out.transformMaskMappingLocalToWorld[26][68] = 10;
    out.transformMaskMappingLocalToWorld[26][69] = 14;
    out.transformMaskMappingLocalToWorld[26][70] = 74;
    out.transformMaskMappingLocalToWorld[26][71] = 78;
    out.transformMaskMappingLocalToWorld[26][72] = 130;
    out.transformMaskMappingLocalToWorld[26][73] = 134;
    out.transformMaskMappingLocalToWorld[26][74] = 194;
    out.transformMaskMappingLocalToWorld[26][75] = 198;
    out.transformMaskMappingLocalToWorld[26][76] = 138;
    out.transformMaskMappingLocalToWorld[26][77] = 142;
    out.transformMaskMappingLocalToWorld[26][78] = 202;
    out.transformMaskMappingLocalToWorld[26][79] = 206;
    out.transformMaskMappingLocalToWorld[26][80] = 3;
    out.transformMaskMappingLocalToWorld[26][81] = 7;
    out.transformMaskMappingLocalToWorld[26][82] = 67;
    out.transformMaskMappingLocalToWorld[26][83] = 71;
    out.transformMaskMappingLocalToWorld[26][84] = 11;
    out.transformMaskMappingLocalToWorld[26][85] = 15;
    out.transformMaskMappingLocalToWorld[26][86] = 75;
    out.transformMaskMappingLocalToWorld[26][87] = 79;
    out.transformMaskMappingLocalToWorld[26][88] = 131;
    out.transformMaskMappingLocalToWorld[26][89] = 135;
    out.transformMaskMappingLocalToWorld[26][90] = 195;
    out.transformMaskMappingLocalToWorld[26][91] = 199;
    out.transformMaskMappingLocalToWorld[26][92] = 139;
    out.transformMaskMappingLocalToWorld[26][93] = 143;
    out.transformMaskMappingLocalToWorld[26][94] = 203;
    out.transformMaskMappingLocalToWorld[26][95] = 207;
    out.transformMaskMappingLocalToWorld[26][96] = 18;
    out.transformMaskMappingLocalToWorld[26][97] = 22;
    out.transformMaskMappingLocalToWorld[26][98] = 82;
    out.transformMaskMappingLocalToWorld[26][99] = 86;
    out.transformMaskMappingLocalToWorld[26][100] = 26;
    out.transformMaskMappingLocalToWorld[26][101] = 30;
    out.transformMaskMappingLocalToWorld[26][102] = 90;
    out.transformMaskMappingLocalToWorld[26][103] = 94;
    out.transformMaskMappingLocalToWorld[26][104] = 146;
    out.transformMaskMappingLocalToWorld[26][105] = 150;
    out.transformMaskMappingLocalToWorld[26][106] = 210;
    out.transformMaskMappingLocalToWorld[26][107] = 214;
    out.transformMaskMappingLocalToWorld[26][108] = 154;
    out.transformMaskMappingLocalToWorld[26][109] = 158;
    out.transformMaskMappingLocalToWorld[26][110] = 218;
    out.transformMaskMappingLocalToWorld[26][111] = 222;
    out.transformMaskMappingLocalToWorld[26][112] = 19;
    out.transformMaskMappingLocalToWorld[26][113] = 23;
    out.transformMaskMappingLocalToWorld[26][114] = 83;
    out.transformMaskMappingLocalToWorld[26][115] = 87;
    out.transformMaskMappingLocalToWorld[26][116] = 27;
    out.transformMaskMappingLocalToWorld[26][117] = 31;
    out.transformMaskMappingLocalToWorld[26][118] = 91;
    out.transformMaskMappingLocalToWorld[26][119] = 95;
    out.transformMaskMappingLocalToWorld[26][120] = 147;
    out.transformMaskMappingLocalToWorld[26][121] = 151;
    out.transformMaskMappingLocalToWorld[26][122] = 211;
    out.transformMaskMappingLocalToWorld[26][123] = 215;
    out.transformMaskMappingLocalToWorld[26][124] = 155;
    out.transformMaskMappingLocalToWorld[26][125] = 159;
    out.transformMaskMappingLocalToWorld[26][126] = 219;
    out.transformMaskMappingLocalToWorld[26][127] = 223;
    out.transformMaskMappingLocalToWorld[26][128] = 32;
    out.transformMaskMappingLocalToWorld[26][129] = 36;
    out.transformMaskMappingLocalToWorld[26][130] = 96;
    out.transformMaskMappingLocalToWorld[26][131] = 100;
    out.transformMaskMappingLocalToWorld[26][132] = 40;
    out.transformMaskMappingLocalToWorld[26][133] = 44;
    out.transformMaskMappingLocalToWorld[26][134] = 104;
    out.transformMaskMappingLocalToWorld[26][135] = 108;
    out.transformMaskMappingLocalToWorld[26][136] = 160;
    out.transformMaskMappingLocalToWorld[26][137] = 164;
    out.transformMaskMappingLocalToWorld[26][138] = 224;
    out.transformMaskMappingLocalToWorld[26][139] = 228;
    out.transformMaskMappingLocalToWorld[26][140] = 168;
    out.transformMaskMappingLocalToWorld[26][141] = 172;
    out.transformMaskMappingLocalToWorld[26][142] = 232;
    out.transformMaskMappingLocalToWorld[26][143] = 236;
    out.transformMaskMappingLocalToWorld[26][144] = 33;
    out.transformMaskMappingLocalToWorld[26][145] = 37;
    out.transformMaskMappingLocalToWorld[26][146] = 97;
    out.transformMaskMappingLocalToWorld[26][147] = 101;
    out.transformMaskMappingLocalToWorld[26][148] = 41;
    out.transformMaskMappingLocalToWorld[26][149] = 45;
    out.transformMaskMappingLocalToWorld[26][150] = 105;
    out.transformMaskMappingLocalToWorld[26][151] = 109;
    out.transformMaskMappingLocalToWorld[26][152] = 161;
    out.transformMaskMappingLocalToWorld[26][153] = 165;
    out.transformMaskMappingLocalToWorld[26][154] = 225;
    out.transformMaskMappingLocalToWorld[26][155] = 229;
    out.transformMaskMappingLocalToWorld[26][156] = 169;
    out.transformMaskMappingLocalToWorld[26][157] = 173;
    out.transformMaskMappingLocalToWorld[26][158] = 233;
    out.transformMaskMappingLocalToWorld[26][159] = 237;
    out.transformMaskMappingLocalToWorld[26][160] = 48;
    out.transformMaskMappingLocalToWorld[26][161] = 52;
    out.transformMaskMappingLocalToWorld[26][162] = 112;
    out.transformMaskMappingLocalToWorld[26][163] = 116;
    out.transformMaskMappingLocalToWorld[26][164] = 56;
    out.transformMaskMappingLocalToWorld[26][165] = 60;
    out.transformMaskMappingLocalToWorld[26][166] = 120;
    out.transformMaskMappingLocalToWorld[26][167] = 124;
    out.transformMaskMappingLocalToWorld[26][168] = 176;
    out.transformMaskMappingLocalToWorld[26][169] = 180;
    out.transformMaskMappingLocalToWorld[26][170] = 240;
    out.transformMaskMappingLocalToWorld[26][171] = 244;
    out.transformMaskMappingLocalToWorld[26][172] = 184;
    out.transformMaskMappingLocalToWorld[26][173] = 188;
    out.transformMaskMappingLocalToWorld[26][174] = 248;
    out.transformMaskMappingLocalToWorld[26][175] = 252;
    out.transformMaskMappingLocalToWorld[26][176] = 49;
    out.transformMaskMappingLocalToWorld[26][177] = 53;
    out.transformMaskMappingLocalToWorld[26][178] = 113;
    out.transformMaskMappingLocalToWorld[26][179] = 117;
    out.transformMaskMappingLocalToWorld[26][180] = 57;
    out.transformMaskMappingLocalToWorld[26][181] = 61;
    out.transformMaskMappingLocalToWorld[26][182] = 121;
    out.transformMaskMappingLocalToWorld[26][183] = 125;
    out.transformMaskMappingLocalToWorld[26][184] = 177;
    out.transformMaskMappingLocalToWorld[26][185] = 181;
    out.transformMaskMappingLocalToWorld[26][186] = 241;
    out.transformMaskMappingLocalToWorld[26][187] = 245;
    out.transformMaskMappingLocalToWorld[26][188] = 185;
    out.transformMaskMappingLocalToWorld[26][189] = 189;
    out.transformMaskMappingLocalToWorld[26][190] = 249;
    out.transformMaskMappingLocalToWorld[26][191] = 253;
    out.transformMaskMappingLocalToWorld[26][192] = 34;
    out.transformMaskMappingLocalToWorld[26][193] = 38;
    out.transformMaskMappingLocalToWorld[26][194] = 98;
    out.transformMaskMappingLocalToWorld[26][195] = 102;
    out.transformMaskMappingLocalToWorld[26][196] = 42;
    out.transformMaskMappingLocalToWorld[26][197] = 46;
    out.transformMaskMappingLocalToWorld[26][198] = 106;
    out.transformMaskMappingLocalToWorld[26][199] = 110;
    out.transformMaskMappingLocalToWorld[26][200] = 162;
    out.transformMaskMappingLocalToWorld[26][201] = 166;
    out.transformMaskMappingLocalToWorld[26][202] = 226;
    out.transformMaskMappingLocalToWorld[26][203] = 230;
    out.transformMaskMappingLocalToWorld[26][204] = 170;
    out.transformMaskMappingLocalToWorld[26][205] = 174;
    out.transformMaskMappingLocalToWorld[26][206] = 234;
    out.transformMaskMappingLocalToWorld[26][207] = 238;
    out.transformMaskMappingLocalToWorld[26][208] = 35;
    out.transformMaskMappingLocalToWorld[26][209] = 39;
    out.transformMaskMappingLocalToWorld[26][210] = 99;
    out.transformMaskMappingLocalToWorld[26][211] = 103;
    out.transformMaskMappingLocalToWorld[26][212] = 43;
    out.transformMaskMappingLocalToWorld[26][213] = 47;
    out.transformMaskMappingLocalToWorld[26][214] = 107;
    out.transformMaskMappingLocalToWorld[26][215] = 111;
    out.transformMaskMappingLocalToWorld[26][216] = 163;
    out.transformMaskMappingLocalToWorld[26][217] = 167;
    out.transformMaskMappingLocalToWorld[26][218] = 227;
    out.transformMaskMappingLocalToWorld[26][219] = 231;
    out.transformMaskMappingLocalToWorld[26][220] = 171;
    out.transformMaskMappingLocalToWorld[26][221] = 175;
    out.transformMaskMappingLocalToWorld[26][222] = 235;
    out.transformMaskMappingLocalToWorld[26][223] = 239;
    out.transformMaskMappingLocalToWorld[26][224] = 50;
    out.transformMaskMappingLocalToWorld[26][225] = 54;
    out.transformMaskMappingLocalToWorld[26][226] = 114;
    out.transformMaskMappingLocalToWorld[26][227] = 118;
    out.transformMaskMappingLocalToWorld[26][228] = 58;
    out.transformMaskMappingLocalToWorld[26][229] = 62;
    out.transformMaskMappingLocalToWorld[26][230] = 122;
    out.transformMaskMappingLocalToWorld[26][231] = 126;
    out.transformMaskMappingLocalToWorld[26][232] = 178;
    out.transformMaskMappingLocalToWorld[26][233] = 182;
    out.transformMaskMappingLocalToWorld[26][234] = 242;
    out.transformMaskMappingLocalToWorld[26][235] = 246;
    out.transformMaskMappingLocalToWorld[26][236] = 186;
    out.transformMaskMappingLocalToWorld[26][237] = 190;
    out.transformMaskMappingLocalToWorld[26][238] = 250;
    out.transformMaskMappingLocalToWorld[26][239] = 254;
    out.transformMaskMappingLocalToWorld[26][240] = 51;
    out.transformMaskMappingLocalToWorld[26][241] = 55;
    out.transformMaskMappingLocalToWorld[26][242] = 115;
    out.transformMaskMappingLocalToWorld[26][243] = 119;
    out.transformMaskMappingLocalToWorld[26][244] = 59;
    out.transformMaskMappingLocalToWorld[26][245] = 63;
    out.transformMaskMappingLocalToWorld[26][246] = 123;
    out.transformMaskMappingLocalToWorld[26][247] = 127;
    out.transformMaskMappingLocalToWorld[26][248] = 179;
    out.transformMaskMappingLocalToWorld[26][249] = 183;
    out.transformMaskMappingLocalToWorld[26][250] = 243;
    out.transformMaskMappingLocalToWorld[26][251] = 247;
    out.transformMaskMappingLocalToWorld[26][252] = 187;
    out.transformMaskMappingLocalToWorld[26][253] = 191;
    out.transformMaskMappingLocalToWorld[26][254] = 251;
    out.transformMaskMappingLocalToWorld[26][255] = 255;
    out.transformMaskMappingLocalToWorld[27][0] = 0;
    out.transformMaskMappingLocalToWorld[27][1] = 8;
    out.transformMaskMappingLocalToWorld[27][2] = 128;
    out.transformMaskMappingLocalToWorld[27][3] = 136;
    out.transformMaskMappingLocalToWorld[27][4] = 4;
    out.transformMaskMappingLocalToWorld[27][5] = 12;
    out.transformMaskMappingLocalToWorld[27][6] = 132;
    out.transformMaskMappingLocalToWorld[27][7] = 140;
    out.transformMaskMappingLocalToWorld[27][8] = 64;
    out.transformMaskMappingLocalToWorld[27][9] = 72;
    out.transformMaskMappingLocalToWorld[27][10] = 192;
    out.transformMaskMappingLocalToWorld[27][11] = 200;
    out.transformMaskMappingLocalToWorld[27][12] = 68;
    out.transformMaskMappingLocalToWorld[27][13] = 76;
    out.transformMaskMappingLocalToWorld[27][14] = 196;
    out.transformMaskMappingLocalToWorld[27][15] = 204;
    out.transformMaskMappingLocalToWorld[27][16] = 2;
    out.transformMaskMappingLocalToWorld[27][17] = 10;
    out.transformMaskMappingLocalToWorld[27][18] = 130;
    out.transformMaskMappingLocalToWorld[27][19] = 138;
    out.transformMaskMappingLocalToWorld[27][20] = 6;
    out.transformMaskMappingLocalToWorld[27][21] = 14;
    out.transformMaskMappingLocalToWorld[27][22] = 134;
    out.transformMaskMappingLocalToWorld[27][23] = 142;
    out.transformMaskMappingLocalToWorld[27][24] = 66;
    out.transformMaskMappingLocalToWorld[27][25] = 74;
    out.transformMaskMappingLocalToWorld[27][26] = 194;
    out.transformMaskMappingLocalToWorld[27][27] = 202;
    out.transformMaskMappingLocalToWorld[27][28] = 70;
    out.transformMaskMappingLocalToWorld[27][29] = 78;
    out.transformMaskMappingLocalToWorld[27][30] = 198;
    out.transformMaskMappingLocalToWorld[27][31] = 206;
    out.transformMaskMappingLocalToWorld[27][32] = 32;
    out.transformMaskMappingLocalToWorld[27][33] = 40;
    out.transformMaskMappingLocalToWorld[27][34] = 160;
    out.transformMaskMappingLocalToWorld[27][35] = 168;
    out.transformMaskMappingLocalToWorld[27][36] = 36;
    out.transformMaskMappingLocalToWorld[27][37] = 44;
    out.transformMaskMappingLocalToWorld[27][38] = 164;
    out.transformMaskMappingLocalToWorld[27][39] = 172;
    out.transformMaskMappingLocalToWorld[27][40] = 96;
    out.transformMaskMappingLocalToWorld[27][41] = 104;
    out.transformMaskMappingLocalToWorld[27][42] = 224;
    out.transformMaskMappingLocalToWorld[27][43] = 232;
    out.transformMaskMappingLocalToWorld[27][44] = 100;
    out.transformMaskMappingLocalToWorld[27][45] = 108;
    out.transformMaskMappingLocalToWorld[27][46] = 228;
    out.transformMaskMappingLocalToWorld[27][47] = 236;
    out.transformMaskMappingLocalToWorld[27][48] = 34;
    out.transformMaskMappingLocalToWorld[27][49] = 42;
    out.transformMaskMappingLocalToWorld[27][50] = 162;
    out.transformMaskMappingLocalToWorld[27][51] = 170;
    out.transformMaskMappingLocalToWorld[27][52] = 38;
    out.transformMaskMappingLocalToWorld[27][53] = 46;
    out.transformMaskMappingLocalToWorld[27][54] = 166;
    out.transformMaskMappingLocalToWorld[27][55] = 174;
    out.transformMaskMappingLocalToWorld[27][56] = 98;
    out.transformMaskMappingLocalToWorld[27][57] = 106;
    out.transformMaskMappingLocalToWorld[27][58] = 226;
    out.transformMaskMappingLocalToWorld[27][59] = 234;
    out.transformMaskMappingLocalToWorld[27][60] = 102;
    out.transformMaskMappingLocalToWorld[27][61] = 110;
    out.transformMaskMappingLocalToWorld[27][62] = 230;
    out.transformMaskMappingLocalToWorld[27][63] = 238;
    out.transformMaskMappingLocalToWorld[27][64] = 1;
    out.transformMaskMappingLocalToWorld[27][65] = 9;
    out.transformMaskMappingLocalToWorld[27][66] = 129;
    out.transformMaskMappingLocalToWorld[27][67] = 137;
    out.transformMaskMappingLocalToWorld[27][68] = 5;
    out.transformMaskMappingLocalToWorld[27][69] = 13;
    out.transformMaskMappingLocalToWorld[27][70] = 133;
    out.transformMaskMappingLocalToWorld[27][71] = 141;
    out.transformMaskMappingLocalToWorld[27][72] = 65;
    out.transformMaskMappingLocalToWorld[27][73] = 73;
    out.transformMaskMappingLocalToWorld[27][74] = 193;
    out.transformMaskMappingLocalToWorld[27][75] = 201;
    out.transformMaskMappingLocalToWorld[27][76] = 69;
    out.transformMaskMappingLocalToWorld[27][77] = 77;
    out.transformMaskMappingLocalToWorld[27][78] = 197;
    out.transformMaskMappingLocalToWorld[27][79] = 205;
    out.transformMaskMappingLocalToWorld[27][80] = 3;
    out.transformMaskMappingLocalToWorld[27][81] = 11;
    out.transformMaskMappingLocalToWorld[27][82] = 131;
    out.transformMaskMappingLocalToWorld[27][83] = 139;
    out.transformMaskMappingLocalToWorld[27][84] = 7;
    out.transformMaskMappingLocalToWorld[27][85] = 15;
    out.transformMaskMappingLocalToWorld[27][86] = 135;
    out.transformMaskMappingLocalToWorld[27][87] = 143;
    out.transformMaskMappingLocalToWorld[27][88] = 67;
    out.transformMaskMappingLocalToWorld[27][89] = 75;
    out.transformMaskMappingLocalToWorld[27][90] = 195;
    out.transformMaskMappingLocalToWorld[27][91] = 203;
    out.transformMaskMappingLocalToWorld[27][92] = 71;
    out.transformMaskMappingLocalToWorld[27][93] = 79;
    out.transformMaskMappingLocalToWorld[27][94] = 199;
    out.transformMaskMappingLocalToWorld[27][95] = 207;
    out.transformMaskMappingLocalToWorld[27][96] = 33;
    out.transformMaskMappingLocalToWorld[27][97] = 41;
    out.transformMaskMappingLocalToWorld[27][98] = 161;
    out.transformMaskMappingLocalToWorld[27][99] = 169;
    out.transformMaskMappingLocalToWorld[27][100] = 37;
    out.transformMaskMappingLocalToWorld[27][101] = 45;
    out.transformMaskMappingLocalToWorld[27][102] = 165;
    out.transformMaskMappingLocalToWorld[27][103] = 173;
    out.transformMaskMappingLocalToWorld[27][104] = 97;
    out.transformMaskMappingLocalToWorld[27][105] = 105;
    out.transformMaskMappingLocalToWorld[27][106] = 225;
    out.transformMaskMappingLocalToWorld[27][107] = 233;
    out.transformMaskMappingLocalToWorld[27][108] = 101;
    out.transformMaskMappingLocalToWorld[27][109] = 109;
    out.transformMaskMappingLocalToWorld[27][110] = 229;
    out.transformMaskMappingLocalToWorld[27][111] = 237;
    out.transformMaskMappingLocalToWorld[27][112] = 35;
    out.transformMaskMappingLocalToWorld[27][113] = 43;
    out.transformMaskMappingLocalToWorld[27][114] = 163;
    out.transformMaskMappingLocalToWorld[27][115] = 171;
    out.transformMaskMappingLocalToWorld[27][116] = 39;
    out.transformMaskMappingLocalToWorld[27][117] = 47;
    out.transformMaskMappingLocalToWorld[27][118] = 167;
    out.transformMaskMappingLocalToWorld[27][119] = 175;
    out.transformMaskMappingLocalToWorld[27][120] = 99;
    out.transformMaskMappingLocalToWorld[27][121] = 107;
    out.transformMaskMappingLocalToWorld[27][122] = 227;
    out.transformMaskMappingLocalToWorld[27][123] = 235;
    out.transformMaskMappingLocalToWorld[27][124] = 103;
    out.transformMaskMappingLocalToWorld[27][125] = 111;
    out.transformMaskMappingLocalToWorld[27][126] = 231;
    out.transformMaskMappingLocalToWorld[27][127] = 239;
    out.transformMaskMappingLocalToWorld[27][128] = 16;
    out.transformMaskMappingLocalToWorld[27][129] = 24;
    out.transformMaskMappingLocalToWorld[27][130] = 144;
    out.transformMaskMappingLocalToWorld[27][131] = 152;
    out.transformMaskMappingLocalToWorld[27][132] = 20;
    out.transformMaskMappingLocalToWorld[27][133] = 28;
    out.transformMaskMappingLocalToWorld[27][134] = 148;
    out.transformMaskMappingLocalToWorld[27][135] = 156;
    out.transformMaskMappingLocalToWorld[27][136] = 80;
    out.transformMaskMappingLocalToWorld[27][137] = 88;
    out.transformMaskMappingLocalToWorld[27][138] = 208;
    out.transformMaskMappingLocalToWorld[27][139] = 216;
    out.transformMaskMappingLocalToWorld[27][140] = 84;
    out.transformMaskMappingLocalToWorld[27][141] = 92;
    out.transformMaskMappingLocalToWorld[27][142] = 212;
    out.transformMaskMappingLocalToWorld[27][143] = 220;
    out.transformMaskMappingLocalToWorld[27][144] = 18;
    out.transformMaskMappingLocalToWorld[27][145] = 26;
    out.transformMaskMappingLocalToWorld[27][146] = 146;
    out.transformMaskMappingLocalToWorld[27][147] = 154;
    out.transformMaskMappingLocalToWorld[27][148] = 22;
    out.transformMaskMappingLocalToWorld[27][149] = 30;
    out.transformMaskMappingLocalToWorld[27][150] = 150;
    out.transformMaskMappingLocalToWorld[27][151] = 158;
    out.transformMaskMappingLocalToWorld[27][152] = 82;
    out.transformMaskMappingLocalToWorld[27][153] = 90;
    out.transformMaskMappingLocalToWorld[27][154] = 210;
    out.transformMaskMappingLocalToWorld[27][155] = 218;
    out.transformMaskMappingLocalToWorld[27][156] = 86;
    out.transformMaskMappingLocalToWorld[27][157] = 94;
    out.transformMaskMappingLocalToWorld[27][158] = 214;
    out.transformMaskMappingLocalToWorld[27][159] = 222;
    out.transformMaskMappingLocalToWorld[27][160] = 48;
    out.transformMaskMappingLocalToWorld[27][161] = 56;
    out.transformMaskMappingLocalToWorld[27][162] = 176;
    out.transformMaskMappingLocalToWorld[27][163] = 184;
    out.transformMaskMappingLocalToWorld[27][164] = 52;
    out.transformMaskMappingLocalToWorld[27][165] = 60;
    out.transformMaskMappingLocalToWorld[27][166] = 180;
    out.transformMaskMappingLocalToWorld[27][167] = 188;
    out.transformMaskMappingLocalToWorld[27][168] = 112;
    out.transformMaskMappingLocalToWorld[27][169] = 120;
    out.transformMaskMappingLocalToWorld[27][170] = 240;
    out.transformMaskMappingLocalToWorld[27][171] = 248;
    out.transformMaskMappingLocalToWorld[27][172] = 116;
    out.transformMaskMappingLocalToWorld[27][173] = 124;
    out.transformMaskMappingLocalToWorld[27][174] = 244;
    out.transformMaskMappingLocalToWorld[27][175] = 252;
    out.transformMaskMappingLocalToWorld[27][176] = 50;
    out.transformMaskMappingLocalToWorld[27][177] = 58;
    out.transformMaskMappingLocalToWorld[27][178] = 178;
    out.transformMaskMappingLocalToWorld[27][179] = 186;
    out.transformMaskMappingLocalToWorld[27][180] = 54;
    out.transformMaskMappingLocalToWorld[27][181] = 62;
    out.transformMaskMappingLocalToWorld[27][182] = 182;
    out.transformMaskMappingLocalToWorld[27][183] = 190;
    out.transformMaskMappingLocalToWorld[27][184] = 114;
    out.transformMaskMappingLocalToWorld[27][185] = 122;
    out.transformMaskMappingLocalToWorld[27][186] = 242;
    out.transformMaskMappingLocalToWorld[27][187] = 250;
    out.transformMaskMappingLocalToWorld[27][188] = 118;
    out.transformMaskMappingLocalToWorld[27][189] = 126;
    out.transformMaskMappingLocalToWorld[27][190] = 246;
    out.transformMaskMappingLocalToWorld[27][191] = 254;
    out.transformMaskMappingLocalToWorld[27][192] = 17;
    out.transformMaskMappingLocalToWorld[27][193] = 25;
    out.transformMaskMappingLocalToWorld[27][194] = 145;
    out.transformMaskMappingLocalToWorld[27][195] = 153;
    out.transformMaskMappingLocalToWorld[27][196] = 21;
    out.transformMaskMappingLocalToWorld[27][197] = 29;
    out.transformMaskMappingLocalToWorld[27][198] = 149;
    out.transformMaskMappingLocalToWorld[27][199] = 157;
    out.transformMaskMappingLocalToWorld[27][200] = 81;
    out.transformMaskMappingLocalToWorld[27][201] = 89;
    out.transformMaskMappingLocalToWorld[27][202] = 209;
    out.transformMaskMappingLocalToWorld[27][203] = 217;
    out.transformMaskMappingLocalToWorld[27][204] = 85;
    out.transformMaskMappingLocalToWorld[27][205] = 93;
    out.transformMaskMappingLocalToWorld[27][206] = 213;
    out.transformMaskMappingLocalToWorld[27][207] = 221;
    out.transformMaskMappingLocalToWorld[27][208] = 19;
    out.transformMaskMappingLocalToWorld[27][209] = 27;
    out.transformMaskMappingLocalToWorld[27][210] = 147;
    out.transformMaskMappingLocalToWorld[27][211] = 155;
    out.transformMaskMappingLocalToWorld[27][212] = 23;
    out.transformMaskMappingLocalToWorld[27][213] = 31;
    out.transformMaskMappingLocalToWorld[27][214] = 151;
    out.transformMaskMappingLocalToWorld[27][215] = 159;
    out.transformMaskMappingLocalToWorld[27][216] = 83;
    out.transformMaskMappingLocalToWorld[27][217] = 91;
    out.transformMaskMappingLocalToWorld[27][218] = 211;
    out.transformMaskMappingLocalToWorld[27][219] = 219;
    out.transformMaskMappingLocalToWorld[27][220] = 87;
    out.transformMaskMappingLocalToWorld[27][221] = 95;
    out.transformMaskMappingLocalToWorld[27][222] = 215;
    out.transformMaskMappingLocalToWorld[27][223] = 223;
    out.transformMaskMappingLocalToWorld[27][224] = 49;
    out.transformMaskMappingLocalToWorld[27][225] = 57;
    out.transformMaskMappingLocalToWorld[27][226] = 177;
    out.transformMaskMappingLocalToWorld[27][227] = 185;
    out.transformMaskMappingLocalToWorld[27][228] = 53;
    out.transformMaskMappingLocalToWorld[27][229] = 61;
    out.transformMaskMappingLocalToWorld[27][230] = 181;
    out.transformMaskMappingLocalToWorld[27][231] = 189;
    out.transformMaskMappingLocalToWorld[27][232] = 113;
    out.transformMaskMappingLocalToWorld[27][233] = 121;
    out.transformMaskMappingLocalToWorld[27][234] = 241;
    out.transformMaskMappingLocalToWorld[27][235] = 249;
    out.transformMaskMappingLocalToWorld[27][236] = 117;
    out.transformMaskMappingLocalToWorld[27][237] = 125;
    out.transformMaskMappingLocalToWorld[27][238] = 245;
    out.transformMaskMappingLocalToWorld[27][239] = 253;
    out.transformMaskMappingLocalToWorld[27][240] = 51;
    out.transformMaskMappingLocalToWorld[27][241] = 59;
    out.transformMaskMappingLocalToWorld[27][242] = 179;
    out.transformMaskMappingLocalToWorld[27][243] = 187;
    out.transformMaskMappingLocalToWorld[27][244] = 55;
    out.transformMaskMappingLocalToWorld[27][245] = 63;
    out.transformMaskMappingLocalToWorld[27][246] = 183;
    out.transformMaskMappingLocalToWorld[27][247] = 191;
    out.transformMaskMappingLocalToWorld[27][248] = 115;
    out.transformMaskMappingLocalToWorld[27][249] = 123;
    out.transformMaskMappingLocalToWorld[27][250] = 243;
    out.transformMaskMappingLocalToWorld[27][251] = 251;
    out.transformMaskMappingLocalToWorld[27][252] = 119;
    out.transformMaskMappingLocalToWorld[27][253] = 127;
    out.transformMaskMappingLocalToWorld[27][254] = 247;
    out.transformMaskMappingLocalToWorld[27][255] = 255;
    out.transformMaskMappingLocalToWorld[28][0] = 0;
    out.transformMaskMappingLocalToWorld[28][1] = 16;
    out.transformMaskMappingLocalToWorld[28][2] = 1;
    out.transformMaskMappingLocalToWorld[28][3] = 17;
    out.transformMaskMappingLocalToWorld[28][4] = 32;
    out.transformMaskMappingLocalToWorld[28][5] = 48;
    out.transformMaskMappingLocalToWorld[28][6] = 33;
    out.transformMaskMappingLocalToWorld[28][7] = 49;
    out.transformMaskMappingLocalToWorld[28][8] = 2;
    out.transformMaskMappingLocalToWorld[28][9] = 18;
    out.transformMaskMappingLocalToWorld[28][10] = 3;
    out.transformMaskMappingLocalToWorld[28][11] = 19;
    out.transformMaskMappingLocalToWorld[28][12] = 34;
    out.transformMaskMappingLocalToWorld[28][13] = 50;
    out.transformMaskMappingLocalToWorld[28][14] = 35;
    out.transformMaskMappingLocalToWorld[28][15] = 51;
    out.transformMaskMappingLocalToWorld[28][16] = 64;
    out.transformMaskMappingLocalToWorld[28][17] = 80;
    out.transformMaskMappingLocalToWorld[28][18] = 65;
    out.transformMaskMappingLocalToWorld[28][19] = 81;
    out.transformMaskMappingLocalToWorld[28][20] = 96;
    out.transformMaskMappingLocalToWorld[28][21] = 112;
    out.transformMaskMappingLocalToWorld[28][22] = 97;
    out.transformMaskMappingLocalToWorld[28][23] = 113;
    out.transformMaskMappingLocalToWorld[28][24] = 66;
    out.transformMaskMappingLocalToWorld[28][25] = 82;
    out.transformMaskMappingLocalToWorld[28][26] = 67;
    out.transformMaskMappingLocalToWorld[28][27] = 83;
    out.transformMaskMappingLocalToWorld[28][28] = 98;
    out.transformMaskMappingLocalToWorld[28][29] = 114;
    out.transformMaskMappingLocalToWorld[28][30] = 99;
    out.transformMaskMappingLocalToWorld[28][31] = 115;
    out.transformMaskMappingLocalToWorld[28][32] = 4;
    out.transformMaskMappingLocalToWorld[28][33] = 20;
    out.transformMaskMappingLocalToWorld[28][34] = 5;
    out.transformMaskMappingLocalToWorld[28][35] = 21;
    out.transformMaskMappingLocalToWorld[28][36] = 36;
    out.transformMaskMappingLocalToWorld[28][37] = 52;
    out.transformMaskMappingLocalToWorld[28][38] = 37;
    out.transformMaskMappingLocalToWorld[28][39] = 53;
    out.transformMaskMappingLocalToWorld[28][40] = 6;
    out.transformMaskMappingLocalToWorld[28][41] = 22;
    out.transformMaskMappingLocalToWorld[28][42] = 7;
    out.transformMaskMappingLocalToWorld[28][43] = 23;
    out.transformMaskMappingLocalToWorld[28][44] = 38;
    out.transformMaskMappingLocalToWorld[28][45] = 54;
    out.transformMaskMappingLocalToWorld[28][46] = 39;
    out.transformMaskMappingLocalToWorld[28][47] = 55;
    out.transformMaskMappingLocalToWorld[28][48] = 68;
    out.transformMaskMappingLocalToWorld[28][49] = 84;
    out.transformMaskMappingLocalToWorld[28][50] = 69;
    out.transformMaskMappingLocalToWorld[28][51] = 85;
    out.transformMaskMappingLocalToWorld[28][52] = 100;
    out.transformMaskMappingLocalToWorld[28][53] = 116;
    out.transformMaskMappingLocalToWorld[28][54] = 101;
    out.transformMaskMappingLocalToWorld[28][55] = 117;
    out.transformMaskMappingLocalToWorld[28][56] = 70;
    out.transformMaskMappingLocalToWorld[28][57] = 86;
    out.transformMaskMappingLocalToWorld[28][58] = 71;
    out.transformMaskMappingLocalToWorld[28][59] = 87;
    out.transformMaskMappingLocalToWorld[28][60] = 102;
    out.transformMaskMappingLocalToWorld[28][61] = 118;
    out.transformMaskMappingLocalToWorld[28][62] = 103;
    out.transformMaskMappingLocalToWorld[28][63] = 119;
    out.transformMaskMappingLocalToWorld[28][64] = 128;
    out.transformMaskMappingLocalToWorld[28][65] = 144;
    out.transformMaskMappingLocalToWorld[28][66] = 129;
    out.transformMaskMappingLocalToWorld[28][67] = 145;
    out.transformMaskMappingLocalToWorld[28][68] = 160;
    out.transformMaskMappingLocalToWorld[28][69] = 176;
    out.transformMaskMappingLocalToWorld[28][70] = 161;
    out.transformMaskMappingLocalToWorld[28][71] = 177;
    out.transformMaskMappingLocalToWorld[28][72] = 130;
    out.transformMaskMappingLocalToWorld[28][73] = 146;
    out.transformMaskMappingLocalToWorld[28][74] = 131;
    out.transformMaskMappingLocalToWorld[28][75] = 147;
    out.transformMaskMappingLocalToWorld[28][76] = 162;
    out.transformMaskMappingLocalToWorld[28][77] = 178;
    out.transformMaskMappingLocalToWorld[28][78] = 163;
    out.transformMaskMappingLocalToWorld[28][79] = 179;
    out.transformMaskMappingLocalToWorld[28][80] = 192;
    out.transformMaskMappingLocalToWorld[28][81] = 208;
    out.transformMaskMappingLocalToWorld[28][82] = 193;
    out.transformMaskMappingLocalToWorld[28][83] = 209;
    out.transformMaskMappingLocalToWorld[28][84] = 224;
    out.transformMaskMappingLocalToWorld[28][85] = 240;
    out.transformMaskMappingLocalToWorld[28][86] = 225;
    out.transformMaskMappingLocalToWorld[28][87] = 241;
    out.transformMaskMappingLocalToWorld[28][88] = 194;
    out.transformMaskMappingLocalToWorld[28][89] = 210;
    out.transformMaskMappingLocalToWorld[28][90] = 195;
    out.transformMaskMappingLocalToWorld[28][91] = 211;
    out.transformMaskMappingLocalToWorld[28][92] = 226;
    out.transformMaskMappingLocalToWorld[28][93] = 242;
    out.transformMaskMappingLocalToWorld[28][94] = 227;
    out.transformMaskMappingLocalToWorld[28][95] = 243;
    out.transformMaskMappingLocalToWorld[28][96] = 132;
    out.transformMaskMappingLocalToWorld[28][97] = 148;
    out.transformMaskMappingLocalToWorld[28][98] = 133;
    out.transformMaskMappingLocalToWorld[28][99] = 149;
    out.transformMaskMappingLocalToWorld[28][100] = 164;
    out.transformMaskMappingLocalToWorld[28][101] = 180;
    out.transformMaskMappingLocalToWorld[28][102] = 165;
    out.transformMaskMappingLocalToWorld[28][103] = 181;
    out.transformMaskMappingLocalToWorld[28][104] = 134;
    out.transformMaskMappingLocalToWorld[28][105] = 150;
    out.transformMaskMappingLocalToWorld[28][106] = 135;
    out.transformMaskMappingLocalToWorld[28][107] = 151;
    out.transformMaskMappingLocalToWorld[28][108] = 166;
    out.transformMaskMappingLocalToWorld[28][109] = 182;
    out.transformMaskMappingLocalToWorld[28][110] = 167;
    out.transformMaskMappingLocalToWorld[28][111] = 183;
    out.transformMaskMappingLocalToWorld[28][112] = 196;
    out.transformMaskMappingLocalToWorld[28][113] = 212;
    out.transformMaskMappingLocalToWorld[28][114] = 197;
    out.transformMaskMappingLocalToWorld[28][115] = 213;
    out.transformMaskMappingLocalToWorld[28][116] = 228;
    out.transformMaskMappingLocalToWorld[28][117] = 244;
    out.transformMaskMappingLocalToWorld[28][118] = 229;
    out.transformMaskMappingLocalToWorld[28][119] = 245;
    out.transformMaskMappingLocalToWorld[28][120] = 198;
    out.transformMaskMappingLocalToWorld[28][121] = 214;
    out.transformMaskMappingLocalToWorld[28][122] = 199;
    out.transformMaskMappingLocalToWorld[28][123] = 215;
    out.transformMaskMappingLocalToWorld[28][124] = 230;
    out.transformMaskMappingLocalToWorld[28][125] = 246;
    out.transformMaskMappingLocalToWorld[28][126] = 231;
    out.transformMaskMappingLocalToWorld[28][127] = 247;
    out.transformMaskMappingLocalToWorld[28][128] = 8;
    out.transformMaskMappingLocalToWorld[28][129] = 24;
    out.transformMaskMappingLocalToWorld[28][130] = 9;
    out.transformMaskMappingLocalToWorld[28][131] = 25;
    out.transformMaskMappingLocalToWorld[28][132] = 40;
    out.transformMaskMappingLocalToWorld[28][133] = 56;
    out.transformMaskMappingLocalToWorld[28][134] = 41;
    out.transformMaskMappingLocalToWorld[28][135] = 57;
    out.transformMaskMappingLocalToWorld[28][136] = 10;
    out.transformMaskMappingLocalToWorld[28][137] = 26;
    out.transformMaskMappingLocalToWorld[28][138] = 11;
    out.transformMaskMappingLocalToWorld[28][139] = 27;
    out.transformMaskMappingLocalToWorld[28][140] = 42;
    out.transformMaskMappingLocalToWorld[28][141] = 58;
    out.transformMaskMappingLocalToWorld[28][142] = 43;
    out.transformMaskMappingLocalToWorld[28][143] = 59;
    out.transformMaskMappingLocalToWorld[28][144] = 72;
    out.transformMaskMappingLocalToWorld[28][145] = 88;
    out.transformMaskMappingLocalToWorld[28][146] = 73;
    out.transformMaskMappingLocalToWorld[28][147] = 89;
    out.transformMaskMappingLocalToWorld[28][148] = 104;
    out.transformMaskMappingLocalToWorld[28][149] = 120;
    out.transformMaskMappingLocalToWorld[28][150] = 105;
    out.transformMaskMappingLocalToWorld[28][151] = 121;
    out.transformMaskMappingLocalToWorld[28][152] = 74;
    out.transformMaskMappingLocalToWorld[28][153] = 90;
    out.transformMaskMappingLocalToWorld[28][154] = 75;
    out.transformMaskMappingLocalToWorld[28][155] = 91;
    out.transformMaskMappingLocalToWorld[28][156] = 106;
    out.transformMaskMappingLocalToWorld[28][157] = 122;
    out.transformMaskMappingLocalToWorld[28][158] = 107;
    out.transformMaskMappingLocalToWorld[28][159] = 123;
    out.transformMaskMappingLocalToWorld[28][160] = 12;
    out.transformMaskMappingLocalToWorld[28][161] = 28;
    out.transformMaskMappingLocalToWorld[28][162] = 13;
    out.transformMaskMappingLocalToWorld[28][163] = 29;
    out.transformMaskMappingLocalToWorld[28][164] = 44;
    out.transformMaskMappingLocalToWorld[28][165] = 60;
    out.transformMaskMappingLocalToWorld[28][166] = 45;
    out.transformMaskMappingLocalToWorld[28][167] = 61;
    out.transformMaskMappingLocalToWorld[28][168] = 14;
    out.transformMaskMappingLocalToWorld[28][169] = 30;
    out.transformMaskMappingLocalToWorld[28][170] = 15;
    out.transformMaskMappingLocalToWorld[28][171] = 31;
    out.transformMaskMappingLocalToWorld[28][172] = 46;
    out.transformMaskMappingLocalToWorld[28][173] = 62;
    out.transformMaskMappingLocalToWorld[28][174] = 47;
    out.transformMaskMappingLocalToWorld[28][175] = 63;
    out.transformMaskMappingLocalToWorld[28][176] = 76;
    out.transformMaskMappingLocalToWorld[28][177] = 92;
    out.transformMaskMappingLocalToWorld[28][178] = 77;
    out.transformMaskMappingLocalToWorld[28][179] = 93;
    out.transformMaskMappingLocalToWorld[28][180] = 108;
    out.transformMaskMappingLocalToWorld[28][181] = 124;
    out.transformMaskMappingLocalToWorld[28][182] = 109;
    out.transformMaskMappingLocalToWorld[28][183] = 125;
    out.transformMaskMappingLocalToWorld[28][184] = 78;
    out.transformMaskMappingLocalToWorld[28][185] = 94;
    out.transformMaskMappingLocalToWorld[28][186] = 79;
    out.transformMaskMappingLocalToWorld[28][187] = 95;
    out.transformMaskMappingLocalToWorld[28][188] = 110;
    out.transformMaskMappingLocalToWorld[28][189] = 126;
    out.transformMaskMappingLocalToWorld[28][190] = 111;
    out.transformMaskMappingLocalToWorld[28][191] = 127;
    out.transformMaskMappingLocalToWorld[28][192] = 136;
    out.transformMaskMappingLocalToWorld[28][193] = 152;
    out.transformMaskMappingLocalToWorld[28][194] = 137;
    out.transformMaskMappingLocalToWorld[28][195] = 153;
    out.transformMaskMappingLocalToWorld[28][196] = 168;
    out.transformMaskMappingLocalToWorld[28][197] = 184;
    out.transformMaskMappingLocalToWorld[28][198] = 169;
    out.transformMaskMappingLocalToWorld[28][199] = 185;
    out.transformMaskMappingLocalToWorld[28][200] = 138;
    out.transformMaskMappingLocalToWorld[28][201] = 154;
    out.transformMaskMappingLocalToWorld[28][202] = 139;
    out.transformMaskMappingLocalToWorld[28][203] = 155;
    out.transformMaskMappingLocalToWorld[28][204] = 170;
    out.transformMaskMappingLocalToWorld[28][205] = 186;
    out.transformMaskMappingLocalToWorld[28][206] = 171;
    out.transformMaskMappingLocalToWorld[28][207] = 187;
    out.transformMaskMappingLocalToWorld[28][208] = 200;
    out.transformMaskMappingLocalToWorld[28][209] = 216;
    out.transformMaskMappingLocalToWorld[28][210] = 201;
    out.transformMaskMappingLocalToWorld[28][211] = 217;
    out.transformMaskMappingLocalToWorld[28][212] = 232;
    out.transformMaskMappingLocalToWorld[28][213] = 248;
    out.transformMaskMappingLocalToWorld[28][214] = 233;
    out.transformMaskMappingLocalToWorld[28][215] = 249;
    out.transformMaskMappingLocalToWorld[28][216] = 202;
    out.transformMaskMappingLocalToWorld[28][217] = 218;
    out.transformMaskMappingLocalToWorld[28][218] = 203;
    out.transformMaskMappingLocalToWorld[28][219] = 219;
    out.transformMaskMappingLocalToWorld[28][220] = 234;
    out.transformMaskMappingLocalToWorld[28][221] = 250;
    out.transformMaskMappingLocalToWorld[28][222] = 235;
    out.transformMaskMappingLocalToWorld[28][223] = 251;
    out.transformMaskMappingLocalToWorld[28][224] = 140;
    out.transformMaskMappingLocalToWorld[28][225] = 156;
    out.transformMaskMappingLocalToWorld[28][226] = 141;
    out.transformMaskMappingLocalToWorld[28][227] = 157;
    out.transformMaskMappingLocalToWorld[28][228] = 172;
    out.transformMaskMappingLocalToWorld[28][229] = 188;
    out.transformMaskMappingLocalToWorld[28][230] = 173;
    out.transformMaskMappingLocalToWorld[28][231] = 189;
    out.transformMaskMappingLocalToWorld[28][232] = 142;
    out.transformMaskMappingLocalToWorld[28][233] = 158;
    out.transformMaskMappingLocalToWorld[28][234] = 143;
    out.transformMaskMappingLocalToWorld[28][235] = 159;
    out.transformMaskMappingLocalToWorld[28][236] = 174;
    out.transformMaskMappingLocalToWorld[28][237] = 190;
    out.transformMaskMappingLocalToWorld[28][238] = 175;
    out.transformMaskMappingLocalToWorld[28][239] = 191;
    out.transformMaskMappingLocalToWorld[28][240] = 204;
    out.transformMaskMappingLocalToWorld[28][241] = 220;
    out.transformMaskMappingLocalToWorld[28][242] = 205;
    out.transformMaskMappingLocalToWorld[28][243] = 221;
    out.transformMaskMappingLocalToWorld[28][244] = 236;
    out.transformMaskMappingLocalToWorld[28][245] = 252;
    out.transformMaskMappingLocalToWorld[28][246] = 237;
    out.transformMaskMappingLocalToWorld[28][247] = 253;
    out.transformMaskMappingLocalToWorld[28][248] = 206;
    out.transformMaskMappingLocalToWorld[28][249] = 222;
    out.transformMaskMappingLocalToWorld[28][250] = 207;
    out.transformMaskMappingLocalToWorld[28][251] = 223;
    out.transformMaskMappingLocalToWorld[28][252] = 238;
    out.transformMaskMappingLocalToWorld[28][253] = 254;
    out.transformMaskMappingLocalToWorld[28][254] = 239;
    out.transformMaskMappingLocalToWorld[28][255] = 255;
    out.transformMaskMappingLocalToWorld[29][0] = 0;
    out.transformMaskMappingLocalToWorld[29][1] = 32;
    out.transformMaskMappingLocalToWorld[29][2] = 2;
    out.transformMaskMappingLocalToWorld[29][3] = 34;
    out.transformMaskMappingLocalToWorld[29][4] = 16;
    out.transformMaskMappingLocalToWorld[29][5] = 48;
    out.transformMaskMappingLocalToWorld[29][6] = 18;
    out.transformMaskMappingLocalToWorld[29][7] = 50;
    out.transformMaskMappingLocalToWorld[29][8] = 1;
    out.transformMaskMappingLocalToWorld[29][9] = 33;
    out.transformMaskMappingLocalToWorld[29][10] = 3;
    out.transformMaskMappingLocalToWorld[29][11] = 35;
    out.transformMaskMappingLocalToWorld[29][12] = 17;
    out.transformMaskMappingLocalToWorld[29][13] = 49;
    out.transformMaskMappingLocalToWorld[29][14] = 19;
    out.transformMaskMappingLocalToWorld[29][15] = 51;
    out.transformMaskMappingLocalToWorld[29][16] = 128;
    out.transformMaskMappingLocalToWorld[29][17] = 160;
    out.transformMaskMappingLocalToWorld[29][18] = 130;
    out.transformMaskMappingLocalToWorld[29][19] = 162;
    out.transformMaskMappingLocalToWorld[29][20] = 144;
    out.transformMaskMappingLocalToWorld[29][21] = 176;
    out.transformMaskMappingLocalToWorld[29][22] = 146;
    out.transformMaskMappingLocalToWorld[29][23] = 178;
    out.transformMaskMappingLocalToWorld[29][24] = 129;
    out.transformMaskMappingLocalToWorld[29][25] = 161;
    out.transformMaskMappingLocalToWorld[29][26] = 131;
    out.transformMaskMappingLocalToWorld[29][27] = 163;
    out.transformMaskMappingLocalToWorld[29][28] = 145;
    out.transformMaskMappingLocalToWorld[29][29] = 177;
    out.transformMaskMappingLocalToWorld[29][30] = 147;
    out.transformMaskMappingLocalToWorld[29][31] = 179;
    out.transformMaskMappingLocalToWorld[29][32] = 8;
    out.transformMaskMappingLocalToWorld[29][33] = 40;
    out.transformMaskMappingLocalToWorld[29][34] = 10;
    out.transformMaskMappingLocalToWorld[29][35] = 42;
    out.transformMaskMappingLocalToWorld[29][36] = 24;
    out.transformMaskMappingLocalToWorld[29][37] = 56;
    out.transformMaskMappingLocalToWorld[29][38] = 26;
    out.transformMaskMappingLocalToWorld[29][39] = 58;
    out.transformMaskMappingLocalToWorld[29][40] = 9;
    out.transformMaskMappingLocalToWorld[29][41] = 41;
    out.transformMaskMappingLocalToWorld[29][42] = 11;
    out.transformMaskMappingLocalToWorld[29][43] = 43;
    out.transformMaskMappingLocalToWorld[29][44] = 25;
    out.transformMaskMappingLocalToWorld[29][45] = 57;
    out.transformMaskMappingLocalToWorld[29][46] = 27;
    out.transformMaskMappingLocalToWorld[29][47] = 59;
    out.transformMaskMappingLocalToWorld[29][48] = 136;
    out.transformMaskMappingLocalToWorld[29][49] = 168;
    out.transformMaskMappingLocalToWorld[29][50] = 138;
    out.transformMaskMappingLocalToWorld[29][51] = 170;
    out.transformMaskMappingLocalToWorld[29][52] = 152;
    out.transformMaskMappingLocalToWorld[29][53] = 184;
    out.transformMaskMappingLocalToWorld[29][54] = 154;
    out.transformMaskMappingLocalToWorld[29][55] = 186;
    out.transformMaskMappingLocalToWorld[29][56] = 137;
    out.transformMaskMappingLocalToWorld[29][57] = 169;
    out.transformMaskMappingLocalToWorld[29][58] = 139;
    out.transformMaskMappingLocalToWorld[29][59] = 171;
    out.transformMaskMappingLocalToWorld[29][60] = 153;
    out.transformMaskMappingLocalToWorld[29][61] = 185;
    out.transformMaskMappingLocalToWorld[29][62] = 155;
    out.transformMaskMappingLocalToWorld[29][63] = 187;
    out.transformMaskMappingLocalToWorld[29][64] = 64;
    out.transformMaskMappingLocalToWorld[29][65] = 96;
    out.transformMaskMappingLocalToWorld[29][66] = 66;
    out.transformMaskMappingLocalToWorld[29][67] = 98;
    out.transformMaskMappingLocalToWorld[29][68] = 80;
    out.transformMaskMappingLocalToWorld[29][69] = 112;
    out.transformMaskMappingLocalToWorld[29][70] = 82;
    out.transformMaskMappingLocalToWorld[29][71] = 114;
    out.transformMaskMappingLocalToWorld[29][72] = 65;
    out.transformMaskMappingLocalToWorld[29][73] = 97;
    out.transformMaskMappingLocalToWorld[29][74] = 67;
    out.transformMaskMappingLocalToWorld[29][75] = 99;
    out.transformMaskMappingLocalToWorld[29][76] = 81;
    out.transformMaskMappingLocalToWorld[29][77] = 113;
    out.transformMaskMappingLocalToWorld[29][78] = 83;
    out.transformMaskMappingLocalToWorld[29][79] = 115;
    out.transformMaskMappingLocalToWorld[29][80] = 192;
    out.transformMaskMappingLocalToWorld[29][81] = 224;
    out.transformMaskMappingLocalToWorld[29][82] = 194;
    out.transformMaskMappingLocalToWorld[29][83] = 226;
    out.transformMaskMappingLocalToWorld[29][84] = 208;
    out.transformMaskMappingLocalToWorld[29][85] = 240;
    out.transformMaskMappingLocalToWorld[29][86] = 210;
    out.transformMaskMappingLocalToWorld[29][87] = 242;
    out.transformMaskMappingLocalToWorld[29][88] = 193;
    out.transformMaskMappingLocalToWorld[29][89] = 225;
    out.transformMaskMappingLocalToWorld[29][90] = 195;
    out.transformMaskMappingLocalToWorld[29][91] = 227;
    out.transformMaskMappingLocalToWorld[29][92] = 209;
    out.transformMaskMappingLocalToWorld[29][93] = 241;
    out.transformMaskMappingLocalToWorld[29][94] = 211;
    out.transformMaskMappingLocalToWorld[29][95] = 243;
    out.transformMaskMappingLocalToWorld[29][96] = 72;
    out.transformMaskMappingLocalToWorld[29][97] = 104;
    out.transformMaskMappingLocalToWorld[29][98] = 74;
    out.transformMaskMappingLocalToWorld[29][99] = 106;
    out.transformMaskMappingLocalToWorld[29][100] = 88;
    out.transformMaskMappingLocalToWorld[29][101] = 120;
    out.transformMaskMappingLocalToWorld[29][102] = 90;
    out.transformMaskMappingLocalToWorld[29][103] = 122;
    out.transformMaskMappingLocalToWorld[29][104] = 73;
    out.transformMaskMappingLocalToWorld[29][105] = 105;
    out.transformMaskMappingLocalToWorld[29][106] = 75;
    out.transformMaskMappingLocalToWorld[29][107] = 107;
    out.transformMaskMappingLocalToWorld[29][108] = 89;
    out.transformMaskMappingLocalToWorld[29][109] = 121;
    out.transformMaskMappingLocalToWorld[29][110] = 91;
    out.transformMaskMappingLocalToWorld[29][111] = 123;
    out.transformMaskMappingLocalToWorld[29][112] = 200;
    out.transformMaskMappingLocalToWorld[29][113] = 232;
    out.transformMaskMappingLocalToWorld[29][114] = 202;
    out.transformMaskMappingLocalToWorld[29][115] = 234;
    out.transformMaskMappingLocalToWorld[29][116] = 216;
    out.transformMaskMappingLocalToWorld[29][117] = 248;
    out.transformMaskMappingLocalToWorld[29][118] = 218;
    out.transformMaskMappingLocalToWorld[29][119] = 250;
    out.transformMaskMappingLocalToWorld[29][120] = 201;
    out.transformMaskMappingLocalToWorld[29][121] = 233;
    out.transformMaskMappingLocalToWorld[29][122] = 203;
    out.transformMaskMappingLocalToWorld[29][123] = 235;
    out.transformMaskMappingLocalToWorld[29][124] = 217;
    out.transformMaskMappingLocalToWorld[29][125] = 249;
    out.transformMaskMappingLocalToWorld[29][126] = 219;
    out.transformMaskMappingLocalToWorld[29][127] = 251;
    out.transformMaskMappingLocalToWorld[29][128] = 4;
    out.transformMaskMappingLocalToWorld[29][129] = 36;
    out.transformMaskMappingLocalToWorld[29][130] = 6;
    out.transformMaskMappingLocalToWorld[29][131] = 38;
    out.transformMaskMappingLocalToWorld[29][132] = 20;
    out.transformMaskMappingLocalToWorld[29][133] = 52;
    out.transformMaskMappingLocalToWorld[29][134] = 22;
    out.transformMaskMappingLocalToWorld[29][135] = 54;
    out.transformMaskMappingLocalToWorld[29][136] = 5;
    out.transformMaskMappingLocalToWorld[29][137] = 37;
    out.transformMaskMappingLocalToWorld[29][138] = 7;
    out.transformMaskMappingLocalToWorld[29][139] = 39;
    out.transformMaskMappingLocalToWorld[29][140] = 21;
    out.transformMaskMappingLocalToWorld[29][141] = 53;
    out.transformMaskMappingLocalToWorld[29][142] = 23;
    out.transformMaskMappingLocalToWorld[29][143] = 55;
    out.transformMaskMappingLocalToWorld[29][144] = 132;
    out.transformMaskMappingLocalToWorld[29][145] = 164;
    out.transformMaskMappingLocalToWorld[29][146] = 134;
    out.transformMaskMappingLocalToWorld[29][147] = 166;
    out.transformMaskMappingLocalToWorld[29][148] = 148;
    out.transformMaskMappingLocalToWorld[29][149] = 180;
    out.transformMaskMappingLocalToWorld[29][150] = 150;
    out.transformMaskMappingLocalToWorld[29][151] = 182;
    out.transformMaskMappingLocalToWorld[29][152] = 133;
    out.transformMaskMappingLocalToWorld[29][153] = 165;
    out.transformMaskMappingLocalToWorld[29][154] = 135;
    out.transformMaskMappingLocalToWorld[29][155] = 167;
    out.transformMaskMappingLocalToWorld[29][156] = 149;
    out.transformMaskMappingLocalToWorld[29][157] = 181;
    out.transformMaskMappingLocalToWorld[29][158] = 151;
    out.transformMaskMappingLocalToWorld[29][159] = 183;
    out.transformMaskMappingLocalToWorld[29][160] = 12;
    out.transformMaskMappingLocalToWorld[29][161] = 44;
    out.transformMaskMappingLocalToWorld[29][162] = 14;
    out.transformMaskMappingLocalToWorld[29][163] = 46;
    out.transformMaskMappingLocalToWorld[29][164] = 28;
    out.transformMaskMappingLocalToWorld[29][165] = 60;
    out.transformMaskMappingLocalToWorld[29][166] = 30;
    out.transformMaskMappingLocalToWorld[29][167] = 62;
    out.transformMaskMappingLocalToWorld[29][168] = 13;
    out.transformMaskMappingLocalToWorld[29][169] = 45;
    out.transformMaskMappingLocalToWorld[29][170] = 15;
    out.transformMaskMappingLocalToWorld[29][171] = 47;
    out.transformMaskMappingLocalToWorld[29][172] = 29;
    out.transformMaskMappingLocalToWorld[29][173] = 61;
    out.transformMaskMappingLocalToWorld[29][174] = 31;
    out.transformMaskMappingLocalToWorld[29][175] = 63;
    out.transformMaskMappingLocalToWorld[29][176] = 140;
    out.transformMaskMappingLocalToWorld[29][177] = 172;
    out.transformMaskMappingLocalToWorld[29][178] = 142;
    out.transformMaskMappingLocalToWorld[29][179] = 174;
    out.transformMaskMappingLocalToWorld[29][180] = 156;
    out.transformMaskMappingLocalToWorld[29][181] = 188;
    out.transformMaskMappingLocalToWorld[29][182] = 158;
    out.transformMaskMappingLocalToWorld[29][183] = 190;
    out.transformMaskMappingLocalToWorld[29][184] = 141;
    out.transformMaskMappingLocalToWorld[29][185] = 173;
    out.transformMaskMappingLocalToWorld[29][186] = 143;
    out.transformMaskMappingLocalToWorld[29][187] = 175;
    out.transformMaskMappingLocalToWorld[29][188] = 157;
    out.transformMaskMappingLocalToWorld[29][189] = 189;
    out.transformMaskMappingLocalToWorld[29][190] = 159;
    out.transformMaskMappingLocalToWorld[29][191] = 191;
    out.transformMaskMappingLocalToWorld[29][192] = 68;
    out.transformMaskMappingLocalToWorld[29][193] = 100;
    out.transformMaskMappingLocalToWorld[29][194] = 70;
    out.transformMaskMappingLocalToWorld[29][195] = 102;
    out.transformMaskMappingLocalToWorld[29][196] = 84;
    out.transformMaskMappingLocalToWorld[29][197] = 116;
    out.transformMaskMappingLocalToWorld[29][198] = 86;
    out.transformMaskMappingLocalToWorld[29][199] = 118;
    out.transformMaskMappingLocalToWorld[29][200] = 69;
    out.transformMaskMappingLocalToWorld[29][201] = 101;
    out.transformMaskMappingLocalToWorld[29][202] = 71;
    out.transformMaskMappingLocalToWorld[29][203] = 103;
    out.transformMaskMappingLocalToWorld[29][204] = 85;
    out.transformMaskMappingLocalToWorld[29][205] = 117;
    out.transformMaskMappingLocalToWorld[29][206] = 87;
    out.transformMaskMappingLocalToWorld[29][207] = 119;
    out.transformMaskMappingLocalToWorld[29][208] = 196;
    out.transformMaskMappingLocalToWorld[29][209] = 228;
    out.transformMaskMappingLocalToWorld[29][210] = 198;
    out.transformMaskMappingLocalToWorld[29][211] = 230;
    out.transformMaskMappingLocalToWorld[29][212] = 212;
    out.transformMaskMappingLocalToWorld[29][213] = 244;
    out.transformMaskMappingLocalToWorld[29][214] = 214;
    out.transformMaskMappingLocalToWorld[29][215] = 246;
    out.transformMaskMappingLocalToWorld[29][216] = 197;
    out.transformMaskMappingLocalToWorld[29][217] = 229;
    out.transformMaskMappingLocalToWorld[29][218] = 199;
    out.transformMaskMappingLocalToWorld[29][219] = 231;
    out.transformMaskMappingLocalToWorld[29][220] = 213;
    out.transformMaskMappingLocalToWorld[29][221] = 245;
    out.transformMaskMappingLocalToWorld[29][222] = 215;
    out.transformMaskMappingLocalToWorld[29][223] = 247;
    out.transformMaskMappingLocalToWorld[29][224] = 76;
    out.transformMaskMappingLocalToWorld[29][225] = 108;
    out.transformMaskMappingLocalToWorld[29][226] = 78;
    out.transformMaskMappingLocalToWorld[29][227] = 110;
    out.transformMaskMappingLocalToWorld[29][228] = 92;
    out.transformMaskMappingLocalToWorld[29][229] = 124;
    out.transformMaskMappingLocalToWorld[29][230] = 94;
    out.transformMaskMappingLocalToWorld[29][231] = 126;
    out.transformMaskMappingLocalToWorld[29][232] = 77;
    out.transformMaskMappingLocalToWorld[29][233] = 109;
    out.transformMaskMappingLocalToWorld[29][234] = 79;
    out.transformMaskMappingLocalToWorld[29][235] = 111;
    out.transformMaskMappingLocalToWorld[29][236] = 93;
    out.transformMaskMappingLocalToWorld[29][237] = 125;
    out.transformMaskMappingLocalToWorld[29][238] = 95;
    out.transformMaskMappingLocalToWorld[29][239] = 127;
    out.transformMaskMappingLocalToWorld[29][240] = 204;
    out.transformMaskMappingLocalToWorld[29][241] = 236;
    out.transformMaskMappingLocalToWorld[29][242] = 206;
    out.transformMaskMappingLocalToWorld[29][243] = 238;
    out.transformMaskMappingLocalToWorld[29][244] = 220;
    out.transformMaskMappingLocalToWorld[29][245] = 252;
    out.transformMaskMappingLocalToWorld[29][246] = 222;
    out.transformMaskMappingLocalToWorld[29][247] = 254;
    out.transformMaskMappingLocalToWorld[29][248] = 205;
    out.transformMaskMappingLocalToWorld[29][249] = 237;
    out.transformMaskMappingLocalToWorld[29][250] = 207;
    out.transformMaskMappingLocalToWorld[29][251] = 239;
    out.transformMaskMappingLocalToWorld[29][252] = 221;
    out.transformMaskMappingLocalToWorld[29][253] = 253;
    out.transformMaskMappingLocalToWorld[29][254] = 223;
    out.transformMaskMappingLocalToWorld[29][255] = 255;
    out.transformMaskMappingLocalToWorld[30][0] = 0;
    out.transformMaskMappingLocalToWorld[30][1] = 64;
    out.transformMaskMappingLocalToWorld[30][2] = 4;
    out.transformMaskMappingLocalToWorld[30][3] = 68;
    out.transformMaskMappingLocalToWorld[30][4] = 128;
    out.transformMaskMappingLocalToWorld[30][5] = 192;
    out.transformMaskMappingLocalToWorld[30][6] = 132;
    out.transformMaskMappingLocalToWorld[30][7] = 196;
    out.transformMaskMappingLocalToWorld[30][8] = 8;
    out.transformMaskMappingLocalToWorld[30][9] = 72;
    out.transformMaskMappingLocalToWorld[30][10] = 12;
    out.transformMaskMappingLocalToWorld[30][11] = 76;
    out.transformMaskMappingLocalToWorld[30][12] = 136;
    out.transformMaskMappingLocalToWorld[30][13] = 200;
    out.transformMaskMappingLocalToWorld[30][14] = 140;
    out.transformMaskMappingLocalToWorld[30][15] = 204;
    out.transformMaskMappingLocalToWorld[30][16] = 16;
    out.transformMaskMappingLocalToWorld[30][17] = 80;
    out.transformMaskMappingLocalToWorld[30][18] = 20;
    out.transformMaskMappingLocalToWorld[30][19] = 84;
    out.transformMaskMappingLocalToWorld[30][20] = 144;
    out.transformMaskMappingLocalToWorld[30][21] = 208;
    out.transformMaskMappingLocalToWorld[30][22] = 148;
    out.transformMaskMappingLocalToWorld[30][23] = 212;
    out.transformMaskMappingLocalToWorld[30][24] = 24;
    out.transformMaskMappingLocalToWorld[30][25] = 88;
    out.transformMaskMappingLocalToWorld[30][26] = 28;
    out.transformMaskMappingLocalToWorld[30][27] = 92;
    out.transformMaskMappingLocalToWorld[30][28] = 152;
    out.transformMaskMappingLocalToWorld[30][29] = 216;
    out.transformMaskMappingLocalToWorld[30][30] = 156;
    out.transformMaskMappingLocalToWorld[30][31] = 220;
    out.transformMaskMappingLocalToWorld[30][32] = 1;
    out.transformMaskMappingLocalToWorld[30][33] = 65;
    out.transformMaskMappingLocalToWorld[30][34] = 5;
    out.transformMaskMappingLocalToWorld[30][35] = 69;
    out.transformMaskMappingLocalToWorld[30][36] = 129;
    out.transformMaskMappingLocalToWorld[30][37] = 193;
    out.transformMaskMappingLocalToWorld[30][38] = 133;
    out.transformMaskMappingLocalToWorld[30][39] = 197;
    out.transformMaskMappingLocalToWorld[30][40] = 9;
    out.transformMaskMappingLocalToWorld[30][41] = 73;
    out.transformMaskMappingLocalToWorld[30][42] = 13;
    out.transformMaskMappingLocalToWorld[30][43] = 77;
    out.transformMaskMappingLocalToWorld[30][44] = 137;
    out.transformMaskMappingLocalToWorld[30][45] = 201;
    out.transformMaskMappingLocalToWorld[30][46] = 141;
    out.transformMaskMappingLocalToWorld[30][47] = 205;
    out.transformMaskMappingLocalToWorld[30][48] = 17;
    out.transformMaskMappingLocalToWorld[30][49] = 81;
    out.transformMaskMappingLocalToWorld[30][50] = 21;
    out.transformMaskMappingLocalToWorld[30][51] = 85;
    out.transformMaskMappingLocalToWorld[30][52] = 145;
    out.transformMaskMappingLocalToWorld[30][53] = 209;
    out.transformMaskMappingLocalToWorld[30][54] = 149;
    out.transformMaskMappingLocalToWorld[30][55] = 213;
    out.transformMaskMappingLocalToWorld[30][56] = 25;
    out.transformMaskMappingLocalToWorld[30][57] = 89;
    out.transformMaskMappingLocalToWorld[30][58] = 29;
    out.transformMaskMappingLocalToWorld[30][59] = 93;
    out.transformMaskMappingLocalToWorld[30][60] = 153;
    out.transformMaskMappingLocalToWorld[30][61] = 217;
    out.transformMaskMappingLocalToWorld[30][62] = 157;
    out.transformMaskMappingLocalToWorld[30][63] = 221;
    out.transformMaskMappingLocalToWorld[30][64] = 32;
    out.transformMaskMappingLocalToWorld[30][65] = 96;
    out.transformMaskMappingLocalToWorld[30][66] = 36;
    out.transformMaskMappingLocalToWorld[30][67] = 100;
    out.transformMaskMappingLocalToWorld[30][68] = 160;
    out.transformMaskMappingLocalToWorld[30][69] = 224;
    out.transformMaskMappingLocalToWorld[30][70] = 164;
    out.transformMaskMappingLocalToWorld[30][71] = 228;
    out.transformMaskMappingLocalToWorld[30][72] = 40;
    out.transformMaskMappingLocalToWorld[30][73] = 104;
    out.transformMaskMappingLocalToWorld[30][74] = 44;
    out.transformMaskMappingLocalToWorld[30][75] = 108;
    out.transformMaskMappingLocalToWorld[30][76] = 168;
    out.transformMaskMappingLocalToWorld[30][77] = 232;
    out.transformMaskMappingLocalToWorld[30][78] = 172;
    out.transformMaskMappingLocalToWorld[30][79] = 236;
    out.transformMaskMappingLocalToWorld[30][80] = 48;
    out.transformMaskMappingLocalToWorld[30][81] = 112;
    out.transformMaskMappingLocalToWorld[30][82] = 52;
    out.transformMaskMappingLocalToWorld[30][83] = 116;
    out.transformMaskMappingLocalToWorld[30][84] = 176;
    out.transformMaskMappingLocalToWorld[30][85] = 240;
    out.transformMaskMappingLocalToWorld[30][86] = 180;
    out.transformMaskMappingLocalToWorld[30][87] = 244;
    out.transformMaskMappingLocalToWorld[30][88] = 56;
    out.transformMaskMappingLocalToWorld[30][89] = 120;
    out.transformMaskMappingLocalToWorld[30][90] = 60;
    out.transformMaskMappingLocalToWorld[30][91] = 124;
    out.transformMaskMappingLocalToWorld[30][92] = 184;
    out.transformMaskMappingLocalToWorld[30][93] = 248;
    out.transformMaskMappingLocalToWorld[30][94] = 188;
    out.transformMaskMappingLocalToWorld[30][95] = 252;
    out.transformMaskMappingLocalToWorld[30][96] = 33;
    out.transformMaskMappingLocalToWorld[30][97] = 97;
    out.transformMaskMappingLocalToWorld[30][98] = 37;
    out.transformMaskMappingLocalToWorld[30][99] = 101;
    out.transformMaskMappingLocalToWorld[30][100] = 161;
    out.transformMaskMappingLocalToWorld[30][101] = 225;
    out.transformMaskMappingLocalToWorld[30][102] = 165;
    out.transformMaskMappingLocalToWorld[30][103] = 229;
    out.transformMaskMappingLocalToWorld[30][104] = 41;
    out.transformMaskMappingLocalToWorld[30][105] = 105;
    out.transformMaskMappingLocalToWorld[30][106] = 45;
    out.transformMaskMappingLocalToWorld[30][107] = 109;
    out.transformMaskMappingLocalToWorld[30][108] = 169;
    out.transformMaskMappingLocalToWorld[30][109] = 233;
    out.transformMaskMappingLocalToWorld[30][110] = 173;
    out.transformMaskMappingLocalToWorld[30][111] = 237;
    out.transformMaskMappingLocalToWorld[30][112] = 49;
    out.transformMaskMappingLocalToWorld[30][113] = 113;
    out.transformMaskMappingLocalToWorld[30][114] = 53;
    out.transformMaskMappingLocalToWorld[30][115] = 117;
    out.transformMaskMappingLocalToWorld[30][116] = 177;
    out.transformMaskMappingLocalToWorld[30][117] = 241;
    out.transformMaskMappingLocalToWorld[30][118] = 181;
    out.transformMaskMappingLocalToWorld[30][119] = 245;
    out.transformMaskMappingLocalToWorld[30][120] = 57;
    out.transformMaskMappingLocalToWorld[30][121] = 121;
    out.transformMaskMappingLocalToWorld[30][122] = 61;
    out.transformMaskMappingLocalToWorld[30][123] = 125;
    out.transformMaskMappingLocalToWorld[30][124] = 185;
    out.transformMaskMappingLocalToWorld[30][125] = 249;
    out.transformMaskMappingLocalToWorld[30][126] = 189;
    out.transformMaskMappingLocalToWorld[30][127] = 253;
    out.transformMaskMappingLocalToWorld[30][128] = 2;
    out.transformMaskMappingLocalToWorld[30][129] = 66;
    out.transformMaskMappingLocalToWorld[30][130] = 6;
    out.transformMaskMappingLocalToWorld[30][131] = 70;
    out.transformMaskMappingLocalToWorld[30][132] = 130;
    out.transformMaskMappingLocalToWorld[30][133] = 194;
    out.transformMaskMappingLocalToWorld[30][134] = 134;
    out.transformMaskMappingLocalToWorld[30][135] = 198;
    out.transformMaskMappingLocalToWorld[30][136] = 10;
    out.transformMaskMappingLocalToWorld[30][137] = 74;
    out.transformMaskMappingLocalToWorld[30][138] = 14;
    out.transformMaskMappingLocalToWorld[30][139] = 78;
    out.transformMaskMappingLocalToWorld[30][140] = 138;
    out.transformMaskMappingLocalToWorld[30][141] = 202;
    out.transformMaskMappingLocalToWorld[30][142] = 142;
    out.transformMaskMappingLocalToWorld[30][143] = 206;
    out.transformMaskMappingLocalToWorld[30][144] = 18;
    out.transformMaskMappingLocalToWorld[30][145] = 82;
    out.transformMaskMappingLocalToWorld[30][146] = 22;
    out.transformMaskMappingLocalToWorld[30][147] = 86;
    out.transformMaskMappingLocalToWorld[30][148] = 146;
    out.transformMaskMappingLocalToWorld[30][149] = 210;
    out.transformMaskMappingLocalToWorld[30][150] = 150;
    out.transformMaskMappingLocalToWorld[30][151] = 214;
    out.transformMaskMappingLocalToWorld[30][152] = 26;
    out.transformMaskMappingLocalToWorld[30][153] = 90;
    out.transformMaskMappingLocalToWorld[30][154] = 30;
    out.transformMaskMappingLocalToWorld[30][155] = 94;
    out.transformMaskMappingLocalToWorld[30][156] = 154;
    out.transformMaskMappingLocalToWorld[30][157] = 218;
    out.transformMaskMappingLocalToWorld[30][158] = 158;
    out.transformMaskMappingLocalToWorld[30][159] = 222;
    out.transformMaskMappingLocalToWorld[30][160] = 3;
    out.transformMaskMappingLocalToWorld[30][161] = 67;
    out.transformMaskMappingLocalToWorld[30][162] = 7;
    out.transformMaskMappingLocalToWorld[30][163] = 71;
    out.transformMaskMappingLocalToWorld[30][164] = 131;
    out.transformMaskMappingLocalToWorld[30][165] = 195;
    out.transformMaskMappingLocalToWorld[30][166] = 135;
    out.transformMaskMappingLocalToWorld[30][167] = 199;
    out.transformMaskMappingLocalToWorld[30][168] = 11;
    out.transformMaskMappingLocalToWorld[30][169] = 75;
    out.transformMaskMappingLocalToWorld[30][170] = 15;
    out.transformMaskMappingLocalToWorld[30][171] = 79;
    out.transformMaskMappingLocalToWorld[30][172] = 139;
    out.transformMaskMappingLocalToWorld[30][173] = 203;
    out.transformMaskMappingLocalToWorld[30][174] = 143;
    out.transformMaskMappingLocalToWorld[30][175] = 207;
    out.transformMaskMappingLocalToWorld[30][176] = 19;
    out.transformMaskMappingLocalToWorld[30][177] = 83;
    out.transformMaskMappingLocalToWorld[30][178] = 23;
    out.transformMaskMappingLocalToWorld[30][179] = 87;
    out.transformMaskMappingLocalToWorld[30][180] = 147;
    out.transformMaskMappingLocalToWorld[30][181] = 211;
    out.transformMaskMappingLocalToWorld[30][182] = 151;
    out.transformMaskMappingLocalToWorld[30][183] = 215;
    out.transformMaskMappingLocalToWorld[30][184] = 27;
    out.transformMaskMappingLocalToWorld[30][185] = 91;
    out.transformMaskMappingLocalToWorld[30][186] = 31;
    out.transformMaskMappingLocalToWorld[30][187] = 95;
    out.transformMaskMappingLocalToWorld[30][188] = 155;
    out.transformMaskMappingLocalToWorld[30][189] = 219;
    out.transformMaskMappingLocalToWorld[30][190] = 159;
    out.transformMaskMappingLocalToWorld[30][191] = 223;
    out.transformMaskMappingLocalToWorld[30][192] = 34;
    out.transformMaskMappingLocalToWorld[30][193] = 98;
    out.transformMaskMappingLocalToWorld[30][194] = 38;
    out.transformMaskMappingLocalToWorld[30][195] = 102;
    out.transformMaskMappingLocalToWorld[30][196] = 162;
    out.transformMaskMappingLocalToWorld[30][197] = 226;
    out.transformMaskMappingLocalToWorld[30][198] = 166;
    out.transformMaskMappingLocalToWorld[30][199] = 230;
    out.transformMaskMappingLocalToWorld[30][200] = 42;
    out.transformMaskMappingLocalToWorld[30][201] = 106;
    out.transformMaskMappingLocalToWorld[30][202] = 46;
    out.transformMaskMappingLocalToWorld[30][203] = 110;
    out.transformMaskMappingLocalToWorld[30][204] = 170;
    out.transformMaskMappingLocalToWorld[30][205] = 234;
    out.transformMaskMappingLocalToWorld[30][206] = 174;
    out.transformMaskMappingLocalToWorld[30][207] = 238;
    out.transformMaskMappingLocalToWorld[30][208] = 50;
    out.transformMaskMappingLocalToWorld[30][209] = 114;
    out.transformMaskMappingLocalToWorld[30][210] = 54;
    out.transformMaskMappingLocalToWorld[30][211] = 118;
    out.transformMaskMappingLocalToWorld[30][212] = 178;
    out.transformMaskMappingLocalToWorld[30][213] = 242;
    out.transformMaskMappingLocalToWorld[30][214] = 182;
    out.transformMaskMappingLocalToWorld[30][215] = 246;
    out.transformMaskMappingLocalToWorld[30][216] = 58;
    out.transformMaskMappingLocalToWorld[30][217] = 122;
    out.transformMaskMappingLocalToWorld[30][218] = 62;
    out.transformMaskMappingLocalToWorld[30][219] = 126;
    out.transformMaskMappingLocalToWorld[30][220] = 186;
    out.transformMaskMappingLocalToWorld[30][221] = 250;
    out.transformMaskMappingLocalToWorld[30][222] = 190;
    out.transformMaskMappingLocalToWorld[30][223] = 254;
    out.transformMaskMappingLocalToWorld[30][224] = 35;
    out.transformMaskMappingLocalToWorld[30][225] = 99;
    out.transformMaskMappingLocalToWorld[30][226] = 39;
    out.transformMaskMappingLocalToWorld[30][227] = 103;
    out.transformMaskMappingLocalToWorld[30][228] = 163;
    out.transformMaskMappingLocalToWorld[30][229] = 227;
    out.transformMaskMappingLocalToWorld[30][230] = 167;
    out.transformMaskMappingLocalToWorld[30][231] = 231;
    out.transformMaskMappingLocalToWorld[30][232] = 43;
    out.transformMaskMappingLocalToWorld[30][233] = 107;
    out.transformMaskMappingLocalToWorld[30][234] = 47;
    out.transformMaskMappingLocalToWorld[30][235] = 111;
    out.transformMaskMappingLocalToWorld[30][236] = 171;
    out.transformMaskMappingLocalToWorld[30][237] = 235;
    out.transformMaskMappingLocalToWorld[30][238] = 175;
    out.transformMaskMappingLocalToWorld[30][239] = 239;
    out.transformMaskMappingLocalToWorld[30][240] = 51;
    out.transformMaskMappingLocalToWorld[30][241] = 115;
    out.transformMaskMappingLocalToWorld[30][242] = 55;
    out.transformMaskMappingLocalToWorld[30][243] = 119;
    out.transformMaskMappingLocalToWorld[30][244] = 179;
    out.transformMaskMappingLocalToWorld[30][245] = 243;
    out.transformMaskMappingLocalToWorld[30][246] = 183;
    out.transformMaskMappingLocalToWorld[30][247] = 247;
    out.transformMaskMappingLocalToWorld[30][248] = 59;
    out.transformMaskMappingLocalToWorld[30][249] = 123;
    out.transformMaskMappingLocalToWorld[30][250] = 63;
    out.transformMaskMappingLocalToWorld[30][251] = 127;
    out.transformMaskMappingLocalToWorld[30][252] = 187;
    out.transformMaskMappingLocalToWorld[30][253] = 251;
    out.transformMaskMappingLocalToWorld[30][254] = 191;
    out.transformMaskMappingLocalToWorld[30][255] = 255;
    out.transformMaskMappingLocalToWorld[31][0] = 0;
    out.transformMaskMappingLocalToWorld[31][1] = 128;
    out.transformMaskMappingLocalToWorld[31][2] = 8;
    out.transformMaskMappingLocalToWorld[31][3] = 136;
    out.transformMaskMappingLocalToWorld[31][4] = 64;
    out.transformMaskMappingLocalToWorld[31][5] = 192;
    out.transformMaskMappingLocalToWorld[31][6] = 72;
    out.transformMaskMappingLocalToWorld[31][7] = 200;
    out.transformMaskMappingLocalToWorld[31][8] = 4;
    out.transformMaskMappingLocalToWorld[31][9] = 132;
    out.transformMaskMappingLocalToWorld[31][10] = 12;
    out.transformMaskMappingLocalToWorld[31][11] = 140;
    out.transformMaskMappingLocalToWorld[31][12] = 68;
    out.transformMaskMappingLocalToWorld[31][13] = 196;
    out.transformMaskMappingLocalToWorld[31][14] = 76;
    out.transformMaskMappingLocalToWorld[31][15] = 204;
    out.transformMaskMappingLocalToWorld[31][16] = 32;
    out.transformMaskMappingLocalToWorld[31][17] = 160;
    out.transformMaskMappingLocalToWorld[31][18] = 40;
    out.transformMaskMappingLocalToWorld[31][19] = 168;
    out.transformMaskMappingLocalToWorld[31][20] = 96;
    out.transformMaskMappingLocalToWorld[31][21] = 224;
    out.transformMaskMappingLocalToWorld[31][22] = 104;
    out.transformMaskMappingLocalToWorld[31][23] = 232;
    out.transformMaskMappingLocalToWorld[31][24] = 36;
    out.transformMaskMappingLocalToWorld[31][25] = 164;
    out.transformMaskMappingLocalToWorld[31][26] = 44;
    out.transformMaskMappingLocalToWorld[31][27] = 172;
    out.transformMaskMappingLocalToWorld[31][28] = 100;
    out.transformMaskMappingLocalToWorld[31][29] = 228;
    out.transformMaskMappingLocalToWorld[31][30] = 108;
    out.transformMaskMappingLocalToWorld[31][31] = 236;
    out.transformMaskMappingLocalToWorld[31][32] = 2;
    out.transformMaskMappingLocalToWorld[31][33] = 130;
    out.transformMaskMappingLocalToWorld[31][34] = 10;
    out.transformMaskMappingLocalToWorld[31][35] = 138;
    out.transformMaskMappingLocalToWorld[31][36] = 66;
    out.transformMaskMappingLocalToWorld[31][37] = 194;
    out.transformMaskMappingLocalToWorld[31][38] = 74;
    out.transformMaskMappingLocalToWorld[31][39] = 202;
    out.transformMaskMappingLocalToWorld[31][40] = 6;
    out.transformMaskMappingLocalToWorld[31][41] = 134;
    out.transformMaskMappingLocalToWorld[31][42] = 14;
    out.transformMaskMappingLocalToWorld[31][43] = 142;
    out.transformMaskMappingLocalToWorld[31][44] = 70;
    out.transformMaskMappingLocalToWorld[31][45] = 198;
    out.transformMaskMappingLocalToWorld[31][46] = 78;
    out.transformMaskMappingLocalToWorld[31][47] = 206;
    out.transformMaskMappingLocalToWorld[31][48] = 34;
    out.transformMaskMappingLocalToWorld[31][49] = 162;
    out.transformMaskMappingLocalToWorld[31][50] = 42;
    out.transformMaskMappingLocalToWorld[31][51] = 170;
    out.transformMaskMappingLocalToWorld[31][52] = 98;
    out.transformMaskMappingLocalToWorld[31][53] = 226;
    out.transformMaskMappingLocalToWorld[31][54] = 106;
    out.transformMaskMappingLocalToWorld[31][55] = 234;
    out.transformMaskMappingLocalToWorld[31][56] = 38;
    out.transformMaskMappingLocalToWorld[31][57] = 166;
    out.transformMaskMappingLocalToWorld[31][58] = 46;
    out.transformMaskMappingLocalToWorld[31][59] = 174;
    out.transformMaskMappingLocalToWorld[31][60] = 102;
    out.transformMaskMappingLocalToWorld[31][61] = 230;
    out.transformMaskMappingLocalToWorld[31][62] = 110;
    out.transformMaskMappingLocalToWorld[31][63] = 238;
    out.transformMaskMappingLocalToWorld[31][64] = 16;
    out.transformMaskMappingLocalToWorld[31][65] = 144;
    out.transformMaskMappingLocalToWorld[31][66] = 24;
    out.transformMaskMappingLocalToWorld[31][67] = 152;
    out.transformMaskMappingLocalToWorld[31][68] = 80;
    out.transformMaskMappingLocalToWorld[31][69] = 208;
    out.transformMaskMappingLocalToWorld[31][70] = 88;
    out.transformMaskMappingLocalToWorld[31][71] = 216;
    out.transformMaskMappingLocalToWorld[31][72] = 20;
    out.transformMaskMappingLocalToWorld[31][73] = 148;
    out.transformMaskMappingLocalToWorld[31][74] = 28;
    out.transformMaskMappingLocalToWorld[31][75] = 156;
    out.transformMaskMappingLocalToWorld[31][76] = 84;
    out.transformMaskMappingLocalToWorld[31][77] = 212;
    out.transformMaskMappingLocalToWorld[31][78] = 92;
    out.transformMaskMappingLocalToWorld[31][79] = 220;
    out.transformMaskMappingLocalToWorld[31][80] = 48;
    out.transformMaskMappingLocalToWorld[31][81] = 176;
    out.transformMaskMappingLocalToWorld[31][82] = 56;
    out.transformMaskMappingLocalToWorld[31][83] = 184;
    out.transformMaskMappingLocalToWorld[31][84] = 112;
    out.transformMaskMappingLocalToWorld[31][85] = 240;
    out.transformMaskMappingLocalToWorld[31][86] = 120;
    out.transformMaskMappingLocalToWorld[31][87] = 248;
    out.transformMaskMappingLocalToWorld[31][88] = 52;
    out.transformMaskMappingLocalToWorld[31][89] = 180;
    out.transformMaskMappingLocalToWorld[31][90] = 60;
    out.transformMaskMappingLocalToWorld[31][91] = 188;
    out.transformMaskMappingLocalToWorld[31][92] = 116;
    out.transformMaskMappingLocalToWorld[31][93] = 244;
    out.transformMaskMappingLocalToWorld[31][94] = 124;
    out.transformMaskMappingLocalToWorld[31][95] = 252;
    out.transformMaskMappingLocalToWorld[31][96] = 18;
    out.transformMaskMappingLocalToWorld[31][97] = 146;
    out.transformMaskMappingLocalToWorld[31][98] = 26;
    out.transformMaskMappingLocalToWorld[31][99] = 154;
    out.transformMaskMappingLocalToWorld[31][100] = 82;
    out.transformMaskMappingLocalToWorld[31][101] = 210;
    out.transformMaskMappingLocalToWorld[31][102] = 90;
    out.transformMaskMappingLocalToWorld[31][103] = 218;
    out.transformMaskMappingLocalToWorld[31][104] = 22;
    out.transformMaskMappingLocalToWorld[31][105] = 150;
    out.transformMaskMappingLocalToWorld[31][106] = 30;
    out.transformMaskMappingLocalToWorld[31][107] = 158;
    out.transformMaskMappingLocalToWorld[31][108] = 86;
    out.transformMaskMappingLocalToWorld[31][109] = 214;
    out.transformMaskMappingLocalToWorld[31][110] = 94;
    out.transformMaskMappingLocalToWorld[31][111] = 222;
    out.transformMaskMappingLocalToWorld[31][112] = 50;
    out.transformMaskMappingLocalToWorld[31][113] = 178;
    out.transformMaskMappingLocalToWorld[31][114] = 58;
    out.transformMaskMappingLocalToWorld[31][115] = 186;
    out.transformMaskMappingLocalToWorld[31][116] = 114;
    out.transformMaskMappingLocalToWorld[31][117] = 242;
    out.transformMaskMappingLocalToWorld[31][118] = 122;
    out.transformMaskMappingLocalToWorld[31][119] = 250;
    out.transformMaskMappingLocalToWorld[31][120] = 54;
    out.transformMaskMappingLocalToWorld[31][121] = 182;
    out.transformMaskMappingLocalToWorld[31][122] = 62;
    out.transformMaskMappingLocalToWorld[31][123] = 190;
    out.transformMaskMappingLocalToWorld[31][124] = 118;
    out.transformMaskMappingLocalToWorld[31][125] = 246;
    out.transformMaskMappingLocalToWorld[31][126] = 126;
    out.transformMaskMappingLocalToWorld[31][127] = 254;
    out.transformMaskMappingLocalToWorld[31][128] = 1;
    out.transformMaskMappingLocalToWorld[31][129] = 129;
    out.transformMaskMappingLocalToWorld[31][130] = 9;
    out.transformMaskMappingLocalToWorld[31][131] = 137;
    out.transformMaskMappingLocalToWorld[31][132] = 65;
    out.transformMaskMappingLocalToWorld[31][133] = 193;
    out.transformMaskMappingLocalToWorld[31][134] = 73;
    out.transformMaskMappingLocalToWorld[31][135] = 201;
    out.transformMaskMappingLocalToWorld[31][136] = 5;
    out.transformMaskMappingLocalToWorld[31][137] = 133;
    out.transformMaskMappingLocalToWorld[31][138] = 13;
    out.transformMaskMappingLocalToWorld[31][139] = 141;
    out.transformMaskMappingLocalToWorld[31][140] = 69;
    out.transformMaskMappingLocalToWorld[31][141] = 197;
    out.transformMaskMappingLocalToWorld[31][142] = 77;
    out.transformMaskMappingLocalToWorld[31][143] = 205;
    out.transformMaskMappingLocalToWorld[31][144] = 33;
    out.transformMaskMappingLocalToWorld[31][145] = 161;
    out.transformMaskMappingLocalToWorld[31][146] = 41;
    out.transformMaskMappingLocalToWorld[31][147] = 169;
    out.transformMaskMappingLocalToWorld[31][148] = 97;
    out.transformMaskMappingLocalToWorld[31][149] = 225;
    out.transformMaskMappingLocalToWorld[31][150] = 105;
    out.transformMaskMappingLocalToWorld[31][151] = 233;
    out.transformMaskMappingLocalToWorld[31][152] = 37;
    out.transformMaskMappingLocalToWorld[31][153] = 165;
    out.transformMaskMappingLocalToWorld[31][154] = 45;
    out.transformMaskMappingLocalToWorld[31][155] = 173;
    out.transformMaskMappingLocalToWorld[31][156] = 101;
    out.transformMaskMappingLocalToWorld[31][157] = 229;
    out.transformMaskMappingLocalToWorld[31][158] = 109;
    out.transformMaskMappingLocalToWorld[31][159] = 237;
    out.transformMaskMappingLocalToWorld[31][160] = 3;
    out.transformMaskMappingLocalToWorld[31][161] = 131;
    out.transformMaskMappingLocalToWorld[31][162] = 11;
    out.transformMaskMappingLocalToWorld[31][163] = 139;
    out.transformMaskMappingLocalToWorld[31][164] = 67;
    out.transformMaskMappingLocalToWorld[31][165] = 195;
    out.transformMaskMappingLocalToWorld[31][166] = 75;
    out.transformMaskMappingLocalToWorld[31][167] = 203;
    out.transformMaskMappingLocalToWorld[31][168] = 7;
    out.transformMaskMappingLocalToWorld[31][169] = 135;
    out.transformMaskMappingLocalToWorld[31][170] = 15;
    out.transformMaskMappingLocalToWorld[31][171] = 143;
    out.transformMaskMappingLocalToWorld[31][172] = 71;
    out.transformMaskMappingLocalToWorld[31][173] = 199;
    out.transformMaskMappingLocalToWorld[31][174] = 79;
    out.transformMaskMappingLocalToWorld[31][175] = 207;
    out.transformMaskMappingLocalToWorld[31][176] = 35;
    out.transformMaskMappingLocalToWorld[31][177] = 163;
    out.transformMaskMappingLocalToWorld[31][178] = 43;
    out.transformMaskMappingLocalToWorld[31][179] = 171;
    out.transformMaskMappingLocalToWorld[31][180] = 99;
    out.transformMaskMappingLocalToWorld[31][181] = 227;
    out.transformMaskMappingLocalToWorld[31][182] = 107;
    out.transformMaskMappingLocalToWorld[31][183] = 235;
    out.transformMaskMappingLocalToWorld[31][184] = 39;
    out.transformMaskMappingLocalToWorld[31][185] = 167;
    out.transformMaskMappingLocalToWorld[31][186] = 47;
    out.transformMaskMappingLocalToWorld[31][187] = 175;
    out.transformMaskMappingLocalToWorld[31][188] = 103;
    out.transformMaskMappingLocalToWorld[31][189] = 231;
    out.transformMaskMappingLocalToWorld[31][190] = 111;
    out.transformMaskMappingLocalToWorld[31][191] = 239;
    out.transformMaskMappingLocalToWorld[31][192] = 17;
    out.transformMaskMappingLocalToWorld[31][193] = 145;
    out.transformMaskMappingLocalToWorld[31][194] = 25;
    out.transformMaskMappingLocalToWorld[31][195] = 153;
    out.transformMaskMappingLocalToWorld[31][196] = 81;
    out.transformMaskMappingLocalToWorld[31][197] = 209;
    out.transformMaskMappingLocalToWorld[31][198] = 89;
    out.transformMaskMappingLocalToWorld[31][199] = 217;
    out.transformMaskMappingLocalToWorld[31][200] = 21;
    out.transformMaskMappingLocalToWorld[31][201] = 149;
    out.transformMaskMappingLocalToWorld[31][202] = 29;
    out.transformMaskMappingLocalToWorld[31][203] = 157;
    out.transformMaskMappingLocalToWorld[31][204] = 85;
    out.transformMaskMappingLocalToWorld[31][205] = 213;
    out.transformMaskMappingLocalToWorld[31][206] = 93;
    out.transformMaskMappingLocalToWorld[31][207] = 221;
    out.transformMaskMappingLocalToWorld[31][208] = 49;
    out.transformMaskMappingLocalToWorld[31][209] = 177;
    out.transformMaskMappingLocalToWorld[31][210] = 57;
    out.transformMaskMappingLocalToWorld[31][211] = 185;
    out.transformMaskMappingLocalToWorld[31][212] = 113;
    out.transformMaskMappingLocalToWorld[31][213] = 241;
    out.transformMaskMappingLocalToWorld[31][214] = 121;
    out.transformMaskMappingLocalToWorld[31][215] = 249;
    out.transformMaskMappingLocalToWorld[31][216] = 53;
    out.transformMaskMappingLocalToWorld[31][217] = 181;
    out.transformMaskMappingLocalToWorld[31][218] = 61;
    out.transformMaskMappingLocalToWorld[31][219] = 189;
    out.transformMaskMappingLocalToWorld[31][220] = 117;
    out.transformMaskMappingLocalToWorld[31][221] = 245;
    out.transformMaskMappingLocalToWorld[31][222] = 125;
    out.transformMaskMappingLocalToWorld[31][223] = 253;
    out.transformMaskMappingLocalToWorld[31][224] = 19;
    out.transformMaskMappingLocalToWorld[31][225] = 147;
    out.transformMaskMappingLocalToWorld[31][226] = 27;
    out.transformMaskMappingLocalToWorld[31][227] = 155;
    out.transformMaskMappingLocalToWorld[31][228] = 83;
    out.transformMaskMappingLocalToWorld[31][229] = 211;
    out.transformMaskMappingLocalToWorld[31][230] = 91;
    out.transformMaskMappingLocalToWorld[31][231] = 219;
    out.transformMaskMappingLocalToWorld[31][232] = 23;
    out.transformMaskMappingLocalToWorld[31][233] = 151;
    out.transformMaskMappingLocalToWorld[31][234] = 31;
    out.transformMaskMappingLocalToWorld[31][235] = 159;
    out.transformMaskMappingLocalToWorld[31][236] = 87;
    out.transformMaskMappingLocalToWorld[31][237] = 215;
    out.transformMaskMappingLocalToWorld[31][238] = 95;
    out.transformMaskMappingLocalToWorld[31][239] = 223;
    out.transformMaskMappingLocalToWorld[31][240] = 51;
    out.transformMaskMappingLocalToWorld[31][241] = 179;
    out.transformMaskMappingLocalToWorld[31][242] = 59;
    out.transformMaskMappingLocalToWorld[31][243] = 187;
    out.transformMaskMappingLocalToWorld[31][244] = 115;
    out.transformMaskMappingLocalToWorld[31][245] = 243;
    out.transformMaskMappingLocalToWorld[31][246] = 123;
    out.transformMaskMappingLocalToWorld[31][247] = 251;
    out.transformMaskMappingLocalToWorld[31][248] = 55;
    out.transformMaskMappingLocalToWorld[31][249] = 183;
    out.transformMaskMappingLocalToWorld[31][250] = 63;
    out.transformMaskMappingLocalToWorld[31][251] = 191;
    out.transformMaskMappingLocalToWorld[31][252] = 119;
    out.transformMaskMappingLocalToWorld[31][253] = 247;
    out.transformMaskMappingLocalToWorld[31][254] = 127;
    out.transformMaskMappingLocalToWorld[31][255] = 255;
    out.transformMaskMappingLocalToWorld[32][0] = 0;
    out.transformMaskMappingLocalToWorld[32][1] = 1;
    out.transformMaskMappingLocalToWorld[32][2] = 4;
    out.transformMaskMappingLocalToWorld[32][3] = 5;
    out.transformMaskMappingLocalToWorld[32][4] = 16;
    out.transformMaskMappingLocalToWorld[32][5] = 17;
    out.transformMaskMappingLocalToWorld[32][6] = 20;
    out.transformMaskMappingLocalToWorld[32][7] = 21;
    out.transformMaskMappingLocalToWorld[32][8] = 64;
    out.transformMaskMappingLocalToWorld[32][9] = 65;
    out.transformMaskMappingLocalToWorld[32][10] = 68;
    out.transformMaskMappingLocalToWorld[32][11] = 69;
    out.transformMaskMappingLocalToWorld[32][12] = 80;
    out.transformMaskMappingLocalToWorld[32][13] = 81;
    out.transformMaskMappingLocalToWorld[32][14] = 84;
    out.transformMaskMappingLocalToWorld[32][15] = 85;
    out.transformMaskMappingLocalToWorld[32][16] = 2;
    out.transformMaskMappingLocalToWorld[32][17] = 3;
    out.transformMaskMappingLocalToWorld[32][18] = 6;
    out.transformMaskMappingLocalToWorld[32][19] = 7;
    out.transformMaskMappingLocalToWorld[32][20] = 18;
    out.transformMaskMappingLocalToWorld[32][21] = 19;
    out.transformMaskMappingLocalToWorld[32][22] = 22;
    out.transformMaskMappingLocalToWorld[32][23] = 23;
    out.transformMaskMappingLocalToWorld[32][24] = 66;
    out.transformMaskMappingLocalToWorld[32][25] = 67;
    out.transformMaskMappingLocalToWorld[32][26] = 70;
    out.transformMaskMappingLocalToWorld[32][27] = 71;
    out.transformMaskMappingLocalToWorld[32][28] = 82;
    out.transformMaskMappingLocalToWorld[32][29] = 83;
    out.transformMaskMappingLocalToWorld[32][30] = 86;
    out.transformMaskMappingLocalToWorld[32][31] = 87;
    out.transformMaskMappingLocalToWorld[32][32] = 8;
    out.transformMaskMappingLocalToWorld[32][33] = 9;
    out.transformMaskMappingLocalToWorld[32][34] = 12;
    out.transformMaskMappingLocalToWorld[32][35] = 13;
    out.transformMaskMappingLocalToWorld[32][36] = 24;
    out.transformMaskMappingLocalToWorld[32][37] = 25;
    out.transformMaskMappingLocalToWorld[32][38] = 28;
    out.transformMaskMappingLocalToWorld[32][39] = 29;
    out.transformMaskMappingLocalToWorld[32][40] = 72;
    out.transformMaskMappingLocalToWorld[32][41] = 73;
    out.transformMaskMappingLocalToWorld[32][42] = 76;
    out.transformMaskMappingLocalToWorld[32][43] = 77;
    out.transformMaskMappingLocalToWorld[32][44] = 88;
    out.transformMaskMappingLocalToWorld[32][45] = 89;
    out.transformMaskMappingLocalToWorld[32][46] = 92;
    out.transformMaskMappingLocalToWorld[32][47] = 93;
    out.transformMaskMappingLocalToWorld[32][48] = 10;
    out.transformMaskMappingLocalToWorld[32][49] = 11;
    out.transformMaskMappingLocalToWorld[32][50] = 14;
    out.transformMaskMappingLocalToWorld[32][51] = 15;
    out.transformMaskMappingLocalToWorld[32][52] = 26;
    out.transformMaskMappingLocalToWorld[32][53] = 27;
    out.transformMaskMappingLocalToWorld[32][54] = 30;
    out.transformMaskMappingLocalToWorld[32][55] = 31;
    out.transformMaskMappingLocalToWorld[32][56] = 74;
    out.transformMaskMappingLocalToWorld[32][57] = 75;
    out.transformMaskMappingLocalToWorld[32][58] = 78;
    out.transformMaskMappingLocalToWorld[32][59] = 79;
    out.transformMaskMappingLocalToWorld[32][60] = 90;
    out.transformMaskMappingLocalToWorld[32][61] = 91;
    out.transformMaskMappingLocalToWorld[32][62] = 94;
    out.transformMaskMappingLocalToWorld[32][63] = 95;
    out.transformMaskMappingLocalToWorld[32][64] = 32;
    out.transformMaskMappingLocalToWorld[32][65] = 33;
    out.transformMaskMappingLocalToWorld[32][66] = 36;
    out.transformMaskMappingLocalToWorld[32][67] = 37;
    out.transformMaskMappingLocalToWorld[32][68] = 48;
    out.transformMaskMappingLocalToWorld[32][69] = 49;
    out.transformMaskMappingLocalToWorld[32][70] = 52;
    out.transformMaskMappingLocalToWorld[32][71] = 53;
    out.transformMaskMappingLocalToWorld[32][72] = 96;
    out.transformMaskMappingLocalToWorld[32][73] = 97;
    out.transformMaskMappingLocalToWorld[32][74] = 100;
    out.transformMaskMappingLocalToWorld[32][75] = 101;
    out.transformMaskMappingLocalToWorld[32][76] = 112;
    out.transformMaskMappingLocalToWorld[32][77] = 113;
    out.transformMaskMappingLocalToWorld[32][78] = 116;
    out.transformMaskMappingLocalToWorld[32][79] = 117;
    out.transformMaskMappingLocalToWorld[32][80] = 34;
    out.transformMaskMappingLocalToWorld[32][81] = 35;
    out.transformMaskMappingLocalToWorld[32][82] = 38;
    out.transformMaskMappingLocalToWorld[32][83] = 39;
    out.transformMaskMappingLocalToWorld[32][84] = 50;
    out.transformMaskMappingLocalToWorld[32][85] = 51;
    out.transformMaskMappingLocalToWorld[32][86] = 54;
    out.transformMaskMappingLocalToWorld[32][87] = 55;
    out.transformMaskMappingLocalToWorld[32][88] = 98;
    out.transformMaskMappingLocalToWorld[32][89] = 99;
    out.transformMaskMappingLocalToWorld[32][90] = 102;
    out.transformMaskMappingLocalToWorld[32][91] = 103;
    out.transformMaskMappingLocalToWorld[32][92] = 114;
    out.transformMaskMappingLocalToWorld[32][93] = 115;
    out.transformMaskMappingLocalToWorld[32][94] = 118;
    out.transformMaskMappingLocalToWorld[32][95] = 119;
    out.transformMaskMappingLocalToWorld[32][96] = 40;
    out.transformMaskMappingLocalToWorld[32][97] = 41;
    out.transformMaskMappingLocalToWorld[32][98] = 44;
    out.transformMaskMappingLocalToWorld[32][99] = 45;
    out.transformMaskMappingLocalToWorld[32][100] = 56;
    out.transformMaskMappingLocalToWorld[32][101] = 57;
    out.transformMaskMappingLocalToWorld[32][102] = 60;
    out.transformMaskMappingLocalToWorld[32][103] = 61;
    out.transformMaskMappingLocalToWorld[32][104] = 104;
    out.transformMaskMappingLocalToWorld[32][105] = 105;
    out.transformMaskMappingLocalToWorld[32][106] = 108;
    out.transformMaskMappingLocalToWorld[32][107] = 109;
    out.transformMaskMappingLocalToWorld[32][108] = 120;
    out.transformMaskMappingLocalToWorld[32][109] = 121;
    out.transformMaskMappingLocalToWorld[32][110] = 124;
    out.transformMaskMappingLocalToWorld[32][111] = 125;
    out.transformMaskMappingLocalToWorld[32][112] = 42;
    out.transformMaskMappingLocalToWorld[32][113] = 43;
    out.transformMaskMappingLocalToWorld[32][114] = 46;
    out.transformMaskMappingLocalToWorld[32][115] = 47;
    out.transformMaskMappingLocalToWorld[32][116] = 58;
    out.transformMaskMappingLocalToWorld[32][117] = 59;
    out.transformMaskMappingLocalToWorld[32][118] = 62;
    out.transformMaskMappingLocalToWorld[32][119] = 63;
    out.transformMaskMappingLocalToWorld[32][120] = 106;
    out.transformMaskMappingLocalToWorld[32][121] = 107;
    out.transformMaskMappingLocalToWorld[32][122] = 110;
    out.transformMaskMappingLocalToWorld[32][123] = 111;
    out.transformMaskMappingLocalToWorld[32][124] = 122;
    out.transformMaskMappingLocalToWorld[32][125] = 123;
    out.transformMaskMappingLocalToWorld[32][126] = 126;
    out.transformMaskMappingLocalToWorld[32][127] = 127;
    out.transformMaskMappingLocalToWorld[32][128] = 128;
    out.transformMaskMappingLocalToWorld[32][129] = 129;
    out.transformMaskMappingLocalToWorld[32][130] = 132;
    out.transformMaskMappingLocalToWorld[32][131] = 133;
    out.transformMaskMappingLocalToWorld[32][132] = 144;
    out.transformMaskMappingLocalToWorld[32][133] = 145;
    out.transformMaskMappingLocalToWorld[32][134] = 148;
    out.transformMaskMappingLocalToWorld[32][135] = 149;
    out.transformMaskMappingLocalToWorld[32][136] = 192;
    out.transformMaskMappingLocalToWorld[32][137] = 193;
    out.transformMaskMappingLocalToWorld[32][138] = 196;
    out.transformMaskMappingLocalToWorld[32][139] = 197;
    out.transformMaskMappingLocalToWorld[32][140] = 208;
    out.transformMaskMappingLocalToWorld[32][141] = 209;
    out.transformMaskMappingLocalToWorld[32][142] = 212;
    out.transformMaskMappingLocalToWorld[32][143] = 213;
    out.transformMaskMappingLocalToWorld[32][144] = 130;
    out.transformMaskMappingLocalToWorld[32][145] = 131;
    out.transformMaskMappingLocalToWorld[32][146] = 134;
    out.transformMaskMappingLocalToWorld[32][147] = 135;
    out.transformMaskMappingLocalToWorld[32][148] = 146;
    out.transformMaskMappingLocalToWorld[32][149] = 147;
    out.transformMaskMappingLocalToWorld[32][150] = 150;
    out.transformMaskMappingLocalToWorld[32][151] = 151;
    out.transformMaskMappingLocalToWorld[32][152] = 194;
    out.transformMaskMappingLocalToWorld[32][153] = 195;
    out.transformMaskMappingLocalToWorld[32][154] = 198;
    out.transformMaskMappingLocalToWorld[32][155] = 199;
    out.transformMaskMappingLocalToWorld[32][156] = 210;
    out.transformMaskMappingLocalToWorld[32][157] = 211;
    out.transformMaskMappingLocalToWorld[32][158] = 214;
    out.transformMaskMappingLocalToWorld[32][159] = 215;
    out.transformMaskMappingLocalToWorld[32][160] = 136;
    out.transformMaskMappingLocalToWorld[32][161] = 137;
    out.transformMaskMappingLocalToWorld[32][162] = 140;
    out.transformMaskMappingLocalToWorld[32][163] = 141;
    out.transformMaskMappingLocalToWorld[32][164] = 152;
    out.transformMaskMappingLocalToWorld[32][165] = 153;
    out.transformMaskMappingLocalToWorld[32][166] = 156;
    out.transformMaskMappingLocalToWorld[32][167] = 157;
    out.transformMaskMappingLocalToWorld[32][168] = 200;
    out.transformMaskMappingLocalToWorld[32][169] = 201;
    out.transformMaskMappingLocalToWorld[32][170] = 204;
    out.transformMaskMappingLocalToWorld[32][171] = 205;
    out.transformMaskMappingLocalToWorld[32][172] = 216;
    out.transformMaskMappingLocalToWorld[32][173] = 217;
    out.transformMaskMappingLocalToWorld[32][174] = 220;
    out.transformMaskMappingLocalToWorld[32][175] = 221;
    out.transformMaskMappingLocalToWorld[32][176] = 138;
    out.transformMaskMappingLocalToWorld[32][177] = 139;
    out.transformMaskMappingLocalToWorld[32][178] = 142;
    out.transformMaskMappingLocalToWorld[32][179] = 143;
    out.transformMaskMappingLocalToWorld[32][180] = 154;
    out.transformMaskMappingLocalToWorld[32][181] = 155;
    out.transformMaskMappingLocalToWorld[32][182] = 158;
    out.transformMaskMappingLocalToWorld[32][183] = 159;
    out.transformMaskMappingLocalToWorld[32][184] = 202;
    out.transformMaskMappingLocalToWorld[32][185] = 203;
    out.transformMaskMappingLocalToWorld[32][186] = 206;
    out.transformMaskMappingLocalToWorld[32][187] = 207;
    out.transformMaskMappingLocalToWorld[32][188] = 218;
    out.transformMaskMappingLocalToWorld[32][189] = 219;
    out.transformMaskMappingLocalToWorld[32][190] = 222;
    out.transformMaskMappingLocalToWorld[32][191] = 223;
    out.transformMaskMappingLocalToWorld[32][192] = 160;
    out.transformMaskMappingLocalToWorld[32][193] = 161;
    out.transformMaskMappingLocalToWorld[32][194] = 164;
    out.transformMaskMappingLocalToWorld[32][195] = 165;
    out.transformMaskMappingLocalToWorld[32][196] = 176;
    out.transformMaskMappingLocalToWorld[32][197] = 177;
    out.transformMaskMappingLocalToWorld[32][198] = 180;
    out.transformMaskMappingLocalToWorld[32][199] = 181;
    out.transformMaskMappingLocalToWorld[32][200] = 224;
    out.transformMaskMappingLocalToWorld[32][201] = 225;
    out.transformMaskMappingLocalToWorld[32][202] = 228;
    out.transformMaskMappingLocalToWorld[32][203] = 229;
    out.transformMaskMappingLocalToWorld[32][204] = 240;
    out.transformMaskMappingLocalToWorld[32][205] = 241;
    out.transformMaskMappingLocalToWorld[32][206] = 244;
    out.transformMaskMappingLocalToWorld[32][207] = 245;
    out.transformMaskMappingLocalToWorld[32][208] = 162;
    out.transformMaskMappingLocalToWorld[32][209] = 163;
    out.transformMaskMappingLocalToWorld[32][210] = 166;
    out.transformMaskMappingLocalToWorld[32][211] = 167;
    out.transformMaskMappingLocalToWorld[32][212] = 178;
    out.transformMaskMappingLocalToWorld[32][213] = 179;
    out.transformMaskMappingLocalToWorld[32][214] = 182;
    out.transformMaskMappingLocalToWorld[32][215] = 183;
    out.transformMaskMappingLocalToWorld[32][216] = 226;
    out.transformMaskMappingLocalToWorld[32][217] = 227;
    out.transformMaskMappingLocalToWorld[32][218] = 230;
    out.transformMaskMappingLocalToWorld[32][219] = 231;
    out.transformMaskMappingLocalToWorld[32][220] = 242;
    out.transformMaskMappingLocalToWorld[32][221] = 243;
    out.transformMaskMappingLocalToWorld[32][222] = 246;
    out.transformMaskMappingLocalToWorld[32][223] = 247;
    out.transformMaskMappingLocalToWorld[32][224] = 168;
    out.transformMaskMappingLocalToWorld[32][225] = 169;
    out.transformMaskMappingLocalToWorld[32][226] = 172;
    out.transformMaskMappingLocalToWorld[32][227] = 173;
    out.transformMaskMappingLocalToWorld[32][228] = 184;
    out.transformMaskMappingLocalToWorld[32][229] = 185;
    out.transformMaskMappingLocalToWorld[32][230] = 188;
    out.transformMaskMappingLocalToWorld[32][231] = 189;
    out.transformMaskMappingLocalToWorld[32][232] = 232;
    out.transformMaskMappingLocalToWorld[32][233] = 233;
    out.transformMaskMappingLocalToWorld[32][234] = 236;
    out.transformMaskMappingLocalToWorld[32][235] = 237;
    out.transformMaskMappingLocalToWorld[32][236] = 248;
    out.transformMaskMappingLocalToWorld[32][237] = 249;
    out.transformMaskMappingLocalToWorld[32][238] = 252;
    out.transformMaskMappingLocalToWorld[32][239] = 253;
    out.transformMaskMappingLocalToWorld[32][240] = 170;
    out.transformMaskMappingLocalToWorld[32][241] = 171;
    out.transformMaskMappingLocalToWorld[32][242] = 174;
    out.transformMaskMappingLocalToWorld[32][243] = 175;
    out.transformMaskMappingLocalToWorld[32][244] = 186;
    out.transformMaskMappingLocalToWorld[32][245] = 187;
    out.transformMaskMappingLocalToWorld[32][246] = 190;
    out.transformMaskMappingLocalToWorld[32][247] = 191;
    out.transformMaskMappingLocalToWorld[32][248] = 234;
    out.transformMaskMappingLocalToWorld[32][249] = 235;
    out.transformMaskMappingLocalToWorld[32][250] = 238;
    out.transformMaskMappingLocalToWorld[32][251] = 239;
    out.transformMaskMappingLocalToWorld[32][252] = 250;
    out.transformMaskMappingLocalToWorld[32][253] = 251;
    out.transformMaskMappingLocalToWorld[32][254] = 254;
    out.transformMaskMappingLocalToWorld[32][255] = 255;
    out.transformMaskMappingLocalToWorld[33][0] = 0;
    out.transformMaskMappingLocalToWorld[33][1] = 2;
    out.transformMaskMappingLocalToWorld[33][2] = 8;
    out.transformMaskMappingLocalToWorld[33][3] = 10;
    out.transformMaskMappingLocalToWorld[33][4] = 32;
    out.transformMaskMappingLocalToWorld[33][5] = 34;
    out.transformMaskMappingLocalToWorld[33][6] = 40;
    out.transformMaskMappingLocalToWorld[33][7] = 42;
    out.transformMaskMappingLocalToWorld[33][8] = 128;
    out.transformMaskMappingLocalToWorld[33][9] = 130;
    out.transformMaskMappingLocalToWorld[33][10] = 136;
    out.transformMaskMappingLocalToWorld[33][11] = 138;
    out.transformMaskMappingLocalToWorld[33][12] = 160;
    out.transformMaskMappingLocalToWorld[33][13] = 162;
    out.transformMaskMappingLocalToWorld[33][14] = 168;
    out.transformMaskMappingLocalToWorld[33][15] = 170;
    out.transformMaskMappingLocalToWorld[33][16] = 1;
    out.transformMaskMappingLocalToWorld[33][17] = 3;
    out.transformMaskMappingLocalToWorld[33][18] = 9;
    out.transformMaskMappingLocalToWorld[33][19] = 11;
    out.transformMaskMappingLocalToWorld[33][20] = 33;
    out.transformMaskMappingLocalToWorld[33][21] = 35;
    out.transformMaskMappingLocalToWorld[33][22] = 41;
    out.transformMaskMappingLocalToWorld[33][23] = 43;
    out.transformMaskMappingLocalToWorld[33][24] = 129;
    out.transformMaskMappingLocalToWorld[33][25] = 131;
    out.transformMaskMappingLocalToWorld[33][26] = 137;
    out.transformMaskMappingLocalToWorld[33][27] = 139;
    out.transformMaskMappingLocalToWorld[33][28] = 161;
    out.transformMaskMappingLocalToWorld[33][29] = 163;
    out.transformMaskMappingLocalToWorld[33][30] = 169;
    out.transformMaskMappingLocalToWorld[33][31] = 171;
    out.transformMaskMappingLocalToWorld[33][32] = 4;
    out.transformMaskMappingLocalToWorld[33][33] = 6;
    out.transformMaskMappingLocalToWorld[33][34] = 12;
    out.transformMaskMappingLocalToWorld[33][35] = 14;
    out.transformMaskMappingLocalToWorld[33][36] = 36;
    out.transformMaskMappingLocalToWorld[33][37] = 38;
    out.transformMaskMappingLocalToWorld[33][38] = 44;
    out.transformMaskMappingLocalToWorld[33][39] = 46;
    out.transformMaskMappingLocalToWorld[33][40] = 132;
    out.transformMaskMappingLocalToWorld[33][41] = 134;
    out.transformMaskMappingLocalToWorld[33][42] = 140;
    out.transformMaskMappingLocalToWorld[33][43] = 142;
    out.transformMaskMappingLocalToWorld[33][44] = 164;
    out.transformMaskMappingLocalToWorld[33][45] = 166;
    out.transformMaskMappingLocalToWorld[33][46] = 172;
    out.transformMaskMappingLocalToWorld[33][47] = 174;
    out.transformMaskMappingLocalToWorld[33][48] = 5;
    out.transformMaskMappingLocalToWorld[33][49] = 7;
    out.transformMaskMappingLocalToWorld[33][50] = 13;
    out.transformMaskMappingLocalToWorld[33][51] = 15;
    out.transformMaskMappingLocalToWorld[33][52] = 37;
    out.transformMaskMappingLocalToWorld[33][53] = 39;
    out.transformMaskMappingLocalToWorld[33][54] = 45;
    out.transformMaskMappingLocalToWorld[33][55] = 47;
    out.transformMaskMappingLocalToWorld[33][56] = 133;
    out.transformMaskMappingLocalToWorld[33][57] = 135;
    out.transformMaskMappingLocalToWorld[33][58] = 141;
    out.transformMaskMappingLocalToWorld[33][59] = 143;
    out.transformMaskMappingLocalToWorld[33][60] = 165;
    out.transformMaskMappingLocalToWorld[33][61] = 167;
    out.transformMaskMappingLocalToWorld[33][62] = 173;
    out.transformMaskMappingLocalToWorld[33][63] = 175;
    out.transformMaskMappingLocalToWorld[33][64] = 16;
    out.transformMaskMappingLocalToWorld[33][65] = 18;
    out.transformMaskMappingLocalToWorld[33][66] = 24;
    out.transformMaskMappingLocalToWorld[33][67] = 26;
    out.transformMaskMappingLocalToWorld[33][68] = 48;
    out.transformMaskMappingLocalToWorld[33][69] = 50;
    out.transformMaskMappingLocalToWorld[33][70] = 56;
    out.transformMaskMappingLocalToWorld[33][71] = 58;
    out.transformMaskMappingLocalToWorld[33][72] = 144;
    out.transformMaskMappingLocalToWorld[33][73] = 146;
    out.transformMaskMappingLocalToWorld[33][74] = 152;
    out.transformMaskMappingLocalToWorld[33][75] = 154;
    out.transformMaskMappingLocalToWorld[33][76] = 176;
    out.transformMaskMappingLocalToWorld[33][77] = 178;
    out.transformMaskMappingLocalToWorld[33][78] = 184;
    out.transformMaskMappingLocalToWorld[33][79] = 186;
    out.transformMaskMappingLocalToWorld[33][80] = 17;
    out.transformMaskMappingLocalToWorld[33][81] = 19;
    out.transformMaskMappingLocalToWorld[33][82] = 25;
    out.transformMaskMappingLocalToWorld[33][83] = 27;
    out.transformMaskMappingLocalToWorld[33][84] = 49;
    out.transformMaskMappingLocalToWorld[33][85] = 51;
    out.transformMaskMappingLocalToWorld[33][86] = 57;
    out.transformMaskMappingLocalToWorld[33][87] = 59;
    out.transformMaskMappingLocalToWorld[33][88] = 145;
    out.transformMaskMappingLocalToWorld[33][89] = 147;
    out.transformMaskMappingLocalToWorld[33][90] = 153;
    out.transformMaskMappingLocalToWorld[33][91] = 155;
    out.transformMaskMappingLocalToWorld[33][92] = 177;
    out.transformMaskMappingLocalToWorld[33][93] = 179;
    out.transformMaskMappingLocalToWorld[33][94] = 185;
    out.transformMaskMappingLocalToWorld[33][95] = 187;
    out.transformMaskMappingLocalToWorld[33][96] = 20;
    out.transformMaskMappingLocalToWorld[33][97] = 22;
    out.transformMaskMappingLocalToWorld[33][98] = 28;
    out.transformMaskMappingLocalToWorld[33][99] = 30;
    out.transformMaskMappingLocalToWorld[33][100] = 52;
    out.transformMaskMappingLocalToWorld[33][101] = 54;
    out.transformMaskMappingLocalToWorld[33][102] = 60;
    out.transformMaskMappingLocalToWorld[33][103] = 62;
    out.transformMaskMappingLocalToWorld[33][104] = 148;
    out.transformMaskMappingLocalToWorld[33][105] = 150;
    out.transformMaskMappingLocalToWorld[33][106] = 156;
    out.transformMaskMappingLocalToWorld[33][107] = 158;
    out.transformMaskMappingLocalToWorld[33][108] = 180;
    out.transformMaskMappingLocalToWorld[33][109] = 182;
    out.transformMaskMappingLocalToWorld[33][110] = 188;
    out.transformMaskMappingLocalToWorld[33][111] = 190;
    out.transformMaskMappingLocalToWorld[33][112] = 21;
    out.transformMaskMappingLocalToWorld[33][113] = 23;
    out.transformMaskMappingLocalToWorld[33][114] = 29;
    out.transformMaskMappingLocalToWorld[33][115] = 31;
    out.transformMaskMappingLocalToWorld[33][116] = 53;
    out.transformMaskMappingLocalToWorld[33][117] = 55;
    out.transformMaskMappingLocalToWorld[33][118] = 61;
    out.transformMaskMappingLocalToWorld[33][119] = 63;
    out.transformMaskMappingLocalToWorld[33][120] = 149;
    out.transformMaskMappingLocalToWorld[33][121] = 151;
    out.transformMaskMappingLocalToWorld[33][122] = 157;
    out.transformMaskMappingLocalToWorld[33][123] = 159;
    out.transformMaskMappingLocalToWorld[33][124] = 181;
    out.transformMaskMappingLocalToWorld[33][125] = 183;
    out.transformMaskMappingLocalToWorld[33][126] = 189;
    out.transformMaskMappingLocalToWorld[33][127] = 191;
    out.transformMaskMappingLocalToWorld[33][128] = 64;
    out.transformMaskMappingLocalToWorld[33][129] = 66;
    out.transformMaskMappingLocalToWorld[33][130] = 72;
    out.transformMaskMappingLocalToWorld[33][131] = 74;
    out.transformMaskMappingLocalToWorld[33][132] = 96;
    out.transformMaskMappingLocalToWorld[33][133] = 98;
    out.transformMaskMappingLocalToWorld[33][134] = 104;
    out.transformMaskMappingLocalToWorld[33][135] = 106;
    out.transformMaskMappingLocalToWorld[33][136] = 192;
    out.transformMaskMappingLocalToWorld[33][137] = 194;
    out.transformMaskMappingLocalToWorld[33][138] = 200;
    out.transformMaskMappingLocalToWorld[33][139] = 202;
    out.transformMaskMappingLocalToWorld[33][140] = 224;
    out.transformMaskMappingLocalToWorld[33][141] = 226;
    out.transformMaskMappingLocalToWorld[33][142] = 232;
    out.transformMaskMappingLocalToWorld[33][143] = 234;
    out.transformMaskMappingLocalToWorld[33][144] = 65;
    out.transformMaskMappingLocalToWorld[33][145] = 67;
    out.transformMaskMappingLocalToWorld[33][146] = 73;
    out.transformMaskMappingLocalToWorld[33][147] = 75;
    out.transformMaskMappingLocalToWorld[33][148] = 97;
    out.transformMaskMappingLocalToWorld[33][149] = 99;
    out.transformMaskMappingLocalToWorld[33][150] = 105;
    out.transformMaskMappingLocalToWorld[33][151] = 107;
    out.transformMaskMappingLocalToWorld[33][152] = 193;
    out.transformMaskMappingLocalToWorld[33][153] = 195;
    out.transformMaskMappingLocalToWorld[33][154] = 201;
    out.transformMaskMappingLocalToWorld[33][155] = 203;
    out.transformMaskMappingLocalToWorld[33][156] = 225;
    out.transformMaskMappingLocalToWorld[33][157] = 227;
    out.transformMaskMappingLocalToWorld[33][158] = 233;
    out.transformMaskMappingLocalToWorld[33][159] = 235;
    out.transformMaskMappingLocalToWorld[33][160] = 68;
    out.transformMaskMappingLocalToWorld[33][161] = 70;
    out.transformMaskMappingLocalToWorld[33][162] = 76;
    out.transformMaskMappingLocalToWorld[33][163] = 78;
    out.transformMaskMappingLocalToWorld[33][164] = 100;
    out.transformMaskMappingLocalToWorld[33][165] = 102;
    out.transformMaskMappingLocalToWorld[33][166] = 108;
    out.transformMaskMappingLocalToWorld[33][167] = 110;
    out.transformMaskMappingLocalToWorld[33][168] = 196;
    out.transformMaskMappingLocalToWorld[33][169] = 198;
    out.transformMaskMappingLocalToWorld[33][170] = 204;
    out.transformMaskMappingLocalToWorld[33][171] = 206;
    out.transformMaskMappingLocalToWorld[33][172] = 228;
    out.transformMaskMappingLocalToWorld[33][173] = 230;
    out.transformMaskMappingLocalToWorld[33][174] = 236;
    out.transformMaskMappingLocalToWorld[33][175] = 238;
    out.transformMaskMappingLocalToWorld[33][176] = 69;
    out.transformMaskMappingLocalToWorld[33][177] = 71;
    out.transformMaskMappingLocalToWorld[33][178] = 77;
    out.transformMaskMappingLocalToWorld[33][179] = 79;
    out.transformMaskMappingLocalToWorld[33][180] = 101;
    out.transformMaskMappingLocalToWorld[33][181] = 103;
    out.transformMaskMappingLocalToWorld[33][182] = 109;
    out.transformMaskMappingLocalToWorld[33][183] = 111;
    out.transformMaskMappingLocalToWorld[33][184] = 197;
    out.transformMaskMappingLocalToWorld[33][185] = 199;
    out.transformMaskMappingLocalToWorld[33][186] = 205;
    out.transformMaskMappingLocalToWorld[33][187] = 207;
    out.transformMaskMappingLocalToWorld[33][188] = 229;
    out.transformMaskMappingLocalToWorld[33][189] = 231;
    out.transformMaskMappingLocalToWorld[33][190] = 237;
    out.transformMaskMappingLocalToWorld[33][191] = 239;
    out.transformMaskMappingLocalToWorld[33][192] = 80;
    out.transformMaskMappingLocalToWorld[33][193] = 82;
    out.transformMaskMappingLocalToWorld[33][194] = 88;
    out.transformMaskMappingLocalToWorld[33][195] = 90;
    out.transformMaskMappingLocalToWorld[33][196] = 112;
    out.transformMaskMappingLocalToWorld[33][197] = 114;
    out.transformMaskMappingLocalToWorld[33][198] = 120;
    out.transformMaskMappingLocalToWorld[33][199] = 122;
    out.transformMaskMappingLocalToWorld[33][200] = 208;
    out.transformMaskMappingLocalToWorld[33][201] = 210;
    out.transformMaskMappingLocalToWorld[33][202] = 216;
    out.transformMaskMappingLocalToWorld[33][203] = 218;
    out.transformMaskMappingLocalToWorld[33][204] = 240;
    out.transformMaskMappingLocalToWorld[33][205] = 242;
    out.transformMaskMappingLocalToWorld[33][206] = 248;
    out.transformMaskMappingLocalToWorld[33][207] = 250;
    out.transformMaskMappingLocalToWorld[33][208] = 81;
    out.transformMaskMappingLocalToWorld[33][209] = 83;
    out.transformMaskMappingLocalToWorld[33][210] = 89;
    out.transformMaskMappingLocalToWorld[33][211] = 91;
    out.transformMaskMappingLocalToWorld[33][212] = 113;
    out.transformMaskMappingLocalToWorld[33][213] = 115;
    out.transformMaskMappingLocalToWorld[33][214] = 121;
    out.transformMaskMappingLocalToWorld[33][215] = 123;
    out.transformMaskMappingLocalToWorld[33][216] = 209;
    out.transformMaskMappingLocalToWorld[33][217] = 211;
    out.transformMaskMappingLocalToWorld[33][218] = 217;
    out.transformMaskMappingLocalToWorld[33][219] = 219;
    out.transformMaskMappingLocalToWorld[33][220] = 241;
    out.transformMaskMappingLocalToWorld[33][221] = 243;
    out.transformMaskMappingLocalToWorld[33][222] = 249;
    out.transformMaskMappingLocalToWorld[33][223] = 251;
    out.transformMaskMappingLocalToWorld[33][224] = 84;
    out.transformMaskMappingLocalToWorld[33][225] = 86;
    out.transformMaskMappingLocalToWorld[33][226] = 92;
    out.transformMaskMappingLocalToWorld[33][227] = 94;
    out.transformMaskMappingLocalToWorld[33][228] = 116;
    out.transformMaskMappingLocalToWorld[33][229] = 118;
    out.transformMaskMappingLocalToWorld[33][230] = 124;
    out.transformMaskMappingLocalToWorld[33][231] = 126;
    out.transformMaskMappingLocalToWorld[33][232] = 212;
    out.transformMaskMappingLocalToWorld[33][233] = 214;
    out.transformMaskMappingLocalToWorld[33][234] = 220;
    out.transformMaskMappingLocalToWorld[33][235] = 222;
    out.transformMaskMappingLocalToWorld[33][236] = 244;
    out.transformMaskMappingLocalToWorld[33][237] = 246;
    out.transformMaskMappingLocalToWorld[33][238] = 252;
    out.transformMaskMappingLocalToWorld[33][239] = 254;
    out.transformMaskMappingLocalToWorld[33][240] = 85;
    out.transformMaskMappingLocalToWorld[33][241] = 87;
    out.transformMaskMappingLocalToWorld[33][242] = 93;
    out.transformMaskMappingLocalToWorld[33][243] = 95;
    out.transformMaskMappingLocalToWorld[33][244] = 117;
    out.transformMaskMappingLocalToWorld[33][245] = 119;
    out.transformMaskMappingLocalToWorld[33][246] = 125;
    out.transformMaskMappingLocalToWorld[33][247] = 127;
    out.transformMaskMappingLocalToWorld[33][248] = 213;
    out.transformMaskMappingLocalToWorld[33][249] = 215;
    out.transformMaskMappingLocalToWorld[33][250] = 221;
    out.transformMaskMappingLocalToWorld[33][251] = 223;
    out.transformMaskMappingLocalToWorld[33][252] = 245;
    out.transformMaskMappingLocalToWorld[33][253] = 247;
    out.transformMaskMappingLocalToWorld[33][254] = 253;
    out.transformMaskMappingLocalToWorld[33][255] = 255;
    out.transformMaskMappingLocalToWorld[34][0] = 0;
    out.transformMaskMappingLocalToWorld[34][1] = 4;
    out.transformMaskMappingLocalToWorld[34][2] = 1;
    out.transformMaskMappingLocalToWorld[34][3] = 5;
    out.transformMaskMappingLocalToWorld[34][4] = 64;
    out.transformMaskMappingLocalToWorld[34][5] = 68;
    out.transformMaskMappingLocalToWorld[34][6] = 65;
    out.transformMaskMappingLocalToWorld[34][7] = 69;
    out.transformMaskMappingLocalToWorld[34][8] = 16;
    out.transformMaskMappingLocalToWorld[34][9] = 20;
    out.transformMaskMappingLocalToWorld[34][10] = 17;
    out.transformMaskMappingLocalToWorld[34][11] = 21;
    out.transformMaskMappingLocalToWorld[34][12] = 80;
    out.transformMaskMappingLocalToWorld[34][13] = 84;
    out.transformMaskMappingLocalToWorld[34][14] = 81;
    out.transformMaskMappingLocalToWorld[34][15] = 85;
    out.transformMaskMappingLocalToWorld[34][16] = 8;
    out.transformMaskMappingLocalToWorld[34][17] = 12;
    out.transformMaskMappingLocalToWorld[34][18] = 9;
    out.transformMaskMappingLocalToWorld[34][19] = 13;
    out.transformMaskMappingLocalToWorld[34][20] = 72;
    out.transformMaskMappingLocalToWorld[34][21] = 76;
    out.transformMaskMappingLocalToWorld[34][22] = 73;
    out.transformMaskMappingLocalToWorld[34][23] = 77;
    out.transformMaskMappingLocalToWorld[34][24] = 24;
    out.transformMaskMappingLocalToWorld[34][25] = 28;
    out.transformMaskMappingLocalToWorld[34][26] = 25;
    out.transformMaskMappingLocalToWorld[34][27] = 29;
    out.transformMaskMappingLocalToWorld[34][28] = 88;
    out.transformMaskMappingLocalToWorld[34][29] = 92;
    out.transformMaskMappingLocalToWorld[34][30] = 89;
    out.transformMaskMappingLocalToWorld[34][31] = 93;
    out.transformMaskMappingLocalToWorld[34][32] = 2;
    out.transformMaskMappingLocalToWorld[34][33] = 6;
    out.transformMaskMappingLocalToWorld[34][34] = 3;
    out.transformMaskMappingLocalToWorld[34][35] = 7;
    out.transformMaskMappingLocalToWorld[34][36] = 66;
    out.transformMaskMappingLocalToWorld[34][37] = 70;
    out.transformMaskMappingLocalToWorld[34][38] = 67;
    out.transformMaskMappingLocalToWorld[34][39] = 71;
    out.transformMaskMappingLocalToWorld[34][40] = 18;
    out.transformMaskMappingLocalToWorld[34][41] = 22;
    out.transformMaskMappingLocalToWorld[34][42] = 19;
    out.transformMaskMappingLocalToWorld[34][43] = 23;
    out.transformMaskMappingLocalToWorld[34][44] = 82;
    out.transformMaskMappingLocalToWorld[34][45] = 86;
    out.transformMaskMappingLocalToWorld[34][46] = 83;
    out.transformMaskMappingLocalToWorld[34][47] = 87;
    out.transformMaskMappingLocalToWorld[34][48] = 10;
    out.transformMaskMappingLocalToWorld[34][49] = 14;
    out.transformMaskMappingLocalToWorld[34][50] = 11;
    out.transformMaskMappingLocalToWorld[34][51] = 15;
    out.transformMaskMappingLocalToWorld[34][52] = 74;
    out.transformMaskMappingLocalToWorld[34][53] = 78;
    out.transformMaskMappingLocalToWorld[34][54] = 75;
    out.transformMaskMappingLocalToWorld[34][55] = 79;
    out.transformMaskMappingLocalToWorld[34][56] = 26;
    out.transformMaskMappingLocalToWorld[34][57] = 30;
    out.transformMaskMappingLocalToWorld[34][58] = 27;
    out.transformMaskMappingLocalToWorld[34][59] = 31;
    out.transformMaskMappingLocalToWorld[34][60] = 90;
    out.transformMaskMappingLocalToWorld[34][61] = 94;
    out.transformMaskMappingLocalToWorld[34][62] = 91;
    out.transformMaskMappingLocalToWorld[34][63] = 95;
    out.transformMaskMappingLocalToWorld[34][64] = 128;
    out.transformMaskMappingLocalToWorld[34][65] = 132;
    out.transformMaskMappingLocalToWorld[34][66] = 129;
    out.transformMaskMappingLocalToWorld[34][67] = 133;
    out.transformMaskMappingLocalToWorld[34][68] = 192;
    out.transformMaskMappingLocalToWorld[34][69] = 196;
    out.transformMaskMappingLocalToWorld[34][70] = 193;
    out.transformMaskMappingLocalToWorld[34][71] = 197;
    out.transformMaskMappingLocalToWorld[34][72] = 144;
    out.transformMaskMappingLocalToWorld[34][73] = 148;
    out.transformMaskMappingLocalToWorld[34][74] = 145;
    out.transformMaskMappingLocalToWorld[34][75] = 149;
    out.transformMaskMappingLocalToWorld[34][76] = 208;
    out.transformMaskMappingLocalToWorld[34][77] = 212;
    out.transformMaskMappingLocalToWorld[34][78] = 209;
    out.transformMaskMappingLocalToWorld[34][79] = 213;
    out.transformMaskMappingLocalToWorld[34][80] = 136;
    out.transformMaskMappingLocalToWorld[34][81] = 140;
    out.transformMaskMappingLocalToWorld[34][82] = 137;
    out.transformMaskMappingLocalToWorld[34][83] = 141;
    out.transformMaskMappingLocalToWorld[34][84] = 200;
    out.transformMaskMappingLocalToWorld[34][85] = 204;
    out.transformMaskMappingLocalToWorld[34][86] = 201;
    out.transformMaskMappingLocalToWorld[34][87] = 205;
    out.transformMaskMappingLocalToWorld[34][88] = 152;
    out.transformMaskMappingLocalToWorld[34][89] = 156;
    out.transformMaskMappingLocalToWorld[34][90] = 153;
    out.transformMaskMappingLocalToWorld[34][91] = 157;
    out.transformMaskMappingLocalToWorld[34][92] = 216;
    out.transformMaskMappingLocalToWorld[34][93] = 220;
    out.transformMaskMappingLocalToWorld[34][94] = 217;
    out.transformMaskMappingLocalToWorld[34][95] = 221;
    out.transformMaskMappingLocalToWorld[34][96] = 130;
    out.transformMaskMappingLocalToWorld[34][97] = 134;
    out.transformMaskMappingLocalToWorld[34][98] = 131;
    out.transformMaskMappingLocalToWorld[34][99] = 135;
    out.transformMaskMappingLocalToWorld[34][100] = 194;
    out.transformMaskMappingLocalToWorld[34][101] = 198;
    out.transformMaskMappingLocalToWorld[34][102] = 195;
    out.transformMaskMappingLocalToWorld[34][103] = 199;
    out.transformMaskMappingLocalToWorld[34][104] = 146;
    out.transformMaskMappingLocalToWorld[34][105] = 150;
    out.transformMaskMappingLocalToWorld[34][106] = 147;
    out.transformMaskMappingLocalToWorld[34][107] = 151;
    out.transformMaskMappingLocalToWorld[34][108] = 210;
    out.transformMaskMappingLocalToWorld[34][109] = 214;
    out.transformMaskMappingLocalToWorld[34][110] = 211;
    out.transformMaskMappingLocalToWorld[34][111] = 215;
    out.transformMaskMappingLocalToWorld[34][112] = 138;
    out.transformMaskMappingLocalToWorld[34][113] = 142;
    out.transformMaskMappingLocalToWorld[34][114] = 139;
    out.transformMaskMappingLocalToWorld[34][115] = 143;
    out.transformMaskMappingLocalToWorld[34][116] = 202;
    out.transformMaskMappingLocalToWorld[34][117] = 206;
    out.transformMaskMappingLocalToWorld[34][118] = 203;
    out.transformMaskMappingLocalToWorld[34][119] = 207;
    out.transformMaskMappingLocalToWorld[34][120] = 154;
    out.transformMaskMappingLocalToWorld[34][121] = 158;
    out.transformMaskMappingLocalToWorld[34][122] = 155;
    out.transformMaskMappingLocalToWorld[34][123] = 159;
    out.transformMaskMappingLocalToWorld[34][124] = 218;
    out.transformMaskMappingLocalToWorld[34][125] = 222;
    out.transformMaskMappingLocalToWorld[34][126] = 219;
    out.transformMaskMappingLocalToWorld[34][127] = 223;
    out.transformMaskMappingLocalToWorld[34][128] = 32;
    out.transformMaskMappingLocalToWorld[34][129] = 36;
    out.transformMaskMappingLocalToWorld[34][130] = 33;
    out.transformMaskMappingLocalToWorld[34][131] = 37;
    out.transformMaskMappingLocalToWorld[34][132] = 96;
    out.transformMaskMappingLocalToWorld[34][133] = 100;
    out.transformMaskMappingLocalToWorld[34][134] = 97;
    out.transformMaskMappingLocalToWorld[34][135] = 101;
    out.transformMaskMappingLocalToWorld[34][136] = 48;
    out.transformMaskMappingLocalToWorld[34][137] = 52;
    out.transformMaskMappingLocalToWorld[34][138] = 49;
    out.transformMaskMappingLocalToWorld[34][139] = 53;
    out.transformMaskMappingLocalToWorld[34][140] = 112;
    out.transformMaskMappingLocalToWorld[34][141] = 116;
    out.transformMaskMappingLocalToWorld[34][142] = 113;
    out.transformMaskMappingLocalToWorld[34][143] = 117;
    out.transformMaskMappingLocalToWorld[34][144] = 40;
    out.transformMaskMappingLocalToWorld[34][145] = 44;
    out.transformMaskMappingLocalToWorld[34][146] = 41;
    out.transformMaskMappingLocalToWorld[34][147] = 45;
    out.transformMaskMappingLocalToWorld[34][148] = 104;
    out.transformMaskMappingLocalToWorld[34][149] = 108;
    out.transformMaskMappingLocalToWorld[34][150] = 105;
    out.transformMaskMappingLocalToWorld[34][151] = 109;
    out.transformMaskMappingLocalToWorld[34][152] = 56;
    out.transformMaskMappingLocalToWorld[34][153] = 60;
    out.transformMaskMappingLocalToWorld[34][154] = 57;
    out.transformMaskMappingLocalToWorld[34][155] = 61;
    out.transformMaskMappingLocalToWorld[34][156] = 120;
    out.transformMaskMappingLocalToWorld[34][157] = 124;
    out.transformMaskMappingLocalToWorld[34][158] = 121;
    out.transformMaskMappingLocalToWorld[34][159] = 125;
    out.transformMaskMappingLocalToWorld[34][160] = 34;
    out.transformMaskMappingLocalToWorld[34][161] = 38;
    out.transformMaskMappingLocalToWorld[34][162] = 35;
    out.transformMaskMappingLocalToWorld[34][163] = 39;
    out.transformMaskMappingLocalToWorld[34][164] = 98;
    out.transformMaskMappingLocalToWorld[34][165] = 102;
    out.transformMaskMappingLocalToWorld[34][166] = 99;
    out.transformMaskMappingLocalToWorld[34][167] = 103;
    out.transformMaskMappingLocalToWorld[34][168] = 50;
    out.transformMaskMappingLocalToWorld[34][169] = 54;
    out.transformMaskMappingLocalToWorld[34][170] = 51;
    out.transformMaskMappingLocalToWorld[34][171] = 55;
    out.transformMaskMappingLocalToWorld[34][172] = 114;
    out.transformMaskMappingLocalToWorld[34][173] = 118;
    out.transformMaskMappingLocalToWorld[34][174] = 115;
    out.transformMaskMappingLocalToWorld[34][175] = 119;
    out.transformMaskMappingLocalToWorld[34][176] = 42;
    out.transformMaskMappingLocalToWorld[34][177] = 46;
    out.transformMaskMappingLocalToWorld[34][178] = 43;
    out.transformMaskMappingLocalToWorld[34][179] = 47;
    out.transformMaskMappingLocalToWorld[34][180] = 106;
    out.transformMaskMappingLocalToWorld[34][181] = 110;
    out.transformMaskMappingLocalToWorld[34][182] = 107;
    out.transformMaskMappingLocalToWorld[34][183] = 111;
    out.transformMaskMappingLocalToWorld[34][184] = 58;
    out.transformMaskMappingLocalToWorld[34][185] = 62;
    out.transformMaskMappingLocalToWorld[34][186] = 59;
    out.transformMaskMappingLocalToWorld[34][187] = 63;
    out.transformMaskMappingLocalToWorld[34][188] = 122;
    out.transformMaskMappingLocalToWorld[34][189] = 126;
    out.transformMaskMappingLocalToWorld[34][190] = 123;
    out.transformMaskMappingLocalToWorld[34][191] = 127;
    out.transformMaskMappingLocalToWorld[34][192] = 160;
    out.transformMaskMappingLocalToWorld[34][193] = 164;
    out.transformMaskMappingLocalToWorld[34][194] = 161;
    out.transformMaskMappingLocalToWorld[34][195] = 165;
    out.transformMaskMappingLocalToWorld[34][196] = 224;
    out.transformMaskMappingLocalToWorld[34][197] = 228;
    out.transformMaskMappingLocalToWorld[34][198] = 225;
    out.transformMaskMappingLocalToWorld[34][199] = 229;
    out.transformMaskMappingLocalToWorld[34][200] = 176;
    out.transformMaskMappingLocalToWorld[34][201] = 180;
    out.transformMaskMappingLocalToWorld[34][202] = 177;
    out.transformMaskMappingLocalToWorld[34][203] = 181;
    out.transformMaskMappingLocalToWorld[34][204] = 240;
    out.transformMaskMappingLocalToWorld[34][205] = 244;
    out.transformMaskMappingLocalToWorld[34][206] = 241;
    out.transformMaskMappingLocalToWorld[34][207] = 245;
    out.transformMaskMappingLocalToWorld[34][208] = 168;
    out.transformMaskMappingLocalToWorld[34][209] = 172;
    out.transformMaskMappingLocalToWorld[34][210] = 169;
    out.transformMaskMappingLocalToWorld[34][211] = 173;
    out.transformMaskMappingLocalToWorld[34][212] = 232;
    out.transformMaskMappingLocalToWorld[34][213] = 236;
    out.transformMaskMappingLocalToWorld[34][214] = 233;
    out.transformMaskMappingLocalToWorld[34][215] = 237;
    out.transformMaskMappingLocalToWorld[34][216] = 184;
    out.transformMaskMappingLocalToWorld[34][217] = 188;
    out.transformMaskMappingLocalToWorld[34][218] = 185;
    out.transformMaskMappingLocalToWorld[34][219] = 189;
    out.transformMaskMappingLocalToWorld[34][220] = 248;
    out.transformMaskMappingLocalToWorld[34][221] = 252;
    out.transformMaskMappingLocalToWorld[34][222] = 249;
    out.transformMaskMappingLocalToWorld[34][223] = 253;
    out.transformMaskMappingLocalToWorld[34][224] = 162;
    out.transformMaskMappingLocalToWorld[34][225] = 166;
    out.transformMaskMappingLocalToWorld[34][226] = 163;
    out.transformMaskMappingLocalToWorld[34][227] = 167;
    out.transformMaskMappingLocalToWorld[34][228] = 226;
    out.transformMaskMappingLocalToWorld[34][229] = 230;
    out.transformMaskMappingLocalToWorld[34][230] = 227;
    out.transformMaskMappingLocalToWorld[34][231] = 231;
    out.transformMaskMappingLocalToWorld[34][232] = 178;
    out.transformMaskMappingLocalToWorld[34][233] = 182;
    out.transformMaskMappingLocalToWorld[34][234] = 179;
    out.transformMaskMappingLocalToWorld[34][235] = 183;
    out.transformMaskMappingLocalToWorld[34][236] = 242;
    out.transformMaskMappingLocalToWorld[34][237] = 246;
    out.transformMaskMappingLocalToWorld[34][238] = 243;
    out.transformMaskMappingLocalToWorld[34][239] = 247;
    out.transformMaskMappingLocalToWorld[34][240] = 170;
    out.transformMaskMappingLocalToWorld[34][241] = 174;
    out.transformMaskMappingLocalToWorld[34][242] = 171;
    out.transformMaskMappingLocalToWorld[34][243] = 175;
    out.transformMaskMappingLocalToWorld[34][244] = 234;
    out.transformMaskMappingLocalToWorld[34][245] = 238;
    out.transformMaskMappingLocalToWorld[34][246] = 235;
    out.transformMaskMappingLocalToWorld[34][247] = 239;
    out.transformMaskMappingLocalToWorld[34][248] = 186;
    out.transformMaskMappingLocalToWorld[34][249] = 190;
    out.transformMaskMappingLocalToWorld[34][250] = 187;
    out.transformMaskMappingLocalToWorld[34][251] = 191;
    out.transformMaskMappingLocalToWorld[34][252] = 250;
    out.transformMaskMappingLocalToWorld[34][253] = 254;
    out.transformMaskMappingLocalToWorld[34][254] = 251;
    out.transformMaskMappingLocalToWorld[34][255] = 255;
    out.transformMaskMappingLocalToWorld[35][0] = 0;
    out.transformMaskMappingLocalToWorld[35][1] = 8;
    out.transformMaskMappingLocalToWorld[35][2] = 2;
    out.transformMaskMappingLocalToWorld[35][3] = 10;
    out.transformMaskMappingLocalToWorld[35][4] = 128;
    out.transformMaskMappingLocalToWorld[35][5] = 136;
    out.transformMaskMappingLocalToWorld[35][6] = 130;
    out.transformMaskMappingLocalToWorld[35][7] = 138;
    out.transformMaskMappingLocalToWorld[35][8] = 32;
    out.transformMaskMappingLocalToWorld[35][9] = 40;
    out.transformMaskMappingLocalToWorld[35][10] = 34;
    out.transformMaskMappingLocalToWorld[35][11] = 42;
    out.transformMaskMappingLocalToWorld[35][12] = 160;
    out.transformMaskMappingLocalToWorld[35][13] = 168;
    out.transformMaskMappingLocalToWorld[35][14] = 162;
    out.transformMaskMappingLocalToWorld[35][15] = 170;
    out.transformMaskMappingLocalToWorld[35][16] = 4;
    out.transformMaskMappingLocalToWorld[35][17] = 12;
    out.transformMaskMappingLocalToWorld[35][18] = 6;
    out.transformMaskMappingLocalToWorld[35][19] = 14;
    out.transformMaskMappingLocalToWorld[35][20] = 132;
    out.transformMaskMappingLocalToWorld[35][21] = 140;
    out.transformMaskMappingLocalToWorld[35][22] = 134;
    out.transformMaskMappingLocalToWorld[35][23] = 142;
    out.transformMaskMappingLocalToWorld[35][24] = 36;
    out.transformMaskMappingLocalToWorld[35][25] = 44;
    out.transformMaskMappingLocalToWorld[35][26] = 38;
    out.transformMaskMappingLocalToWorld[35][27] = 46;
    out.transformMaskMappingLocalToWorld[35][28] = 164;
    out.transformMaskMappingLocalToWorld[35][29] = 172;
    out.transformMaskMappingLocalToWorld[35][30] = 166;
    out.transformMaskMappingLocalToWorld[35][31] = 174;
    out.transformMaskMappingLocalToWorld[35][32] = 1;
    out.transformMaskMappingLocalToWorld[35][33] = 9;
    out.transformMaskMappingLocalToWorld[35][34] = 3;
    out.transformMaskMappingLocalToWorld[35][35] = 11;
    out.transformMaskMappingLocalToWorld[35][36] = 129;
    out.transformMaskMappingLocalToWorld[35][37] = 137;
    out.transformMaskMappingLocalToWorld[35][38] = 131;
    out.transformMaskMappingLocalToWorld[35][39] = 139;
    out.transformMaskMappingLocalToWorld[35][40] = 33;
    out.transformMaskMappingLocalToWorld[35][41] = 41;
    out.transformMaskMappingLocalToWorld[35][42] = 35;
    out.transformMaskMappingLocalToWorld[35][43] = 43;
    out.transformMaskMappingLocalToWorld[35][44] = 161;
    out.transformMaskMappingLocalToWorld[35][45] = 169;
    out.transformMaskMappingLocalToWorld[35][46] = 163;
    out.transformMaskMappingLocalToWorld[35][47] = 171;
    out.transformMaskMappingLocalToWorld[35][48] = 5;
    out.transformMaskMappingLocalToWorld[35][49] = 13;
    out.transformMaskMappingLocalToWorld[35][50] = 7;
    out.transformMaskMappingLocalToWorld[35][51] = 15;
    out.transformMaskMappingLocalToWorld[35][52] = 133;
    out.transformMaskMappingLocalToWorld[35][53] = 141;
    out.transformMaskMappingLocalToWorld[35][54] = 135;
    out.transformMaskMappingLocalToWorld[35][55] = 143;
    out.transformMaskMappingLocalToWorld[35][56] = 37;
    out.transformMaskMappingLocalToWorld[35][57] = 45;
    out.transformMaskMappingLocalToWorld[35][58] = 39;
    out.transformMaskMappingLocalToWorld[35][59] = 47;
    out.transformMaskMappingLocalToWorld[35][60] = 165;
    out.transformMaskMappingLocalToWorld[35][61] = 173;
    out.transformMaskMappingLocalToWorld[35][62] = 167;
    out.transformMaskMappingLocalToWorld[35][63] = 175;
    out.transformMaskMappingLocalToWorld[35][64] = 64;
    out.transformMaskMappingLocalToWorld[35][65] = 72;
    out.transformMaskMappingLocalToWorld[35][66] = 66;
    out.transformMaskMappingLocalToWorld[35][67] = 74;
    out.transformMaskMappingLocalToWorld[35][68] = 192;
    out.transformMaskMappingLocalToWorld[35][69] = 200;
    out.transformMaskMappingLocalToWorld[35][70] = 194;
    out.transformMaskMappingLocalToWorld[35][71] = 202;
    out.transformMaskMappingLocalToWorld[35][72] = 96;
    out.transformMaskMappingLocalToWorld[35][73] = 104;
    out.transformMaskMappingLocalToWorld[35][74] = 98;
    out.transformMaskMappingLocalToWorld[35][75] = 106;
    out.transformMaskMappingLocalToWorld[35][76] = 224;
    out.transformMaskMappingLocalToWorld[35][77] = 232;
    out.transformMaskMappingLocalToWorld[35][78] = 226;
    out.transformMaskMappingLocalToWorld[35][79] = 234;
    out.transformMaskMappingLocalToWorld[35][80] = 68;
    out.transformMaskMappingLocalToWorld[35][81] = 76;
    out.transformMaskMappingLocalToWorld[35][82] = 70;
    out.transformMaskMappingLocalToWorld[35][83] = 78;
    out.transformMaskMappingLocalToWorld[35][84] = 196;
    out.transformMaskMappingLocalToWorld[35][85] = 204;
    out.transformMaskMappingLocalToWorld[35][86] = 198;
    out.transformMaskMappingLocalToWorld[35][87] = 206;
    out.transformMaskMappingLocalToWorld[35][88] = 100;
    out.transformMaskMappingLocalToWorld[35][89] = 108;
    out.transformMaskMappingLocalToWorld[35][90] = 102;
    out.transformMaskMappingLocalToWorld[35][91] = 110;
    out.transformMaskMappingLocalToWorld[35][92] = 228;
    out.transformMaskMappingLocalToWorld[35][93] = 236;
    out.transformMaskMappingLocalToWorld[35][94] = 230;
    out.transformMaskMappingLocalToWorld[35][95] = 238;
    out.transformMaskMappingLocalToWorld[35][96] = 65;
    out.transformMaskMappingLocalToWorld[35][97] = 73;
    out.transformMaskMappingLocalToWorld[35][98] = 67;
    out.transformMaskMappingLocalToWorld[35][99] = 75;
    out.transformMaskMappingLocalToWorld[35][100] = 193;
    out.transformMaskMappingLocalToWorld[35][101] = 201;
    out.transformMaskMappingLocalToWorld[35][102] = 195;
    out.transformMaskMappingLocalToWorld[35][103] = 203;
    out.transformMaskMappingLocalToWorld[35][104] = 97;
    out.transformMaskMappingLocalToWorld[35][105] = 105;
    out.transformMaskMappingLocalToWorld[35][106] = 99;
    out.transformMaskMappingLocalToWorld[35][107] = 107;
    out.transformMaskMappingLocalToWorld[35][108] = 225;
    out.transformMaskMappingLocalToWorld[35][109] = 233;
    out.transformMaskMappingLocalToWorld[35][110] = 227;
    out.transformMaskMappingLocalToWorld[35][111] = 235;
    out.transformMaskMappingLocalToWorld[35][112] = 69;
    out.transformMaskMappingLocalToWorld[35][113] = 77;
    out.transformMaskMappingLocalToWorld[35][114] = 71;
    out.transformMaskMappingLocalToWorld[35][115] = 79;
    out.transformMaskMappingLocalToWorld[35][116] = 197;
    out.transformMaskMappingLocalToWorld[35][117] = 205;
    out.transformMaskMappingLocalToWorld[35][118] = 199;
    out.transformMaskMappingLocalToWorld[35][119] = 207;
    out.transformMaskMappingLocalToWorld[35][120] = 101;
    out.transformMaskMappingLocalToWorld[35][121] = 109;
    out.transformMaskMappingLocalToWorld[35][122] = 103;
    out.transformMaskMappingLocalToWorld[35][123] = 111;
    out.transformMaskMappingLocalToWorld[35][124] = 229;
    out.transformMaskMappingLocalToWorld[35][125] = 237;
    out.transformMaskMappingLocalToWorld[35][126] = 231;
    out.transformMaskMappingLocalToWorld[35][127] = 239;
    out.transformMaskMappingLocalToWorld[35][128] = 16;
    out.transformMaskMappingLocalToWorld[35][129] = 24;
    out.transformMaskMappingLocalToWorld[35][130] = 18;
    out.transformMaskMappingLocalToWorld[35][131] = 26;
    out.transformMaskMappingLocalToWorld[35][132] = 144;
    out.transformMaskMappingLocalToWorld[35][133] = 152;
    out.transformMaskMappingLocalToWorld[35][134] = 146;
    out.transformMaskMappingLocalToWorld[35][135] = 154;
    out.transformMaskMappingLocalToWorld[35][136] = 48;
    out.transformMaskMappingLocalToWorld[35][137] = 56;
    out.transformMaskMappingLocalToWorld[35][138] = 50;
    out.transformMaskMappingLocalToWorld[35][139] = 58;
    out.transformMaskMappingLocalToWorld[35][140] = 176;
    out.transformMaskMappingLocalToWorld[35][141] = 184;
    out.transformMaskMappingLocalToWorld[35][142] = 178;
    out.transformMaskMappingLocalToWorld[35][143] = 186;
    out.transformMaskMappingLocalToWorld[35][144] = 20;
    out.transformMaskMappingLocalToWorld[35][145] = 28;
    out.transformMaskMappingLocalToWorld[35][146] = 22;
    out.transformMaskMappingLocalToWorld[35][147] = 30;
    out.transformMaskMappingLocalToWorld[35][148] = 148;
    out.transformMaskMappingLocalToWorld[35][149] = 156;
    out.transformMaskMappingLocalToWorld[35][150] = 150;
    out.transformMaskMappingLocalToWorld[35][151] = 158;
    out.transformMaskMappingLocalToWorld[35][152] = 52;
    out.transformMaskMappingLocalToWorld[35][153] = 60;
    out.transformMaskMappingLocalToWorld[35][154] = 54;
    out.transformMaskMappingLocalToWorld[35][155] = 62;
    out.transformMaskMappingLocalToWorld[35][156] = 180;
    out.transformMaskMappingLocalToWorld[35][157] = 188;
    out.transformMaskMappingLocalToWorld[35][158] = 182;
    out.transformMaskMappingLocalToWorld[35][159] = 190;
    out.transformMaskMappingLocalToWorld[35][160] = 17;
    out.transformMaskMappingLocalToWorld[35][161] = 25;
    out.transformMaskMappingLocalToWorld[35][162] = 19;
    out.transformMaskMappingLocalToWorld[35][163] = 27;
    out.transformMaskMappingLocalToWorld[35][164] = 145;
    out.transformMaskMappingLocalToWorld[35][165] = 153;
    out.transformMaskMappingLocalToWorld[35][166] = 147;
    out.transformMaskMappingLocalToWorld[35][167] = 155;
    out.transformMaskMappingLocalToWorld[35][168] = 49;
    out.transformMaskMappingLocalToWorld[35][169] = 57;
    out.transformMaskMappingLocalToWorld[35][170] = 51;
    out.transformMaskMappingLocalToWorld[35][171] = 59;
    out.transformMaskMappingLocalToWorld[35][172] = 177;
    out.transformMaskMappingLocalToWorld[35][173] = 185;
    out.transformMaskMappingLocalToWorld[35][174] = 179;
    out.transformMaskMappingLocalToWorld[35][175] = 187;
    out.transformMaskMappingLocalToWorld[35][176] = 21;
    out.transformMaskMappingLocalToWorld[35][177] = 29;
    out.transformMaskMappingLocalToWorld[35][178] = 23;
    out.transformMaskMappingLocalToWorld[35][179] = 31;
    out.transformMaskMappingLocalToWorld[35][180] = 149;
    out.transformMaskMappingLocalToWorld[35][181] = 157;
    out.transformMaskMappingLocalToWorld[35][182] = 151;
    out.transformMaskMappingLocalToWorld[35][183] = 159;
    out.transformMaskMappingLocalToWorld[35][184] = 53;
    out.transformMaskMappingLocalToWorld[35][185] = 61;
    out.transformMaskMappingLocalToWorld[35][186] = 55;
    out.transformMaskMappingLocalToWorld[35][187] = 63;
    out.transformMaskMappingLocalToWorld[35][188] = 181;
    out.transformMaskMappingLocalToWorld[35][189] = 189;
    out.transformMaskMappingLocalToWorld[35][190] = 183;
    out.transformMaskMappingLocalToWorld[35][191] = 191;
    out.transformMaskMappingLocalToWorld[35][192] = 80;
    out.transformMaskMappingLocalToWorld[35][193] = 88;
    out.transformMaskMappingLocalToWorld[35][194] = 82;
    out.transformMaskMappingLocalToWorld[35][195] = 90;
    out.transformMaskMappingLocalToWorld[35][196] = 208;
    out.transformMaskMappingLocalToWorld[35][197] = 216;
    out.transformMaskMappingLocalToWorld[35][198] = 210;
    out.transformMaskMappingLocalToWorld[35][199] = 218;
    out.transformMaskMappingLocalToWorld[35][200] = 112;
    out.transformMaskMappingLocalToWorld[35][201] = 120;
    out.transformMaskMappingLocalToWorld[35][202] = 114;
    out.transformMaskMappingLocalToWorld[35][203] = 122;
    out.transformMaskMappingLocalToWorld[35][204] = 240;
    out.transformMaskMappingLocalToWorld[35][205] = 248;
    out.transformMaskMappingLocalToWorld[35][206] = 242;
    out.transformMaskMappingLocalToWorld[35][207] = 250;
    out.transformMaskMappingLocalToWorld[35][208] = 84;
    out.transformMaskMappingLocalToWorld[35][209] = 92;
    out.transformMaskMappingLocalToWorld[35][210] = 86;
    out.transformMaskMappingLocalToWorld[35][211] = 94;
    out.transformMaskMappingLocalToWorld[35][212] = 212;
    out.transformMaskMappingLocalToWorld[35][213] = 220;
    out.transformMaskMappingLocalToWorld[35][214] = 214;
    out.transformMaskMappingLocalToWorld[35][215] = 222;
    out.transformMaskMappingLocalToWorld[35][216] = 116;
    out.transformMaskMappingLocalToWorld[35][217] = 124;
    out.transformMaskMappingLocalToWorld[35][218] = 118;
    out.transformMaskMappingLocalToWorld[35][219] = 126;
    out.transformMaskMappingLocalToWorld[35][220] = 244;
    out.transformMaskMappingLocalToWorld[35][221] = 252;
    out.transformMaskMappingLocalToWorld[35][222] = 246;
    out.transformMaskMappingLocalToWorld[35][223] = 254;
    out.transformMaskMappingLocalToWorld[35][224] = 81;
    out.transformMaskMappingLocalToWorld[35][225] = 89;
    out.transformMaskMappingLocalToWorld[35][226] = 83;
    out.transformMaskMappingLocalToWorld[35][227] = 91;
    out.transformMaskMappingLocalToWorld[35][228] = 209;
    out.transformMaskMappingLocalToWorld[35][229] = 217;
    out.transformMaskMappingLocalToWorld[35][230] = 211;
    out.transformMaskMappingLocalToWorld[35][231] = 219;
    out.transformMaskMappingLocalToWorld[35][232] = 113;
    out.transformMaskMappingLocalToWorld[35][233] = 121;
    out.transformMaskMappingLocalToWorld[35][234] = 115;
    out.transformMaskMappingLocalToWorld[35][235] = 123;
    out.transformMaskMappingLocalToWorld[35][236] = 241;
    out.transformMaskMappingLocalToWorld[35][237] = 249;
    out.transformMaskMappingLocalToWorld[35][238] = 243;
    out.transformMaskMappingLocalToWorld[35][239] = 251;
    out.transformMaskMappingLocalToWorld[35][240] = 85;
    out.transformMaskMappingLocalToWorld[35][241] = 93;
    out.transformMaskMappingLocalToWorld[35][242] = 87;
    out.transformMaskMappingLocalToWorld[35][243] = 95;
    out.transformMaskMappingLocalToWorld[35][244] = 213;
    out.transformMaskMappingLocalToWorld[35][245] = 221;
    out.transformMaskMappingLocalToWorld[35][246] = 215;
    out.transformMaskMappingLocalToWorld[35][247] = 223;
    out.transformMaskMappingLocalToWorld[35][248] = 117;
    out.transformMaskMappingLocalToWorld[35][249] = 125;
    out.transformMaskMappingLocalToWorld[35][250] = 119;
    out.transformMaskMappingLocalToWorld[35][251] = 127;
    out.transformMaskMappingLocalToWorld[35][252] = 245;
    out.transformMaskMappingLocalToWorld[35][253] = 253;
    out.transformMaskMappingLocalToWorld[35][254] = 247;
    out.transformMaskMappingLocalToWorld[35][255] = 255;
    out.transformMaskMappingLocalToWorld[36][0] = 0;
    out.transformMaskMappingLocalToWorld[36][1] = 16;
    out.transformMaskMappingLocalToWorld[36][2] = 64;
    out.transformMaskMappingLocalToWorld[36][3] = 80;
    out.transformMaskMappingLocalToWorld[36][4] = 1;
    out.transformMaskMappingLocalToWorld[36][5] = 17;
    out.transformMaskMappingLocalToWorld[36][6] = 65;
    out.transformMaskMappingLocalToWorld[36][7] = 81;
    out.transformMaskMappingLocalToWorld[36][8] = 4;
    out.transformMaskMappingLocalToWorld[36][9] = 20;
    out.transformMaskMappingLocalToWorld[36][10] = 68;
    out.transformMaskMappingLocalToWorld[36][11] = 84;
    out.transformMaskMappingLocalToWorld[36][12] = 5;
    out.transformMaskMappingLocalToWorld[36][13] = 21;
    out.transformMaskMappingLocalToWorld[36][14] = 69;
    out.transformMaskMappingLocalToWorld[36][15] = 85;
    out.transformMaskMappingLocalToWorld[36][16] = 32;
    out.transformMaskMappingLocalToWorld[36][17] = 48;
    out.transformMaskMappingLocalToWorld[36][18] = 96;
    out.transformMaskMappingLocalToWorld[36][19] = 112;
    out.transformMaskMappingLocalToWorld[36][20] = 33;
    out.transformMaskMappingLocalToWorld[36][21] = 49;
    out.transformMaskMappingLocalToWorld[36][22] = 97;
    out.transformMaskMappingLocalToWorld[36][23] = 113;
    out.transformMaskMappingLocalToWorld[36][24] = 36;
    out.transformMaskMappingLocalToWorld[36][25] = 52;
    out.transformMaskMappingLocalToWorld[36][26] = 100;
    out.transformMaskMappingLocalToWorld[36][27] = 116;
    out.transformMaskMappingLocalToWorld[36][28] = 37;
    out.transformMaskMappingLocalToWorld[36][29] = 53;
    out.transformMaskMappingLocalToWorld[36][30] = 101;
    out.transformMaskMappingLocalToWorld[36][31] = 117;
    out.transformMaskMappingLocalToWorld[36][32] = 128;
    out.transformMaskMappingLocalToWorld[36][33] = 144;
    out.transformMaskMappingLocalToWorld[36][34] = 192;
    out.transformMaskMappingLocalToWorld[36][35] = 208;
    out.transformMaskMappingLocalToWorld[36][36] = 129;
    out.transformMaskMappingLocalToWorld[36][37] = 145;
    out.transformMaskMappingLocalToWorld[36][38] = 193;
    out.transformMaskMappingLocalToWorld[36][39] = 209;
    out.transformMaskMappingLocalToWorld[36][40] = 132;
    out.transformMaskMappingLocalToWorld[36][41] = 148;
    out.transformMaskMappingLocalToWorld[36][42] = 196;
    out.transformMaskMappingLocalToWorld[36][43] = 212;
    out.transformMaskMappingLocalToWorld[36][44] = 133;
    out.transformMaskMappingLocalToWorld[36][45] = 149;
    out.transformMaskMappingLocalToWorld[36][46] = 197;
    out.transformMaskMappingLocalToWorld[36][47] = 213;
    out.transformMaskMappingLocalToWorld[36][48] = 160;
    out.transformMaskMappingLocalToWorld[36][49] = 176;
    out.transformMaskMappingLocalToWorld[36][50] = 224;
    out.transformMaskMappingLocalToWorld[36][51] = 240;
    out.transformMaskMappingLocalToWorld[36][52] = 161;
    out.transformMaskMappingLocalToWorld[36][53] = 177;
    out.transformMaskMappingLocalToWorld[36][54] = 225;
    out.transformMaskMappingLocalToWorld[36][55] = 241;
    out.transformMaskMappingLocalToWorld[36][56] = 164;
    out.transformMaskMappingLocalToWorld[36][57] = 180;
    out.transformMaskMappingLocalToWorld[36][58] = 228;
    out.transformMaskMappingLocalToWorld[36][59] = 244;
    out.transformMaskMappingLocalToWorld[36][60] = 165;
    out.transformMaskMappingLocalToWorld[36][61] = 181;
    out.transformMaskMappingLocalToWorld[36][62] = 229;
    out.transformMaskMappingLocalToWorld[36][63] = 245;
    out.transformMaskMappingLocalToWorld[36][64] = 2;
    out.transformMaskMappingLocalToWorld[36][65] = 18;
    out.transformMaskMappingLocalToWorld[36][66] = 66;
    out.transformMaskMappingLocalToWorld[36][67] = 82;
    out.transformMaskMappingLocalToWorld[36][68] = 3;
    out.transformMaskMappingLocalToWorld[36][69] = 19;
    out.transformMaskMappingLocalToWorld[36][70] = 67;
    out.transformMaskMappingLocalToWorld[36][71] = 83;
    out.transformMaskMappingLocalToWorld[36][72] = 6;
    out.transformMaskMappingLocalToWorld[36][73] = 22;
    out.transformMaskMappingLocalToWorld[36][74] = 70;
    out.transformMaskMappingLocalToWorld[36][75] = 86;
    out.transformMaskMappingLocalToWorld[36][76] = 7;
    out.transformMaskMappingLocalToWorld[36][77] = 23;
    out.transformMaskMappingLocalToWorld[36][78] = 71;
    out.transformMaskMappingLocalToWorld[36][79] = 87;
    out.transformMaskMappingLocalToWorld[36][80] = 34;
    out.transformMaskMappingLocalToWorld[36][81] = 50;
    out.transformMaskMappingLocalToWorld[36][82] = 98;
    out.transformMaskMappingLocalToWorld[36][83] = 114;
    out.transformMaskMappingLocalToWorld[36][84] = 35;
    out.transformMaskMappingLocalToWorld[36][85] = 51;
    out.transformMaskMappingLocalToWorld[36][86] = 99;
    out.transformMaskMappingLocalToWorld[36][87] = 115;
    out.transformMaskMappingLocalToWorld[36][88] = 38;
    out.transformMaskMappingLocalToWorld[36][89] = 54;
    out.transformMaskMappingLocalToWorld[36][90] = 102;
    out.transformMaskMappingLocalToWorld[36][91] = 118;
    out.transformMaskMappingLocalToWorld[36][92] = 39;
    out.transformMaskMappingLocalToWorld[36][93] = 55;
    out.transformMaskMappingLocalToWorld[36][94] = 103;
    out.transformMaskMappingLocalToWorld[36][95] = 119;
    out.transformMaskMappingLocalToWorld[36][96] = 130;
    out.transformMaskMappingLocalToWorld[36][97] = 146;
    out.transformMaskMappingLocalToWorld[36][98] = 194;
    out.transformMaskMappingLocalToWorld[36][99] = 210;
    out.transformMaskMappingLocalToWorld[36][100] = 131;
    out.transformMaskMappingLocalToWorld[36][101] = 147;
    out.transformMaskMappingLocalToWorld[36][102] = 195;
    out.transformMaskMappingLocalToWorld[36][103] = 211;
    out.transformMaskMappingLocalToWorld[36][104] = 134;
    out.transformMaskMappingLocalToWorld[36][105] = 150;
    out.transformMaskMappingLocalToWorld[36][106] = 198;
    out.transformMaskMappingLocalToWorld[36][107] = 214;
    out.transformMaskMappingLocalToWorld[36][108] = 135;
    out.transformMaskMappingLocalToWorld[36][109] = 151;
    out.transformMaskMappingLocalToWorld[36][110] = 199;
    out.transformMaskMappingLocalToWorld[36][111] = 215;
    out.transformMaskMappingLocalToWorld[36][112] = 162;
    out.transformMaskMappingLocalToWorld[36][113] = 178;
    out.transformMaskMappingLocalToWorld[36][114] = 226;
    out.transformMaskMappingLocalToWorld[36][115] = 242;
    out.transformMaskMappingLocalToWorld[36][116] = 163;
    out.transformMaskMappingLocalToWorld[36][117] = 179;
    out.transformMaskMappingLocalToWorld[36][118] = 227;
    out.transformMaskMappingLocalToWorld[36][119] = 243;
    out.transformMaskMappingLocalToWorld[36][120] = 166;
    out.transformMaskMappingLocalToWorld[36][121] = 182;
    out.transformMaskMappingLocalToWorld[36][122] = 230;
    out.transformMaskMappingLocalToWorld[36][123] = 246;
    out.transformMaskMappingLocalToWorld[36][124] = 167;
    out.transformMaskMappingLocalToWorld[36][125] = 183;
    out.transformMaskMappingLocalToWorld[36][126] = 231;
    out.transformMaskMappingLocalToWorld[36][127] = 247;
    out.transformMaskMappingLocalToWorld[36][128] = 8;
    out.transformMaskMappingLocalToWorld[36][129] = 24;
    out.transformMaskMappingLocalToWorld[36][130] = 72;
    out.transformMaskMappingLocalToWorld[36][131] = 88;
    out.transformMaskMappingLocalToWorld[36][132] = 9;
    out.transformMaskMappingLocalToWorld[36][133] = 25;
    out.transformMaskMappingLocalToWorld[36][134] = 73;
    out.transformMaskMappingLocalToWorld[36][135] = 89;
    out.transformMaskMappingLocalToWorld[36][136] = 12;
    out.transformMaskMappingLocalToWorld[36][137] = 28;
    out.transformMaskMappingLocalToWorld[36][138] = 76;
    out.transformMaskMappingLocalToWorld[36][139] = 92;
    out.transformMaskMappingLocalToWorld[36][140] = 13;
    out.transformMaskMappingLocalToWorld[36][141] = 29;
    out.transformMaskMappingLocalToWorld[36][142] = 77;
    out.transformMaskMappingLocalToWorld[36][143] = 93;
    out.transformMaskMappingLocalToWorld[36][144] = 40;
    out.transformMaskMappingLocalToWorld[36][145] = 56;
    out.transformMaskMappingLocalToWorld[36][146] = 104;
    out.transformMaskMappingLocalToWorld[36][147] = 120;
    out.transformMaskMappingLocalToWorld[36][148] = 41;
    out.transformMaskMappingLocalToWorld[36][149] = 57;
    out.transformMaskMappingLocalToWorld[36][150] = 105;
    out.transformMaskMappingLocalToWorld[36][151] = 121;
    out.transformMaskMappingLocalToWorld[36][152] = 44;
    out.transformMaskMappingLocalToWorld[36][153] = 60;
    out.transformMaskMappingLocalToWorld[36][154] = 108;
    out.transformMaskMappingLocalToWorld[36][155] = 124;
    out.transformMaskMappingLocalToWorld[36][156] = 45;
    out.transformMaskMappingLocalToWorld[36][157] = 61;
    out.transformMaskMappingLocalToWorld[36][158] = 109;
    out.transformMaskMappingLocalToWorld[36][159] = 125;
    out.transformMaskMappingLocalToWorld[36][160] = 136;
    out.transformMaskMappingLocalToWorld[36][161] = 152;
    out.transformMaskMappingLocalToWorld[36][162] = 200;
    out.transformMaskMappingLocalToWorld[36][163] = 216;
    out.transformMaskMappingLocalToWorld[36][164] = 137;
    out.transformMaskMappingLocalToWorld[36][165] = 153;
    out.transformMaskMappingLocalToWorld[36][166] = 201;
    out.transformMaskMappingLocalToWorld[36][167] = 217;
    out.transformMaskMappingLocalToWorld[36][168] = 140;
    out.transformMaskMappingLocalToWorld[36][169] = 156;
    out.transformMaskMappingLocalToWorld[36][170] = 204;
    out.transformMaskMappingLocalToWorld[36][171] = 220;
    out.transformMaskMappingLocalToWorld[36][172] = 141;
    out.transformMaskMappingLocalToWorld[36][173] = 157;
    out.transformMaskMappingLocalToWorld[36][174] = 205;
    out.transformMaskMappingLocalToWorld[36][175] = 221;
    out.transformMaskMappingLocalToWorld[36][176] = 168;
    out.transformMaskMappingLocalToWorld[36][177] = 184;
    out.transformMaskMappingLocalToWorld[36][178] = 232;
    out.transformMaskMappingLocalToWorld[36][179] = 248;
    out.transformMaskMappingLocalToWorld[36][180] = 169;
    out.transformMaskMappingLocalToWorld[36][181] = 185;
    out.transformMaskMappingLocalToWorld[36][182] = 233;
    out.transformMaskMappingLocalToWorld[36][183] = 249;
    out.transformMaskMappingLocalToWorld[36][184] = 172;
    out.transformMaskMappingLocalToWorld[36][185] = 188;
    out.transformMaskMappingLocalToWorld[36][186] = 236;
    out.transformMaskMappingLocalToWorld[36][187] = 252;
    out.transformMaskMappingLocalToWorld[36][188] = 173;
    out.transformMaskMappingLocalToWorld[36][189] = 189;
    out.transformMaskMappingLocalToWorld[36][190] = 237;
    out.transformMaskMappingLocalToWorld[36][191] = 253;
    out.transformMaskMappingLocalToWorld[36][192] = 10;
    out.transformMaskMappingLocalToWorld[36][193] = 26;
    out.transformMaskMappingLocalToWorld[36][194] = 74;
    out.transformMaskMappingLocalToWorld[36][195] = 90;
    out.transformMaskMappingLocalToWorld[36][196] = 11;
    out.transformMaskMappingLocalToWorld[36][197] = 27;
    out.transformMaskMappingLocalToWorld[36][198] = 75;
    out.transformMaskMappingLocalToWorld[36][199] = 91;
    out.transformMaskMappingLocalToWorld[36][200] = 14;
    out.transformMaskMappingLocalToWorld[36][201] = 30;
    out.transformMaskMappingLocalToWorld[36][202] = 78;
    out.transformMaskMappingLocalToWorld[36][203] = 94;
    out.transformMaskMappingLocalToWorld[36][204] = 15;
    out.transformMaskMappingLocalToWorld[36][205] = 31;
    out.transformMaskMappingLocalToWorld[36][206] = 79;
    out.transformMaskMappingLocalToWorld[36][207] = 95;
    out.transformMaskMappingLocalToWorld[36][208] = 42;
    out.transformMaskMappingLocalToWorld[36][209] = 58;
    out.transformMaskMappingLocalToWorld[36][210] = 106;
    out.transformMaskMappingLocalToWorld[36][211] = 122;
    out.transformMaskMappingLocalToWorld[36][212] = 43;
    out.transformMaskMappingLocalToWorld[36][213] = 59;
    out.transformMaskMappingLocalToWorld[36][214] = 107;
    out.transformMaskMappingLocalToWorld[36][215] = 123;
    out.transformMaskMappingLocalToWorld[36][216] = 46;
    out.transformMaskMappingLocalToWorld[36][217] = 62;
    out.transformMaskMappingLocalToWorld[36][218] = 110;
    out.transformMaskMappingLocalToWorld[36][219] = 126;
    out.transformMaskMappingLocalToWorld[36][220] = 47;
    out.transformMaskMappingLocalToWorld[36][221] = 63;
    out.transformMaskMappingLocalToWorld[36][222] = 111;
    out.transformMaskMappingLocalToWorld[36][223] = 127;
    out.transformMaskMappingLocalToWorld[36][224] = 138;
    out.transformMaskMappingLocalToWorld[36][225] = 154;
    out.transformMaskMappingLocalToWorld[36][226] = 202;
    out.transformMaskMappingLocalToWorld[36][227] = 218;
    out.transformMaskMappingLocalToWorld[36][228] = 139;
    out.transformMaskMappingLocalToWorld[36][229] = 155;
    out.transformMaskMappingLocalToWorld[36][230] = 203;
    out.transformMaskMappingLocalToWorld[36][231] = 219;
    out.transformMaskMappingLocalToWorld[36][232] = 142;
    out.transformMaskMappingLocalToWorld[36][233] = 158;
    out.transformMaskMappingLocalToWorld[36][234] = 206;
    out.transformMaskMappingLocalToWorld[36][235] = 222;
    out.transformMaskMappingLocalToWorld[36][236] = 143;
    out.transformMaskMappingLocalToWorld[36][237] = 159;
    out.transformMaskMappingLocalToWorld[36][238] = 207;
    out.transformMaskMappingLocalToWorld[36][239] = 223;
    out.transformMaskMappingLocalToWorld[36][240] = 170;
    out.transformMaskMappingLocalToWorld[36][241] = 186;
    out.transformMaskMappingLocalToWorld[36][242] = 234;
    out.transformMaskMappingLocalToWorld[36][243] = 250;
    out.transformMaskMappingLocalToWorld[36][244] = 171;
    out.transformMaskMappingLocalToWorld[36][245] = 187;
    out.transformMaskMappingLocalToWorld[36][246] = 235;
    out.transformMaskMappingLocalToWorld[36][247] = 251;
    out.transformMaskMappingLocalToWorld[36][248] = 174;
    out.transformMaskMappingLocalToWorld[36][249] = 190;
    out.transformMaskMappingLocalToWorld[36][250] = 238;
    out.transformMaskMappingLocalToWorld[36][251] = 254;
    out.transformMaskMappingLocalToWorld[36][252] = 175;
    out.transformMaskMappingLocalToWorld[36][253] = 191;
    out.transformMaskMappingLocalToWorld[36][254] = 239;
    out.transformMaskMappingLocalToWorld[36][255] = 255;
    out.transformMaskMappingLocalToWorld[37][0] = 0;
    out.transformMaskMappingLocalToWorld[37][1] = 32;
    out.transformMaskMappingLocalToWorld[37][2] = 128;
    out.transformMaskMappingLocalToWorld[37][3] = 160;
    out.transformMaskMappingLocalToWorld[37][4] = 2;
    out.transformMaskMappingLocalToWorld[37][5] = 34;
    out.transformMaskMappingLocalToWorld[37][6] = 130;
    out.transformMaskMappingLocalToWorld[37][7] = 162;
    out.transformMaskMappingLocalToWorld[37][8] = 8;
    out.transformMaskMappingLocalToWorld[37][9] = 40;
    out.transformMaskMappingLocalToWorld[37][10] = 136;
    out.transformMaskMappingLocalToWorld[37][11] = 168;
    out.transformMaskMappingLocalToWorld[37][12] = 10;
    out.transformMaskMappingLocalToWorld[37][13] = 42;
    out.transformMaskMappingLocalToWorld[37][14] = 138;
    out.transformMaskMappingLocalToWorld[37][15] = 170;
    out.transformMaskMappingLocalToWorld[37][16] = 16;
    out.transformMaskMappingLocalToWorld[37][17] = 48;
    out.transformMaskMappingLocalToWorld[37][18] = 144;
    out.transformMaskMappingLocalToWorld[37][19] = 176;
    out.transformMaskMappingLocalToWorld[37][20] = 18;
    out.transformMaskMappingLocalToWorld[37][21] = 50;
    out.transformMaskMappingLocalToWorld[37][22] = 146;
    out.transformMaskMappingLocalToWorld[37][23] = 178;
    out.transformMaskMappingLocalToWorld[37][24] = 24;
    out.transformMaskMappingLocalToWorld[37][25] = 56;
    out.transformMaskMappingLocalToWorld[37][26] = 152;
    out.transformMaskMappingLocalToWorld[37][27] = 184;
    out.transformMaskMappingLocalToWorld[37][28] = 26;
    out.transformMaskMappingLocalToWorld[37][29] = 58;
    out.transformMaskMappingLocalToWorld[37][30] = 154;
    out.transformMaskMappingLocalToWorld[37][31] = 186;
    out.transformMaskMappingLocalToWorld[37][32] = 64;
    out.transformMaskMappingLocalToWorld[37][33] = 96;
    out.transformMaskMappingLocalToWorld[37][34] = 192;
    out.transformMaskMappingLocalToWorld[37][35] = 224;
    out.transformMaskMappingLocalToWorld[37][36] = 66;
    out.transformMaskMappingLocalToWorld[37][37] = 98;
    out.transformMaskMappingLocalToWorld[37][38] = 194;
    out.transformMaskMappingLocalToWorld[37][39] = 226;
    out.transformMaskMappingLocalToWorld[37][40] = 72;
    out.transformMaskMappingLocalToWorld[37][41] = 104;
    out.transformMaskMappingLocalToWorld[37][42] = 200;
    out.transformMaskMappingLocalToWorld[37][43] = 232;
    out.transformMaskMappingLocalToWorld[37][44] = 74;
    out.transformMaskMappingLocalToWorld[37][45] = 106;
    out.transformMaskMappingLocalToWorld[37][46] = 202;
    out.transformMaskMappingLocalToWorld[37][47] = 234;
    out.transformMaskMappingLocalToWorld[37][48] = 80;
    out.transformMaskMappingLocalToWorld[37][49] = 112;
    out.transformMaskMappingLocalToWorld[37][50] = 208;
    out.transformMaskMappingLocalToWorld[37][51] = 240;
    out.transformMaskMappingLocalToWorld[37][52] = 82;
    out.transformMaskMappingLocalToWorld[37][53] = 114;
    out.transformMaskMappingLocalToWorld[37][54] = 210;
    out.transformMaskMappingLocalToWorld[37][55] = 242;
    out.transformMaskMappingLocalToWorld[37][56] = 88;
    out.transformMaskMappingLocalToWorld[37][57] = 120;
    out.transformMaskMappingLocalToWorld[37][58] = 216;
    out.transformMaskMappingLocalToWorld[37][59] = 248;
    out.transformMaskMappingLocalToWorld[37][60] = 90;
    out.transformMaskMappingLocalToWorld[37][61] = 122;
    out.transformMaskMappingLocalToWorld[37][62] = 218;
    out.transformMaskMappingLocalToWorld[37][63] = 250;
    out.transformMaskMappingLocalToWorld[37][64] = 1;
    out.transformMaskMappingLocalToWorld[37][65] = 33;
    out.transformMaskMappingLocalToWorld[37][66] = 129;
    out.transformMaskMappingLocalToWorld[37][67] = 161;
    out.transformMaskMappingLocalToWorld[37][68] = 3;
    out.transformMaskMappingLocalToWorld[37][69] = 35;
    out.transformMaskMappingLocalToWorld[37][70] = 131;
    out.transformMaskMappingLocalToWorld[37][71] = 163;
    out.transformMaskMappingLocalToWorld[37][72] = 9;
    out.transformMaskMappingLocalToWorld[37][73] = 41;
    out.transformMaskMappingLocalToWorld[37][74] = 137;
    out.transformMaskMappingLocalToWorld[37][75] = 169;
    out.transformMaskMappingLocalToWorld[37][76] = 11;
    out.transformMaskMappingLocalToWorld[37][77] = 43;
    out.transformMaskMappingLocalToWorld[37][78] = 139;
    out.transformMaskMappingLocalToWorld[37][79] = 171;
    out.transformMaskMappingLocalToWorld[37][80] = 17;
    out.transformMaskMappingLocalToWorld[37][81] = 49;
    out.transformMaskMappingLocalToWorld[37][82] = 145;
    out.transformMaskMappingLocalToWorld[37][83] = 177;
    out.transformMaskMappingLocalToWorld[37][84] = 19;
    out.transformMaskMappingLocalToWorld[37][85] = 51;
    out.transformMaskMappingLocalToWorld[37][86] = 147;
    out.transformMaskMappingLocalToWorld[37][87] = 179;
    out.transformMaskMappingLocalToWorld[37][88] = 25;
    out.transformMaskMappingLocalToWorld[37][89] = 57;
    out.transformMaskMappingLocalToWorld[37][90] = 153;
    out.transformMaskMappingLocalToWorld[37][91] = 185;
    out.transformMaskMappingLocalToWorld[37][92] = 27;
    out.transformMaskMappingLocalToWorld[37][93] = 59;
    out.transformMaskMappingLocalToWorld[37][94] = 155;
    out.transformMaskMappingLocalToWorld[37][95] = 187;
    out.transformMaskMappingLocalToWorld[37][96] = 65;
    out.transformMaskMappingLocalToWorld[37][97] = 97;
    out.transformMaskMappingLocalToWorld[37][98] = 193;
    out.transformMaskMappingLocalToWorld[37][99] = 225;
    out.transformMaskMappingLocalToWorld[37][100] = 67;
    out.transformMaskMappingLocalToWorld[37][101] = 99;
    out.transformMaskMappingLocalToWorld[37][102] = 195;
    out.transformMaskMappingLocalToWorld[37][103] = 227;
    out.transformMaskMappingLocalToWorld[37][104] = 73;
    out.transformMaskMappingLocalToWorld[37][105] = 105;
    out.transformMaskMappingLocalToWorld[37][106] = 201;
    out.transformMaskMappingLocalToWorld[37][107] = 233;
    out.transformMaskMappingLocalToWorld[37][108] = 75;
    out.transformMaskMappingLocalToWorld[37][109] = 107;
    out.transformMaskMappingLocalToWorld[37][110] = 203;
    out.transformMaskMappingLocalToWorld[37][111] = 235;
    out.transformMaskMappingLocalToWorld[37][112] = 81;
    out.transformMaskMappingLocalToWorld[37][113] = 113;
    out.transformMaskMappingLocalToWorld[37][114] = 209;
    out.transformMaskMappingLocalToWorld[37][115] = 241;
    out.transformMaskMappingLocalToWorld[37][116] = 83;
    out.transformMaskMappingLocalToWorld[37][117] = 115;
    out.transformMaskMappingLocalToWorld[37][118] = 211;
    out.transformMaskMappingLocalToWorld[37][119] = 243;
    out.transformMaskMappingLocalToWorld[37][120] = 89;
    out.transformMaskMappingLocalToWorld[37][121] = 121;
    out.transformMaskMappingLocalToWorld[37][122] = 217;
    out.transformMaskMappingLocalToWorld[37][123] = 249;
    out.transformMaskMappingLocalToWorld[37][124] = 91;
    out.transformMaskMappingLocalToWorld[37][125] = 123;
    out.transformMaskMappingLocalToWorld[37][126] = 219;
    out.transformMaskMappingLocalToWorld[37][127] = 251;
    out.transformMaskMappingLocalToWorld[37][128] = 4;
    out.transformMaskMappingLocalToWorld[37][129] = 36;
    out.transformMaskMappingLocalToWorld[37][130] = 132;
    out.transformMaskMappingLocalToWorld[37][131] = 164;
    out.transformMaskMappingLocalToWorld[37][132] = 6;
    out.transformMaskMappingLocalToWorld[37][133] = 38;
    out.transformMaskMappingLocalToWorld[37][134] = 134;
    out.transformMaskMappingLocalToWorld[37][135] = 166;
    out.transformMaskMappingLocalToWorld[37][136] = 12;
    out.transformMaskMappingLocalToWorld[37][137] = 44;
    out.transformMaskMappingLocalToWorld[37][138] = 140;
    out.transformMaskMappingLocalToWorld[37][139] = 172;
    out.transformMaskMappingLocalToWorld[37][140] = 14;
    out.transformMaskMappingLocalToWorld[37][141] = 46;
    out.transformMaskMappingLocalToWorld[37][142] = 142;
    out.transformMaskMappingLocalToWorld[37][143] = 174;
    out.transformMaskMappingLocalToWorld[37][144] = 20;
    out.transformMaskMappingLocalToWorld[37][145] = 52;
    out.transformMaskMappingLocalToWorld[37][146] = 148;
    out.transformMaskMappingLocalToWorld[37][147] = 180;
    out.transformMaskMappingLocalToWorld[37][148] = 22;
    out.transformMaskMappingLocalToWorld[37][149] = 54;
    out.transformMaskMappingLocalToWorld[37][150] = 150;
    out.transformMaskMappingLocalToWorld[37][151] = 182;
    out.transformMaskMappingLocalToWorld[37][152] = 28;
    out.transformMaskMappingLocalToWorld[37][153] = 60;
    out.transformMaskMappingLocalToWorld[37][154] = 156;
    out.transformMaskMappingLocalToWorld[37][155] = 188;
    out.transformMaskMappingLocalToWorld[37][156] = 30;
    out.transformMaskMappingLocalToWorld[37][157] = 62;
    out.transformMaskMappingLocalToWorld[37][158] = 158;
    out.transformMaskMappingLocalToWorld[37][159] = 190;
    out.transformMaskMappingLocalToWorld[37][160] = 68;
    out.transformMaskMappingLocalToWorld[37][161] = 100;
    out.transformMaskMappingLocalToWorld[37][162] = 196;
    out.transformMaskMappingLocalToWorld[37][163] = 228;
    out.transformMaskMappingLocalToWorld[37][164] = 70;
    out.transformMaskMappingLocalToWorld[37][165] = 102;
    out.transformMaskMappingLocalToWorld[37][166] = 198;
    out.transformMaskMappingLocalToWorld[37][167] = 230;
    out.transformMaskMappingLocalToWorld[37][168] = 76;
    out.transformMaskMappingLocalToWorld[37][169] = 108;
    out.transformMaskMappingLocalToWorld[37][170] = 204;
    out.transformMaskMappingLocalToWorld[37][171] = 236;
    out.transformMaskMappingLocalToWorld[37][172] = 78;
    out.transformMaskMappingLocalToWorld[37][173] = 110;
    out.transformMaskMappingLocalToWorld[37][174] = 206;
    out.transformMaskMappingLocalToWorld[37][175] = 238;
    out.transformMaskMappingLocalToWorld[37][176] = 84;
    out.transformMaskMappingLocalToWorld[37][177] = 116;
    out.transformMaskMappingLocalToWorld[37][178] = 212;
    out.transformMaskMappingLocalToWorld[37][179] = 244;
    out.transformMaskMappingLocalToWorld[37][180] = 86;
    out.transformMaskMappingLocalToWorld[37][181] = 118;
    out.transformMaskMappingLocalToWorld[37][182] = 214;
    out.transformMaskMappingLocalToWorld[37][183] = 246;
    out.transformMaskMappingLocalToWorld[37][184] = 92;
    out.transformMaskMappingLocalToWorld[37][185] = 124;
    out.transformMaskMappingLocalToWorld[37][186] = 220;
    out.transformMaskMappingLocalToWorld[37][187] = 252;
    out.transformMaskMappingLocalToWorld[37][188] = 94;
    out.transformMaskMappingLocalToWorld[37][189] = 126;
    out.transformMaskMappingLocalToWorld[37][190] = 222;
    out.transformMaskMappingLocalToWorld[37][191] = 254;
    out.transformMaskMappingLocalToWorld[37][192] = 5;
    out.transformMaskMappingLocalToWorld[37][193] = 37;
    out.transformMaskMappingLocalToWorld[37][194] = 133;
    out.transformMaskMappingLocalToWorld[37][195] = 165;
    out.transformMaskMappingLocalToWorld[37][196] = 7;
    out.transformMaskMappingLocalToWorld[37][197] = 39;
    out.transformMaskMappingLocalToWorld[37][198] = 135;
    out.transformMaskMappingLocalToWorld[37][199] = 167;
    out.transformMaskMappingLocalToWorld[37][200] = 13;
    out.transformMaskMappingLocalToWorld[37][201] = 45;
    out.transformMaskMappingLocalToWorld[37][202] = 141;
    out.transformMaskMappingLocalToWorld[37][203] = 173;
    out.transformMaskMappingLocalToWorld[37][204] = 15;
    out.transformMaskMappingLocalToWorld[37][205] = 47;
    out.transformMaskMappingLocalToWorld[37][206] = 143;
    out.transformMaskMappingLocalToWorld[37][207] = 175;
    out.transformMaskMappingLocalToWorld[37][208] = 21;
    out.transformMaskMappingLocalToWorld[37][209] = 53;
    out.transformMaskMappingLocalToWorld[37][210] = 149;
    out.transformMaskMappingLocalToWorld[37][211] = 181;
    out.transformMaskMappingLocalToWorld[37][212] = 23;
    out.transformMaskMappingLocalToWorld[37][213] = 55;
    out.transformMaskMappingLocalToWorld[37][214] = 151;
    out.transformMaskMappingLocalToWorld[37][215] = 183;
    out.transformMaskMappingLocalToWorld[37][216] = 29;
    out.transformMaskMappingLocalToWorld[37][217] = 61;
    out.transformMaskMappingLocalToWorld[37][218] = 157;
    out.transformMaskMappingLocalToWorld[37][219] = 189;
    out.transformMaskMappingLocalToWorld[37][220] = 31;
    out.transformMaskMappingLocalToWorld[37][221] = 63;
    out.transformMaskMappingLocalToWorld[37][222] = 159;
    out.transformMaskMappingLocalToWorld[37][223] = 191;
    out.transformMaskMappingLocalToWorld[37][224] = 69;
    out.transformMaskMappingLocalToWorld[37][225] = 101;
    out.transformMaskMappingLocalToWorld[37][226] = 197;
    out.transformMaskMappingLocalToWorld[37][227] = 229;
    out.transformMaskMappingLocalToWorld[37][228] = 71;
    out.transformMaskMappingLocalToWorld[37][229] = 103;
    out.transformMaskMappingLocalToWorld[37][230] = 199;
    out.transformMaskMappingLocalToWorld[37][231] = 231;
    out.transformMaskMappingLocalToWorld[37][232] = 77;
    out.transformMaskMappingLocalToWorld[37][233] = 109;
    out.transformMaskMappingLocalToWorld[37][234] = 205;
    out.transformMaskMappingLocalToWorld[37][235] = 237;
    out.transformMaskMappingLocalToWorld[37][236] = 79;
    out.transformMaskMappingLocalToWorld[37][237] = 111;
    out.transformMaskMappingLocalToWorld[37][238] = 207;
    out.transformMaskMappingLocalToWorld[37][239] = 239;
    out.transformMaskMappingLocalToWorld[37][240] = 85;
    out.transformMaskMappingLocalToWorld[37][241] = 117;
    out.transformMaskMappingLocalToWorld[37][242] = 213;
    out.transformMaskMappingLocalToWorld[37][243] = 245;
    out.transformMaskMappingLocalToWorld[37][244] = 87;
    out.transformMaskMappingLocalToWorld[37][245] = 119;
    out.transformMaskMappingLocalToWorld[37][246] = 215;
    out.transformMaskMappingLocalToWorld[37][247] = 247;
    out.transformMaskMappingLocalToWorld[37][248] = 93;
    out.transformMaskMappingLocalToWorld[37][249] = 125;
    out.transformMaskMappingLocalToWorld[37][250] = 221;
    out.transformMaskMappingLocalToWorld[37][251] = 253;
    out.transformMaskMappingLocalToWorld[37][252] = 95;
    out.transformMaskMappingLocalToWorld[37][253] = 127;
    out.transformMaskMappingLocalToWorld[37][254] = 223;
    out.transformMaskMappingLocalToWorld[37][255] = 255;
    out.transformMaskMappingLocalToWorld[38][0] = 0;
    out.transformMaskMappingLocalToWorld[38][1] = 64;
    out.transformMaskMappingLocalToWorld[38][2] = 16;
    out.transformMaskMappingLocalToWorld[38][3] = 80;
    out.transformMaskMappingLocalToWorld[38][4] = 4;
    out.transformMaskMappingLocalToWorld[38][5] = 68;
    out.transformMaskMappingLocalToWorld[38][6] = 20;
    out.transformMaskMappingLocalToWorld[38][7] = 84;
    out.transformMaskMappingLocalToWorld[38][8] = 1;
    out.transformMaskMappingLocalToWorld[38][9] = 65;
    out.transformMaskMappingLocalToWorld[38][10] = 17;
    out.transformMaskMappingLocalToWorld[38][11] = 81;
    out.transformMaskMappingLocalToWorld[38][12] = 5;
    out.transformMaskMappingLocalToWorld[38][13] = 69;
    out.transformMaskMappingLocalToWorld[38][14] = 21;
    out.transformMaskMappingLocalToWorld[38][15] = 85;
    out.transformMaskMappingLocalToWorld[38][16] = 128;
    out.transformMaskMappingLocalToWorld[38][17] = 192;
    out.transformMaskMappingLocalToWorld[38][18] = 144;
    out.transformMaskMappingLocalToWorld[38][19] = 208;
    out.transformMaskMappingLocalToWorld[38][20] = 132;
    out.transformMaskMappingLocalToWorld[38][21] = 196;
    out.transformMaskMappingLocalToWorld[38][22] = 148;
    out.transformMaskMappingLocalToWorld[38][23] = 212;
    out.transformMaskMappingLocalToWorld[38][24] = 129;
    out.transformMaskMappingLocalToWorld[38][25] = 193;
    out.transformMaskMappingLocalToWorld[38][26] = 145;
    out.transformMaskMappingLocalToWorld[38][27] = 209;
    out.transformMaskMappingLocalToWorld[38][28] = 133;
    out.transformMaskMappingLocalToWorld[38][29] = 197;
    out.transformMaskMappingLocalToWorld[38][30] = 149;
    out.transformMaskMappingLocalToWorld[38][31] = 213;
    out.transformMaskMappingLocalToWorld[38][32] = 32;
    out.transformMaskMappingLocalToWorld[38][33] = 96;
    out.transformMaskMappingLocalToWorld[38][34] = 48;
    out.transformMaskMappingLocalToWorld[38][35] = 112;
    out.transformMaskMappingLocalToWorld[38][36] = 36;
    out.transformMaskMappingLocalToWorld[38][37] = 100;
    out.transformMaskMappingLocalToWorld[38][38] = 52;
    out.transformMaskMappingLocalToWorld[38][39] = 116;
    out.transformMaskMappingLocalToWorld[38][40] = 33;
    out.transformMaskMappingLocalToWorld[38][41] = 97;
    out.transformMaskMappingLocalToWorld[38][42] = 49;
    out.transformMaskMappingLocalToWorld[38][43] = 113;
    out.transformMaskMappingLocalToWorld[38][44] = 37;
    out.transformMaskMappingLocalToWorld[38][45] = 101;
    out.transformMaskMappingLocalToWorld[38][46] = 53;
    out.transformMaskMappingLocalToWorld[38][47] = 117;
    out.transformMaskMappingLocalToWorld[38][48] = 160;
    out.transformMaskMappingLocalToWorld[38][49] = 224;
    out.transformMaskMappingLocalToWorld[38][50] = 176;
    out.transformMaskMappingLocalToWorld[38][51] = 240;
    out.transformMaskMappingLocalToWorld[38][52] = 164;
    out.transformMaskMappingLocalToWorld[38][53] = 228;
    out.transformMaskMappingLocalToWorld[38][54] = 180;
    out.transformMaskMappingLocalToWorld[38][55] = 244;
    out.transformMaskMappingLocalToWorld[38][56] = 161;
    out.transformMaskMappingLocalToWorld[38][57] = 225;
    out.transformMaskMappingLocalToWorld[38][58] = 177;
    out.transformMaskMappingLocalToWorld[38][59] = 241;
    out.transformMaskMappingLocalToWorld[38][60] = 165;
    out.transformMaskMappingLocalToWorld[38][61] = 229;
    out.transformMaskMappingLocalToWorld[38][62] = 181;
    out.transformMaskMappingLocalToWorld[38][63] = 245;
    out.transformMaskMappingLocalToWorld[38][64] = 8;
    out.transformMaskMappingLocalToWorld[38][65] = 72;
    out.transformMaskMappingLocalToWorld[38][66] = 24;
    out.transformMaskMappingLocalToWorld[38][67] = 88;
    out.transformMaskMappingLocalToWorld[38][68] = 12;
    out.transformMaskMappingLocalToWorld[38][69] = 76;
    out.transformMaskMappingLocalToWorld[38][70] = 28;
    out.transformMaskMappingLocalToWorld[38][71] = 92;
    out.transformMaskMappingLocalToWorld[38][72] = 9;
    out.transformMaskMappingLocalToWorld[38][73] = 73;
    out.transformMaskMappingLocalToWorld[38][74] = 25;
    out.transformMaskMappingLocalToWorld[38][75] = 89;
    out.transformMaskMappingLocalToWorld[38][76] = 13;
    out.transformMaskMappingLocalToWorld[38][77] = 77;
    out.transformMaskMappingLocalToWorld[38][78] = 29;
    out.transformMaskMappingLocalToWorld[38][79] = 93;
    out.transformMaskMappingLocalToWorld[38][80] = 136;
    out.transformMaskMappingLocalToWorld[38][81] = 200;
    out.transformMaskMappingLocalToWorld[38][82] = 152;
    out.transformMaskMappingLocalToWorld[38][83] = 216;
    out.transformMaskMappingLocalToWorld[38][84] = 140;
    out.transformMaskMappingLocalToWorld[38][85] = 204;
    out.transformMaskMappingLocalToWorld[38][86] = 156;
    out.transformMaskMappingLocalToWorld[38][87] = 220;
    out.transformMaskMappingLocalToWorld[38][88] = 137;
    out.transformMaskMappingLocalToWorld[38][89] = 201;
    out.transformMaskMappingLocalToWorld[38][90] = 153;
    out.transformMaskMappingLocalToWorld[38][91] = 217;
    out.transformMaskMappingLocalToWorld[38][92] = 141;
    out.transformMaskMappingLocalToWorld[38][93] = 205;
    out.transformMaskMappingLocalToWorld[38][94] = 157;
    out.transformMaskMappingLocalToWorld[38][95] = 221;
    out.transformMaskMappingLocalToWorld[38][96] = 40;
    out.transformMaskMappingLocalToWorld[38][97] = 104;
    out.transformMaskMappingLocalToWorld[38][98] = 56;
    out.transformMaskMappingLocalToWorld[38][99] = 120;
    out.transformMaskMappingLocalToWorld[38][100] = 44;
    out.transformMaskMappingLocalToWorld[38][101] = 108;
    out.transformMaskMappingLocalToWorld[38][102] = 60;
    out.transformMaskMappingLocalToWorld[38][103] = 124;
    out.transformMaskMappingLocalToWorld[38][104] = 41;
    out.transformMaskMappingLocalToWorld[38][105] = 105;
    out.transformMaskMappingLocalToWorld[38][106] = 57;
    out.transformMaskMappingLocalToWorld[38][107] = 121;
    out.transformMaskMappingLocalToWorld[38][108] = 45;
    out.transformMaskMappingLocalToWorld[38][109] = 109;
    out.transformMaskMappingLocalToWorld[38][110] = 61;
    out.transformMaskMappingLocalToWorld[38][111] = 125;
    out.transformMaskMappingLocalToWorld[38][112] = 168;
    out.transformMaskMappingLocalToWorld[38][113] = 232;
    out.transformMaskMappingLocalToWorld[38][114] = 184;
    out.transformMaskMappingLocalToWorld[38][115] = 248;
    out.transformMaskMappingLocalToWorld[38][116] = 172;
    out.transformMaskMappingLocalToWorld[38][117] = 236;
    out.transformMaskMappingLocalToWorld[38][118] = 188;
    out.transformMaskMappingLocalToWorld[38][119] = 252;
    out.transformMaskMappingLocalToWorld[38][120] = 169;
    out.transformMaskMappingLocalToWorld[38][121] = 233;
    out.transformMaskMappingLocalToWorld[38][122] = 185;
    out.transformMaskMappingLocalToWorld[38][123] = 249;
    out.transformMaskMappingLocalToWorld[38][124] = 173;
    out.transformMaskMappingLocalToWorld[38][125] = 237;
    out.transformMaskMappingLocalToWorld[38][126] = 189;
    out.transformMaskMappingLocalToWorld[38][127] = 253;
    out.transformMaskMappingLocalToWorld[38][128] = 2;
    out.transformMaskMappingLocalToWorld[38][129] = 66;
    out.transformMaskMappingLocalToWorld[38][130] = 18;
    out.transformMaskMappingLocalToWorld[38][131] = 82;
    out.transformMaskMappingLocalToWorld[38][132] = 6;
    out.transformMaskMappingLocalToWorld[38][133] = 70;
    out.transformMaskMappingLocalToWorld[38][134] = 22;
    out.transformMaskMappingLocalToWorld[38][135] = 86;
    out.transformMaskMappingLocalToWorld[38][136] = 3;
    out.transformMaskMappingLocalToWorld[38][137] = 67;
    out.transformMaskMappingLocalToWorld[38][138] = 19;
    out.transformMaskMappingLocalToWorld[38][139] = 83;
    out.transformMaskMappingLocalToWorld[38][140] = 7;
    out.transformMaskMappingLocalToWorld[38][141] = 71;
    out.transformMaskMappingLocalToWorld[38][142] = 23;
    out.transformMaskMappingLocalToWorld[38][143] = 87;
    out.transformMaskMappingLocalToWorld[38][144] = 130;
    out.transformMaskMappingLocalToWorld[38][145] = 194;
    out.transformMaskMappingLocalToWorld[38][146] = 146;
    out.transformMaskMappingLocalToWorld[38][147] = 210;
    out.transformMaskMappingLocalToWorld[38][148] = 134;
    out.transformMaskMappingLocalToWorld[38][149] = 198;
    out.transformMaskMappingLocalToWorld[38][150] = 150;
    out.transformMaskMappingLocalToWorld[38][151] = 214;
    out.transformMaskMappingLocalToWorld[38][152] = 131;
    out.transformMaskMappingLocalToWorld[38][153] = 195;
    out.transformMaskMappingLocalToWorld[38][154] = 147;
    out.transformMaskMappingLocalToWorld[38][155] = 211;
    out.transformMaskMappingLocalToWorld[38][156] = 135;
    out.transformMaskMappingLocalToWorld[38][157] = 199;
    out.transformMaskMappingLocalToWorld[38][158] = 151;
    out.transformMaskMappingLocalToWorld[38][159] = 215;
    out.transformMaskMappingLocalToWorld[38][160] = 34;
    out.transformMaskMappingLocalToWorld[38][161] = 98;
    out.transformMaskMappingLocalToWorld[38][162] = 50;
    out.transformMaskMappingLocalToWorld[38][163] = 114;
    out.transformMaskMappingLocalToWorld[38][164] = 38;
    out.transformMaskMappingLocalToWorld[38][165] = 102;
    out.transformMaskMappingLocalToWorld[38][166] = 54;
    out.transformMaskMappingLocalToWorld[38][167] = 118;
    out.transformMaskMappingLocalToWorld[38][168] = 35;
    out.transformMaskMappingLocalToWorld[38][169] = 99;
    out.transformMaskMappingLocalToWorld[38][170] = 51;
    out.transformMaskMappingLocalToWorld[38][171] = 115;
    out.transformMaskMappingLocalToWorld[38][172] = 39;
    out.transformMaskMappingLocalToWorld[38][173] = 103;
    out.transformMaskMappingLocalToWorld[38][174] = 55;
    out.transformMaskMappingLocalToWorld[38][175] = 119;
    out.transformMaskMappingLocalToWorld[38][176] = 162;
    out.transformMaskMappingLocalToWorld[38][177] = 226;
    out.transformMaskMappingLocalToWorld[38][178] = 178;
    out.transformMaskMappingLocalToWorld[38][179] = 242;
    out.transformMaskMappingLocalToWorld[38][180] = 166;
    out.transformMaskMappingLocalToWorld[38][181] = 230;
    out.transformMaskMappingLocalToWorld[38][182] = 182;
    out.transformMaskMappingLocalToWorld[38][183] = 246;
    out.transformMaskMappingLocalToWorld[38][184] = 163;
    out.transformMaskMappingLocalToWorld[38][185] = 227;
    out.transformMaskMappingLocalToWorld[38][186] = 179;
    out.transformMaskMappingLocalToWorld[38][187] = 243;
    out.transformMaskMappingLocalToWorld[38][188] = 167;
    out.transformMaskMappingLocalToWorld[38][189] = 231;
    out.transformMaskMappingLocalToWorld[38][190] = 183;
    out.transformMaskMappingLocalToWorld[38][191] = 247;
    out.transformMaskMappingLocalToWorld[38][192] = 10;
    out.transformMaskMappingLocalToWorld[38][193] = 74;
    out.transformMaskMappingLocalToWorld[38][194] = 26;
    out.transformMaskMappingLocalToWorld[38][195] = 90;
    out.transformMaskMappingLocalToWorld[38][196] = 14;
    out.transformMaskMappingLocalToWorld[38][197] = 78;
    out.transformMaskMappingLocalToWorld[38][198] = 30;
    out.transformMaskMappingLocalToWorld[38][199] = 94;
    out.transformMaskMappingLocalToWorld[38][200] = 11;
    out.transformMaskMappingLocalToWorld[38][201] = 75;
    out.transformMaskMappingLocalToWorld[38][202] = 27;
    out.transformMaskMappingLocalToWorld[38][203] = 91;
    out.transformMaskMappingLocalToWorld[38][204] = 15;
    out.transformMaskMappingLocalToWorld[38][205] = 79;
    out.transformMaskMappingLocalToWorld[38][206] = 31;
    out.transformMaskMappingLocalToWorld[38][207] = 95;
    out.transformMaskMappingLocalToWorld[38][208] = 138;
    out.transformMaskMappingLocalToWorld[38][209] = 202;
    out.transformMaskMappingLocalToWorld[38][210] = 154;
    out.transformMaskMappingLocalToWorld[38][211] = 218;
    out.transformMaskMappingLocalToWorld[38][212] = 142;
    out.transformMaskMappingLocalToWorld[38][213] = 206;
    out.transformMaskMappingLocalToWorld[38][214] = 158;
    out.transformMaskMappingLocalToWorld[38][215] = 222;
    out.transformMaskMappingLocalToWorld[38][216] = 139;
    out.transformMaskMappingLocalToWorld[38][217] = 203;
    out.transformMaskMappingLocalToWorld[38][218] = 155;
    out.transformMaskMappingLocalToWorld[38][219] = 219;
    out.transformMaskMappingLocalToWorld[38][220] = 143;
    out.transformMaskMappingLocalToWorld[38][221] = 207;
    out.transformMaskMappingLocalToWorld[38][222] = 159;
    out.transformMaskMappingLocalToWorld[38][223] = 223;
    out.transformMaskMappingLocalToWorld[38][224] = 42;
    out.transformMaskMappingLocalToWorld[38][225] = 106;
    out.transformMaskMappingLocalToWorld[38][226] = 58;
    out.transformMaskMappingLocalToWorld[38][227] = 122;
    out.transformMaskMappingLocalToWorld[38][228] = 46;
    out.transformMaskMappingLocalToWorld[38][229] = 110;
    out.transformMaskMappingLocalToWorld[38][230] = 62;
    out.transformMaskMappingLocalToWorld[38][231] = 126;
    out.transformMaskMappingLocalToWorld[38][232] = 43;
    out.transformMaskMappingLocalToWorld[38][233] = 107;
    out.transformMaskMappingLocalToWorld[38][234] = 59;
    out.transformMaskMappingLocalToWorld[38][235] = 123;
    out.transformMaskMappingLocalToWorld[38][236] = 47;
    out.transformMaskMappingLocalToWorld[38][237] = 111;
    out.transformMaskMappingLocalToWorld[38][238] = 63;
    out.transformMaskMappingLocalToWorld[38][239] = 127;
    out.transformMaskMappingLocalToWorld[38][240] = 170;
    out.transformMaskMappingLocalToWorld[38][241] = 234;
    out.transformMaskMappingLocalToWorld[38][242] = 186;
    out.transformMaskMappingLocalToWorld[38][243] = 250;
    out.transformMaskMappingLocalToWorld[38][244] = 174;
    out.transformMaskMappingLocalToWorld[38][245] = 238;
    out.transformMaskMappingLocalToWorld[38][246] = 190;
    out.transformMaskMappingLocalToWorld[38][247] = 254;
    out.transformMaskMappingLocalToWorld[38][248] = 171;
    out.transformMaskMappingLocalToWorld[38][249] = 235;
    out.transformMaskMappingLocalToWorld[38][250] = 187;
    out.transformMaskMappingLocalToWorld[38][251] = 251;
    out.transformMaskMappingLocalToWorld[38][252] = 175;
    out.transformMaskMappingLocalToWorld[38][253] = 239;
    out.transformMaskMappingLocalToWorld[38][254] = 191;
    out.transformMaskMappingLocalToWorld[38][255] = 255;
    out.transformMaskMappingLocalToWorld[39][0] = 0;
    out.transformMaskMappingLocalToWorld[39][1] = 128;
    out.transformMaskMappingLocalToWorld[39][2] = 32;
    out.transformMaskMappingLocalToWorld[39][3] = 160;
    out.transformMaskMappingLocalToWorld[39][4] = 8;
    out.transformMaskMappingLocalToWorld[39][5] = 136;
    out.transformMaskMappingLocalToWorld[39][6] = 40;
    out.transformMaskMappingLocalToWorld[39][7] = 168;
    out.transformMaskMappingLocalToWorld[39][8] = 2;
    out.transformMaskMappingLocalToWorld[39][9] = 130;
    out.transformMaskMappingLocalToWorld[39][10] = 34;
    out.transformMaskMappingLocalToWorld[39][11] = 162;
    out.transformMaskMappingLocalToWorld[39][12] = 10;
    out.transformMaskMappingLocalToWorld[39][13] = 138;
    out.transformMaskMappingLocalToWorld[39][14] = 42;
    out.transformMaskMappingLocalToWorld[39][15] = 170;
    out.transformMaskMappingLocalToWorld[39][16] = 64;
    out.transformMaskMappingLocalToWorld[39][17] = 192;
    out.transformMaskMappingLocalToWorld[39][18] = 96;
    out.transformMaskMappingLocalToWorld[39][19] = 224;
    out.transformMaskMappingLocalToWorld[39][20] = 72;
    out.transformMaskMappingLocalToWorld[39][21] = 200;
    out.transformMaskMappingLocalToWorld[39][22] = 104;
    out.transformMaskMappingLocalToWorld[39][23] = 232;
    out.transformMaskMappingLocalToWorld[39][24] = 66;
    out.transformMaskMappingLocalToWorld[39][25] = 194;
    out.transformMaskMappingLocalToWorld[39][26] = 98;
    out.transformMaskMappingLocalToWorld[39][27] = 226;
    out.transformMaskMappingLocalToWorld[39][28] = 74;
    out.transformMaskMappingLocalToWorld[39][29] = 202;
    out.transformMaskMappingLocalToWorld[39][30] = 106;
    out.transformMaskMappingLocalToWorld[39][31] = 234;
    out.transformMaskMappingLocalToWorld[39][32] = 16;
    out.transformMaskMappingLocalToWorld[39][33] = 144;
    out.transformMaskMappingLocalToWorld[39][34] = 48;
    out.transformMaskMappingLocalToWorld[39][35] = 176;
    out.transformMaskMappingLocalToWorld[39][36] = 24;
    out.transformMaskMappingLocalToWorld[39][37] = 152;
    out.transformMaskMappingLocalToWorld[39][38] = 56;
    out.transformMaskMappingLocalToWorld[39][39] = 184;
    out.transformMaskMappingLocalToWorld[39][40] = 18;
    out.transformMaskMappingLocalToWorld[39][41] = 146;
    out.transformMaskMappingLocalToWorld[39][42] = 50;
    out.transformMaskMappingLocalToWorld[39][43] = 178;
    out.transformMaskMappingLocalToWorld[39][44] = 26;
    out.transformMaskMappingLocalToWorld[39][45] = 154;
    out.transformMaskMappingLocalToWorld[39][46] = 58;
    out.transformMaskMappingLocalToWorld[39][47] = 186;
    out.transformMaskMappingLocalToWorld[39][48] = 80;
    out.transformMaskMappingLocalToWorld[39][49] = 208;
    out.transformMaskMappingLocalToWorld[39][50] = 112;
    out.transformMaskMappingLocalToWorld[39][51] = 240;
    out.transformMaskMappingLocalToWorld[39][52] = 88;
    out.transformMaskMappingLocalToWorld[39][53] = 216;
    out.transformMaskMappingLocalToWorld[39][54] = 120;
    out.transformMaskMappingLocalToWorld[39][55] = 248;
    out.transformMaskMappingLocalToWorld[39][56] = 82;
    out.transformMaskMappingLocalToWorld[39][57] = 210;
    out.transformMaskMappingLocalToWorld[39][58] = 114;
    out.transformMaskMappingLocalToWorld[39][59] = 242;
    out.transformMaskMappingLocalToWorld[39][60] = 90;
    out.transformMaskMappingLocalToWorld[39][61] = 218;
    out.transformMaskMappingLocalToWorld[39][62] = 122;
    out.transformMaskMappingLocalToWorld[39][63] = 250;
    out.transformMaskMappingLocalToWorld[39][64] = 4;
    out.transformMaskMappingLocalToWorld[39][65] = 132;
    out.transformMaskMappingLocalToWorld[39][66] = 36;
    out.transformMaskMappingLocalToWorld[39][67] = 164;
    out.transformMaskMappingLocalToWorld[39][68] = 12;
    out.transformMaskMappingLocalToWorld[39][69] = 140;
    out.transformMaskMappingLocalToWorld[39][70] = 44;
    out.transformMaskMappingLocalToWorld[39][71] = 172;
    out.transformMaskMappingLocalToWorld[39][72] = 6;
    out.transformMaskMappingLocalToWorld[39][73] = 134;
    out.transformMaskMappingLocalToWorld[39][74] = 38;
    out.transformMaskMappingLocalToWorld[39][75] = 166;
    out.transformMaskMappingLocalToWorld[39][76] = 14;
    out.transformMaskMappingLocalToWorld[39][77] = 142;
    out.transformMaskMappingLocalToWorld[39][78] = 46;
    out.transformMaskMappingLocalToWorld[39][79] = 174;
    out.transformMaskMappingLocalToWorld[39][80] = 68;
    out.transformMaskMappingLocalToWorld[39][81] = 196;
    out.transformMaskMappingLocalToWorld[39][82] = 100;
    out.transformMaskMappingLocalToWorld[39][83] = 228;
    out.transformMaskMappingLocalToWorld[39][84] = 76;
    out.transformMaskMappingLocalToWorld[39][85] = 204;
    out.transformMaskMappingLocalToWorld[39][86] = 108;
    out.transformMaskMappingLocalToWorld[39][87] = 236;
    out.transformMaskMappingLocalToWorld[39][88] = 70;
    out.transformMaskMappingLocalToWorld[39][89] = 198;
    out.transformMaskMappingLocalToWorld[39][90] = 102;
    out.transformMaskMappingLocalToWorld[39][91] = 230;
    out.transformMaskMappingLocalToWorld[39][92] = 78;
    out.transformMaskMappingLocalToWorld[39][93] = 206;
    out.transformMaskMappingLocalToWorld[39][94] = 110;
    out.transformMaskMappingLocalToWorld[39][95] = 238;
    out.transformMaskMappingLocalToWorld[39][96] = 20;
    out.transformMaskMappingLocalToWorld[39][97] = 148;
    out.transformMaskMappingLocalToWorld[39][98] = 52;
    out.transformMaskMappingLocalToWorld[39][99] = 180;
    out.transformMaskMappingLocalToWorld[39][100] = 28;
    out.transformMaskMappingLocalToWorld[39][101] = 156;
    out.transformMaskMappingLocalToWorld[39][102] = 60;
    out.transformMaskMappingLocalToWorld[39][103] = 188;
    out.transformMaskMappingLocalToWorld[39][104] = 22;
    out.transformMaskMappingLocalToWorld[39][105] = 150;
    out.transformMaskMappingLocalToWorld[39][106] = 54;
    out.transformMaskMappingLocalToWorld[39][107] = 182;
    out.transformMaskMappingLocalToWorld[39][108] = 30;
    out.transformMaskMappingLocalToWorld[39][109] = 158;
    out.transformMaskMappingLocalToWorld[39][110] = 62;
    out.transformMaskMappingLocalToWorld[39][111] = 190;
    out.transformMaskMappingLocalToWorld[39][112] = 84;
    out.transformMaskMappingLocalToWorld[39][113] = 212;
    out.transformMaskMappingLocalToWorld[39][114] = 116;
    out.transformMaskMappingLocalToWorld[39][115] = 244;
    out.transformMaskMappingLocalToWorld[39][116] = 92;
    out.transformMaskMappingLocalToWorld[39][117] = 220;
    out.transformMaskMappingLocalToWorld[39][118] = 124;
    out.transformMaskMappingLocalToWorld[39][119] = 252;
    out.transformMaskMappingLocalToWorld[39][120] = 86;
    out.transformMaskMappingLocalToWorld[39][121] = 214;
    out.transformMaskMappingLocalToWorld[39][122] = 118;
    out.transformMaskMappingLocalToWorld[39][123] = 246;
    out.transformMaskMappingLocalToWorld[39][124] = 94;
    out.transformMaskMappingLocalToWorld[39][125] = 222;
    out.transformMaskMappingLocalToWorld[39][126] = 126;
    out.transformMaskMappingLocalToWorld[39][127] = 254;
    out.transformMaskMappingLocalToWorld[39][128] = 1;
    out.transformMaskMappingLocalToWorld[39][129] = 129;
    out.transformMaskMappingLocalToWorld[39][130] = 33;
    out.transformMaskMappingLocalToWorld[39][131] = 161;
    out.transformMaskMappingLocalToWorld[39][132] = 9;
    out.transformMaskMappingLocalToWorld[39][133] = 137;
    out.transformMaskMappingLocalToWorld[39][134] = 41;
    out.transformMaskMappingLocalToWorld[39][135] = 169;
    out.transformMaskMappingLocalToWorld[39][136] = 3;
    out.transformMaskMappingLocalToWorld[39][137] = 131;
    out.transformMaskMappingLocalToWorld[39][138] = 35;
    out.transformMaskMappingLocalToWorld[39][139] = 163;
    out.transformMaskMappingLocalToWorld[39][140] = 11;
    out.transformMaskMappingLocalToWorld[39][141] = 139;
    out.transformMaskMappingLocalToWorld[39][142] = 43;
    out.transformMaskMappingLocalToWorld[39][143] = 171;
    out.transformMaskMappingLocalToWorld[39][144] = 65;
    out.transformMaskMappingLocalToWorld[39][145] = 193;
    out.transformMaskMappingLocalToWorld[39][146] = 97;
    out.transformMaskMappingLocalToWorld[39][147] = 225;
    out.transformMaskMappingLocalToWorld[39][148] = 73;
    out.transformMaskMappingLocalToWorld[39][149] = 201;
    out.transformMaskMappingLocalToWorld[39][150] = 105;
    out.transformMaskMappingLocalToWorld[39][151] = 233;
    out.transformMaskMappingLocalToWorld[39][152] = 67;
    out.transformMaskMappingLocalToWorld[39][153] = 195;
    out.transformMaskMappingLocalToWorld[39][154] = 99;
    out.transformMaskMappingLocalToWorld[39][155] = 227;
    out.transformMaskMappingLocalToWorld[39][156] = 75;
    out.transformMaskMappingLocalToWorld[39][157] = 203;
    out.transformMaskMappingLocalToWorld[39][158] = 107;
    out.transformMaskMappingLocalToWorld[39][159] = 235;
    out.transformMaskMappingLocalToWorld[39][160] = 17;
    out.transformMaskMappingLocalToWorld[39][161] = 145;
    out.transformMaskMappingLocalToWorld[39][162] = 49;
    out.transformMaskMappingLocalToWorld[39][163] = 177;
    out.transformMaskMappingLocalToWorld[39][164] = 25;
    out.transformMaskMappingLocalToWorld[39][165] = 153;
    out.transformMaskMappingLocalToWorld[39][166] = 57;
    out.transformMaskMappingLocalToWorld[39][167] = 185;
    out.transformMaskMappingLocalToWorld[39][168] = 19;
    out.transformMaskMappingLocalToWorld[39][169] = 147;
    out.transformMaskMappingLocalToWorld[39][170] = 51;
    out.transformMaskMappingLocalToWorld[39][171] = 179;
    out.transformMaskMappingLocalToWorld[39][172] = 27;
    out.transformMaskMappingLocalToWorld[39][173] = 155;
    out.transformMaskMappingLocalToWorld[39][174] = 59;
    out.transformMaskMappingLocalToWorld[39][175] = 187;
    out.transformMaskMappingLocalToWorld[39][176] = 81;
    out.transformMaskMappingLocalToWorld[39][177] = 209;
    out.transformMaskMappingLocalToWorld[39][178] = 113;
    out.transformMaskMappingLocalToWorld[39][179] = 241;
    out.transformMaskMappingLocalToWorld[39][180] = 89;
    out.transformMaskMappingLocalToWorld[39][181] = 217;
    out.transformMaskMappingLocalToWorld[39][182] = 121;
    out.transformMaskMappingLocalToWorld[39][183] = 249;
    out.transformMaskMappingLocalToWorld[39][184] = 83;
    out.transformMaskMappingLocalToWorld[39][185] = 211;
    out.transformMaskMappingLocalToWorld[39][186] = 115;
    out.transformMaskMappingLocalToWorld[39][187] = 243;
    out.transformMaskMappingLocalToWorld[39][188] = 91;
    out.transformMaskMappingLocalToWorld[39][189] = 219;
    out.transformMaskMappingLocalToWorld[39][190] = 123;
    out.transformMaskMappingLocalToWorld[39][191] = 251;
    out.transformMaskMappingLocalToWorld[39][192] = 5;
    out.transformMaskMappingLocalToWorld[39][193] = 133;
    out.transformMaskMappingLocalToWorld[39][194] = 37;
    out.transformMaskMappingLocalToWorld[39][195] = 165;
    out.transformMaskMappingLocalToWorld[39][196] = 13;
    out.transformMaskMappingLocalToWorld[39][197] = 141;
    out.transformMaskMappingLocalToWorld[39][198] = 45;
    out.transformMaskMappingLocalToWorld[39][199] = 173;
    out.transformMaskMappingLocalToWorld[39][200] = 7;
    out.transformMaskMappingLocalToWorld[39][201] = 135;
    out.transformMaskMappingLocalToWorld[39][202] = 39;
    out.transformMaskMappingLocalToWorld[39][203] = 167;
    out.transformMaskMappingLocalToWorld[39][204] = 15;
    out.transformMaskMappingLocalToWorld[39][205] = 143;
    out.transformMaskMappingLocalToWorld[39][206] = 47;
    out.transformMaskMappingLocalToWorld[39][207] = 175;
    out.transformMaskMappingLocalToWorld[39][208] = 69;
    out.transformMaskMappingLocalToWorld[39][209] = 197;
    out.transformMaskMappingLocalToWorld[39][210] = 101;
    out.transformMaskMappingLocalToWorld[39][211] = 229;
    out.transformMaskMappingLocalToWorld[39][212] = 77;
    out.transformMaskMappingLocalToWorld[39][213] = 205;
    out.transformMaskMappingLocalToWorld[39][214] = 109;
    out.transformMaskMappingLocalToWorld[39][215] = 237;
    out.transformMaskMappingLocalToWorld[39][216] = 71;
    out.transformMaskMappingLocalToWorld[39][217] = 199;
    out.transformMaskMappingLocalToWorld[39][218] = 103;
    out.transformMaskMappingLocalToWorld[39][219] = 231;
    out.transformMaskMappingLocalToWorld[39][220] = 79;
    out.transformMaskMappingLocalToWorld[39][221] = 207;
    out.transformMaskMappingLocalToWorld[39][222] = 111;
    out.transformMaskMappingLocalToWorld[39][223] = 239;
    out.transformMaskMappingLocalToWorld[39][224] = 21;
    out.transformMaskMappingLocalToWorld[39][225] = 149;
    out.transformMaskMappingLocalToWorld[39][226] = 53;
    out.transformMaskMappingLocalToWorld[39][227] = 181;
    out.transformMaskMappingLocalToWorld[39][228] = 29;
    out.transformMaskMappingLocalToWorld[39][229] = 157;
    out.transformMaskMappingLocalToWorld[39][230] = 61;
    out.transformMaskMappingLocalToWorld[39][231] = 189;
    out.transformMaskMappingLocalToWorld[39][232] = 23;
    out.transformMaskMappingLocalToWorld[39][233] = 151;
    out.transformMaskMappingLocalToWorld[39][234] = 55;
    out.transformMaskMappingLocalToWorld[39][235] = 183;
    out.transformMaskMappingLocalToWorld[39][236] = 31;
    out.transformMaskMappingLocalToWorld[39][237] = 159;
    out.transformMaskMappingLocalToWorld[39][238] = 63;
    out.transformMaskMappingLocalToWorld[39][239] = 191;
    out.transformMaskMappingLocalToWorld[39][240] = 85;
    out.transformMaskMappingLocalToWorld[39][241] = 213;
    out.transformMaskMappingLocalToWorld[39][242] = 117;
    out.transformMaskMappingLocalToWorld[39][243] = 245;
    out.transformMaskMappingLocalToWorld[39][244] = 93;
    out.transformMaskMappingLocalToWorld[39][245] = 221;
    out.transformMaskMappingLocalToWorld[39][246] = 125;
    out.transformMaskMappingLocalToWorld[39][247] = 253;
    out.transformMaskMappingLocalToWorld[39][248] = 87;
    out.transformMaskMappingLocalToWorld[39][249] = 215;
    out.transformMaskMappingLocalToWorld[39][250] = 119;
    out.transformMaskMappingLocalToWorld[39][251] = 247;
    out.transformMaskMappingLocalToWorld[39][252] = 95;
    out.transformMaskMappingLocalToWorld[39][253] = 223;
    out.transformMaskMappingLocalToWorld[39][254] = 127;
    out.transformMaskMappingLocalToWorld[39][255] = 255;
    out.transformMaskMappingLocalToWorld[40][0] = 0;
    out.transformMaskMappingLocalToWorld[40][1] = 1;
    out.transformMaskMappingLocalToWorld[40][2] = 16;
    out.transformMaskMappingLocalToWorld[40][3] = 17;
    out.transformMaskMappingLocalToWorld[40][4] = 4;
    out.transformMaskMappingLocalToWorld[40][5] = 5;
    out.transformMaskMappingLocalToWorld[40][6] = 20;
    out.transformMaskMappingLocalToWorld[40][7] = 21;
    out.transformMaskMappingLocalToWorld[40][8] = 64;
    out.transformMaskMappingLocalToWorld[40][9] = 65;
    out.transformMaskMappingLocalToWorld[40][10] = 80;
    out.transformMaskMappingLocalToWorld[40][11] = 81;
    out.transformMaskMappingLocalToWorld[40][12] = 68;
    out.transformMaskMappingLocalToWorld[40][13] = 69;
    out.transformMaskMappingLocalToWorld[40][14] = 84;
    out.transformMaskMappingLocalToWorld[40][15] = 85;
    out.transformMaskMappingLocalToWorld[40][16] = 2;
    out.transformMaskMappingLocalToWorld[40][17] = 3;
    out.transformMaskMappingLocalToWorld[40][18] = 18;
    out.transformMaskMappingLocalToWorld[40][19] = 19;
    out.transformMaskMappingLocalToWorld[40][20] = 6;
    out.transformMaskMappingLocalToWorld[40][21] = 7;
    out.transformMaskMappingLocalToWorld[40][22] = 22;
    out.transformMaskMappingLocalToWorld[40][23] = 23;
    out.transformMaskMappingLocalToWorld[40][24] = 66;
    out.transformMaskMappingLocalToWorld[40][25] = 67;
    out.transformMaskMappingLocalToWorld[40][26] = 82;
    out.transformMaskMappingLocalToWorld[40][27] = 83;
    out.transformMaskMappingLocalToWorld[40][28] = 70;
    out.transformMaskMappingLocalToWorld[40][29] = 71;
    out.transformMaskMappingLocalToWorld[40][30] = 86;
    out.transformMaskMappingLocalToWorld[40][31] = 87;
    out.transformMaskMappingLocalToWorld[40][32] = 32;
    out.transformMaskMappingLocalToWorld[40][33] = 33;
    out.transformMaskMappingLocalToWorld[40][34] = 48;
    out.transformMaskMappingLocalToWorld[40][35] = 49;
    out.transformMaskMappingLocalToWorld[40][36] = 36;
    out.transformMaskMappingLocalToWorld[40][37] = 37;
    out.transformMaskMappingLocalToWorld[40][38] = 52;
    out.transformMaskMappingLocalToWorld[40][39] = 53;
    out.transformMaskMappingLocalToWorld[40][40] = 96;
    out.transformMaskMappingLocalToWorld[40][41] = 97;
    out.transformMaskMappingLocalToWorld[40][42] = 112;
    out.transformMaskMappingLocalToWorld[40][43] = 113;
    out.transformMaskMappingLocalToWorld[40][44] = 100;
    out.transformMaskMappingLocalToWorld[40][45] = 101;
    out.transformMaskMappingLocalToWorld[40][46] = 116;
    out.transformMaskMappingLocalToWorld[40][47] = 117;
    out.transformMaskMappingLocalToWorld[40][48] = 34;
    out.transformMaskMappingLocalToWorld[40][49] = 35;
    out.transformMaskMappingLocalToWorld[40][50] = 50;
    out.transformMaskMappingLocalToWorld[40][51] = 51;
    out.transformMaskMappingLocalToWorld[40][52] = 38;
    out.transformMaskMappingLocalToWorld[40][53] = 39;
    out.transformMaskMappingLocalToWorld[40][54] = 54;
    out.transformMaskMappingLocalToWorld[40][55] = 55;
    out.transformMaskMappingLocalToWorld[40][56] = 98;
    out.transformMaskMappingLocalToWorld[40][57] = 99;
    out.transformMaskMappingLocalToWorld[40][58] = 114;
    out.transformMaskMappingLocalToWorld[40][59] = 115;
    out.transformMaskMappingLocalToWorld[40][60] = 102;
    out.transformMaskMappingLocalToWorld[40][61] = 103;
    out.transformMaskMappingLocalToWorld[40][62] = 118;
    out.transformMaskMappingLocalToWorld[40][63] = 119;
    out.transformMaskMappingLocalToWorld[40][64] = 8;
    out.transformMaskMappingLocalToWorld[40][65] = 9;
    out.transformMaskMappingLocalToWorld[40][66] = 24;
    out.transformMaskMappingLocalToWorld[40][67] = 25;
    out.transformMaskMappingLocalToWorld[40][68] = 12;
    out.transformMaskMappingLocalToWorld[40][69] = 13;
    out.transformMaskMappingLocalToWorld[40][70] = 28;
    out.transformMaskMappingLocalToWorld[40][71] = 29;
    out.transformMaskMappingLocalToWorld[40][72] = 72;
    out.transformMaskMappingLocalToWorld[40][73] = 73;
    out.transformMaskMappingLocalToWorld[40][74] = 88;
    out.transformMaskMappingLocalToWorld[40][75] = 89;
    out.transformMaskMappingLocalToWorld[40][76] = 76;
    out.transformMaskMappingLocalToWorld[40][77] = 77;
    out.transformMaskMappingLocalToWorld[40][78] = 92;
    out.transformMaskMappingLocalToWorld[40][79] = 93;
    out.transformMaskMappingLocalToWorld[40][80] = 10;
    out.transformMaskMappingLocalToWorld[40][81] = 11;
    out.transformMaskMappingLocalToWorld[40][82] = 26;
    out.transformMaskMappingLocalToWorld[40][83] = 27;
    out.transformMaskMappingLocalToWorld[40][84] = 14;
    out.transformMaskMappingLocalToWorld[40][85] = 15;
    out.transformMaskMappingLocalToWorld[40][86] = 30;
    out.transformMaskMappingLocalToWorld[40][87] = 31;
    out.transformMaskMappingLocalToWorld[40][88] = 74;
    out.transformMaskMappingLocalToWorld[40][89] = 75;
    out.transformMaskMappingLocalToWorld[40][90] = 90;
    out.transformMaskMappingLocalToWorld[40][91] = 91;
    out.transformMaskMappingLocalToWorld[40][92] = 78;
    out.transformMaskMappingLocalToWorld[40][93] = 79;
    out.transformMaskMappingLocalToWorld[40][94] = 94;
    out.transformMaskMappingLocalToWorld[40][95] = 95;
    out.transformMaskMappingLocalToWorld[40][96] = 40;
    out.transformMaskMappingLocalToWorld[40][97] = 41;
    out.transformMaskMappingLocalToWorld[40][98] = 56;
    out.transformMaskMappingLocalToWorld[40][99] = 57;
    out.transformMaskMappingLocalToWorld[40][100] = 44;
    out.transformMaskMappingLocalToWorld[40][101] = 45;
    out.transformMaskMappingLocalToWorld[40][102] = 60;
    out.transformMaskMappingLocalToWorld[40][103] = 61;
    out.transformMaskMappingLocalToWorld[40][104] = 104;
    out.transformMaskMappingLocalToWorld[40][105] = 105;
    out.transformMaskMappingLocalToWorld[40][106] = 120;
    out.transformMaskMappingLocalToWorld[40][107] = 121;
    out.transformMaskMappingLocalToWorld[40][108] = 108;
    out.transformMaskMappingLocalToWorld[40][109] = 109;
    out.transformMaskMappingLocalToWorld[40][110] = 124;
    out.transformMaskMappingLocalToWorld[40][111] = 125;
    out.transformMaskMappingLocalToWorld[40][112] = 42;
    out.transformMaskMappingLocalToWorld[40][113] = 43;
    out.transformMaskMappingLocalToWorld[40][114] = 58;
    out.transformMaskMappingLocalToWorld[40][115] = 59;
    out.transformMaskMappingLocalToWorld[40][116] = 46;
    out.transformMaskMappingLocalToWorld[40][117] = 47;
    out.transformMaskMappingLocalToWorld[40][118] = 62;
    out.transformMaskMappingLocalToWorld[40][119] = 63;
    out.transformMaskMappingLocalToWorld[40][120] = 106;
    out.transformMaskMappingLocalToWorld[40][121] = 107;
    out.transformMaskMappingLocalToWorld[40][122] = 122;
    out.transformMaskMappingLocalToWorld[40][123] = 123;
    out.transformMaskMappingLocalToWorld[40][124] = 110;
    out.transformMaskMappingLocalToWorld[40][125] = 111;
    out.transformMaskMappingLocalToWorld[40][126] = 126;
    out.transformMaskMappingLocalToWorld[40][127] = 127;
    out.transformMaskMappingLocalToWorld[40][128] = 128;
    out.transformMaskMappingLocalToWorld[40][129] = 129;
    out.transformMaskMappingLocalToWorld[40][130] = 144;
    out.transformMaskMappingLocalToWorld[40][131] = 145;
    out.transformMaskMappingLocalToWorld[40][132] = 132;
    out.transformMaskMappingLocalToWorld[40][133] = 133;
    out.transformMaskMappingLocalToWorld[40][134] = 148;
    out.transformMaskMappingLocalToWorld[40][135] = 149;
    out.transformMaskMappingLocalToWorld[40][136] = 192;
    out.transformMaskMappingLocalToWorld[40][137] = 193;
    out.transformMaskMappingLocalToWorld[40][138] = 208;
    out.transformMaskMappingLocalToWorld[40][139] = 209;
    out.transformMaskMappingLocalToWorld[40][140] = 196;
    out.transformMaskMappingLocalToWorld[40][141] = 197;
    out.transformMaskMappingLocalToWorld[40][142] = 212;
    out.transformMaskMappingLocalToWorld[40][143] = 213;
    out.transformMaskMappingLocalToWorld[40][144] = 130;
    out.transformMaskMappingLocalToWorld[40][145] = 131;
    out.transformMaskMappingLocalToWorld[40][146] = 146;
    out.transformMaskMappingLocalToWorld[40][147] = 147;
    out.transformMaskMappingLocalToWorld[40][148] = 134;
    out.transformMaskMappingLocalToWorld[40][149] = 135;
    out.transformMaskMappingLocalToWorld[40][150] = 150;
    out.transformMaskMappingLocalToWorld[40][151] = 151;
    out.transformMaskMappingLocalToWorld[40][152] = 194;
    out.transformMaskMappingLocalToWorld[40][153] = 195;
    out.transformMaskMappingLocalToWorld[40][154] = 210;
    out.transformMaskMappingLocalToWorld[40][155] = 211;
    out.transformMaskMappingLocalToWorld[40][156] = 198;
    out.transformMaskMappingLocalToWorld[40][157] = 199;
    out.transformMaskMappingLocalToWorld[40][158] = 214;
    out.transformMaskMappingLocalToWorld[40][159] = 215;
    out.transformMaskMappingLocalToWorld[40][160] = 160;
    out.transformMaskMappingLocalToWorld[40][161] = 161;
    out.transformMaskMappingLocalToWorld[40][162] = 176;
    out.transformMaskMappingLocalToWorld[40][163] = 177;
    out.transformMaskMappingLocalToWorld[40][164] = 164;
    out.transformMaskMappingLocalToWorld[40][165] = 165;
    out.transformMaskMappingLocalToWorld[40][166] = 180;
    out.transformMaskMappingLocalToWorld[40][167] = 181;
    out.transformMaskMappingLocalToWorld[40][168] = 224;
    out.transformMaskMappingLocalToWorld[40][169] = 225;
    out.transformMaskMappingLocalToWorld[40][170] = 240;
    out.transformMaskMappingLocalToWorld[40][171] = 241;
    out.transformMaskMappingLocalToWorld[40][172] = 228;
    out.transformMaskMappingLocalToWorld[40][173] = 229;
    out.transformMaskMappingLocalToWorld[40][174] = 244;
    out.transformMaskMappingLocalToWorld[40][175] = 245;
    out.transformMaskMappingLocalToWorld[40][176] = 162;
    out.transformMaskMappingLocalToWorld[40][177] = 163;
    out.transformMaskMappingLocalToWorld[40][178] = 178;
    out.transformMaskMappingLocalToWorld[40][179] = 179;
    out.transformMaskMappingLocalToWorld[40][180] = 166;
    out.transformMaskMappingLocalToWorld[40][181] = 167;
    out.transformMaskMappingLocalToWorld[40][182] = 182;
    out.transformMaskMappingLocalToWorld[40][183] = 183;
    out.transformMaskMappingLocalToWorld[40][184] = 226;
    out.transformMaskMappingLocalToWorld[40][185] = 227;
    out.transformMaskMappingLocalToWorld[40][186] = 242;
    out.transformMaskMappingLocalToWorld[40][187] = 243;
    out.transformMaskMappingLocalToWorld[40][188] = 230;
    out.transformMaskMappingLocalToWorld[40][189] = 231;
    out.transformMaskMappingLocalToWorld[40][190] = 246;
    out.transformMaskMappingLocalToWorld[40][191] = 247;
    out.transformMaskMappingLocalToWorld[40][192] = 136;
    out.transformMaskMappingLocalToWorld[40][193] = 137;
    out.transformMaskMappingLocalToWorld[40][194] = 152;
    out.transformMaskMappingLocalToWorld[40][195] = 153;
    out.transformMaskMappingLocalToWorld[40][196] = 140;
    out.transformMaskMappingLocalToWorld[40][197] = 141;
    out.transformMaskMappingLocalToWorld[40][198] = 156;
    out.transformMaskMappingLocalToWorld[40][199] = 157;
    out.transformMaskMappingLocalToWorld[40][200] = 200;
    out.transformMaskMappingLocalToWorld[40][201] = 201;
    out.transformMaskMappingLocalToWorld[40][202] = 216;
    out.transformMaskMappingLocalToWorld[40][203] = 217;
    out.transformMaskMappingLocalToWorld[40][204] = 204;
    out.transformMaskMappingLocalToWorld[40][205] = 205;
    out.transformMaskMappingLocalToWorld[40][206] = 220;
    out.transformMaskMappingLocalToWorld[40][207] = 221;
    out.transformMaskMappingLocalToWorld[40][208] = 138;
    out.transformMaskMappingLocalToWorld[40][209] = 139;
    out.transformMaskMappingLocalToWorld[40][210] = 154;
    out.transformMaskMappingLocalToWorld[40][211] = 155;
    out.transformMaskMappingLocalToWorld[40][212] = 142;
    out.transformMaskMappingLocalToWorld[40][213] = 143;
    out.transformMaskMappingLocalToWorld[40][214] = 158;
    out.transformMaskMappingLocalToWorld[40][215] = 159;
    out.transformMaskMappingLocalToWorld[40][216] = 202;
    out.transformMaskMappingLocalToWorld[40][217] = 203;
    out.transformMaskMappingLocalToWorld[40][218] = 218;
    out.transformMaskMappingLocalToWorld[40][219] = 219;
    out.transformMaskMappingLocalToWorld[40][220] = 206;
    out.transformMaskMappingLocalToWorld[40][221] = 207;
    out.transformMaskMappingLocalToWorld[40][222] = 222;
    out.transformMaskMappingLocalToWorld[40][223] = 223;
    out.transformMaskMappingLocalToWorld[40][224] = 168;
    out.transformMaskMappingLocalToWorld[40][225] = 169;
    out.transformMaskMappingLocalToWorld[40][226] = 184;
    out.transformMaskMappingLocalToWorld[40][227] = 185;
    out.transformMaskMappingLocalToWorld[40][228] = 172;
    out.transformMaskMappingLocalToWorld[40][229] = 173;
    out.transformMaskMappingLocalToWorld[40][230] = 188;
    out.transformMaskMappingLocalToWorld[40][231] = 189;
    out.transformMaskMappingLocalToWorld[40][232] = 232;
    out.transformMaskMappingLocalToWorld[40][233] = 233;
    out.transformMaskMappingLocalToWorld[40][234] = 248;
    out.transformMaskMappingLocalToWorld[40][235] = 249;
    out.transformMaskMappingLocalToWorld[40][236] = 236;
    out.transformMaskMappingLocalToWorld[40][237] = 237;
    out.transformMaskMappingLocalToWorld[40][238] = 252;
    out.transformMaskMappingLocalToWorld[40][239] = 253;
    out.transformMaskMappingLocalToWorld[40][240] = 170;
    out.transformMaskMappingLocalToWorld[40][241] = 171;
    out.transformMaskMappingLocalToWorld[40][242] = 186;
    out.transformMaskMappingLocalToWorld[40][243] = 187;
    out.transformMaskMappingLocalToWorld[40][244] = 174;
    out.transformMaskMappingLocalToWorld[40][245] = 175;
    out.transformMaskMappingLocalToWorld[40][246] = 190;
    out.transformMaskMappingLocalToWorld[40][247] = 191;
    out.transformMaskMappingLocalToWorld[40][248] = 234;
    out.transformMaskMappingLocalToWorld[40][249] = 235;
    out.transformMaskMappingLocalToWorld[40][250] = 250;
    out.transformMaskMappingLocalToWorld[40][251] = 251;
    out.transformMaskMappingLocalToWorld[40][252] = 238;
    out.transformMaskMappingLocalToWorld[40][253] = 239;
    out.transformMaskMappingLocalToWorld[40][254] = 254;
    out.transformMaskMappingLocalToWorld[40][255] = 255;
    out.transformMaskMappingLocalToWorld[41][0] = 0;
    out.transformMaskMappingLocalToWorld[41][1] = 2;
    out.transformMaskMappingLocalToWorld[41][2] = 32;
    out.transformMaskMappingLocalToWorld[41][3] = 34;
    out.transformMaskMappingLocalToWorld[41][4] = 8;
    out.transformMaskMappingLocalToWorld[41][5] = 10;
    out.transformMaskMappingLocalToWorld[41][6] = 40;
    out.transformMaskMappingLocalToWorld[41][7] = 42;
    out.transformMaskMappingLocalToWorld[41][8] = 128;
    out.transformMaskMappingLocalToWorld[41][9] = 130;
    out.transformMaskMappingLocalToWorld[41][10] = 160;
    out.transformMaskMappingLocalToWorld[41][11] = 162;
    out.transformMaskMappingLocalToWorld[41][12] = 136;
    out.transformMaskMappingLocalToWorld[41][13] = 138;
    out.transformMaskMappingLocalToWorld[41][14] = 168;
    out.transformMaskMappingLocalToWorld[41][15] = 170;
    out.transformMaskMappingLocalToWorld[41][16] = 1;
    out.transformMaskMappingLocalToWorld[41][17] = 3;
    out.transformMaskMappingLocalToWorld[41][18] = 33;
    out.transformMaskMappingLocalToWorld[41][19] = 35;
    out.transformMaskMappingLocalToWorld[41][20] = 9;
    out.transformMaskMappingLocalToWorld[41][21] = 11;
    out.transformMaskMappingLocalToWorld[41][22] = 41;
    out.transformMaskMappingLocalToWorld[41][23] = 43;
    out.transformMaskMappingLocalToWorld[41][24] = 129;
    out.transformMaskMappingLocalToWorld[41][25] = 131;
    out.transformMaskMappingLocalToWorld[41][26] = 161;
    out.transformMaskMappingLocalToWorld[41][27] = 163;
    out.transformMaskMappingLocalToWorld[41][28] = 137;
    out.transformMaskMappingLocalToWorld[41][29] = 139;
    out.transformMaskMappingLocalToWorld[41][30] = 169;
    out.transformMaskMappingLocalToWorld[41][31] = 171;
    out.transformMaskMappingLocalToWorld[41][32] = 16;
    out.transformMaskMappingLocalToWorld[41][33] = 18;
    out.transformMaskMappingLocalToWorld[41][34] = 48;
    out.transformMaskMappingLocalToWorld[41][35] = 50;
    out.transformMaskMappingLocalToWorld[41][36] = 24;
    out.transformMaskMappingLocalToWorld[41][37] = 26;
    out.transformMaskMappingLocalToWorld[41][38] = 56;
    out.transformMaskMappingLocalToWorld[41][39] = 58;
    out.transformMaskMappingLocalToWorld[41][40] = 144;
    out.transformMaskMappingLocalToWorld[41][41] = 146;
    out.transformMaskMappingLocalToWorld[41][42] = 176;
    out.transformMaskMappingLocalToWorld[41][43] = 178;
    out.transformMaskMappingLocalToWorld[41][44] = 152;
    out.transformMaskMappingLocalToWorld[41][45] = 154;
    out.transformMaskMappingLocalToWorld[41][46] = 184;
    out.transformMaskMappingLocalToWorld[41][47] = 186;
    out.transformMaskMappingLocalToWorld[41][48] = 17;
    out.transformMaskMappingLocalToWorld[41][49] = 19;
    out.transformMaskMappingLocalToWorld[41][50] = 49;
    out.transformMaskMappingLocalToWorld[41][51] = 51;
    out.transformMaskMappingLocalToWorld[41][52] = 25;
    out.transformMaskMappingLocalToWorld[41][53] = 27;
    out.transformMaskMappingLocalToWorld[41][54] = 57;
    out.transformMaskMappingLocalToWorld[41][55] = 59;
    out.transformMaskMappingLocalToWorld[41][56] = 145;
    out.transformMaskMappingLocalToWorld[41][57] = 147;
    out.transformMaskMappingLocalToWorld[41][58] = 177;
    out.transformMaskMappingLocalToWorld[41][59] = 179;
    out.transformMaskMappingLocalToWorld[41][60] = 153;
    out.transformMaskMappingLocalToWorld[41][61] = 155;
    out.transformMaskMappingLocalToWorld[41][62] = 185;
    out.transformMaskMappingLocalToWorld[41][63] = 187;
    out.transformMaskMappingLocalToWorld[41][64] = 4;
    out.transformMaskMappingLocalToWorld[41][65] = 6;
    out.transformMaskMappingLocalToWorld[41][66] = 36;
    out.transformMaskMappingLocalToWorld[41][67] = 38;
    out.transformMaskMappingLocalToWorld[41][68] = 12;
    out.transformMaskMappingLocalToWorld[41][69] = 14;
    out.transformMaskMappingLocalToWorld[41][70] = 44;
    out.transformMaskMappingLocalToWorld[41][71] = 46;
    out.transformMaskMappingLocalToWorld[41][72] = 132;
    out.transformMaskMappingLocalToWorld[41][73] = 134;
    out.transformMaskMappingLocalToWorld[41][74] = 164;
    out.transformMaskMappingLocalToWorld[41][75] = 166;
    out.transformMaskMappingLocalToWorld[41][76] = 140;
    out.transformMaskMappingLocalToWorld[41][77] = 142;
    out.transformMaskMappingLocalToWorld[41][78] = 172;
    out.transformMaskMappingLocalToWorld[41][79] = 174;
    out.transformMaskMappingLocalToWorld[41][80] = 5;
    out.transformMaskMappingLocalToWorld[41][81] = 7;
    out.transformMaskMappingLocalToWorld[41][82] = 37;
    out.transformMaskMappingLocalToWorld[41][83] = 39;
    out.transformMaskMappingLocalToWorld[41][84] = 13;
    out.transformMaskMappingLocalToWorld[41][85] = 15;
    out.transformMaskMappingLocalToWorld[41][86] = 45;
    out.transformMaskMappingLocalToWorld[41][87] = 47;
    out.transformMaskMappingLocalToWorld[41][88] = 133;
    out.transformMaskMappingLocalToWorld[41][89] = 135;
    out.transformMaskMappingLocalToWorld[41][90] = 165;
    out.transformMaskMappingLocalToWorld[41][91] = 167;
    out.transformMaskMappingLocalToWorld[41][92] = 141;
    out.transformMaskMappingLocalToWorld[41][93] = 143;
    out.transformMaskMappingLocalToWorld[41][94] = 173;
    out.transformMaskMappingLocalToWorld[41][95] = 175;
    out.transformMaskMappingLocalToWorld[41][96] = 20;
    out.transformMaskMappingLocalToWorld[41][97] = 22;
    out.transformMaskMappingLocalToWorld[41][98] = 52;
    out.transformMaskMappingLocalToWorld[41][99] = 54;
    out.transformMaskMappingLocalToWorld[41][100] = 28;
    out.transformMaskMappingLocalToWorld[41][101] = 30;
    out.transformMaskMappingLocalToWorld[41][102] = 60;
    out.transformMaskMappingLocalToWorld[41][103] = 62;
    out.transformMaskMappingLocalToWorld[41][104] = 148;
    out.transformMaskMappingLocalToWorld[41][105] = 150;
    out.transformMaskMappingLocalToWorld[41][106] = 180;
    out.transformMaskMappingLocalToWorld[41][107] = 182;
    out.transformMaskMappingLocalToWorld[41][108] = 156;
    out.transformMaskMappingLocalToWorld[41][109] = 158;
    out.transformMaskMappingLocalToWorld[41][110] = 188;
    out.transformMaskMappingLocalToWorld[41][111] = 190;
    out.transformMaskMappingLocalToWorld[41][112] = 21;
    out.transformMaskMappingLocalToWorld[41][113] = 23;
    out.transformMaskMappingLocalToWorld[41][114] = 53;
    out.transformMaskMappingLocalToWorld[41][115] = 55;
    out.transformMaskMappingLocalToWorld[41][116] = 29;
    out.transformMaskMappingLocalToWorld[41][117] = 31;
    out.transformMaskMappingLocalToWorld[41][118] = 61;
    out.transformMaskMappingLocalToWorld[41][119] = 63;
    out.transformMaskMappingLocalToWorld[41][120] = 149;
    out.transformMaskMappingLocalToWorld[41][121] = 151;
    out.transformMaskMappingLocalToWorld[41][122] = 181;
    out.transformMaskMappingLocalToWorld[41][123] = 183;
    out.transformMaskMappingLocalToWorld[41][124] = 157;
    out.transformMaskMappingLocalToWorld[41][125] = 159;
    out.transformMaskMappingLocalToWorld[41][126] = 189;
    out.transformMaskMappingLocalToWorld[41][127] = 191;
    out.transformMaskMappingLocalToWorld[41][128] = 64;
    out.transformMaskMappingLocalToWorld[41][129] = 66;
    out.transformMaskMappingLocalToWorld[41][130] = 96;
    out.transformMaskMappingLocalToWorld[41][131] = 98;
    out.transformMaskMappingLocalToWorld[41][132] = 72;
    out.transformMaskMappingLocalToWorld[41][133] = 74;
    out.transformMaskMappingLocalToWorld[41][134] = 104;
    out.transformMaskMappingLocalToWorld[41][135] = 106;
    out.transformMaskMappingLocalToWorld[41][136] = 192;
    out.transformMaskMappingLocalToWorld[41][137] = 194;
    out.transformMaskMappingLocalToWorld[41][138] = 224;
    out.transformMaskMappingLocalToWorld[41][139] = 226;
    out.transformMaskMappingLocalToWorld[41][140] = 200;
    out.transformMaskMappingLocalToWorld[41][141] = 202;
    out.transformMaskMappingLocalToWorld[41][142] = 232;
    out.transformMaskMappingLocalToWorld[41][143] = 234;
    out.transformMaskMappingLocalToWorld[41][144] = 65;
    out.transformMaskMappingLocalToWorld[41][145] = 67;
    out.transformMaskMappingLocalToWorld[41][146] = 97;
    out.transformMaskMappingLocalToWorld[41][147] = 99;
    out.transformMaskMappingLocalToWorld[41][148] = 73;
    out.transformMaskMappingLocalToWorld[41][149] = 75;
    out.transformMaskMappingLocalToWorld[41][150] = 105;
    out.transformMaskMappingLocalToWorld[41][151] = 107;
    out.transformMaskMappingLocalToWorld[41][152] = 193;
    out.transformMaskMappingLocalToWorld[41][153] = 195;
    out.transformMaskMappingLocalToWorld[41][154] = 225;
    out.transformMaskMappingLocalToWorld[41][155] = 227;
    out.transformMaskMappingLocalToWorld[41][156] = 201;
    out.transformMaskMappingLocalToWorld[41][157] = 203;
    out.transformMaskMappingLocalToWorld[41][158] = 233;
    out.transformMaskMappingLocalToWorld[41][159] = 235;
    out.transformMaskMappingLocalToWorld[41][160] = 80;
    out.transformMaskMappingLocalToWorld[41][161] = 82;
    out.transformMaskMappingLocalToWorld[41][162] = 112;
    out.transformMaskMappingLocalToWorld[41][163] = 114;
    out.transformMaskMappingLocalToWorld[41][164] = 88;
    out.transformMaskMappingLocalToWorld[41][165] = 90;
    out.transformMaskMappingLocalToWorld[41][166] = 120;
    out.transformMaskMappingLocalToWorld[41][167] = 122;
    out.transformMaskMappingLocalToWorld[41][168] = 208;
    out.transformMaskMappingLocalToWorld[41][169] = 210;
    out.transformMaskMappingLocalToWorld[41][170] = 240;
    out.transformMaskMappingLocalToWorld[41][171] = 242;
    out.transformMaskMappingLocalToWorld[41][172] = 216;
    out.transformMaskMappingLocalToWorld[41][173] = 218;
    out.transformMaskMappingLocalToWorld[41][174] = 248;
    out.transformMaskMappingLocalToWorld[41][175] = 250;
    out.transformMaskMappingLocalToWorld[41][176] = 81;
    out.transformMaskMappingLocalToWorld[41][177] = 83;
    out.transformMaskMappingLocalToWorld[41][178] = 113;
    out.transformMaskMappingLocalToWorld[41][179] = 115;
    out.transformMaskMappingLocalToWorld[41][180] = 89;
    out.transformMaskMappingLocalToWorld[41][181] = 91;
    out.transformMaskMappingLocalToWorld[41][182] = 121;
    out.transformMaskMappingLocalToWorld[41][183] = 123;
    out.transformMaskMappingLocalToWorld[41][184] = 209;
    out.transformMaskMappingLocalToWorld[41][185] = 211;
    out.transformMaskMappingLocalToWorld[41][186] = 241;
    out.transformMaskMappingLocalToWorld[41][187] = 243;
    out.transformMaskMappingLocalToWorld[41][188] = 217;
    out.transformMaskMappingLocalToWorld[41][189] = 219;
    out.transformMaskMappingLocalToWorld[41][190] = 249;
    out.transformMaskMappingLocalToWorld[41][191] = 251;
    out.transformMaskMappingLocalToWorld[41][192] = 68;
    out.transformMaskMappingLocalToWorld[41][193] = 70;
    out.transformMaskMappingLocalToWorld[41][194] = 100;
    out.transformMaskMappingLocalToWorld[41][195] = 102;
    out.transformMaskMappingLocalToWorld[41][196] = 76;
    out.transformMaskMappingLocalToWorld[41][197] = 78;
    out.transformMaskMappingLocalToWorld[41][198] = 108;
    out.transformMaskMappingLocalToWorld[41][199] = 110;
    out.transformMaskMappingLocalToWorld[41][200] = 196;
    out.transformMaskMappingLocalToWorld[41][201] = 198;
    out.transformMaskMappingLocalToWorld[41][202] = 228;
    out.transformMaskMappingLocalToWorld[41][203] = 230;
    out.transformMaskMappingLocalToWorld[41][204] = 204;
    out.transformMaskMappingLocalToWorld[41][205] = 206;
    out.transformMaskMappingLocalToWorld[41][206] = 236;
    out.transformMaskMappingLocalToWorld[41][207] = 238;
    out.transformMaskMappingLocalToWorld[41][208] = 69;
    out.transformMaskMappingLocalToWorld[41][209] = 71;
    out.transformMaskMappingLocalToWorld[41][210] = 101;
    out.transformMaskMappingLocalToWorld[41][211] = 103;
    out.transformMaskMappingLocalToWorld[41][212] = 77;
    out.transformMaskMappingLocalToWorld[41][213] = 79;
    out.transformMaskMappingLocalToWorld[41][214] = 109;
    out.transformMaskMappingLocalToWorld[41][215] = 111;
    out.transformMaskMappingLocalToWorld[41][216] = 197;
    out.transformMaskMappingLocalToWorld[41][217] = 199;
    out.transformMaskMappingLocalToWorld[41][218] = 229;
    out.transformMaskMappingLocalToWorld[41][219] = 231;
    out.transformMaskMappingLocalToWorld[41][220] = 205;
    out.transformMaskMappingLocalToWorld[41][221] = 207;
    out.transformMaskMappingLocalToWorld[41][222] = 237;
    out.transformMaskMappingLocalToWorld[41][223] = 239;
    out.transformMaskMappingLocalToWorld[41][224] = 84;
    out.transformMaskMappingLocalToWorld[41][225] = 86;
    out.transformMaskMappingLocalToWorld[41][226] = 116;
    out.transformMaskMappingLocalToWorld[41][227] = 118;
    out.transformMaskMappingLocalToWorld[41][228] = 92;
    out.transformMaskMappingLocalToWorld[41][229] = 94;
    out.transformMaskMappingLocalToWorld[41][230] = 124;
    out.transformMaskMappingLocalToWorld[41][231] = 126;
    out.transformMaskMappingLocalToWorld[41][232] = 212;
    out.transformMaskMappingLocalToWorld[41][233] = 214;
    out.transformMaskMappingLocalToWorld[41][234] = 244;
    out.transformMaskMappingLocalToWorld[41][235] = 246;
    out.transformMaskMappingLocalToWorld[41][236] = 220;
    out.transformMaskMappingLocalToWorld[41][237] = 222;
    out.transformMaskMappingLocalToWorld[41][238] = 252;
    out.transformMaskMappingLocalToWorld[41][239] = 254;
    out.transformMaskMappingLocalToWorld[41][240] = 85;
    out.transformMaskMappingLocalToWorld[41][241] = 87;
    out.transformMaskMappingLocalToWorld[41][242] = 117;
    out.transformMaskMappingLocalToWorld[41][243] = 119;
    out.transformMaskMappingLocalToWorld[41][244] = 93;
    out.transformMaskMappingLocalToWorld[41][245] = 95;
    out.transformMaskMappingLocalToWorld[41][246] = 125;
    out.transformMaskMappingLocalToWorld[41][247] = 127;
    out.transformMaskMappingLocalToWorld[41][248] = 213;
    out.transformMaskMappingLocalToWorld[41][249] = 215;
    out.transformMaskMappingLocalToWorld[41][250] = 245;
    out.transformMaskMappingLocalToWorld[41][251] = 247;
    out.transformMaskMappingLocalToWorld[41][252] = 221;
    out.transformMaskMappingLocalToWorld[41][253] = 223;
    out.transformMaskMappingLocalToWorld[41][254] = 253;
    out.transformMaskMappingLocalToWorld[41][255] = 255;
    out.transformMaskMappingLocalToWorld[42][0] = 0;
    out.transformMaskMappingLocalToWorld[42][1] = 4;
    out.transformMaskMappingLocalToWorld[42][2] = 64;
    out.transformMaskMappingLocalToWorld[42][3] = 68;
    out.transformMaskMappingLocalToWorld[42][4] = 1;
    out.transformMaskMappingLocalToWorld[42][5] = 5;
    out.transformMaskMappingLocalToWorld[42][6] = 65;
    out.transformMaskMappingLocalToWorld[42][7] = 69;
    out.transformMaskMappingLocalToWorld[42][8] = 16;
    out.transformMaskMappingLocalToWorld[42][9] = 20;
    out.transformMaskMappingLocalToWorld[42][10] = 80;
    out.transformMaskMappingLocalToWorld[42][11] = 84;
    out.transformMaskMappingLocalToWorld[42][12] = 17;
    out.transformMaskMappingLocalToWorld[42][13] = 21;
    out.transformMaskMappingLocalToWorld[42][14] = 81;
    out.transformMaskMappingLocalToWorld[42][15] = 85;
    out.transformMaskMappingLocalToWorld[42][16] = 8;
    out.transformMaskMappingLocalToWorld[42][17] = 12;
    out.transformMaskMappingLocalToWorld[42][18] = 72;
    out.transformMaskMappingLocalToWorld[42][19] = 76;
    out.transformMaskMappingLocalToWorld[42][20] = 9;
    out.transformMaskMappingLocalToWorld[42][21] = 13;
    out.transformMaskMappingLocalToWorld[42][22] = 73;
    out.transformMaskMappingLocalToWorld[42][23] = 77;
    out.transformMaskMappingLocalToWorld[42][24] = 24;
    out.transformMaskMappingLocalToWorld[42][25] = 28;
    out.transformMaskMappingLocalToWorld[42][26] = 88;
    out.transformMaskMappingLocalToWorld[42][27] = 92;
    out.transformMaskMappingLocalToWorld[42][28] = 25;
    out.transformMaskMappingLocalToWorld[42][29] = 29;
    out.transformMaskMappingLocalToWorld[42][30] = 89;
    out.transformMaskMappingLocalToWorld[42][31] = 93;
    out.transformMaskMappingLocalToWorld[42][32] = 128;
    out.transformMaskMappingLocalToWorld[42][33] = 132;
    out.transformMaskMappingLocalToWorld[42][34] = 192;
    out.transformMaskMappingLocalToWorld[42][35] = 196;
    out.transformMaskMappingLocalToWorld[42][36] = 129;
    out.transformMaskMappingLocalToWorld[42][37] = 133;
    out.transformMaskMappingLocalToWorld[42][38] = 193;
    out.transformMaskMappingLocalToWorld[42][39] = 197;
    out.transformMaskMappingLocalToWorld[42][40] = 144;
    out.transformMaskMappingLocalToWorld[42][41] = 148;
    out.transformMaskMappingLocalToWorld[42][42] = 208;
    out.transformMaskMappingLocalToWorld[42][43] = 212;
    out.transformMaskMappingLocalToWorld[42][44] = 145;
    out.transformMaskMappingLocalToWorld[42][45] = 149;
    out.transformMaskMappingLocalToWorld[42][46] = 209;
    out.transformMaskMappingLocalToWorld[42][47] = 213;
    out.transformMaskMappingLocalToWorld[42][48] = 136;
    out.transformMaskMappingLocalToWorld[42][49] = 140;
    out.transformMaskMappingLocalToWorld[42][50] = 200;
    out.transformMaskMappingLocalToWorld[42][51] = 204;
    out.transformMaskMappingLocalToWorld[42][52] = 137;
    out.transformMaskMappingLocalToWorld[42][53] = 141;
    out.transformMaskMappingLocalToWorld[42][54] = 201;
    out.transformMaskMappingLocalToWorld[42][55] = 205;
    out.transformMaskMappingLocalToWorld[42][56] = 152;
    out.transformMaskMappingLocalToWorld[42][57] = 156;
    out.transformMaskMappingLocalToWorld[42][58] = 216;
    out.transformMaskMappingLocalToWorld[42][59] = 220;
    out.transformMaskMappingLocalToWorld[42][60] = 153;
    out.transformMaskMappingLocalToWorld[42][61] = 157;
    out.transformMaskMappingLocalToWorld[42][62] = 217;
    out.transformMaskMappingLocalToWorld[42][63] = 221;
    out.transformMaskMappingLocalToWorld[42][64] = 2;
    out.transformMaskMappingLocalToWorld[42][65] = 6;
    out.transformMaskMappingLocalToWorld[42][66] = 66;
    out.transformMaskMappingLocalToWorld[42][67] = 70;
    out.transformMaskMappingLocalToWorld[42][68] = 3;
    out.transformMaskMappingLocalToWorld[42][69] = 7;
    out.transformMaskMappingLocalToWorld[42][70] = 67;
    out.transformMaskMappingLocalToWorld[42][71] = 71;
    out.transformMaskMappingLocalToWorld[42][72] = 18;
    out.transformMaskMappingLocalToWorld[42][73] = 22;
    out.transformMaskMappingLocalToWorld[42][74] = 82;
    out.transformMaskMappingLocalToWorld[42][75] = 86;
    out.transformMaskMappingLocalToWorld[42][76] = 19;
    out.transformMaskMappingLocalToWorld[42][77] = 23;
    out.transformMaskMappingLocalToWorld[42][78] = 83;
    out.transformMaskMappingLocalToWorld[42][79] = 87;
    out.transformMaskMappingLocalToWorld[42][80] = 10;
    out.transformMaskMappingLocalToWorld[42][81] = 14;
    out.transformMaskMappingLocalToWorld[42][82] = 74;
    out.transformMaskMappingLocalToWorld[42][83] = 78;
    out.transformMaskMappingLocalToWorld[42][84] = 11;
    out.transformMaskMappingLocalToWorld[42][85] = 15;
    out.transformMaskMappingLocalToWorld[42][86] = 75;
    out.transformMaskMappingLocalToWorld[42][87] = 79;
    out.transformMaskMappingLocalToWorld[42][88] = 26;
    out.transformMaskMappingLocalToWorld[42][89] = 30;
    out.transformMaskMappingLocalToWorld[42][90] = 90;
    out.transformMaskMappingLocalToWorld[42][91] = 94;
    out.transformMaskMappingLocalToWorld[42][92] = 27;
    out.transformMaskMappingLocalToWorld[42][93] = 31;
    out.transformMaskMappingLocalToWorld[42][94] = 91;
    out.transformMaskMappingLocalToWorld[42][95] = 95;
    out.transformMaskMappingLocalToWorld[42][96] = 130;
    out.transformMaskMappingLocalToWorld[42][97] = 134;
    out.transformMaskMappingLocalToWorld[42][98] = 194;
    out.transformMaskMappingLocalToWorld[42][99] = 198;
    out.transformMaskMappingLocalToWorld[42][100] = 131;
    out.transformMaskMappingLocalToWorld[42][101] = 135;
    out.transformMaskMappingLocalToWorld[42][102] = 195;
    out.transformMaskMappingLocalToWorld[42][103] = 199;
    out.transformMaskMappingLocalToWorld[42][104] = 146;
    out.transformMaskMappingLocalToWorld[42][105] = 150;
    out.transformMaskMappingLocalToWorld[42][106] = 210;
    out.transformMaskMappingLocalToWorld[42][107] = 214;
    out.transformMaskMappingLocalToWorld[42][108] = 147;
    out.transformMaskMappingLocalToWorld[42][109] = 151;
    out.transformMaskMappingLocalToWorld[42][110] = 211;
    out.transformMaskMappingLocalToWorld[42][111] = 215;
    out.transformMaskMappingLocalToWorld[42][112] = 138;
    out.transformMaskMappingLocalToWorld[42][113] = 142;
    out.transformMaskMappingLocalToWorld[42][114] = 202;
    out.transformMaskMappingLocalToWorld[42][115] = 206;
    out.transformMaskMappingLocalToWorld[42][116] = 139;
    out.transformMaskMappingLocalToWorld[42][117] = 143;
    out.transformMaskMappingLocalToWorld[42][118] = 203;
    out.transformMaskMappingLocalToWorld[42][119] = 207;
    out.transformMaskMappingLocalToWorld[42][120] = 154;
    out.transformMaskMappingLocalToWorld[42][121] = 158;
    out.transformMaskMappingLocalToWorld[42][122] = 218;
    out.transformMaskMappingLocalToWorld[42][123] = 222;
    out.transformMaskMappingLocalToWorld[42][124] = 155;
    out.transformMaskMappingLocalToWorld[42][125] = 159;
    out.transformMaskMappingLocalToWorld[42][126] = 219;
    out.transformMaskMappingLocalToWorld[42][127] = 223;
    out.transformMaskMappingLocalToWorld[42][128] = 32;
    out.transformMaskMappingLocalToWorld[42][129] = 36;
    out.transformMaskMappingLocalToWorld[42][130] = 96;
    out.transformMaskMappingLocalToWorld[42][131] = 100;
    out.transformMaskMappingLocalToWorld[42][132] = 33;
    out.transformMaskMappingLocalToWorld[42][133] = 37;
    out.transformMaskMappingLocalToWorld[42][134] = 97;
    out.transformMaskMappingLocalToWorld[42][135] = 101;
    out.transformMaskMappingLocalToWorld[42][136] = 48;
    out.transformMaskMappingLocalToWorld[42][137] = 52;
    out.transformMaskMappingLocalToWorld[42][138] = 112;
    out.transformMaskMappingLocalToWorld[42][139] = 116;
    out.transformMaskMappingLocalToWorld[42][140] = 49;
    out.transformMaskMappingLocalToWorld[42][141] = 53;
    out.transformMaskMappingLocalToWorld[42][142] = 113;
    out.transformMaskMappingLocalToWorld[42][143] = 117;
    out.transformMaskMappingLocalToWorld[42][144] = 40;
    out.transformMaskMappingLocalToWorld[42][145] = 44;
    out.transformMaskMappingLocalToWorld[42][146] = 104;
    out.transformMaskMappingLocalToWorld[42][147] = 108;
    out.transformMaskMappingLocalToWorld[42][148] = 41;
    out.transformMaskMappingLocalToWorld[42][149] = 45;
    out.transformMaskMappingLocalToWorld[42][150] = 105;
    out.transformMaskMappingLocalToWorld[42][151] = 109;
    out.transformMaskMappingLocalToWorld[42][152] = 56;
    out.transformMaskMappingLocalToWorld[42][153] = 60;
    out.transformMaskMappingLocalToWorld[42][154] = 120;
    out.transformMaskMappingLocalToWorld[42][155] = 124;
    out.transformMaskMappingLocalToWorld[42][156] = 57;
    out.transformMaskMappingLocalToWorld[42][157] = 61;
    out.transformMaskMappingLocalToWorld[42][158] = 121;
    out.transformMaskMappingLocalToWorld[42][159] = 125;
    out.transformMaskMappingLocalToWorld[42][160] = 160;
    out.transformMaskMappingLocalToWorld[42][161] = 164;
    out.transformMaskMappingLocalToWorld[42][162] = 224;
    out.transformMaskMappingLocalToWorld[42][163] = 228;
    out.transformMaskMappingLocalToWorld[42][164] = 161;
    out.transformMaskMappingLocalToWorld[42][165] = 165;
    out.transformMaskMappingLocalToWorld[42][166] = 225;
    out.transformMaskMappingLocalToWorld[42][167] = 229;
    out.transformMaskMappingLocalToWorld[42][168] = 176;
    out.transformMaskMappingLocalToWorld[42][169] = 180;
    out.transformMaskMappingLocalToWorld[42][170] = 240;
    out.transformMaskMappingLocalToWorld[42][171] = 244;
    out.transformMaskMappingLocalToWorld[42][172] = 177;
    out.transformMaskMappingLocalToWorld[42][173] = 181;
    out.transformMaskMappingLocalToWorld[42][174] = 241;
    out.transformMaskMappingLocalToWorld[42][175] = 245;
    out.transformMaskMappingLocalToWorld[42][176] = 168;
    out.transformMaskMappingLocalToWorld[42][177] = 172;
    out.transformMaskMappingLocalToWorld[42][178] = 232;
    out.transformMaskMappingLocalToWorld[42][179] = 236;
    out.transformMaskMappingLocalToWorld[42][180] = 169;
    out.transformMaskMappingLocalToWorld[42][181] = 173;
    out.transformMaskMappingLocalToWorld[42][182] = 233;
    out.transformMaskMappingLocalToWorld[42][183] = 237;
    out.transformMaskMappingLocalToWorld[42][184] = 184;
    out.transformMaskMappingLocalToWorld[42][185] = 188;
    out.transformMaskMappingLocalToWorld[42][186] = 248;
    out.transformMaskMappingLocalToWorld[42][187] = 252;
    out.transformMaskMappingLocalToWorld[42][188] = 185;
    out.transformMaskMappingLocalToWorld[42][189] = 189;
    out.transformMaskMappingLocalToWorld[42][190] = 249;
    out.transformMaskMappingLocalToWorld[42][191] = 253;
    out.transformMaskMappingLocalToWorld[42][192] = 34;
    out.transformMaskMappingLocalToWorld[42][193] = 38;
    out.transformMaskMappingLocalToWorld[42][194] = 98;
    out.transformMaskMappingLocalToWorld[42][195] = 102;
    out.transformMaskMappingLocalToWorld[42][196] = 35;
    out.transformMaskMappingLocalToWorld[42][197] = 39;
    out.transformMaskMappingLocalToWorld[42][198] = 99;
    out.transformMaskMappingLocalToWorld[42][199] = 103;
    out.transformMaskMappingLocalToWorld[42][200] = 50;
    out.transformMaskMappingLocalToWorld[42][201] = 54;
    out.transformMaskMappingLocalToWorld[42][202] = 114;
    out.transformMaskMappingLocalToWorld[42][203] = 118;
    out.transformMaskMappingLocalToWorld[42][204] = 51;
    out.transformMaskMappingLocalToWorld[42][205] = 55;
    out.transformMaskMappingLocalToWorld[42][206] = 115;
    out.transformMaskMappingLocalToWorld[42][207] = 119;
    out.transformMaskMappingLocalToWorld[42][208] = 42;
    out.transformMaskMappingLocalToWorld[42][209] = 46;
    out.transformMaskMappingLocalToWorld[42][210] = 106;
    out.transformMaskMappingLocalToWorld[42][211] = 110;
    out.transformMaskMappingLocalToWorld[42][212] = 43;
    out.transformMaskMappingLocalToWorld[42][213] = 47;
    out.transformMaskMappingLocalToWorld[42][214] = 107;
    out.transformMaskMappingLocalToWorld[42][215] = 111;
    out.transformMaskMappingLocalToWorld[42][216] = 58;
    out.transformMaskMappingLocalToWorld[42][217] = 62;
    out.transformMaskMappingLocalToWorld[42][218] = 122;
    out.transformMaskMappingLocalToWorld[42][219] = 126;
    out.transformMaskMappingLocalToWorld[42][220] = 59;
    out.transformMaskMappingLocalToWorld[42][221] = 63;
    out.transformMaskMappingLocalToWorld[42][222] = 123;
    out.transformMaskMappingLocalToWorld[42][223] = 127;
    out.transformMaskMappingLocalToWorld[42][224] = 162;
    out.transformMaskMappingLocalToWorld[42][225] = 166;
    out.transformMaskMappingLocalToWorld[42][226] = 226;
    out.transformMaskMappingLocalToWorld[42][227] = 230;
    out.transformMaskMappingLocalToWorld[42][228] = 163;
    out.transformMaskMappingLocalToWorld[42][229] = 167;
    out.transformMaskMappingLocalToWorld[42][230] = 227;
    out.transformMaskMappingLocalToWorld[42][231] = 231;
    out.transformMaskMappingLocalToWorld[42][232] = 178;
    out.transformMaskMappingLocalToWorld[42][233] = 182;
    out.transformMaskMappingLocalToWorld[42][234] = 242;
    out.transformMaskMappingLocalToWorld[42][235] = 246;
    out.transformMaskMappingLocalToWorld[42][236] = 179;
    out.transformMaskMappingLocalToWorld[42][237] = 183;
    out.transformMaskMappingLocalToWorld[42][238] = 243;
    out.transformMaskMappingLocalToWorld[42][239] = 247;
    out.transformMaskMappingLocalToWorld[42][240] = 170;
    out.transformMaskMappingLocalToWorld[42][241] = 174;
    out.transformMaskMappingLocalToWorld[42][242] = 234;
    out.transformMaskMappingLocalToWorld[42][243] = 238;
    out.transformMaskMappingLocalToWorld[42][244] = 171;
    out.transformMaskMappingLocalToWorld[42][245] = 175;
    out.transformMaskMappingLocalToWorld[42][246] = 235;
    out.transformMaskMappingLocalToWorld[42][247] = 239;
    out.transformMaskMappingLocalToWorld[42][248] = 186;
    out.transformMaskMappingLocalToWorld[42][249] = 190;
    out.transformMaskMappingLocalToWorld[42][250] = 250;
    out.transformMaskMappingLocalToWorld[42][251] = 254;
    out.transformMaskMappingLocalToWorld[42][252] = 187;
    out.transformMaskMappingLocalToWorld[42][253] = 191;
    out.transformMaskMappingLocalToWorld[42][254] = 251;
    out.transformMaskMappingLocalToWorld[42][255] = 255;
    out.transformMaskMappingLocalToWorld[43][0] = 0;
    out.transformMaskMappingLocalToWorld[43][1] = 8;
    out.transformMaskMappingLocalToWorld[43][2] = 128;
    out.transformMaskMappingLocalToWorld[43][3] = 136;
    out.transformMaskMappingLocalToWorld[43][4] = 2;
    out.transformMaskMappingLocalToWorld[43][5] = 10;
    out.transformMaskMappingLocalToWorld[43][6] = 130;
    out.transformMaskMappingLocalToWorld[43][7] = 138;
    out.transformMaskMappingLocalToWorld[43][8] = 32;
    out.transformMaskMappingLocalToWorld[43][9] = 40;
    out.transformMaskMappingLocalToWorld[43][10] = 160;
    out.transformMaskMappingLocalToWorld[43][11] = 168;
    out.transformMaskMappingLocalToWorld[43][12] = 34;
    out.transformMaskMappingLocalToWorld[43][13] = 42;
    out.transformMaskMappingLocalToWorld[43][14] = 162;
    out.transformMaskMappingLocalToWorld[43][15] = 170;
    out.transformMaskMappingLocalToWorld[43][16] = 4;
    out.transformMaskMappingLocalToWorld[43][17] = 12;
    out.transformMaskMappingLocalToWorld[43][18] = 132;
    out.transformMaskMappingLocalToWorld[43][19] = 140;
    out.transformMaskMappingLocalToWorld[43][20] = 6;
    out.transformMaskMappingLocalToWorld[43][21] = 14;
    out.transformMaskMappingLocalToWorld[43][22] = 134;
    out.transformMaskMappingLocalToWorld[43][23] = 142;
    out.transformMaskMappingLocalToWorld[43][24] = 36;
    out.transformMaskMappingLocalToWorld[43][25] = 44;
    out.transformMaskMappingLocalToWorld[43][26] = 164;
    out.transformMaskMappingLocalToWorld[43][27] = 172;
    out.transformMaskMappingLocalToWorld[43][28] = 38;
    out.transformMaskMappingLocalToWorld[43][29] = 46;
    out.transformMaskMappingLocalToWorld[43][30] = 166;
    out.transformMaskMappingLocalToWorld[43][31] = 174;
    out.transformMaskMappingLocalToWorld[43][32] = 64;
    out.transformMaskMappingLocalToWorld[43][33] = 72;
    out.transformMaskMappingLocalToWorld[43][34] = 192;
    out.transformMaskMappingLocalToWorld[43][35] = 200;
    out.transformMaskMappingLocalToWorld[43][36] = 66;
    out.transformMaskMappingLocalToWorld[43][37] = 74;
    out.transformMaskMappingLocalToWorld[43][38] = 194;
    out.transformMaskMappingLocalToWorld[43][39] = 202;
    out.transformMaskMappingLocalToWorld[43][40] = 96;
    out.transformMaskMappingLocalToWorld[43][41] = 104;
    out.transformMaskMappingLocalToWorld[43][42] = 224;
    out.transformMaskMappingLocalToWorld[43][43] = 232;
    out.transformMaskMappingLocalToWorld[43][44] = 98;
    out.transformMaskMappingLocalToWorld[43][45] = 106;
    out.transformMaskMappingLocalToWorld[43][46] = 226;
    out.transformMaskMappingLocalToWorld[43][47] = 234;
    out.transformMaskMappingLocalToWorld[43][48] = 68;
    out.transformMaskMappingLocalToWorld[43][49] = 76;
    out.transformMaskMappingLocalToWorld[43][50] = 196;
    out.transformMaskMappingLocalToWorld[43][51] = 204;
    out.transformMaskMappingLocalToWorld[43][52] = 70;
    out.transformMaskMappingLocalToWorld[43][53] = 78;
    out.transformMaskMappingLocalToWorld[43][54] = 198;
    out.transformMaskMappingLocalToWorld[43][55] = 206;
    out.transformMaskMappingLocalToWorld[43][56] = 100;
    out.transformMaskMappingLocalToWorld[43][57] = 108;
    out.transformMaskMappingLocalToWorld[43][58] = 228;
    out.transformMaskMappingLocalToWorld[43][59] = 236;
    out.transformMaskMappingLocalToWorld[43][60] = 102;
    out.transformMaskMappingLocalToWorld[43][61] = 110;
    out.transformMaskMappingLocalToWorld[43][62] = 230;
    out.transformMaskMappingLocalToWorld[43][63] = 238;
    out.transformMaskMappingLocalToWorld[43][64] = 1;
    out.transformMaskMappingLocalToWorld[43][65] = 9;
    out.transformMaskMappingLocalToWorld[43][66] = 129;
    out.transformMaskMappingLocalToWorld[43][67] = 137;
    out.transformMaskMappingLocalToWorld[43][68] = 3;
    out.transformMaskMappingLocalToWorld[43][69] = 11;
    out.transformMaskMappingLocalToWorld[43][70] = 131;
    out.transformMaskMappingLocalToWorld[43][71] = 139;
    out.transformMaskMappingLocalToWorld[43][72] = 33;
    out.transformMaskMappingLocalToWorld[43][73] = 41;
    out.transformMaskMappingLocalToWorld[43][74] = 161;
    out.transformMaskMappingLocalToWorld[43][75] = 169;
    out.transformMaskMappingLocalToWorld[43][76] = 35;
    out.transformMaskMappingLocalToWorld[43][77] = 43;
    out.transformMaskMappingLocalToWorld[43][78] = 163;
    out.transformMaskMappingLocalToWorld[43][79] = 171;
    out.transformMaskMappingLocalToWorld[43][80] = 5;
    out.transformMaskMappingLocalToWorld[43][81] = 13;
    out.transformMaskMappingLocalToWorld[43][82] = 133;
    out.transformMaskMappingLocalToWorld[43][83] = 141;
    out.transformMaskMappingLocalToWorld[43][84] = 7;
    out.transformMaskMappingLocalToWorld[43][85] = 15;
    out.transformMaskMappingLocalToWorld[43][86] = 135;
    out.transformMaskMappingLocalToWorld[43][87] = 143;
    out.transformMaskMappingLocalToWorld[43][88] = 37;
    out.transformMaskMappingLocalToWorld[43][89] = 45;
    out.transformMaskMappingLocalToWorld[43][90] = 165;
    out.transformMaskMappingLocalToWorld[43][91] = 173;
    out.transformMaskMappingLocalToWorld[43][92] = 39;
    out.transformMaskMappingLocalToWorld[43][93] = 47;
    out.transformMaskMappingLocalToWorld[43][94] = 167;
    out.transformMaskMappingLocalToWorld[43][95] = 175;
    out.transformMaskMappingLocalToWorld[43][96] = 65;
    out.transformMaskMappingLocalToWorld[43][97] = 73;
    out.transformMaskMappingLocalToWorld[43][98] = 193;
    out.transformMaskMappingLocalToWorld[43][99] = 201;
    out.transformMaskMappingLocalToWorld[43][100] = 67;
    out.transformMaskMappingLocalToWorld[43][101] = 75;
    out.transformMaskMappingLocalToWorld[43][102] = 195;
    out.transformMaskMappingLocalToWorld[43][103] = 203;
    out.transformMaskMappingLocalToWorld[43][104] = 97;
    out.transformMaskMappingLocalToWorld[43][105] = 105;
    out.transformMaskMappingLocalToWorld[43][106] = 225;
    out.transformMaskMappingLocalToWorld[43][107] = 233;
    out.transformMaskMappingLocalToWorld[43][108] = 99;
    out.transformMaskMappingLocalToWorld[43][109] = 107;
    out.transformMaskMappingLocalToWorld[43][110] = 227;
    out.transformMaskMappingLocalToWorld[43][111] = 235;
    out.transformMaskMappingLocalToWorld[43][112] = 69;
    out.transformMaskMappingLocalToWorld[43][113] = 77;
    out.transformMaskMappingLocalToWorld[43][114] = 197;
    out.transformMaskMappingLocalToWorld[43][115] = 205;
    out.transformMaskMappingLocalToWorld[43][116] = 71;
    out.transformMaskMappingLocalToWorld[43][117] = 79;
    out.transformMaskMappingLocalToWorld[43][118] = 199;
    out.transformMaskMappingLocalToWorld[43][119] = 207;
    out.transformMaskMappingLocalToWorld[43][120] = 101;
    out.transformMaskMappingLocalToWorld[43][121] = 109;
    out.transformMaskMappingLocalToWorld[43][122] = 229;
    out.transformMaskMappingLocalToWorld[43][123] = 237;
    out.transformMaskMappingLocalToWorld[43][124] = 103;
    out.transformMaskMappingLocalToWorld[43][125] = 111;
    out.transformMaskMappingLocalToWorld[43][126] = 231;
    out.transformMaskMappingLocalToWorld[43][127] = 239;
    out.transformMaskMappingLocalToWorld[43][128] = 16;
    out.transformMaskMappingLocalToWorld[43][129] = 24;
    out.transformMaskMappingLocalToWorld[43][130] = 144;
    out.transformMaskMappingLocalToWorld[43][131] = 152;
    out.transformMaskMappingLocalToWorld[43][132] = 18;
    out.transformMaskMappingLocalToWorld[43][133] = 26;
    out.transformMaskMappingLocalToWorld[43][134] = 146;
    out.transformMaskMappingLocalToWorld[43][135] = 154;
    out.transformMaskMappingLocalToWorld[43][136] = 48;
    out.transformMaskMappingLocalToWorld[43][137] = 56;
    out.transformMaskMappingLocalToWorld[43][138] = 176;
    out.transformMaskMappingLocalToWorld[43][139] = 184;
    out.transformMaskMappingLocalToWorld[43][140] = 50;
    out.transformMaskMappingLocalToWorld[43][141] = 58;
    out.transformMaskMappingLocalToWorld[43][142] = 178;
    out.transformMaskMappingLocalToWorld[43][143] = 186;
    out.transformMaskMappingLocalToWorld[43][144] = 20;
    out.transformMaskMappingLocalToWorld[43][145] = 28;
    out.transformMaskMappingLocalToWorld[43][146] = 148;
    out.transformMaskMappingLocalToWorld[43][147] = 156;
    out.transformMaskMappingLocalToWorld[43][148] = 22;
    out.transformMaskMappingLocalToWorld[43][149] = 30;
    out.transformMaskMappingLocalToWorld[43][150] = 150;
    out.transformMaskMappingLocalToWorld[43][151] = 158;
    out.transformMaskMappingLocalToWorld[43][152] = 52;
    out.transformMaskMappingLocalToWorld[43][153] = 60;
    out.transformMaskMappingLocalToWorld[43][154] = 180;
    out.transformMaskMappingLocalToWorld[43][155] = 188;
    out.transformMaskMappingLocalToWorld[43][156] = 54;
    out.transformMaskMappingLocalToWorld[43][157] = 62;
    out.transformMaskMappingLocalToWorld[43][158] = 182;
    out.transformMaskMappingLocalToWorld[43][159] = 190;
    out.transformMaskMappingLocalToWorld[43][160] = 80;
    out.transformMaskMappingLocalToWorld[43][161] = 88;
    out.transformMaskMappingLocalToWorld[43][162] = 208;
    out.transformMaskMappingLocalToWorld[43][163] = 216;
    out.transformMaskMappingLocalToWorld[43][164] = 82;
    out.transformMaskMappingLocalToWorld[43][165] = 90;
    out.transformMaskMappingLocalToWorld[43][166] = 210;
    out.transformMaskMappingLocalToWorld[43][167] = 218;
    out.transformMaskMappingLocalToWorld[43][168] = 112;
    out.transformMaskMappingLocalToWorld[43][169] = 120;
    out.transformMaskMappingLocalToWorld[43][170] = 240;
    out.transformMaskMappingLocalToWorld[43][171] = 248;
    out.transformMaskMappingLocalToWorld[43][172] = 114;
    out.transformMaskMappingLocalToWorld[43][173] = 122;
    out.transformMaskMappingLocalToWorld[43][174] = 242;
    out.transformMaskMappingLocalToWorld[43][175] = 250;
    out.transformMaskMappingLocalToWorld[43][176] = 84;
    out.transformMaskMappingLocalToWorld[43][177] = 92;
    out.transformMaskMappingLocalToWorld[43][178] = 212;
    out.transformMaskMappingLocalToWorld[43][179] = 220;
    out.transformMaskMappingLocalToWorld[43][180] = 86;
    out.transformMaskMappingLocalToWorld[43][181] = 94;
    out.transformMaskMappingLocalToWorld[43][182] = 214;
    out.transformMaskMappingLocalToWorld[43][183] = 222;
    out.transformMaskMappingLocalToWorld[43][184] = 116;
    out.transformMaskMappingLocalToWorld[43][185] = 124;
    out.transformMaskMappingLocalToWorld[43][186] = 244;
    out.transformMaskMappingLocalToWorld[43][187] = 252;
    out.transformMaskMappingLocalToWorld[43][188] = 118;
    out.transformMaskMappingLocalToWorld[43][189] = 126;
    out.transformMaskMappingLocalToWorld[43][190] = 246;
    out.transformMaskMappingLocalToWorld[43][191] = 254;
    out.transformMaskMappingLocalToWorld[43][192] = 17;
    out.transformMaskMappingLocalToWorld[43][193] = 25;
    out.transformMaskMappingLocalToWorld[43][194] = 145;
    out.transformMaskMappingLocalToWorld[43][195] = 153;
    out.transformMaskMappingLocalToWorld[43][196] = 19;
    out.transformMaskMappingLocalToWorld[43][197] = 27;
    out.transformMaskMappingLocalToWorld[43][198] = 147;
    out.transformMaskMappingLocalToWorld[43][199] = 155;
    out.transformMaskMappingLocalToWorld[43][200] = 49;
    out.transformMaskMappingLocalToWorld[43][201] = 57;
    out.transformMaskMappingLocalToWorld[43][202] = 177;
    out.transformMaskMappingLocalToWorld[43][203] = 185;
    out.transformMaskMappingLocalToWorld[43][204] = 51;
    out.transformMaskMappingLocalToWorld[43][205] = 59;
    out.transformMaskMappingLocalToWorld[43][206] = 179;
    out.transformMaskMappingLocalToWorld[43][207] = 187;
    out.transformMaskMappingLocalToWorld[43][208] = 21;
    out.transformMaskMappingLocalToWorld[43][209] = 29;
    out.transformMaskMappingLocalToWorld[43][210] = 149;
    out.transformMaskMappingLocalToWorld[43][211] = 157;
    out.transformMaskMappingLocalToWorld[43][212] = 23;
    out.transformMaskMappingLocalToWorld[43][213] = 31;
    out.transformMaskMappingLocalToWorld[43][214] = 151;
    out.transformMaskMappingLocalToWorld[43][215] = 159;
    out.transformMaskMappingLocalToWorld[43][216] = 53;
    out.transformMaskMappingLocalToWorld[43][217] = 61;
    out.transformMaskMappingLocalToWorld[43][218] = 181;
    out.transformMaskMappingLocalToWorld[43][219] = 189;
    out.transformMaskMappingLocalToWorld[43][220] = 55;
    out.transformMaskMappingLocalToWorld[43][221] = 63;
    out.transformMaskMappingLocalToWorld[43][222] = 183;
    out.transformMaskMappingLocalToWorld[43][223] = 191;
    out.transformMaskMappingLocalToWorld[43][224] = 81;
    out.transformMaskMappingLocalToWorld[43][225] = 89;
    out.transformMaskMappingLocalToWorld[43][226] = 209;
    out.transformMaskMappingLocalToWorld[43][227] = 217;
    out.transformMaskMappingLocalToWorld[43][228] = 83;
    out.transformMaskMappingLocalToWorld[43][229] = 91;
    out.transformMaskMappingLocalToWorld[43][230] = 211;
    out.transformMaskMappingLocalToWorld[43][231] = 219;
    out.transformMaskMappingLocalToWorld[43][232] = 113;
    out.transformMaskMappingLocalToWorld[43][233] = 121;
    out.transformMaskMappingLocalToWorld[43][234] = 241;
    out.transformMaskMappingLocalToWorld[43][235] = 249;
    out.transformMaskMappingLocalToWorld[43][236] = 115;
    out.transformMaskMappingLocalToWorld[43][237] = 123;
    out.transformMaskMappingLocalToWorld[43][238] = 243;
    out.transformMaskMappingLocalToWorld[43][239] = 251;
    out.transformMaskMappingLocalToWorld[43][240] = 85;
    out.transformMaskMappingLocalToWorld[43][241] = 93;
    out.transformMaskMappingLocalToWorld[43][242] = 213;
    out.transformMaskMappingLocalToWorld[43][243] = 221;
    out.transformMaskMappingLocalToWorld[43][244] = 87;
    out.transformMaskMappingLocalToWorld[43][245] = 95;
    out.transformMaskMappingLocalToWorld[43][246] = 215;
    out.transformMaskMappingLocalToWorld[43][247] = 223;
    out.transformMaskMappingLocalToWorld[43][248] = 117;
    out.transformMaskMappingLocalToWorld[43][249] = 125;
    out.transformMaskMappingLocalToWorld[43][250] = 245;
    out.transformMaskMappingLocalToWorld[43][251] = 253;
    out.transformMaskMappingLocalToWorld[43][252] = 119;
    out.transformMaskMappingLocalToWorld[43][253] = 127;
    out.transformMaskMappingLocalToWorld[43][254] = 247;
    out.transformMaskMappingLocalToWorld[43][255] = 255;
    out.transformMaskMappingLocalToWorld[44][0] = 0;
    out.transformMaskMappingLocalToWorld[44][1] = 16;
    out.transformMaskMappingLocalToWorld[44][2] = 1;
    out.transformMaskMappingLocalToWorld[44][3] = 17;
    out.transformMaskMappingLocalToWorld[44][4] = 64;
    out.transformMaskMappingLocalToWorld[44][5] = 80;
    out.transformMaskMappingLocalToWorld[44][6] = 65;
    out.transformMaskMappingLocalToWorld[44][7] = 81;
    out.transformMaskMappingLocalToWorld[44][8] = 4;
    out.transformMaskMappingLocalToWorld[44][9] = 20;
    out.transformMaskMappingLocalToWorld[44][10] = 5;
    out.transformMaskMappingLocalToWorld[44][11] = 21;
    out.transformMaskMappingLocalToWorld[44][12] = 68;
    out.transformMaskMappingLocalToWorld[44][13] = 84;
    out.transformMaskMappingLocalToWorld[44][14] = 69;
    out.transformMaskMappingLocalToWorld[44][15] = 85;
    out.transformMaskMappingLocalToWorld[44][16] = 32;
    out.transformMaskMappingLocalToWorld[44][17] = 48;
    out.transformMaskMappingLocalToWorld[44][18] = 33;
    out.transformMaskMappingLocalToWorld[44][19] = 49;
    out.transformMaskMappingLocalToWorld[44][20] = 96;
    out.transformMaskMappingLocalToWorld[44][21] = 112;
    out.transformMaskMappingLocalToWorld[44][22] = 97;
    out.transformMaskMappingLocalToWorld[44][23] = 113;
    out.transformMaskMappingLocalToWorld[44][24] = 36;
    out.transformMaskMappingLocalToWorld[44][25] = 52;
    out.transformMaskMappingLocalToWorld[44][26] = 37;
    out.transformMaskMappingLocalToWorld[44][27] = 53;
    out.transformMaskMappingLocalToWorld[44][28] = 100;
    out.transformMaskMappingLocalToWorld[44][29] = 116;
    out.transformMaskMappingLocalToWorld[44][30] = 101;
    out.transformMaskMappingLocalToWorld[44][31] = 117;
    out.transformMaskMappingLocalToWorld[44][32] = 2;
    out.transformMaskMappingLocalToWorld[44][33] = 18;
    out.transformMaskMappingLocalToWorld[44][34] = 3;
    out.transformMaskMappingLocalToWorld[44][35] = 19;
    out.transformMaskMappingLocalToWorld[44][36] = 66;
    out.transformMaskMappingLocalToWorld[44][37] = 82;
    out.transformMaskMappingLocalToWorld[44][38] = 67;
    out.transformMaskMappingLocalToWorld[44][39] = 83;
    out.transformMaskMappingLocalToWorld[44][40] = 6;
    out.transformMaskMappingLocalToWorld[44][41] = 22;
    out.transformMaskMappingLocalToWorld[44][42] = 7;
    out.transformMaskMappingLocalToWorld[44][43] = 23;
    out.transformMaskMappingLocalToWorld[44][44] = 70;
    out.transformMaskMappingLocalToWorld[44][45] = 86;
    out.transformMaskMappingLocalToWorld[44][46] = 71;
    out.transformMaskMappingLocalToWorld[44][47] = 87;
    out.transformMaskMappingLocalToWorld[44][48] = 34;
    out.transformMaskMappingLocalToWorld[44][49] = 50;
    out.transformMaskMappingLocalToWorld[44][50] = 35;
    out.transformMaskMappingLocalToWorld[44][51] = 51;
    out.transformMaskMappingLocalToWorld[44][52] = 98;
    out.transformMaskMappingLocalToWorld[44][53] = 114;
    out.transformMaskMappingLocalToWorld[44][54] = 99;
    out.transformMaskMappingLocalToWorld[44][55] = 115;
    out.transformMaskMappingLocalToWorld[44][56] = 38;
    out.transformMaskMappingLocalToWorld[44][57] = 54;
    out.transformMaskMappingLocalToWorld[44][58] = 39;
    out.transformMaskMappingLocalToWorld[44][59] = 55;
    out.transformMaskMappingLocalToWorld[44][60] = 102;
    out.transformMaskMappingLocalToWorld[44][61] = 118;
    out.transformMaskMappingLocalToWorld[44][62] = 103;
    out.transformMaskMappingLocalToWorld[44][63] = 119;
    out.transformMaskMappingLocalToWorld[44][64] = 128;
    out.transformMaskMappingLocalToWorld[44][65] = 144;
    out.transformMaskMappingLocalToWorld[44][66] = 129;
    out.transformMaskMappingLocalToWorld[44][67] = 145;
    out.transformMaskMappingLocalToWorld[44][68] = 192;
    out.transformMaskMappingLocalToWorld[44][69] = 208;
    out.transformMaskMappingLocalToWorld[44][70] = 193;
    out.transformMaskMappingLocalToWorld[44][71] = 209;
    out.transformMaskMappingLocalToWorld[44][72] = 132;
    out.transformMaskMappingLocalToWorld[44][73] = 148;
    out.transformMaskMappingLocalToWorld[44][74] = 133;
    out.transformMaskMappingLocalToWorld[44][75] = 149;
    out.transformMaskMappingLocalToWorld[44][76] = 196;
    out.transformMaskMappingLocalToWorld[44][77] = 212;
    out.transformMaskMappingLocalToWorld[44][78] = 197;
    out.transformMaskMappingLocalToWorld[44][79] = 213;
    out.transformMaskMappingLocalToWorld[44][80] = 160;
    out.transformMaskMappingLocalToWorld[44][81] = 176;
    out.transformMaskMappingLocalToWorld[44][82] = 161;
    out.transformMaskMappingLocalToWorld[44][83] = 177;
    out.transformMaskMappingLocalToWorld[44][84] = 224;
    out.transformMaskMappingLocalToWorld[44][85] = 240;
    out.transformMaskMappingLocalToWorld[44][86] = 225;
    out.transformMaskMappingLocalToWorld[44][87] = 241;
    out.transformMaskMappingLocalToWorld[44][88] = 164;
    out.transformMaskMappingLocalToWorld[44][89] = 180;
    out.transformMaskMappingLocalToWorld[44][90] = 165;
    out.transformMaskMappingLocalToWorld[44][91] = 181;
    out.transformMaskMappingLocalToWorld[44][92] = 228;
    out.transformMaskMappingLocalToWorld[44][93] = 244;
    out.transformMaskMappingLocalToWorld[44][94] = 229;
    out.transformMaskMappingLocalToWorld[44][95] = 245;
    out.transformMaskMappingLocalToWorld[44][96] = 130;
    out.transformMaskMappingLocalToWorld[44][97] = 146;
    out.transformMaskMappingLocalToWorld[44][98] = 131;
    out.transformMaskMappingLocalToWorld[44][99] = 147;
    out.transformMaskMappingLocalToWorld[44][100] = 194;
    out.transformMaskMappingLocalToWorld[44][101] = 210;
    out.transformMaskMappingLocalToWorld[44][102] = 195;
    out.transformMaskMappingLocalToWorld[44][103] = 211;
    out.transformMaskMappingLocalToWorld[44][104] = 134;
    out.transformMaskMappingLocalToWorld[44][105] = 150;
    out.transformMaskMappingLocalToWorld[44][106] = 135;
    out.transformMaskMappingLocalToWorld[44][107] = 151;
    out.transformMaskMappingLocalToWorld[44][108] = 198;
    out.transformMaskMappingLocalToWorld[44][109] = 214;
    out.transformMaskMappingLocalToWorld[44][110] = 199;
    out.transformMaskMappingLocalToWorld[44][111] = 215;
    out.transformMaskMappingLocalToWorld[44][112] = 162;
    out.transformMaskMappingLocalToWorld[44][113] = 178;
    out.transformMaskMappingLocalToWorld[44][114] = 163;
    out.transformMaskMappingLocalToWorld[44][115] = 179;
    out.transformMaskMappingLocalToWorld[44][116] = 226;
    out.transformMaskMappingLocalToWorld[44][117] = 242;
    out.transformMaskMappingLocalToWorld[44][118] = 227;
    out.transformMaskMappingLocalToWorld[44][119] = 243;
    out.transformMaskMappingLocalToWorld[44][120] = 166;
    out.transformMaskMappingLocalToWorld[44][121] = 182;
    out.transformMaskMappingLocalToWorld[44][122] = 167;
    out.transformMaskMappingLocalToWorld[44][123] = 183;
    out.transformMaskMappingLocalToWorld[44][124] = 230;
    out.transformMaskMappingLocalToWorld[44][125] = 246;
    out.transformMaskMappingLocalToWorld[44][126] = 231;
    out.transformMaskMappingLocalToWorld[44][127] = 247;
    out.transformMaskMappingLocalToWorld[44][128] = 8;
    out.transformMaskMappingLocalToWorld[44][129] = 24;
    out.transformMaskMappingLocalToWorld[44][130] = 9;
    out.transformMaskMappingLocalToWorld[44][131] = 25;
    out.transformMaskMappingLocalToWorld[44][132] = 72;
    out.transformMaskMappingLocalToWorld[44][133] = 88;
    out.transformMaskMappingLocalToWorld[44][134] = 73;
    out.transformMaskMappingLocalToWorld[44][135] = 89;
    out.transformMaskMappingLocalToWorld[44][136] = 12;
    out.transformMaskMappingLocalToWorld[44][137] = 28;
    out.transformMaskMappingLocalToWorld[44][138] = 13;
    out.transformMaskMappingLocalToWorld[44][139] = 29;
    out.transformMaskMappingLocalToWorld[44][140] = 76;
    out.transformMaskMappingLocalToWorld[44][141] = 92;
    out.transformMaskMappingLocalToWorld[44][142] = 77;
    out.transformMaskMappingLocalToWorld[44][143] = 93;
    out.transformMaskMappingLocalToWorld[44][144] = 40;
    out.transformMaskMappingLocalToWorld[44][145] = 56;
    out.transformMaskMappingLocalToWorld[44][146] = 41;
    out.transformMaskMappingLocalToWorld[44][147] = 57;
    out.transformMaskMappingLocalToWorld[44][148] = 104;
    out.transformMaskMappingLocalToWorld[44][149] = 120;
    out.transformMaskMappingLocalToWorld[44][150] = 105;
    out.transformMaskMappingLocalToWorld[44][151] = 121;
    out.transformMaskMappingLocalToWorld[44][152] = 44;
    out.transformMaskMappingLocalToWorld[44][153] = 60;
    out.transformMaskMappingLocalToWorld[44][154] = 45;
    out.transformMaskMappingLocalToWorld[44][155] = 61;
    out.transformMaskMappingLocalToWorld[44][156] = 108;
    out.transformMaskMappingLocalToWorld[44][157] = 124;
    out.transformMaskMappingLocalToWorld[44][158] = 109;
    out.transformMaskMappingLocalToWorld[44][159] = 125;
    out.transformMaskMappingLocalToWorld[44][160] = 10;
    out.transformMaskMappingLocalToWorld[44][161] = 26;
    out.transformMaskMappingLocalToWorld[44][162] = 11;
    out.transformMaskMappingLocalToWorld[44][163] = 27;
    out.transformMaskMappingLocalToWorld[44][164] = 74;
    out.transformMaskMappingLocalToWorld[44][165] = 90;
    out.transformMaskMappingLocalToWorld[44][166] = 75;
    out.transformMaskMappingLocalToWorld[44][167] = 91;
    out.transformMaskMappingLocalToWorld[44][168] = 14;
    out.transformMaskMappingLocalToWorld[44][169] = 30;
    out.transformMaskMappingLocalToWorld[44][170] = 15;
    out.transformMaskMappingLocalToWorld[44][171] = 31;
    out.transformMaskMappingLocalToWorld[44][172] = 78;
    out.transformMaskMappingLocalToWorld[44][173] = 94;
    out.transformMaskMappingLocalToWorld[44][174] = 79;
    out.transformMaskMappingLocalToWorld[44][175] = 95;
    out.transformMaskMappingLocalToWorld[44][176] = 42;
    out.transformMaskMappingLocalToWorld[44][177] = 58;
    out.transformMaskMappingLocalToWorld[44][178] = 43;
    out.transformMaskMappingLocalToWorld[44][179] = 59;
    out.transformMaskMappingLocalToWorld[44][180] = 106;
    out.transformMaskMappingLocalToWorld[44][181] = 122;
    out.transformMaskMappingLocalToWorld[44][182] = 107;
    out.transformMaskMappingLocalToWorld[44][183] = 123;
    out.transformMaskMappingLocalToWorld[44][184] = 46;
    out.transformMaskMappingLocalToWorld[44][185] = 62;
    out.transformMaskMappingLocalToWorld[44][186] = 47;
    out.transformMaskMappingLocalToWorld[44][187] = 63;
    out.transformMaskMappingLocalToWorld[44][188] = 110;
    out.transformMaskMappingLocalToWorld[44][189] = 126;
    out.transformMaskMappingLocalToWorld[44][190] = 111;
    out.transformMaskMappingLocalToWorld[44][191] = 127;
    out.transformMaskMappingLocalToWorld[44][192] = 136;
    out.transformMaskMappingLocalToWorld[44][193] = 152;
    out.transformMaskMappingLocalToWorld[44][194] = 137;
    out.transformMaskMappingLocalToWorld[44][195] = 153;
    out.transformMaskMappingLocalToWorld[44][196] = 200;
    out.transformMaskMappingLocalToWorld[44][197] = 216;
    out.transformMaskMappingLocalToWorld[44][198] = 201;
    out.transformMaskMappingLocalToWorld[44][199] = 217;
    out.transformMaskMappingLocalToWorld[44][200] = 140;
    out.transformMaskMappingLocalToWorld[44][201] = 156;
    out.transformMaskMappingLocalToWorld[44][202] = 141;
    out.transformMaskMappingLocalToWorld[44][203] = 157;
    out.transformMaskMappingLocalToWorld[44][204] = 204;
    out.transformMaskMappingLocalToWorld[44][205] = 220;
    out.transformMaskMappingLocalToWorld[44][206] = 205;
    out.transformMaskMappingLocalToWorld[44][207] = 221;
    out.transformMaskMappingLocalToWorld[44][208] = 168;
    out.transformMaskMappingLocalToWorld[44][209] = 184;
    out.transformMaskMappingLocalToWorld[44][210] = 169;
    out.transformMaskMappingLocalToWorld[44][211] = 185;
    out.transformMaskMappingLocalToWorld[44][212] = 232;
    out.transformMaskMappingLocalToWorld[44][213] = 248;
    out.transformMaskMappingLocalToWorld[44][214] = 233;
    out.transformMaskMappingLocalToWorld[44][215] = 249;
    out.transformMaskMappingLocalToWorld[44][216] = 172;
    out.transformMaskMappingLocalToWorld[44][217] = 188;
    out.transformMaskMappingLocalToWorld[44][218] = 173;
    out.transformMaskMappingLocalToWorld[44][219] = 189;
    out.transformMaskMappingLocalToWorld[44][220] = 236;
    out.transformMaskMappingLocalToWorld[44][221] = 252;
    out.transformMaskMappingLocalToWorld[44][222] = 237;
    out.transformMaskMappingLocalToWorld[44][223] = 253;
    out.transformMaskMappingLocalToWorld[44][224] = 138;
    out.transformMaskMappingLocalToWorld[44][225] = 154;
    out.transformMaskMappingLocalToWorld[44][226] = 139;
    out.transformMaskMappingLocalToWorld[44][227] = 155;
    out.transformMaskMappingLocalToWorld[44][228] = 202;
    out.transformMaskMappingLocalToWorld[44][229] = 218;
    out.transformMaskMappingLocalToWorld[44][230] = 203;
    out.transformMaskMappingLocalToWorld[44][231] = 219;
    out.transformMaskMappingLocalToWorld[44][232] = 142;
    out.transformMaskMappingLocalToWorld[44][233] = 158;
    out.transformMaskMappingLocalToWorld[44][234] = 143;
    out.transformMaskMappingLocalToWorld[44][235] = 159;
    out.transformMaskMappingLocalToWorld[44][236] = 206;
    out.transformMaskMappingLocalToWorld[44][237] = 222;
    out.transformMaskMappingLocalToWorld[44][238] = 207;
    out.transformMaskMappingLocalToWorld[44][239] = 223;
    out.transformMaskMappingLocalToWorld[44][240] = 170;
    out.transformMaskMappingLocalToWorld[44][241] = 186;
    out.transformMaskMappingLocalToWorld[44][242] = 171;
    out.transformMaskMappingLocalToWorld[44][243] = 187;
    out.transformMaskMappingLocalToWorld[44][244] = 234;
    out.transformMaskMappingLocalToWorld[44][245] = 250;
    out.transformMaskMappingLocalToWorld[44][246] = 235;
    out.transformMaskMappingLocalToWorld[44][247] = 251;
    out.transformMaskMappingLocalToWorld[44][248] = 174;
    out.transformMaskMappingLocalToWorld[44][249] = 190;
    out.transformMaskMappingLocalToWorld[44][250] = 175;
    out.transformMaskMappingLocalToWorld[44][251] = 191;
    out.transformMaskMappingLocalToWorld[44][252] = 238;
    out.transformMaskMappingLocalToWorld[44][253] = 254;
    out.transformMaskMappingLocalToWorld[44][254] = 239;
    out.transformMaskMappingLocalToWorld[44][255] = 255;
    out.transformMaskMappingLocalToWorld[45][0] = 0;
    out.transformMaskMappingLocalToWorld[45][1] = 32;
    out.transformMaskMappingLocalToWorld[45][2] = 2;
    out.transformMaskMappingLocalToWorld[45][3] = 34;
    out.transformMaskMappingLocalToWorld[45][4] = 128;
    out.transformMaskMappingLocalToWorld[45][5] = 160;
    out.transformMaskMappingLocalToWorld[45][6] = 130;
    out.transformMaskMappingLocalToWorld[45][7] = 162;
    out.transformMaskMappingLocalToWorld[45][8] = 8;
    out.transformMaskMappingLocalToWorld[45][9] = 40;
    out.transformMaskMappingLocalToWorld[45][10] = 10;
    out.transformMaskMappingLocalToWorld[45][11] = 42;
    out.transformMaskMappingLocalToWorld[45][12] = 136;
    out.transformMaskMappingLocalToWorld[45][13] = 168;
    out.transformMaskMappingLocalToWorld[45][14] = 138;
    out.transformMaskMappingLocalToWorld[45][15] = 170;
    out.transformMaskMappingLocalToWorld[45][16] = 16;
    out.transformMaskMappingLocalToWorld[45][17] = 48;
    out.transformMaskMappingLocalToWorld[45][18] = 18;
    out.transformMaskMappingLocalToWorld[45][19] = 50;
    out.transformMaskMappingLocalToWorld[45][20] = 144;
    out.transformMaskMappingLocalToWorld[45][21] = 176;
    out.transformMaskMappingLocalToWorld[45][22] = 146;
    out.transformMaskMappingLocalToWorld[45][23] = 178;
    out.transformMaskMappingLocalToWorld[45][24] = 24;
    out.transformMaskMappingLocalToWorld[45][25] = 56;
    out.transformMaskMappingLocalToWorld[45][26] = 26;
    out.transformMaskMappingLocalToWorld[45][27] = 58;
    out.transformMaskMappingLocalToWorld[45][28] = 152;
    out.transformMaskMappingLocalToWorld[45][29] = 184;
    out.transformMaskMappingLocalToWorld[45][30] = 154;
    out.transformMaskMappingLocalToWorld[45][31] = 186;
    out.transformMaskMappingLocalToWorld[45][32] = 1;
    out.transformMaskMappingLocalToWorld[45][33] = 33;
    out.transformMaskMappingLocalToWorld[45][34] = 3;
    out.transformMaskMappingLocalToWorld[45][35] = 35;
    out.transformMaskMappingLocalToWorld[45][36] = 129;
    out.transformMaskMappingLocalToWorld[45][37] = 161;
    out.transformMaskMappingLocalToWorld[45][38] = 131;
    out.transformMaskMappingLocalToWorld[45][39] = 163;
    out.transformMaskMappingLocalToWorld[45][40] = 9;
    out.transformMaskMappingLocalToWorld[45][41] = 41;
    out.transformMaskMappingLocalToWorld[45][42] = 11;
    out.transformMaskMappingLocalToWorld[45][43] = 43;
    out.transformMaskMappingLocalToWorld[45][44] = 137;
    out.transformMaskMappingLocalToWorld[45][45] = 169;
    out.transformMaskMappingLocalToWorld[45][46] = 139;
    out.transformMaskMappingLocalToWorld[45][47] = 171;
    out.transformMaskMappingLocalToWorld[45][48] = 17;
    out.transformMaskMappingLocalToWorld[45][49] = 49;
    out.transformMaskMappingLocalToWorld[45][50] = 19;
    out.transformMaskMappingLocalToWorld[45][51] = 51;
    out.transformMaskMappingLocalToWorld[45][52] = 145;
    out.transformMaskMappingLocalToWorld[45][53] = 177;
    out.transformMaskMappingLocalToWorld[45][54] = 147;
    out.transformMaskMappingLocalToWorld[45][55] = 179;
    out.transformMaskMappingLocalToWorld[45][56] = 25;
    out.transformMaskMappingLocalToWorld[45][57] = 57;
    out.transformMaskMappingLocalToWorld[45][58] = 27;
    out.transformMaskMappingLocalToWorld[45][59] = 59;
    out.transformMaskMappingLocalToWorld[45][60] = 153;
    out.transformMaskMappingLocalToWorld[45][61] = 185;
    out.transformMaskMappingLocalToWorld[45][62] = 155;
    out.transformMaskMappingLocalToWorld[45][63] = 187;
    out.transformMaskMappingLocalToWorld[45][64] = 64;
    out.transformMaskMappingLocalToWorld[45][65] = 96;
    out.transformMaskMappingLocalToWorld[45][66] = 66;
    out.transformMaskMappingLocalToWorld[45][67] = 98;
    out.transformMaskMappingLocalToWorld[45][68] = 192;
    out.transformMaskMappingLocalToWorld[45][69] = 224;
    out.transformMaskMappingLocalToWorld[45][70] = 194;
    out.transformMaskMappingLocalToWorld[45][71] = 226;
    out.transformMaskMappingLocalToWorld[45][72] = 72;
    out.transformMaskMappingLocalToWorld[45][73] = 104;
    out.transformMaskMappingLocalToWorld[45][74] = 74;
    out.transformMaskMappingLocalToWorld[45][75] = 106;
    out.transformMaskMappingLocalToWorld[45][76] = 200;
    out.transformMaskMappingLocalToWorld[45][77] = 232;
    out.transformMaskMappingLocalToWorld[45][78] = 202;
    out.transformMaskMappingLocalToWorld[45][79] = 234;
    out.transformMaskMappingLocalToWorld[45][80] = 80;
    out.transformMaskMappingLocalToWorld[45][81] = 112;
    out.transformMaskMappingLocalToWorld[45][82] = 82;
    out.transformMaskMappingLocalToWorld[45][83] = 114;
    out.transformMaskMappingLocalToWorld[45][84] = 208;
    out.transformMaskMappingLocalToWorld[45][85] = 240;
    out.transformMaskMappingLocalToWorld[45][86] = 210;
    out.transformMaskMappingLocalToWorld[45][87] = 242;
    out.transformMaskMappingLocalToWorld[45][88] = 88;
    out.transformMaskMappingLocalToWorld[45][89] = 120;
    out.transformMaskMappingLocalToWorld[45][90] = 90;
    out.transformMaskMappingLocalToWorld[45][91] = 122;
    out.transformMaskMappingLocalToWorld[45][92] = 216;
    out.transformMaskMappingLocalToWorld[45][93] = 248;
    out.transformMaskMappingLocalToWorld[45][94] = 218;
    out.transformMaskMappingLocalToWorld[45][95] = 250;
    out.transformMaskMappingLocalToWorld[45][96] = 65;
    out.transformMaskMappingLocalToWorld[45][97] = 97;
    out.transformMaskMappingLocalToWorld[45][98] = 67;
    out.transformMaskMappingLocalToWorld[45][99] = 99;
    out.transformMaskMappingLocalToWorld[45][100] = 193;
    out.transformMaskMappingLocalToWorld[45][101] = 225;
    out.transformMaskMappingLocalToWorld[45][102] = 195;
    out.transformMaskMappingLocalToWorld[45][103] = 227;
    out.transformMaskMappingLocalToWorld[45][104] = 73;
    out.transformMaskMappingLocalToWorld[45][105] = 105;
    out.transformMaskMappingLocalToWorld[45][106] = 75;
    out.transformMaskMappingLocalToWorld[45][107] = 107;
    out.transformMaskMappingLocalToWorld[45][108] = 201;
    out.transformMaskMappingLocalToWorld[45][109] = 233;
    out.transformMaskMappingLocalToWorld[45][110] = 203;
    out.transformMaskMappingLocalToWorld[45][111] = 235;
    out.transformMaskMappingLocalToWorld[45][112] = 81;
    out.transformMaskMappingLocalToWorld[45][113] = 113;
    out.transformMaskMappingLocalToWorld[45][114] = 83;
    out.transformMaskMappingLocalToWorld[45][115] = 115;
    out.transformMaskMappingLocalToWorld[45][116] = 209;
    out.transformMaskMappingLocalToWorld[45][117] = 241;
    out.transformMaskMappingLocalToWorld[45][118] = 211;
    out.transformMaskMappingLocalToWorld[45][119] = 243;
    out.transformMaskMappingLocalToWorld[45][120] = 89;
    out.transformMaskMappingLocalToWorld[45][121] = 121;
    out.transformMaskMappingLocalToWorld[45][122] = 91;
    out.transformMaskMappingLocalToWorld[45][123] = 123;
    out.transformMaskMappingLocalToWorld[45][124] = 217;
    out.transformMaskMappingLocalToWorld[45][125] = 249;
    out.transformMaskMappingLocalToWorld[45][126] = 219;
    out.transformMaskMappingLocalToWorld[45][127] = 251;
    out.transformMaskMappingLocalToWorld[45][128] = 4;
    out.transformMaskMappingLocalToWorld[45][129] = 36;
    out.transformMaskMappingLocalToWorld[45][130] = 6;
    out.transformMaskMappingLocalToWorld[45][131] = 38;
    out.transformMaskMappingLocalToWorld[45][132] = 132;
    out.transformMaskMappingLocalToWorld[45][133] = 164;
    out.transformMaskMappingLocalToWorld[45][134] = 134;
    out.transformMaskMappingLocalToWorld[45][135] = 166;
    out.transformMaskMappingLocalToWorld[45][136] = 12;
    out.transformMaskMappingLocalToWorld[45][137] = 44;
    out.transformMaskMappingLocalToWorld[45][138] = 14;
    out.transformMaskMappingLocalToWorld[45][139] = 46;
    out.transformMaskMappingLocalToWorld[45][140] = 140;
    out.transformMaskMappingLocalToWorld[45][141] = 172;
    out.transformMaskMappingLocalToWorld[45][142] = 142;
    out.transformMaskMappingLocalToWorld[45][143] = 174;
    out.transformMaskMappingLocalToWorld[45][144] = 20;
    out.transformMaskMappingLocalToWorld[45][145] = 52;
    out.transformMaskMappingLocalToWorld[45][146] = 22;
    out.transformMaskMappingLocalToWorld[45][147] = 54;
    out.transformMaskMappingLocalToWorld[45][148] = 148;
    out.transformMaskMappingLocalToWorld[45][149] = 180;
    out.transformMaskMappingLocalToWorld[45][150] = 150;
    out.transformMaskMappingLocalToWorld[45][151] = 182;
    out.transformMaskMappingLocalToWorld[45][152] = 28;
    out.transformMaskMappingLocalToWorld[45][153] = 60;
    out.transformMaskMappingLocalToWorld[45][154] = 30;
    out.transformMaskMappingLocalToWorld[45][155] = 62;
    out.transformMaskMappingLocalToWorld[45][156] = 156;
    out.transformMaskMappingLocalToWorld[45][157] = 188;
    out.transformMaskMappingLocalToWorld[45][158] = 158;
    out.transformMaskMappingLocalToWorld[45][159] = 190;
    out.transformMaskMappingLocalToWorld[45][160] = 5;
    out.transformMaskMappingLocalToWorld[45][161] = 37;
    out.transformMaskMappingLocalToWorld[45][162] = 7;
    out.transformMaskMappingLocalToWorld[45][163] = 39;
    out.transformMaskMappingLocalToWorld[45][164] = 133;
    out.transformMaskMappingLocalToWorld[45][165] = 165;
    out.transformMaskMappingLocalToWorld[45][166] = 135;
    out.transformMaskMappingLocalToWorld[45][167] = 167;
    out.transformMaskMappingLocalToWorld[45][168] = 13;
    out.transformMaskMappingLocalToWorld[45][169] = 45;
    out.transformMaskMappingLocalToWorld[45][170] = 15;
    out.transformMaskMappingLocalToWorld[45][171] = 47;
    out.transformMaskMappingLocalToWorld[45][172] = 141;
    out.transformMaskMappingLocalToWorld[45][173] = 173;
    out.transformMaskMappingLocalToWorld[45][174] = 143;
    out.transformMaskMappingLocalToWorld[45][175] = 175;
    out.transformMaskMappingLocalToWorld[45][176] = 21;
    out.transformMaskMappingLocalToWorld[45][177] = 53;
    out.transformMaskMappingLocalToWorld[45][178] = 23;
    out.transformMaskMappingLocalToWorld[45][179] = 55;
    out.transformMaskMappingLocalToWorld[45][180] = 149;
    out.transformMaskMappingLocalToWorld[45][181] = 181;
    out.transformMaskMappingLocalToWorld[45][182] = 151;
    out.transformMaskMappingLocalToWorld[45][183] = 183;
    out.transformMaskMappingLocalToWorld[45][184] = 29;
    out.transformMaskMappingLocalToWorld[45][185] = 61;
    out.transformMaskMappingLocalToWorld[45][186] = 31;
    out.transformMaskMappingLocalToWorld[45][187] = 63;
    out.transformMaskMappingLocalToWorld[45][188] = 157;
    out.transformMaskMappingLocalToWorld[45][189] = 189;
    out.transformMaskMappingLocalToWorld[45][190] = 159;
    out.transformMaskMappingLocalToWorld[45][191] = 191;
    out.transformMaskMappingLocalToWorld[45][192] = 68;
    out.transformMaskMappingLocalToWorld[45][193] = 100;
    out.transformMaskMappingLocalToWorld[45][194] = 70;
    out.transformMaskMappingLocalToWorld[45][195] = 102;
    out.transformMaskMappingLocalToWorld[45][196] = 196;
    out.transformMaskMappingLocalToWorld[45][197] = 228;
    out.transformMaskMappingLocalToWorld[45][198] = 198;
    out.transformMaskMappingLocalToWorld[45][199] = 230;
    out.transformMaskMappingLocalToWorld[45][200] = 76;
    out.transformMaskMappingLocalToWorld[45][201] = 108;
    out.transformMaskMappingLocalToWorld[45][202] = 78;
    out.transformMaskMappingLocalToWorld[45][203] = 110;
    out.transformMaskMappingLocalToWorld[45][204] = 204;
    out.transformMaskMappingLocalToWorld[45][205] = 236;
    out.transformMaskMappingLocalToWorld[45][206] = 206;
    out.transformMaskMappingLocalToWorld[45][207] = 238;
    out.transformMaskMappingLocalToWorld[45][208] = 84;
    out.transformMaskMappingLocalToWorld[45][209] = 116;
    out.transformMaskMappingLocalToWorld[45][210] = 86;
    out.transformMaskMappingLocalToWorld[45][211] = 118;
    out.transformMaskMappingLocalToWorld[45][212] = 212;
    out.transformMaskMappingLocalToWorld[45][213] = 244;
    out.transformMaskMappingLocalToWorld[45][214] = 214;
    out.transformMaskMappingLocalToWorld[45][215] = 246;
    out.transformMaskMappingLocalToWorld[45][216] = 92;
    out.transformMaskMappingLocalToWorld[45][217] = 124;
    out.transformMaskMappingLocalToWorld[45][218] = 94;
    out.transformMaskMappingLocalToWorld[45][219] = 126;
    out.transformMaskMappingLocalToWorld[45][220] = 220;
    out.transformMaskMappingLocalToWorld[45][221] = 252;
    out.transformMaskMappingLocalToWorld[45][222] = 222;
    out.transformMaskMappingLocalToWorld[45][223] = 254;
    out.transformMaskMappingLocalToWorld[45][224] = 69;
    out.transformMaskMappingLocalToWorld[45][225] = 101;
    out.transformMaskMappingLocalToWorld[45][226] = 71;
    out.transformMaskMappingLocalToWorld[45][227] = 103;
    out.transformMaskMappingLocalToWorld[45][228] = 197;
    out.transformMaskMappingLocalToWorld[45][229] = 229;
    out.transformMaskMappingLocalToWorld[45][230] = 199;
    out.transformMaskMappingLocalToWorld[45][231] = 231;
    out.transformMaskMappingLocalToWorld[45][232] = 77;
    out.transformMaskMappingLocalToWorld[45][233] = 109;
    out.transformMaskMappingLocalToWorld[45][234] = 79;
    out.transformMaskMappingLocalToWorld[45][235] = 111;
    out.transformMaskMappingLocalToWorld[45][236] = 205;
    out.transformMaskMappingLocalToWorld[45][237] = 237;
    out.transformMaskMappingLocalToWorld[45][238] = 207;
    out.transformMaskMappingLocalToWorld[45][239] = 239;
    out.transformMaskMappingLocalToWorld[45][240] = 85;
    out.transformMaskMappingLocalToWorld[45][241] = 117;
    out.transformMaskMappingLocalToWorld[45][242] = 87;
    out.transformMaskMappingLocalToWorld[45][243] = 119;
    out.transformMaskMappingLocalToWorld[45][244] = 213;
    out.transformMaskMappingLocalToWorld[45][245] = 245;
    out.transformMaskMappingLocalToWorld[45][246] = 215;
    out.transformMaskMappingLocalToWorld[45][247] = 247;
    out.transformMaskMappingLocalToWorld[45][248] = 93;
    out.transformMaskMappingLocalToWorld[45][249] = 125;
    out.transformMaskMappingLocalToWorld[45][250] = 95;
    out.transformMaskMappingLocalToWorld[45][251] = 127;
    out.transformMaskMappingLocalToWorld[45][252] = 221;
    out.transformMaskMappingLocalToWorld[45][253] = 253;
    out.transformMaskMappingLocalToWorld[45][254] = 223;
    out.transformMaskMappingLocalToWorld[45][255] = 255;
    out.transformMaskMappingLocalToWorld[46][0] = 0;
    out.transformMaskMappingLocalToWorld[46][1] = 64;
    out.transformMaskMappingLocalToWorld[46][2] = 4;
    out.transformMaskMappingLocalToWorld[46][3] = 68;
    out.transformMaskMappingLocalToWorld[46][4] = 16;
    out.transformMaskMappingLocalToWorld[46][5] = 80;
    out.transformMaskMappingLocalToWorld[46][6] = 20;
    out.transformMaskMappingLocalToWorld[46][7] = 84;
    out.transformMaskMappingLocalToWorld[46][8] = 1;
    out.transformMaskMappingLocalToWorld[46][9] = 65;
    out.transformMaskMappingLocalToWorld[46][10] = 5;
    out.transformMaskMappingLocalToWorld[46][11] = 69;
    out.transformMaskMappingLocalToWorld[46][12] = 17;
    out.transformMaskMappingLocalToWorld[46][13] = 81;
    out.transformMaskMappingLocalToWorld[46][14] = 21;
    out.transformMaskMappingLocalToWorld[46][15] = 85;
    out.transformMaskMappingLocalToWorld[46][16] = 128;
    out.transformMaskMappingLocalToWorld[46][17] = 192;
    out.transformMaskMappingLocalToWorld[46][18] = 132;
    out.transformMaskMappingLocalToWorld[46][19] = 196;
    out.transformMaskMappingLocalToWorld[46][20] = 144;
    out.transformMaskMappingLocalToWorld[46][21] = 208;
    out.transformMaskMappingLocalToWorld[46][22] = 148;
    out.transformMaskMappingLocalToWorld[46][23] = 212;
    out.transformMaskMappingLocalToWorld[46][24] = 129;
    out.transformMaskMappingLocalToWorld[46][25] = 193;
    out.transformMaskMappingLocalToWorld[46][26] = 133;
    out.transformMaskMappingLocalToWorld[46][27] = 197;
    out.transformMaskMappingLocalToWorld[46][28] = 145;
    out.transformMaskMappingLocalToWorld[46][29] = 209;
    out.transformMaskMappingLocalToWorld[46][30] = 149;
    out.transformMaskMappingLocalToWorld[46][31] = 213;
    out.transformMaskMappingLocalToWorld[46][32] = 8;
    out.transformMaskMappingLocalToWorld[46][33] = 72;
    out.transformMaskMappingLocalToWorld[46][34] = 12;
    out.transformMaskMappingLocalToWorld[46][35] = 76;
    out.transformMaskMappingLocalToWorld[46][36] = 24;
    out.transformMaskMappingLocalToWorld[46][37] = 88;
    out.transformMaskMappingLocalToWorld[46][38] = 28;
    out.transformMaskMappingLocalToWorld[46][39] = 92;
    out.transformMaskMappingLocalToWorld[46][40] = 9;
    out.transformMaskMappingLocalToWorld[46][41] = 73;
    out.transformMaskMappingLocalToWorld[46][42] = 13;
    out.transformMaskMappingLocalToWorld[46][43] = 77;
    out.transformMaskMappingLocalToWorld[46][44] = 25;
    out.transformMaskMappingLocalToWorld[46][45] = 89;
    out.transformMaskMappingLocalToWorld[46][46] = 29;
    out.transformMaskMappingLocalToWorld[46][47] = 93;
    out.transformMaskMappingLocalToWorld[46][48] = 136;
    out.transformMaskMappingLocalToWorld[46][49] = 200;
    out.transformMaskMappingLocalToWorld[46][50] = 140;
    out.transformMaskMappingLocalToWorld[46][51] = 204;
    out.transformMaskMappingLocalToWorld[46][52] = 152;
    out.transformMaskMappingLocalToWorld[46][53] = 216;
    out.transformMaskMappingLocalToWorld[46][54] = 156;
    out.transformMaskMappingLocalToWorld[46][55] = 220;
    out.transformMaskMappingLocalToWorld[46][56] = 137;
    out.transformMaskMappingLocalToWorld[46][57] = 201;
    out.transformMaskMappingLocalToWorld[46][58] = 141;
    out.transformMaskMappingLocalToWorld[46][59] = 205;
    out.transformMaskMappingLocalToWorld[46][60] = 153;
    out.transformMaskMappingLocalToWorld[46][61] = 217;
    out.transformMaskMappingLocalToWorld[46][62] = 157;
    out.transformMaskMappingLocalToWorld[46][63] = 221;
    out.transformMaskMappingLocalToWorld[46][64] = 32;
    out.transformMaskMappingLocalToWorld[46][65] = 96;
    out.transformMaskMappingLocalToWorld[46][66] = 36;
    out.transformMaskMappingLocalToWorld[46][67] = 100;
    out.transformMaskMappingLocalToWorld[46][68] = 48;
    out.transformMaskMappingLocalToWorld[46][69] = 112;
    out.transformMaskMappingLocalToWorld[46][70] = 52;
    out.transformMaskMappingLocalToWorld[46][71] = 116;
    out.transformMaskMappingLocalToWorld[46][72] = 33;
    out.transformMaskMappingLocalToWorld[46][73] = 97;
    out.transformMaskMappingLocalToWorld[46][74] = 37;
    out.transformMaskMappingLocalToWorld[46][75] = 101;
    out.transformMaskMappingLocalToWorld[46][76] = 49;
    out.transformMaskMappingLocalToWorld[46][77] = 113;
    out.transformMaskMappingLocalToWorld[46][78] = 53;
    out.transformMaskMappingLocalToWorld[46][79] = 117;
    out.transformMaskMappingLocalToWorld[46][80] = 160;
    out.transformMaskMappingLocalToWorld[46][81] = 224;
    out.transformMaskMappingLocalToWorld[46][82] = 164;
    out.transformMaskMappingLocalToWorld[46][83] = 228;
    out.transformMaskMappingLocalToWorld[46][84] = 176;
    out.transformMaskMappingLocalToWorld[46][85] = 240;
    out.transformMaskMappingLocalToWorld[46][86] = 180;
    out.transformMaskMappingLocalToWorld[46][87] = 244;
    out.transformMaskMappingLocalToWorld[46][88] = 161;
    out.transformMaskMappingLocalToWorld[46][89] = 225;
    out.transformMaskMappingLocalToWorld[46][90] = 165;
    out.transformMaskMappingLocalToWorld[46][91] = 229;
    out.transformMaskMappingLocalToWorld[46][92] = 177;
    out.transformMaskMappingLocalToWorld[46][93] = 241;
    out.transformMaskMappingLocalToWorld[46][94] = 181;
    out.transformMaskMappingLocalToWorld[46][95] = 245;
    out.transformMaskMappingLocalToWorld[46][96] = 40;
    out.transformMaskMappingLocalToWorld[46][97] = 104;
    out.transformMaskMappingLocalToWorld[46][98] = 44;
    out.transformMaskMappingLocalToWorld[46][99] = 108;
    out.transformMaskMappingLocalToWorld[46][100] = 56;
    out.transformMaskMappingLocalToWorld[46][101] = 120;
    out.transformMaskMappingLocalToWorld[46][102] = 60;
    out.transformMaskMappingLocalToWorld[46][103] = 124;
    out.transformMaskMappingLocalToWorld[46][104] = 41;
    out.transformMaskMappingLocalToWorld[46][105] = 105;
    out.transformMaskMappingLocalToWorld[46][106] = 45;
    out.transformMaskMappingLocalToWorld[46][107] = 109;
    out.transformMaskMappingLocalToWorld[46][108] = 57;
    out.transformMaskMappingLocalToWorld[46][109] = 121;
    out.transformMaskMappingLocalToWorld[46][110] = 61;
    out.transformMaskMappingLocalToWorld[46][111] = 125;
    out.transformMaskMappingLocalToWorld[46][112] = 168;
    out.transformMaskMappingLocalToWorld[46][113] = 232;
    out.transformMaskMappingLocalToWorld[46][114] = 172;
    out.transformMaskMappingLocalToWorld[46][115] = 236;
    out.transformMaskMappingLocalToWorld[46][116] = 184;
    out.transformMaskMappingLocalToWorld[46][117] = 248;
    out.transformMaskMappingLocalToWorld[46][118] = 188;
    out.transformMaskMappingLocalToWorld[46][119] = 252;
    out.transformMaskMappingLocalToWorld[46][120] = 169;
    out.transformMaskMappingLocalToWorld[46][121] = 233;
    out.transformMaskMappingLocalToWorld[46][122] = 173;
    out.transformMaskMappingLocalToWorld[46][123] = 237;
    out.transformMaskMappingLocalToWorld[46][124] = 185;
    out.transformMaskMappingLocalToWorld[46][125] = 249;
    out.transformMaskMappingLocalToWorld[46][126] = 189;
    out.transformMaskMappingLocalToWorld[46][127] = 253;
    out.transformMaskMappingLocalToWorld[46][128] = 2;
    out.transformMaskMappingLocalToWorld[46][129] = 66;
    out.transformMaskMappingLocalToWorld[46][130] = 6;
    out.transformMaskMappingLocalToWorld[46][131] = 70;
    out.transformMaskMappingLocalToWorld[46][132] = 18;
    out.transformMaskMappingLocalToWorld[46][133] = 82;
    out.transformMaskMappingLocalToWorld[46][134] = 22;
    out.transformMaskMappingLocalToWorld[46][135] = 86;
    out.transformMaskMappingLocalToWorld[46][136] = 3;
    out.transformMaskMappingLocalToWorld[46][137] = 67;
    out.transformMaskMappingLocalToWorld[46][138] = 7;
    out.transformMaskMappingLocalToWorld[46][139] = 71;
    out.transformMaskMappingLocalToWorld[46][140] = 19;
    out.transformMaskMappingLocalToWorld[46][141] = 83;
    out.transformMaskMappingLocalToWorld[46][142] = 23;
    out.transformMaskMappingLocalToWorld[46][143] = 87;
    out.transformMaskMappingLocalToWorld[46][144] = 130;
    out.transformMaskMappingLocalToWorld[46][145] = 194;
    out.transformMaskMappingLocalToWorld[46][146] = 134;
    out.transformMaskMappingLocalToWorld[46][147] = 198;
    out.transformMaskMappingLocalToWorld[46][148] = 146;
    out.transformMaskMappingLocalToWorld[46][149] = 210;
    out.transformMaskMappingLocalToWorld[46][150] = 150;
    out.transformMaskMappingLocalToWorld[46][151] = 214;
    out.transformMaskMappingLocalToWorld[46][152] = 131;
    out.transformMaskMappingLocalToWorld[46][153] = 195;
    out.transformMaskMappingLocalToWorld[46][154] = 135;
    out.transformMaskMappingLocalToWorld[46][155] = 199;
    out.transformMaskMappingLocalToWorld[46][156] = 147;
    out.transformMaskMappingLocalToWorld[46][157] = 211;
    out.transformMaskMappingLocalToWorld[46][158] = 151;
    out.transformMaskMappingLocalToWorld[46][159] = 215;
    out.transformMaskMappingLocalToWorld[46][160] = 10;
    out.transformMaskMappingLocalToWorld[46][161] = 74;
    out.transformMaskMappingLocalToWorld[46][162] = 14;
    out.transformMaskMappingLocalToWorld[46][163] = 78;
    out.transformMaskMappingLocalToWorld[46][164] = 26;
    out.transformMaskMappingLocalToWorld[46][165] = 90;
    out.transformMaskMappingLocalToWorld[46][166] = 30;
    out.transformMaskMappingLocalToWorld[46][167] = 94;
    out.transformMaskMappingLocalToWorld[46][168] = 11;
    out.transformMaskMappingLocalToWorld[46][169] = 75;
    out.transformMaskMappingLocalToWorld[46][170] = 15;
    out.transformMaskMappingLocalToWorld[46][171] = 79;
    out.transformMaskMappingLocalToWorld[46][172] = 27;
    out.transformMaskMappingLocalToWorld[46][173] = 91;
    out.transformMaskMappingLocalToWorld[46][174] = 31;
    out.transformMaskMappingLocalToWorld[46][175] = 95;
    out.transformMaskMappingLocalToWorld[46][176] = 138;
    out.transformMaskMappingLocalToWorld[46][177] = 202;
    out.transformMaskMappingLocalToWorld[46][178] = 142;
    out.transformMaskMappingLocalToWorld[46][179] = 206;
    out.transformMaskMappingLocalToWorld[46][180] = 154;
    out.transformMaskMappingLocalToWorld[46][181] = 218;
    out.transformMaskMappingLocalToWorld[46][182] = 158;
    out.transformMaskMappingLocalToWorld[46][183] = 222;
    out.transformMaskMappingLocalToWorld[46][184] = 139;
    out.transformMaskMappingLocalToWorld[46][185] = 203;
    out.transformMaskMappingLocalToWorld[46][186] = 143;
    out.transformMaskMappingLocalToWorld[46][187] = 207;
    out.transformMaskMappingLocalToWorld[46][188] = 155;
    out.transformMaskMappingLocalToWorld[46][189] = 219;
    out.transformMaskMappingLocalToWorld[46][190] = 159;
    out.transformMaskMappingLocalToWorld[46][191] = 223;
    out.transformMaskMappingLocalToWorld[46][192] = 34;
    out.transformMaskMappingLocalToWorld[46][193] = 98;
    out.transformMaskMappingLocalToWorld[46][194] = 38;
    out.transformMaskMappingLocalToWorld[46][195] = 102;
    out.transformMaskMappingLocalToWorld[46][196] = 50;
    out.transformMaskMappingLocalToWorld[46][197] = 114;
    out.transformMaskMappingLocalToWorld[46][198] = 54;
    out.transformMaskMappingLocalToWorld[46][199] = 118;
    out.transformMaskMappingLocalToWorld[46][200] = 35;
    out.transformMaskMappingLocalToWorld[46][201] = 99;
    out.transformMaskMappingLocalToWorld[46][202] = 39;
    out.transformMaskMappingLocalToWorld[46][203] = 103;
    out.transformMaskMappingLocalToWorld[46][204] = 51;
    out.transformMaskMappingLocalToWorld[46][205] = 115;
    out.transformMaskMappingLocalToWorld[46][206] = 55;
    out.transformMaskMappingLocalToWorld[46][207] = 119;
    out.transformMaskMappingLocalToWorld[46][208] = 162;
    out.transformMaskMappingLocalToWorld[46][209] = 226;
    out.transformMaskMappingLocalToWorld[46][210] = 166;
    out.transformMaskMappingLocalToWorld[46][211] = 230;
    out.transformMaskMappingLocalToWorld[46][212] = 178;
    out.transformMaskMappingLocalToWorld[46][213] = 242;
    out.transformMaskMappingLocalToWorld[46][214] = 182;
    out.transformMaskMappingLocalToWorld[46][215] = 246;
    out.transformMaskMappingLocalToWorld[46][216] = 163;
    out.transformMaskMappingLocalToWorld[46][217] = 227;
    out.transformMaskMappingLocalToWorld[46][218] = 167;
    out.transformMaskMappingLocalToWorld[46][219] = 231;
    out.transformMaskMappingLocalToWorld[46][220] = 179;
    out.transformMaskMappingLocalToWorld[46][221] = 243;
    out.transformMaskMappingLocalToWorld[46][222] = 183;
    out.transformMaskMappingLocalToWorld[46][223] = 247;
    out.transformMaskMappingLocalToWorld[46][224] = 42;
    out.transformMaskMappingLocalToWorld[46][225] = 106;
    out.transformMaskMappingLocalToWorld[46][226] = 46;
    out.transformMaskMappingLocalToWorld[46][227] = 110;
    out.transformMaskMappingLocalToWorld[46][228] = 58;
    out.transformMaskMappingLocalToWorld[46][229] = 122;
    out.transformMaskMappingLocalToWorld[46][230] = 62;
    out.transformMaskMappingLocalToWorld[46][231] = 126;
    out.transformMaskMappingLocalToWorld[46][232] = 43;
    out.transformMaskMappingLocalToWorld[46][233] = 107;
    out.transformMaskMappingLocalToWorld[46][234] = 47;
    out.transformMaskMappingLocalToWorld[46][235] = 111;
    out.transformMaskMappingLocalToWorld[46][236] = 59;
    out.transformMaskMappingLocalToWorld[46][237] = 123;
    out.transformMaskMappingLocalToWorld[46][238] = 63;
    out.transformMaskMappingLocalToWorld[46][239] = 127;
    out.transformMaskMappingLocalToWorld[46][240] = 170;
    out.transformMaskMappingLocalToWorld[46][241] = 234;
    out.transformMaskMappingLocalToWorld[46][242] = 174;
    out.transformMaskMappingLocalToWorld[46][243] = 238;
    out.transformMaskMappingLocalToWorld[46][244] = 186;
    out.transformMaskMappingLocalToWorld[46][245] = 250;
    out.transformMaskMappingLocalToWorld[46][246] = 190;
    out.transformMaskMappingLocalToWorld[46][247] = 254;
    out.transformMaskMappingLocalToWorld[46][248] = 171;
    out.transformMaskMappingLocalToWorld[46][249] = 235;
    out.transformMaskMappingLocalToWorld[46][250] = 175;
    out.transformMaskMappingLocalToWorld[46][251] = 239;
    out.transformMaskMappingLocalToWorld[46][252] = 187;
    out.transformMaskMappingLocalToWorld[46][253] = 251;
    out.transformMaskMappingLocalToWorld[46][254] = 191;
    out.transformMaskMappingLocalToWorld[46][255] = 255;
    out.transformMaskMappingLocalToWorld[47][0] = 0;
    out.transformMaskMappingLocalToWorld[47][1] = 128;
    out.transformMaskMappingLocalToWorld[47][2] = 8;
    out.transformMaskMappingLocalToWorld[47][3] = 136;
    out.transformMaskMappingLocalToWorld[47][4] = 32;
    out.transformMaskMappingLocalToWorld[47][5] = 160;
    out.transformMaskMappingLocalToWorld[47][6] = 40;
    out.transformMaskMappingLocalToWorld[47][7] = 168;
    out.transformMaskMappingLocalToWorld[47][8] = 2;
    out.transformMaskMappingLocalToWorld[47][9] = 130;
    out.transformMaskMappingLocalToWorld[47][10] = 10;
    out.transformMaskMappingLocalToWorld[47][11] = 138;
    out.transformMaskMappingLocalToWorld[47][12] = 34;
    out.transformMaskMappingLocalToWorld[47][13] = 162;
    out.transformMaskMappingLocalToWorld[47][14] = 42;
    out.transformMaskMappingLocalToWorld[47][15] = 170;
    out.transformMaskMappingLocalToWorld[47][16] = 64;
    out.transformMaskMappingLocalToWorld[47][17] = 192;
    out.transformMaskMappingLocalToWorld[47][18] = 72;
    out.transformMaskMappingLocalToWorld[47][19] = 200;
    out.transformMaskMappingLocalToWorld[47][20] = 96;
    out.transformMaskMappingLocalToWorld[47][21] = 224;
    out.transformMaskMappingLocalToWorld[47][22] = 104;
    out.transformMaskMappingLocalToWorld[47][23] = 232;
    out.transformMaskMappingLocalToWorld[47][24] = 66;
    out.transformMaskMappingLocalToWorld[47][25] = 194;
    out.transformMaskMappingLocalToWorld[47][26] = 74;
    out.transformMaskMappingLocalToWorld[47][27] = 202;
    out.transformMaskMappingLocalToWorld[47][28] = 98;
    out.transformMaskMappingLocalToWorld[47][29] = 226;
    out.transformMaskMappingLocalToWorld[47][30] = 106;
    out.transformMaskMappingLocalToWorld[47][31] = 234;
    out.transformMaskMappingLocalToWorld[47][32] = 4;
    out.transformMaskMappingLocalToWorld[47][33] = 132;
    out.transformMaskMappingLocalToWorld[47][34] = 12;
    out.transformMaskMappingLocalToWorld[47][35] = 140;
    out.transformMaskMappingLocalToWorld[47][36] = 36;
    out.transformMaskMappingLocalToWorld[47][37] = 164;
    out.transformMaskMappingLocalToWorld[47][38] = 44;
    out.transformMaskMappingLocalToWorld[47][39] = 172;
    out.transformMaskMappingLocalToWorld[47][40] = 6;
    out.transformMaskMappingLocalToWorld[47][41] = 134;
    out.transformMaskMappingLocalToWorld[47][42] = 14;
    out.transformMaskMappingLocalToWorld[47][43] = 142;
    out.transformMaskMappingLocalToWorld[47][44] = 38;
    out.transformMaskMappingLocalToWorld[47][45] = 166;
    out.transformMaskMappingLocalToWorld[47][46] = 46;
    out.transformMaskMappingLocalToWorld[47][47] = 174;
    out.transformMaskMappingLocalToWorld[47][48] = 68;
    out.transformMaskMappingLocalToWorld[47][49] = 196;
    out.transformMaskMappingLocalToWorld[47][50] = 76;
    out.transformMaskMappingLocalToWorld[47][51] = 204;
    out.transformMaskMappingLocalToWorld[47][52] = 100;
    out.transformMaskMappingLocalToWorld[47][53] = 228;
    out.transformMaskMappingLocalToWorld[47][54] = 108;
    out.transformMaskMappingLocalToWorld[47][55] = 236;
    out.transformMaskMappingLocalToWorld[47][56] = 70;
    out.transformMaskMappingLocalToWorld[47][57] = 198;
    out.transformMaskMappingLocalToWorld[47][58] = 78;
    out.transformMaskMappingLocalToWorld[47][59] = 206;
    out.transformMaskMappingLocalToWorld[47][60] = 102;
    out.transformMaskMappingLocalToWorld[47][61] = 230;
    out.transformMaskMappingLocalToWorld[47][62] = 110;
    out.transformMaskMappingLocalToWorld[47][63] = 238;
    out.transformMaskMappingLocalToWorld[47][64] = 16;
    out.transformMaskMappingLocalToWorld[47][65] = 144;
    out.transformMaskMappingLocalToWorld[47][66] = 24;
    out.transformMaskMappingLocalToWorld[47][67] = 152;
    out.transformMaskMappingLocalToWorld[47][68] = 48;
    out.transformMaskMappingLocalToWorld[47][69] = 176;
    out.transformMaskMappingLocalToWorld[47][70] = 56;
    out.transformMaskMappingLocalToWorld[47][71] = 184;
    out.transformMaskMappingLocalToWorld[47][72] = 18;
    out.transformMaskMappingLocalToWorld[47][73] = 146;
    out.transformMaskMappingLocalToWorld[47][74] = 26;
    out.transformMaskMappingLocalToWorld[47][75] = 154;
    out.transformMaskMappingLocalToWorld[47][76] = 50;
    out.transformMaskMappingLocalToWorld[47][77] = 178;
    out.transformMaskMappingLocalToWorld[47][78] = 58;
    out.transformMaskMappingLocalToWorld[47][79] = 186;
    out.transformMaskMappingLocalToWorld[47][80] = 80;
    out.transformMaskMappingLocalToWorld[47][81] = 208;
    out.transformMaskMappingLocalToWorld[47][82] = 88;
    out.transformMaskMappingLocalToWorld[47][83] = 216;
    out.transformMaskMappingLocalToWorld[47][84] = 112;
    out.transformMaskMappingLocalToWorld[47][85] = 240;
    out.transformMaskMappingLocalToWorld[47][86] = 120;
    out.transformMaskMappingLocalToWorld[47][87] = 248;
    out.transformMaskMappingLocalToWorld[47][88] = 82;
    out.transformMaskMappingLocalToWorld[47][89] = 210;
    out.transformMaskMappingLocalToWorld[47][90] = 90;
    out.transformMaskMappingLocalToWorld[47][91] = 218;
    out.transformMaskMappingLocalToWorld[47][92] = 114;
    out.transformMaskMappingLocalToWorld[47][93] = 242;
    out.transformMaskMappingLocalToWorld[47][94] = 122;
    out.transformMaskMappingLocalToWorld[47][95] = 250;
    out.transformMaskMappingLocalToWorld[47][96] = 20;
    out.transformMaskMappingLocalToWorld[47][97] = 148;
    out.transformMaskMappingLocalToWorld[47][98] = 28;
    out.transformMaskMappingLocalToWorld[47][99] = 156;
    out.transformMaskMappingLocalToWorld[47][100] = 52;
    out.transformMaskMappingLocalToWorld[47][101] = 180;
    out.transformMaskMappingLocalToWorld[47][102] = 60;
    out.transformMaskMappingLocalToWorld[47][103] = 188;
    out.transformMaskMappingLocalToWorld[47][104] = 22;
    out.transformMaskMappingLocalToWorld[47][105] = 150;
    out.transformMaskMappingLocalToWorld[47][106] = 30;
    out.transformMaskMappingLocalToWorld[47][107] = 158;
    out.transformMaskMappingLocalToWorld[47][108] = 54;
    out.transformMaskMappingLocalToWorld[47][109] = 182;
    out.transformMaskMappingLocalToWorld[47][110] = 62;
    out.transformMaskMappingLocalToWorld[47][111] = 190;
    out.transformMaskMappingLocalToWorld[47][112] = 84;
    out.transformMaskMappingLocalToWorld[47][113] = 212;
    out.transformMaskMappingLocalToWorld[47][114] = 92;
    out.transformMaskMappingLocalToWorld[47][115] = 220;
    out.transformMaskMappingLocalToWorld[47][116] = 116;
    out.transformMaskMappingLocalToWorld[47][117] = 244;
    out.transformMaskMappingLocalToWorld[47][118] = 124;
    out.transformMaskMappingLocalToWorld[47][119] = 252;
    out.transformMaskMappingLocalToWorld[47][120] = 86;
    out.transformMaskMappingLocalToWorld[47][121] = 214;
    out.transformMaskMappingLocalToWorld[47][122] = 94;
    out.transformMaskMappingLocalToWorld[47][123] = 222;
    out.transformMaskMappingLocalToWorld[47][124] = 118;
    out.transformMaskMappingLocalToWorld[47][125] = 246;
    out.transformMaskMappingLocalToWorld[47][126] = 126;
    out.transformMaskMappingLocalToWorld[47][127] = 254;
    out.transformMaskMappingLocalToWorld[47][128] = 1;
    out.transformMaskMappingLocalToWorld[47][129] = 129;
    out.transformMaskMappingLocalToWorld[47][130] = 9;
    out.transformMaskMappingLocalToWorld[47][131] = 137;
    out.transformMaskMappingLocalToWorld[47][132] = 33;
    out.transformMaskMappingLocalToWorld[47][133] = 161;
    out.transformMaskMappingLocalToWorld[47][134] = 41;
    out.transformMaskMappingLocalToWorld[47][135] = 169;
    out.transformMaskMappingLocalToWorld[47][136] = 3;
    out.transformMaskMappingLocalToWorld[47][137] = 131;
    out.transformMaskMappingLocalToWorld[47][138] = 11;
    out.transformMaskMappingLocalToWorld[47][139] = 139;
    out.transformMaskMappingLocalToWorld[47][140] = 35;
    out.transformMaskMappingLocalToWorld[47][141] = 163;
    out.transformMaskMappingLocalToWorld[47][142] = 43;
    out.transformMaskMappingLocalToWorld[47][143] = 171;
    out.transformMaskMappingLocalToWorld[47][144] = 65;
    out.transformMaskMappingLocalToWorld[47][145] = 193;
    out.transformMaskMappingLocalToWorld[47][146] = 73;
    out.transformMaskMappingLocalToWorld[47][147] = 201;
    out.transformMaskMappingLocalToWorld[47][148] = 97;
    out.transformMaskMappingLocalToWorld[47][149] = 225;
    out.transformMaskMappingLocalToWorld[47][150] = 105;
    out.transformMaskMappingLocalToWorld[47][151] = 233;
    out.transformMaskMappingLocalToWorld[47][152] = 67;
    out.transformMaskMappingLocalToWorld[47][153] = 195;
    out.transformMaskMappingLocalToWorld[47][154] = 75;
    out.transformMaskMappingLocalToWorld[47][155] = 203;
    out.transformMaskMappingLocalToWorld[47][156] = 99;
    out.transformMaskMappingLocalToWorld[47][157] = 227;
    out.transformMaskMappingLocalToWorld[47][158] = 107;
    out.transformMaskMappingLocalToWorld[47][159] = 235;
    out.transformMaskMappingLocalToWorld[47][160] = 5;
    out.transformMaskMappingLocalToWorld[47][161] = 133;
    out.transformMaskMappingLocalToWorld[47][162] = 13;
    out.transformMaskMappingLocalToWorld[47][163] = 141;
    out.transformMaskMappingLocalToWorld[47][164] = 37;
    out.transformMaskMappingLocalToWorld[47][165] = 165;
    out.transformMaskMappingLocalToWorld[47][166] = 45;
    out.transformMaskMappingLocalToWorld[47][167] = 173;
    out.transformMaskMappingLocalToWorld[47][168] = 7;
    out.transformMaskMappingLocalToWorld[47][169] = 135;
    out.transformMaskMappingLocalToWorld[47][170] = 15;
    out.transformMaskMappingLocalToWorld[47][171] = 143;
    out.transformMaskMappingLocalToWorld[47][172] = 39;
    out.transformMaskMappingLocalToWorld[47][173] = 167;
    out.transformMaskMappingLocalToWorld[47][174] = 47;
    out.transformMaskMappingLocalToWorld[47][175] = 175;
    out.transformMaskMappingLocalToWorld[47][176] = 69;
    out.transformMaskMappingLocalToWorld[47][177] = 197;
    out.transformMaskMappingLocalToWorld[47][178] = 77;
    out.transformMaskMappingLocalToWorld[47][179] = 205;
    out.transformMaskMappingLocalToWorld[47][180] = 101;
    out.transformMaskMappingLocalToWorld[47][181] = 229;
    out.transformMaskMappingLocalToWorld[47][182] = 109;
    out.transformMaskMappingLocalToWorld[47][183] = 237;
    out.transformMaskMappingLocalToWorld[47][184] = 71;
    out.transformMaskMappingLocalToWorld[47][185] = 199;
    out.transformMaskMappingLocalToWorld[47][186] = 79;
    out.transformMaskMappingLocalToWorld[47][187] = 207;
    out.transformMaskMappingLocalToWorld[47][188] = 103;
    out.transformMaskMappingLocalToWorld[47][189] = 231;
    out.transformMaskMappingLocalToWorld[47][190] = 111;
    out.transformMaskMappingLocalToWorld[47][191] = 239;
    out.transformMaskMappingLocalToWorld[47][192] = 17;
    out.transformMaskMappingLocalToWorld[47][193] = 145;
    out.transformMaskMappingLocalToWorld[47][194] = 25;
    out.transformMaskMappingLocalToWorld[47][195] = 153;
    out.transformMaskMappingLocalToWorld[47][196] = 49;
    out.transformMaskMappingLocalToWorld[47][197] = 177;
    out.transformMaskMappingLocalToWorld[47][198] = 57;
    out.transformMaskMappingLocalToWorld[47][199] = 185;
    out.transformMaskMappingLocalToWorld[47][200] = 19;
    out.transformMaskMappingLocalToWorld[47][201] = 147;
    out.transformMaskMappingLocalToWorld[47][202] = 27;
    out.transformMaskMappingLocalToWorld[47][203] = 155;
    out.transformMaskMappingLocalToWorld[47][204] = 51;
    out.transformMaskMappingLocalToWorld[47][205] = 179;
    out.transformMaskMappingLocalToWorld[47][206] = 59;
    out.transformMaskMappingLocalToWorld[47][207] = 187;
    out.transformMaskMappingLocalToWorld[47][208] = 81;
    out.transformMaskMappingLocalToWorld[47][209] = 209;
    out.transformMaskMappingLocalToWorld[47][210] = 89;
    out.transformMaskMappingLocalToWorld[47][211] = 217;
    out.transformMaskMappingLocalToWorld[47][212] = 113;
    out.transformMaskMappingLocalToWorld[47][213] = 241;
    out.transformMaskMappingLocalToWorld[47][214] = 121;
    out.transformMaskMappingLocalToWorld[47][215] = 249;
    out.transformMaskMappingLocalToWorld[47][216] = 83;
    out.transformMaskMappingLocalToWorld[47][217] = 211;
    out.transformMaskMappingLocalToWorld[47][218] = 91;
    out.transformMaskMappingLocalToWorld[47][219] = 219;
    out.transformMaskMappingLocalToWorld[47][220] = 115;
    out.transformMaskMappingLocalToWorld[47][221] = 243;
    out.transformMaskMappingLocalToWorld[47][222] = 123;
    out.transformMaskMappingLocalToWorld[47][223] = 251;
    out.transformMaskMappingLocalToWorld[47][224] = 21;
    out.transformMaskMappingLocalToWorld[47][225] = 149;
    out.transformMaskMappingLocalToWorld[47][226] = 29;
    out.transformMaskMappingLocalToWorld[47][227] = 157;
    out.transformMaskMappingLocalToWorld[47][228] = 53;
    out.transformMaskMappingLocalToWorld[47][229] = 181;
    out.transformMaskMappingLocalToWorld[47][230] = 61;
    out.transformMaskMappingLocalToWorld[47][231] = 189;
    out.transformMaskMappingLocalToWorld[47][232] = 23;
    out.transformMaskMappingLocalToWorld[47][233] = 151;
    out.transformMaskMappingLocalToWorld[47][234] = 31;
    out.transformMaskMappingLocalToWorld[47][235] = 159;
    out.transformMaskMappingLocalToWorld[47][236] = 55;
    out.transformMaskMappingLocalToWorld[47][237] = 183;
    out.transformMaskMappingLocalToWorld[47][238] = 63;
    out.transformMaskMappingLocalToWorld[47][239] = 191;
    out.transformMaskMappingLocalToWorld[47][240] = 85;
    out.transformMaskMappingLocalToWorld[47][241] = 213;
    out.transformMaskMappingLocalToWorld[47][242] = 93;
    out.transformMaskMappingLocalToWorld[47][243] = 221;
    out.transformMaskMappingLocalToWorld[47][244] = 117;
    out.transformMaskMappingLocalToWorld[47][245] = 245;
    out.transformMaskMappingLocalToWorld[47][246] = 125;
    out.transformMaskMappingLocalToWorld[47][247] = 253;
    out.transformMaskMappingLocalToWorld[47][248] = 87;
    out.transformMaskMappingLocalToWorld[47][249] = 215;
    out.transformMaskMappingLocalToWorld[47][250] = 95;
    out.transformMaskMappingLocalToWorld[47][251] = 223;
    out.transformMaskMappingLocalToWorld[47][252] = 119;
    out.transformMaskMappingLocalToWorld[47][253] = 247;
    out.transformMaskMappingLocalToWorld[47][254] = 127;
    out.transformMaskMappingLocalToWorld[47][255] = 255;
    out.transformCombineTable[0][0] = 0;
    out.transformCombineTable[0][1] = 1;
    out.transformCombineTable[0][2] = 2;
    out.transformCombineTable[0][3] = 3;
    out.transformCombineTable[0][4] = 4;
    out.transformCombineTable[0][5] = 5;
    out.transformCombineTable[0][6] = 6;
    out.transformCombineTable[0][7] = 7;
    out.transformCombineTable[0][8] = 8;
    out.transformCombineTable[0][9] = 9;
    out.transformCombineTable[0][10] = 10;
    out.transformCombineTable[0][11] = 11;
    out.transformCombineTable[0][12] = 12;
    out.transformCombineTable[0][13] = 13;
    out.transformCombineTable[0][14] = 14;
    out.transformCombineTable[0][15] = 15;
    out.transformCombineTable[0][16] = 16;
    out.transformCombineTable[0][17] = 17;
    out.transformCombineTable[0][18] = 18;
    out.transformCombineTable[0][19] = 19;
    out.transformCombineTable[0][20] = 20;
    out.transformCombineTable[0][21] = 21;
    out.transformCombineTable[0][22] = 22;
    out.transformCombineTable[0][23] = 23;
    out.transformCombineTable[0][24] = 24;
    out.transformCombineTable[0][25] = 25;
    out.transformCombineTable[0][26] = 26;
    out.transformCombineTable[0][27] = 27;
    out.transformCombineTable[0][28] = 28;
    out.transformCombineTable[0][29] = 29;
    out.transformCombineTable[0][30] = 30;
    out.transformCombineTable[0][31] = 31;
    out.transformCombineTable[0][32] = 32;
    out.transformCombineTable[0][33] = 33;
    out.transformCombineTable[0][34] = 34;
    out.transformCombineTable[0][35] = 35;
    out.transformCombineTable[0][36] = 36;
    out.transformCombineTable[0][37] = 37;
    out.transformCombineTable[0][38] = 38;
    out.transformCombineTable[0][39] = 39;
    out.transformCombineTable[0][40] = 40;
    out.transformCombineTable[0][41] = 41;
    out.transformCombineTable[0][42] = 42;
    out.transformCombineTable[0][43] = 43;
    out.transformCombineTable[0][44] = 44;
    out.transformCombineTable[0][45] = 45;
    out.transformCombineTable[0][46] = 46;
    out.transformCombineTable[0][47] = 47;
    out.transformCombineTable[1][0] = 1;
    out.transformCombineTable[1][1] = 0;
    out.transformCombineTable[1][2] = 3;
    out.transformCombineTable[1][3] = 2;
    out.transformCombineTable[1][4] = 5;
    out.transformCombineTable[1][5] = 4;
    out.transformCombineTable[1][6] = 7;
    out.transformCombineTable[1][7] = 6;
    out.transformCombineTable[1][8] = 9;
    out.transformCombineTable[1][9] = 8;
    out.transformCombineTable[1][10] = 11;
    out.transformCombineTable[1][11] = 10;
    out.transformCombineTable[1][12] = 13;
    out.transformCombineTable[1][13] = 12;
    out.transformCombineTable[1][14] = 15;
    out.transformCombineTable[1][15] = 14;
    out.transformCombineTable[1][16] = 17;
    out.transformCombineTable[1][17] = 16;
    out.transformCombineTable[1][18] = 19;
    out.transformCombineTable[1][19] = 18;
    out.transformCombineTable[1][20] = 21;
    out.transformCombineTable[1][21] = 20;
    out.transformCombineTable[1][22] = 23;
    out.transformCombineTable[1][23] = 22;
    out.transformCombineTable[1][24] = 25;
    out.transformCombineTable[1][25] = 24;
    out.transformCombineTable[1][26] = 27;
    out.transformCombineTable[1][27] = 26;
    out.transformCombineTable[1][28] = 29;
    out.transformCombineTable[1][29] = 28;
    out.transformCombineTable[1][30] = 31;
    out.transformCombineTable[1][31] = 30;
    out.transformCombineTable[1][32] = 33;
    out.transformCombineTable[1][33] = 32;
    out.transformCombineTable[1][34] = 35;
    out.transformCombineTable[1][35] = 34;
    out.transformCombineTable[1][36] = 37;
    out.transformCombineTable[1][37] = 36;
    out.transformCombineTable[1][38] = 39;
    out.transformCombineTable[1][39] = 38;
    out.transformCombineTable[1][40] = 41;
    out.transformCombineTable[1][41] = 40;
    out.transformCombineTable[1][42] = 43;
    out.transformCombineTable[1][43] = 42;
    out.transformCombineTable[1][44] = 45;
    out.transformCombineTable[1][45] = 44;
    out.transformCombineTable[1][46] = 47;
    out.transformCombineTable[1][47] = 46;
    out.transformCombineTable[2][0] = 2;
    out.transformCombineTable[2][1] = 3;
    out.transformCombineTable[2][2] = 0;
    out.transformCombineTable[2][3] = 1;
    out.transformCombineTable[2][4] = 6;
    out.transformCombineTable[2][5] = 7;
    out.transformCombineTable[2][6] = 4;
    out.transformCombineTable[2][7] = 5;
    out.transformCombineTable[2][8] = 10;
    out.transformCombineTable[2][9] = 11;
    out.transformCombineTable[2][10] = 8;
    out.transformCombineTable[2][11] = 9;
    out.transformCombineTable[2][12] = 14;
    out.transformCombineTable[2][13] = 15;
    out.transformCombineTable[2][14] = 12;
    out.transformCombineTable[2][15] = 13;
    out.transformCombineTable[2][16] = 18;
    out.transformCombineTable[2][17] = 19;
    out.transformCombineTable[2][18] = 16;
    out.transformCombineTable[2][19] = 17;
    out.transformCombineTable[2][20] = 22;
    out.transformCombineTable[2][21] = 23;
    out.transformCombineTable[2][22] = 20;
    out.transformCombineTable[2][23] = 21;
    out.transformCombineTable[2][24] = 26;
    out.transformCombineTable[2][25] = 27;
    out.transformCombineTable[2][26] = 24;
    out.transformCombineTable[2][27] = 25;
    out.transformCombineTable[2][28] = 30;
    out.transformCombineTable[2][29] = 31;
    out.transformCombineTable[2][30] = 28;
    out.transformCombineTable[2][31] = 29;
    out.transformCombineTable[2][32] = 34;
    out.transformCombineTable[2][33] = 35;
    out.transformCombineTable[2][34] = 32;
    out.transformCombineTable[2][35] = 33;
    out.transformCombineTable[2][36] = 38;
    out.transformCombineTable[2][37] = 39;
    out.transformCombineTable[2][38] = 36;
    out.transformCombineTable[2][39] = 37;
    out.transformCombineTable[2][40] = 42;
    out.transformCombineTable[2][41] = 43;
    out.transformCombineTable[2][42] = 40;
    out.transformCombineTable[2][43] = 41;
    out.transformCombineTable[2][44] = 46;
    out.transformCombineTable[2][45] = 47;
    out.transformCombineTable[2][46] = 44;
    out.transformCombineTable[2][47] = 45;
    out.transformCombineTable[3][0] = 3;
    out.transformCombineTable[3][1] = 2;
    out.transformCombineTable[3][2] = 1;
    out.transformCombineTable[3][3] = 0;
    out.transformCombineTable[3][4] = 7;
    out.transformCombineTable[3][5] = 6;
    out.transformCombineTable[3][6] = 5;
    out.transformCombineTable[3][7] = 4;
    out.transformCombineTable[3][8] = 11;
    out.transformCombineTable[3][9] = 10;
    out.transformCombineTable[3][10] = 9;
    out.transformCombineTable[3][11] = 8;
    out.transformCombineTable[3][12] = 15;
    out.transformCombineTable[3][13] = 14;
    out.transformCombineTable[3][14] = 13;
    out.transformCombineTable[3][15] = 12;
    out.transformCombineTable[3][16] = 19;
    out.transformCombineTable[3][17] = 18;
    out.transformCombineTable[3][18] = 17;
    out.transformCombineTable[3][19] = 16;
    out.transformCombineTable[3][20] = 23;
    out.transformCombineTable[3][21] = 22;
    out.transformCombineTable[3][22] = 21;
    out.transformCombineTable[3][23] = 20;
    out.transformCombineTable[3][24] = 27;
    out.transformCombineTable[3][25] = 26;
    out.transformCombineTable[3][26] = 25;
    out.transformCombineTable[3][27] = 24;
    out.transformCombineTable[3][28] = 31;
    out.transformCombineTable[3][29] = 30;
    out.transformCombineTable[3][30] = 29;
    out.transformCombineTable[3][31] = 28;
    out.transformCombineTable[3][32] = 35;
    out.transformCombineTable[3][33] = 34;
    out.transformCombineTable[3][34] = 33;
    out.transformCombineTable[3][35] = 32;
    out.transformCombineTable[3][36] = 39;
    out.transformCombineTable[3][37] = 38;
    out.transformCombineTable[3][38] = 37;
    out.transformCombineTable[3][39] = 36;
    out.transformCombineTable[3][40] = 43;
    out.transformCombineTable[3][41] = 42;
    out.transformCombineTable[3][42] = 41;
    out.transformCombineTable[3][43] = 40;
    out.transformCombineTable[3][44] = 47;
    out.transformCombineTable[3][45] = 46;
    out.transformCombineTable[3][46] = 45;
    out.transformCombineTable[3][47] = 44;
    out.transformCombineTable[4][0] = 4;
    out.transformCombineTable[4][1] = 5;
    out.transformCombineTable[4][2] = 6;
    out.transformCombineTable[4][3] = 7;
    out.transformCombineTable[4][4] = 0;
    out.transformCombineTable[4][5] = 1;
    out.transformCombineTable[4][6] = 2;
    out.transformCombineTable[4][7] = 3;
    out.transformCombineTable[4][8] = 12;
    out.transformCombineTable[4][9] = 13;
    out.transformCombineTable[4][10] = 14;
    out.transformCombineTable[4][11] = 15;
    out.transformCombineTable[4][12] = 8;
    out.transformCombineTable[4][13] = 9;
    out.transformCombineTable[4][14] = 10;
    out.transformCombineTable[4][15] = 11;
    out.transformCombineTable[4][16] = 20;
    out.transformCombineTable[4][17] = 21;
    out.transformCombineTable[4][18] = 22;
    out.transformCombineTable[4][19] = 23;
    out.transformCombineTable[4][20] = 16;
    out.transformCombineTable[4][21] = 17;
    out.transformCombineTable[4][22] = 18;
    out.transformCombineTable[4][23] = 19;
    out.transformCombineTable[4][24] = 28;
    out.transformCombineTable[4][25] = 29;
    out.transformCombineTable[4][26] = 30;
    out.transformCombineTable[4][27] = 31;
    out.transformCombineTable[4][28] = 24;
    out.transformCombineTable[4][29] = 25;
    out.transformCombineTable[4][30] = 26;
    out.transformCombineTable[4][31] = 27;
    out.transformCombineTable[4][32] = 36;
    out.transformCombineTable[4][33] = 37;
    out.transformCombineTable[4][34] = 38;
    out.transformCombineTable[4][35] = 39;
    out.transformCombineTable[4][36] = 32;
    out.transformCombineTable[4][37] = 33;
    out.transformCombineTable[4][38] = 34;
    out.transformCombineTable[4][39] = 35;
    out.transformCombineTable[4][40] = 44;
    out.transformCombineTable[4][41] = 45;
    out.transformCombineTable[4][42] = 46;
    out.transformCombineTable[4][43] = 47;
    out.transformCombineTable[4][44] = 40;
    out.transformCombineTable[4][45] = 41;
    out.transformCombineTable[4][46] = 42;
    out.transformCombineTable[4][47] = 43;
    out.transformCombineTable[5][0] = 5;
    out.transformCombineTable[5][1] = 4;
    out.transformCombineTable[5][2] = 7;
    out.transformCombineTable[5][3] = 6;
    out.transformCombineTable[5][4] = 1;
    out.transformCombineTable[5][5] = 0;
    out.transformCombineTable[5][6] = 3;
    out.transformCombineTable[5][7] = 2;
    out.transformCombineTable[5][8] = 13;
    out.transformCombineTable[5][9] = 12;
    out.transformCombineTable[5][10] = 15;
    out.transformCombineTable[5][11] = 14;
    out.transformCombineTable[5][12] = 9;
    out.transformCombineTable[5][13] = 8;
    out.transformCombineTable[5][14] = 11;
    out.transformCombineTable[5][15] = 10;
    out.transformCombineTable[5][16] = 21;
    out.transformCombineTable[5][17] = 20;
    out.transformCombineTable[5][18] = 23;
    out.transformCombineTable[5][19] = 22;
    out.transformCombineTable[5][20] = 17;
    out.transformCombineTable[5][21] = 16;
    out.transformCombineTable[5][22] = 19;
    out.transformCombineTable[5][23] = 18;
    out.transformCombineTable[5][24] = 29;
    out.transformCombineTable[5][25] = 28;
    out.transformCombineTable[5][26] = 31;
    out.transformCombineTable[5][27] = 30;
    out.transformCombineTable[5][28] = 25;
    out.transformCombineTable[5][29] = 24;
    out.transformCombineTable[5][30] = 27;
    out.transformCombineTable[5][31] = 26;
    out.transformCombineTable[5][32] = 37;
    out.transformCombineTable[5][33] = 36;
    out.transformCombineTable[5][34] = 39;
    out.transformCombineTable[5][35] = 38;
    out.transformCombineTable[5][36] = 33;
    out.transformCombineTable[5][37] = 32;
    out.transformCombineTable[5][38] = 35;
    out.transformCombineTable[5][39] = 34;
    out.transformCombineTable[5][40] = 45;
    out.transformCombineTable[5][41] = 44;
    out.transformCombineTable[5][42] = 47;
    out.transformCombineTable[5][43] = 46;
    out.transformCombineTable[5][44] = 41;
    out.transformCombineTable[5][45] = 40;
    out.transformCombineTable[5][46] = 43;
    out.transformCombineTable[5][47] = 42;
    out.transformCombineTable[6][0] = 6;
    out.transformCombineTable[6][1] = 7;
    out.transformCombineTable[6][2] = 4;
    out.transformCombineTable[6][3] = 5;
    out.transformCombineTable[6][4] = 2;
    out.transformCombineTable[6][5] = 3;
    out.transformCombineTable[6][6] = 0;
    out.transformCombineTable[6][7] = 1;
    out.transformCombineTable[6][8] = 14;
    out.transformCombineTable[6][9] = 15;
    out.transformCombineTable[6][10] = 12;
    out.transformCombineTable[6][11] = 13;
    out.transformCombineTable[6][12] = 10;
    out.transformCombineTable[6][13] = 11;
    out.transformCombineTable[6][14] = 8;
    out.transformCombineTable[6][15] = 9;
    out.transformCombineTable[6][16] = 22;
    out.transformCombineTable[6][17] = 23;
    out.transformCombineTable[6][18] = 20;
    out.transformCombineTable[6][19] = 21;
    out.transformCombineTable[6][20] = 18;
    out.transformCombineTable[6][21] = 19;
    out.transformCombineTable[6][22] = 16;
    out.transformCombineTable[6][23] = 17;
    out.transformCombineTable[6][24] = 30;
    out.transformCombineTable[6][25] = 31;
    out.transformCombineTable[6][26] = 28;
    out.transformCombineTable[6][27] = 29;
    out.transformCombineTable[6][28] = 26;
    out.transformCombineTable[6][29] = 27;
    out.transformCombineTable[6][30] = 24;
    out.transformCombineTable[6][31] = 25;
    out.transformCombineTable[6][32] = 38;
    out.transformCombineTable[6][33] = 39;
    out.transformCombineTable[6][34] = 36;
    out.transformCombineTable[6][35] = 37;
    out.transformCombineTable[6][36] = 34;
    out.transformCombineTable[6][37] = 35;
    out.transformCombineTable[6][38] = 32;
    out.transformCombineTable[6][39] = 33;
    out.transformCombineTable[6][40] = 46;
    out.transformCombineTable[6][41] = 47;
    out.transformCombineTable[6][42] = 44;
    out.transformCombineTable[6][43] = 45;
    out.transformCombineTable[6][44] = 42;
    out.transformCombineTable[6][45] = 43;
    out.transformCombineTable[6][46] = 40;
    out.transformCombineTable[6][47] = 41;
    out.transformCombineTable[7][0] = 7;
    out.transformCombineTable[7][1] = 6;
    out.transformCombineTable[7][2] = 5;
    out.transformCombineTable[7][3] = 4;
    out.transformCombineTable[7][4] = 3;
    out.transformCombineTable[7][5] = 2;
    out.transformCombineTable[7][6] = 1;
    out.transformCombineTable[7][7] = 0;
    out.transformCombineTable[7][8] = 15;
    out.transformCombineTable[7][9] = 14;
    out.transformCombineTable[7][10] = 13;
    out.transformCombineTable[7][11] = 12;
    out.transformCombineTable[7][12] = 11;
    out.transformCombineTable[7][13] = 10;
    out.transformCombineTable[7][14] = 9;
    out.transformCombineTable[7][15] = 8;
    out.transformCombineTable[7][16] = 23;
    out.transformCombineTable[7][17] = 22;
    out.transformCombineTable[7][18] = 21;
    out.transformCombineTable[7][19] = 20;
    out.transformCombineTable[7][20] = 19;
    out.transformCombineTable[7][21] = 18;
    out.transformCombineTable[7][22] = 17;
    out.transformCombineTable[7][23] = 16;
    out.transformCombineTable[7][24] = 31;
    out.transformCombineTable[7][25] = 30;
    out.transformCombineTable[7][26] = 29;
    out.transformCombineTable[7][27] = 28;
    out.transformCombineTable[7][28] = 27;
    out.transformCombineTable[7][29] = 26;
    out.transformCombineTable[7][30] = 25;
    out.transformCombineTable[7][31] = 24;
    out.transformCombineTable[7][32] = 39;
    out.transformCombineTable[7][33] = 38;
    out.transformCombineTable[7][34] = 37;
    out.transformCombineTable[7][35] = 36;
    out.transformCombineTable[7][36] = 35;
    out.transformCombineTable[7][37] = 34;
    out.transformCombineTable[7][38] = 33;
    out.transformCombineTable[7][39] = 32;
    out.transformCombineTable[7][40] = 47;
    out.transformCombineTable[7][41] = 46;
    out.transformCombineTable[7][42] = 45;
    out.transformCombineTable[7][43] = 44;
    out.transformCombineTable[7][44] = 43;
    out.transformCombineTable[7][45] = 42;
    out.transformCombineTable[7][46] = 41;
    out.transformCombineTable[7][47] = 40;
    out.transformCombineTable[8][0] = 8;
    out.transformCombineTable[8][1] = 9;
    out.transformCombineTable[8][2] = 12;
    out.transformCombineTable[8][3] = 13;
    out.transformCombineTable[8][4] = 10;
    out.transformCombineTable[8][5] = 11;
    out.transformCombineTable[8][6] = 14;
    out.transformCombineTable[8][7] = 15;
    out.transformCombineTable[8][8] = 0;
    out.transformCombineTable[8][9] = 1;
    out.transformCombineTable[8][10] = 4;
    out.transformCombineTable[8][11] = 5;
    out.transformCombineTable[8][12] = 2;
    out.transformCombineTable[8][13] = 3;
    out.transformCombineTable[8][14] = 6;
    out.transformCombineTable[8][15] = 7;
    out.transformCombineTable[8][16] = 24;
    out.transformCombineTable[8][17] = 25;
    out.transformCombineTable[8][18] = 28;
    out.transformCombineTable[8][19] = 29;
    out.transformCombineTable[8][20] = 26;
    out.transformCombineTable[8][21] = 27;
    out.transformCombineTable[8][22] = 30;
    out.transformCombineTable[8][23] = 31;
    out.transformCombineTable[8][24] = 16;
    out.transformCombineTable[8][25] = 17;
    out.transformCombineTable[8][26] = 20;
    out.transformCombineTable[8][27] = 21;
    out.transformCombineTable[8][28] = 18;
    out.transformCombineTable[8][29] = 19;
    out.transformCombineTable[8][30] = 22;
    out.transformCombineTable[8][31] = 23;
    out.transformCombineTable[8][32] = 40;
    out.transformCombineTable[8][33] = 41;
    out.transformCombineTable[8][34] = 44;
    out.transformCombineTable[8][35] = 45;
    out.transformCombineTable[8][36] = 42;
    out.transformCombineTable[8][37] = 43;
    out.transformCombineTable[8][38] = 46;
    out.transformCombineTable[8][39] = 47;
    out.transformCombineTable[8][40] = 32;
    out.transformCombineTable[8][41] = 33;
    out.transformCombineTable[8][42] = 36;
    out.transformCombineTable[8][43] = 37;
    out.transformCombineTable[8][44] = 34;
    out.transformCombineTable[8][45] = 35;
    out.transformCombineTable[8][46] = 38;
    out.transformCombineTable[8][47] = 39;
    out.transformCombineTable[9][0] = 9;
    out.transformCombineTable[9][1] = 8;
    out.transformCombineTable[9][2] = 13;
    out.transformCombineTable[9][3] = 12;
    out.transformCombineTable[9][4] = 11;
    out.transformCombineTable[9][5] = 10;
    out.transformCombineTable[9][6] = 15;
    out.transformCombineTable[9][7] = 14;
    out.transformCombineTable[9][8] = 1;
    out.transformCombineTable[9][9] = 0;
    out.transformCombineTable[9][10] = 5;
    out.transformCombineTable[9][11] = 4;
    out.transformCombineTable[9][12] = 3;
    out.transformCombineTable[9][13] = 2;
    out.transformCombineTable[9][14] = 7;
    out.transformCombineTable[9][15] = 6;
    out.transformCombineTable[9][16] = 25;
    out.transformCombineTable[9][17] = 24;
    out.transformCombineTable[9][18] = 29;
    out.transformCombineTable[9][19] = 28;
    out.transformCombineTable[9][20] = 27;
    out.transformCombineTable[9][21] = 26;
    out.transformCombineTable[9][22] = 31;
    out.transformCombineTable[9][23] = 30;
    out.transformCombineTable[9][24] = 17;
    out.transformCombineTable[9][25] = 16;
    out.transformCombineTable[9][26] = 21;
    out.transformCombineTable[9][27] = 20;
    out.transformCombineTable[9][28] = 19;
    out.transformCombineTable[9][29] = 18;
    out.transformCombineTable[9][30] = 23;
    out.transformCombineTable[9][31] = 22;
    out.transformCombineTable[9][32] = 41;
    out.transformCombineTable[9][33] = 40;
    out.transformCombineTable[9][34] = 45;
    out.transformCombineTable[9][35] = 44;
    out.transformCombineTable[9][36] = 43;
    out.transformCombineTable[9][37] = 42;
    out.transformCombineTable[9][38] = 47;
    out.transformCombineTable[9][39] = 46;
    out.transformCombineTable[9][40] = 33;
    out.transformCombineTable[9][41] = 32;
    out.transformCombineTable[9][42] = 37;
    out.transformCombineTable[9][43] = 36;
    out.transformCombineTable[9][44] = 35;
    out.transformCombineTable[9][45] = 34;
    out.transformCombineTable[9][46] = 39;
    out.transformCombineTable[9][47] = 38;
    out.transformCombineTable[10][0] = 10;
    out.transformCombineTable[10][1] = 11;
    out.transformCombineTable[10][2] = 14;
    out.transformCombineTable[10][3] = 15;
    out.transformCombineTable[10][4] = 8;
    out.transformCombineTable[10][5] = 9;
    out.transformCombineTable[10][6] = 12;
    out.transformCombineTable[10][7] = 13;
    out.transformCombineTable[10][8] = 2;
    out.transformCombineTable[10][9] = 3;
    out.transformCombineTable[10][10] = 6;
    out.transformCombineTable[10][11] = 7;
    out.transformCombineTable[10][12] = 0;
    out.transformCombineTable[10][13] = 1;
    out.transformCombineTable[10][14] = 4;
    out.transformCombineTable[10][15] = 5;
    out.transformCombineTable[10][16] = 26;
    out.transformCombineTable[10][17] = 27;
    out.transformCombineTable[10][18] = 30;
    out.transformCombineTable[10][19] = 31;
    out.transformCombineTable[10][20] = 24;
    out.transformCombineTable[10][21] = 25;
    out.transformCombineTable[10][22] = 28;
    out.transformCombineTable[10][23] = 29;
    out.transformCombineTable[10][24] = 18;
    out.transformCombineTable[10][25] = 19;
    out.transformCombineTable[10][26] = 22;
    out.transformCombineTable[10][27] = 23;
    out.transformCombineTable[10][28] = 16;
    out.transformCombineTable[10][29] = 17;
    out.transformCombineTable[10][30] = 20;
    out.transformCombineTable[10][31] = 21;
    out.transformCombineTable[10][32] = 42;
    out.transformCombineTable[10][33] = 43;
    out.transformCombineTable[10][34] = 46;
    out.transformCombineTable[10][35] = 47;
    out.transformCombineTable[10][36] = 40;
    out.transformCombineTable[10][37] = 41;
    out.transformCombineTable[10][38] = 44;
    out.transformCombineTable[10][39] = 45;
    out.transformCombineTable[10][40] = 34;
    out.transformCombineTable[10][41] = 35;
    out.transformCombineTable[10][42] = 38;
    out.transformCombineTable[10][43] = 39;
    out.transformCombineTable[10][44] = 32;
    out.transformCombineTable[10][45] = 33;
    out.transformCombineTable[10][46] = 36;
    out.transformCombineTable[10][47] = 37;
    out.transformCombineTable[11][0] = 11;
    out.transformCombineTable[11][1] = 10;
    out.transformCombineTable[11][2] = 15;
    out.transformCombineTable[11][3] = 14;
    out.transformCombineTable[11][4] = 9;
    out.transformCombineTable[11][5] = 8;
    out.transformCombineTable[11][6] = 13;
    out.transformCombineTable[11][7] = 12;
    out.transformCombineTable[11][8] = 3;
    out.transformCombineTable[11][9] = 2;
    out.transformCombineTable[11][10] = 7;
    out.transformCombineTable[11][11] = 6;
    out.transformCombineTable[11][12] = 1;
    out.transformCombineTable[11][13] = 0;
    out.transformCombineTable[11][14] = 5;
    out.transformCombineTable[11][15] = 4;
    out.transformCombineTable[11][16] = 27;
    out.transformCombineTable[11][17] = 26;
    out.transformCombineTable[11][18] = 31;
    out.transformCombineTable[11][19] = 30;
    out.transformCombineTable[11][20] = 25;
    out.transformCombineTable[11][21] = 24;
    out.transformCombineTable[11][22] = 29;
    out.transformCombineTable[11][23] = 28;
    out.transformCombineTable[11][24] = 19;
    out.transformCombineTable[11][25] = 18;
    out.transformCombineTable[11][26] = 23;
    out.transformCombineTable[11][27] = 22;
    out.transformCombineTable[11][28] = 17;
    out.transformCombineTable[11][29] = 16;
    out.transformCombineTable[11][30] = 21;
    out.transformCombineTable[11][31] = 20;
    out.transformCombineTable[11][32] = 43;
    out.transformCombineTable[11][33] = 42;
    out.transformCombineTable[11][34] = 47;
    out.transformCombineTable[11][35] = 46;
    out.transformCombineTable[11][36] = 41;
    out.transformCombineTable[11][37] = 40;
    out.transformCombineTable[11][38] = 45;
    out.transformCombineTable[11][39] = 44;
    out.transformCombineTable[11][40] = 35;
    out.transformCombineTable[11][41] = 34;
    out.transformCombineTable[11][42] = 39;
    out.transformCombineTable[11][43] = 38;
    out.transformCombineTable[11][44] = 33;
    out.transformCombineTable[11][45] = 32;
    out.transformCombineTable[11][46] = 37;
    out.transformCombineTable[11][47] = 36;
    out.transformCombineTable[12][0] = 12;
    out.transformCombineTable[12][1] = 13;
    out.transformCombineTable[12][2] = 8;
    out.transformCombineTable[12][3] = 9;
    out.transformCombineTable[12][4] = 14;
    out.transformCombineTable[12][5] = 15;
    out.transformCombineTable[12][6] = 10;
    out.transformCombineTable[12][7] = 11;
    out.transformCombineTable[12][8] = 4;
    out.transformCombineTable[12][9] = 5;
    out.transformCombineTable[12][10] = 0;
    out.transformCombineTable[12][11] = 1;
    out.transformCombineTable[12][12] = 6;
    out.transformCombineTable[12][13] = 7;
    out.transformCombineTable[12][14] = 2;
    out.transformCombineTable[12][15] = 3;
    out.transformCombineTable[12][16] = 28;
    out.transformCombineTable[12][17] = 29;
    out.transformCombineTable[12][18] = 24;
    out.transformCombineTable[12][19] = 25;
    out.transformCombineTable[12][20] = 30;
    out.transformCombineTable[12][21] = 31;
    out.transformCombineTable[12][22] = 26;
    out.transformCombineTable[12][23] = 27;
    out.transformCombineTable[12][24] = 20;
    out.transformCombineTable[12][25] = 21;
    out.transformCombineTable[12][26] = 16;
    out.transformCombineTable[12][27] = 17;
    out.transformCombineTable[12][28] = 22;
    out.transformCombineTable[12][29] = 23;
    out.transformCombineTable[12][30] = 18;
    out.transformCombineTable[12][31] = 19;
    out.transformCombineTable[12][32] = 44;
    out.transformCombineTable[12][33] = 45;
    out.transformCombineTable[12][34] = 40;
    out.transformCombineTable[12][35] = 41;
    out.transformCombineTable[12][36] = 46;
    out.transformCombineTable[12][37] = 47;
    out.transformCombineTable[12][38] = 42;
    out.transformCombineTable[12][39] = 43;
    out.transformCombineTable[12][40] = 36;
    out.transformCombineTable[12][41] = 37;
    out.transformCombineTable[12][42] = 32;
    out.transformCombineTable[12][43] = 33;
    out.transformCombineTable[12][44] = 38;
    out.transformCombineTable[12][45] = 39;
    out.transformCombineTable[12][46] = 34;
    out.transformCombineTable[12][47] = 35;
    out.transformCombineTable[13][0] = 13;
    out.transformCombineTable[13][1] = 12;
    out.transformCombineTable[13][2] = 9;
    out.transformCombineTable[13][3] = 8;
    out.transformCombineTable[13][4] = 15;
    out.transformCombineTable[13][5] = 14;
    out.transformCombineTable[13][6] = 11;
    out.transformCombineTable[13][7] = 10;
    out.transformCombineTable[13][8] = 5;
    out.transformCombineTable[13][9] = 4;
    out.transformCombineTable[13][10] = 1;
    out.transformCombineTable[13][11] = 0;
    out.transformCombineTable[13][12] = 7;
    out.transformCombineTable[13][13] = 6;
    out.transformCombineTable[13][14] = 3;
    out.transformCombineTable[13][15] = 2;
    out.transformCombineTable[13][16] = 29;
    out.transformCombineTable[13][17] = 28;
    out.transformCombineTable[13][18] = 25;
    out.transformCombineTable[13][19] = 24;
    out.transformCombineTable[13][20] = 31;
    out.transformCombineTable[13][21] = 30;
    out.transformCombineTable[13][22] = 27;
    out.transformCombineTable[13][23] = 26;
    out.transformCombineTable[13][24] = 21;
    out.transformCombineTable[13][25] = 20;
    out.transformCombineTable[13][26] = 17;
    out.transformCombineTable[13][27] = 16;
    out.transformCombineTable[13][28] = 23;
    out.transformCombineTable[13][29] = 22;
    out.transformCombineTable[13][30] = 19;
    out.transformCombineTable[13][31] = 18;
    out.transformCombineTable[13][32] = 45;
    out.transformCombineTable[13][33] = 44;
    out.transformCombineTable[13][34] = 41;
    out.transformCombineTable[13][35] = 40;
    out.transformCombineTable[13][36] = 47;
    out.transformCombineTable[13][37] = 46;
    out.transformCombineTable[13][38] = 43;
    out.transformCombineTable[13][39] = 42;
    out.transformCombineTable[13][40] = 37;
    out.transformCombineTable[13][41] = 36;
    out.transformCombineTable[13][42] = 33;
    out.transformCombineTable[13][43] = 32;
    out.transformCombineTable[13][44] = 39;
    out.transformCombineTable[13][45] = 38;
    out.transformCombineTable[13][46] = 35;
    out.transformCombineTable[13][47] = 34;
    out.transformCombineTable[14][0] = 14;
    out.transformCombineTable[14][1] = 15;
    out.transformCombineTable[14][2] = 10;
    out.transformCombineTable[14][3] = 11;
    out.transformCombineTable[14][4] = 12;
    out.transformCombineTable[14][5] = 13;
    out.transformCombineTable[14][6] = 8;
    out.transformCombineTable[14][7] = 9;
    out.transformCombineTable[14][8] = 6;
    out.transformCombineTable[14][9] = 7;
    out.transformCombineTable[14][10] = 2;
    out.transformCombineTable[14][11] = 3;
    out.transformCombineTable[14][12] = 4;
    out.transformCombineTable[14][13] = 5;
    out.transformCombineTable[14][14] = 0;
    out.transformCombineTable[14][15] = 1;
    out.transformCombineTable[14][16] = 30;
    out.transformCombineTable[14][17] = 31;
    out.transformCombineTable[14][18] = 26;
    out.transformCombineTable[14][19] = 27;
    out.transformCombineTable[14][20] = 28;
    out.transformCombineTable[14][21] = 29;
    out.transformCombineTable[14][22] = 24;
    out.transformCombineTable[14][23] = 25;
    out.transformCombineTable[14][24] = 22;
    out.transformCombineTable[14][25] = 23;
    out.transformCombineTable[14][26] = 18;
    out.transformCombineTable[14][27] = 19;
    out.transformCombineTable[14][28] = 20;
    out.transformCombineTable[14][29] = 21;
    out.transformCombineTable[14][30] = 16;
    out.transformCombineTable[14][31] = 17;
    out.transformCombineTable[14][32] = 46;
    out.transformCombineTable[14][33] = 47;
    out.transformCombineTable[14][34] = 42;
    out.transformCombineTable[14][35] = 43;
    out.transformCombineTable[14][36] = 44;
    out.transformCombineTable[14][37] = 45;
    out.transformCombineTable[14][38] = 40;
    out.transformCombineTable[14][39] = 41;
    out.transformCombineTable[14][40] = 38;
    out.transformCombineTable[14][41] = 39;
    out.transformCombineTable[14][42] = 34;
    out.transformCombineTable[14][43] = 35;
    out.transformCombineTable[14][44] = 36;
    out.transformCombineTable[14][45] = 37;
    out.transformCombineTable[14][46] = 32;
    out.transformCombineTable[14][47] = 33;
    out.transformCombineTable[15][0] = 15;
    out.transformCombineTable[15][1] = 14;
    out.transformCombineTable[15][2] = 11;
    out.transformCombineTable[15][3] = 10;
    out.transformCombineTable[15][4] = 13;
    out.transformCombineTable[15][5] = 12;
    out.transformCombineTable[15][6] = 9;
    out.transformCombineTable[15][7] = 8;
    out.transformCombineTable[15][8] = 7;
    out.transformCombineTable[15][9] = 6;
    out.transformCombineTable[15][10] = 3;
    out.transformCombineTable[15][11] = 2;
    out.transformCombineTable[15][12] = 5;
    out.transformCombineTable[15][13] = 4;
    out.transformCombineTable[15][14] = 1;
    out.transformCombineTable[15][15] = 0;
    out.transformCombineTable[15][16] = 31;
    out.transformCombineTable[15][17] = 30;
    out.transformCombineTable[15][18] = 27;
    out.transformCombineTable[15][19] = 26;
    out.transformCombineTable[15][20] = 29;
    out.transformCombineTable[15][21] = 28;
    out.transformCombineTable[15][22] = 25;
    out.transformCombineTable[15][23] = 24;
    out.transformCombineTable[15][24] = 23;
    out.transformCombineTable[15][25] = 22;
    out.transformCombineTable[15][26] = 19;
    out.transformCombineTable[15][27] = 18;
    out.transformCombineTable[15][28] = 21;
    out.transformCombineTable[15][29] = 20;
    out.transformCombineTable[15][30] = 17;
    out.transformCombineTable[15][31] = 16;
    out.transformCombineTable[15][32] = 47;
    out.transformCombineTable[15][33] = 46;
    out.transformCombineTable[15][34] = 43;
    out.transformCombineTable[15][35] = 42;
    out.transformCombineTable[15][36] = 45;
    out.transformCombineTable[15][37] = 44;
    out.transformCombineTable[15][38] = 41;
    out.transformCombineTable[15][39] = 40;
    out.transformCombineTable[15][40] = 39;
    out.transformCombineTable[15][41] = 38;
    out.transformCombineTable[15][42] = 35;
    out.transformCombineTable[15][43] = 34;
    out.transformCombineTable[15][44] = 37;
    out.transformCombineTable[15][45] = 36;
    out.transformCombineTable[15][46] = 33;
    out.transformCombineTable[15][47] = 32;
    out.transformCombineTable[16][0] = 16;
    out.transformCombineTable[16][1] = 18;
    out.transformCombineTable[16][2] = 17;
    out.transformCombineTable[16][3] = 19;
    out.transformCombineTable[16][4] = 20;
    out.transformCombineTable[16][5] = 22;
    out.transformCombineTable[16][6] = 21;
    out.transformCombineTable[16][7] = 23;
    out.transformCombineTable[16][8] = 32;
    out.transformCombineTable[16][9] = 34;
    out.transformCombineTable[16][10] = 33;
    out.transformCombineTable[16][11] = 35;
    out.transformCombineTable[16][12] = 36;
    out.transformCombineTable[16][13] = 38;
    out.transformCombineTable[16][14] = 37;
    out.transformCombineTable[16][15] = 39;
    out.transformCombineTable[16][16] = 0;
    out.transformCombineTable[16][17] = 2;
    out.transformCombineTable[16][18] = 1;
    out.transformCombineTable[16][19] = 3;
    out.transformCombineTable[16][20] = 4;
    out.transformCombineTable[16][21] = 6;
    out.transformCombineTable[16][22] = 5;
    out.transformCombineTable[16][23] = 7;
    out.transformCombineTable[16][24] = 40;
    out.transformCombineTable[16][25] = 42;
    out.transformCombineTable[16][26] = 41;
    out.transformCombineTable[16][27] = 43;
    out.transformCombineTable[16][28] = 44;
    out.transformCombineTable[16][29] = 46;
    out.transformCombineTable[16][30] = 45;
    out.transformCombineTable[16][31] = 47;
    out.transformCombineTable[16][32] = 8;
    out.transformCombineTable[16][33] = 10;
    out.transformCombineTable[16][34] = 9;
    out.transformCombineTable[16][35] = 11;
    out.transformCombineTable[16][36] = 12;
    out.transformCombineTable[16][37] = 14;
    out.transformCombineTable[16][38] = 13;
    out.transformCombineTable[16][39] = 15;
    out.transformCombineTable[16][40] = 24;
    out.transformCombineTable[16][41] = 26;
    out.transformCombineTable[16][42] = 25;
    out.transformCombineTable[16][43] = 27;
    out.transformCombineTable[16][44] = 28;
    out.transformCombineTable[16][45] = 30;
    out.transformCombineTable[16][46] = 29;
    out.transformCombineTable[16][47] = 31;
    out.transformCombineTable[17][0] = 17;
    out.transformCombineTable[17][1] = 19;
    out.transformCombineTable[17][2] = 16;
    out.transformCombineTable[17][3] = 18;
    out.transformCombineTable[17][4] = 21;
    out.transformCombineTable[17][5] = 23;
    out.transformCombineTable[17][6] = 20;
    out.transformCombineTable[17][7] = 22;
    out.transformCombineTable[17][8] = 33;
    out.transformCombineTable[17][9] = 35;
    out.transformCombineTable[17][10] = 32;
    out.transformCombineTable[17][11] = 34;
    out.transformCombineTable[17][12] = 37;
    out.transformCombineTable[17][13] = 39;
    out.transformCombineTable[17][14] = 36;
    out.transformCombineTable[17][15] = 38;
    out.transformCombineTable[17][16] = 1;
    out.transformCombineTable[17][17] = 3;
    out.transformCombineTable[17][18] = 0;
    out.transformCombineTable[17][19] = 2;
    out.transformCombineTable[17][20] = 5;
    out.transformCombineTable[17][21] = 7;
    out.transformCombineTable[17][22] = 4;
    out.transformCombineTable[17][23] = 6;
    out.transformCombineTable[17][24] = 41;
    out.transformCombineTable[17][25] = 43;
    out.transformCombineTable[17][26] = 40;
    out.transformCombineTable[17][27] = 42;
    out.transformCombineTable[17][28] = 45;
    out.transformCombineTable[17][29] = 47;
    out.transformCombineTable[17][30] = 44;
    out.transformCombineTable[17][31] = 46;
    out.transformCombineTable[17][32] = 9;
    out.transformCombineTable[17][33] = 11;
    out.transformCombineTable[17][34] = 8;
    out.transformCombineTable[17][35] = 10;
    out.transformCombineTable[17][36] = 13;
    out.transformCombineTable[17][37] = 15;
    out.transformCombineTable[17][38] = 12;
    out.transformCombineTable[17][39] = 14;
    out.transformCombineTable[17][40] = 25;
    out.transformCombineTable[17][41] = 27;
    out.transformCombineTable[17][42] = 24;
    out.transformCombineTable[17][43] = 26;
    out.transformCombineTable[17][44] = 29;
    out.transformCombineTable[17][45] = 31;
    out.transformCombineTable[17][46] = 28;
    out.transformCombineTable[17][47] = 30;
    out.transformCombineTable[18][0] = 18;
    out.transformCombineTable[18][1] = 16;
    out.transformCombineTable[18][2] = 19;
    out.transformCombineTable[18][3] = 17;
    out.transformCombineTable[18][4] = 22;
    out.transformCombineTable[18][5] = 20;
    out.transformCombineTable[18][6] = 23;
    out.transformCombineTable[18][7] = 21;
    out.transformCombineTable[18][8] = 34;
    out.transformCombineTable[18][9] = 32;
    out.transformCombineTable[18][10] = 35;
    out.transformCombineTable[18][11] = 33;
    out.transformCombineTable[18][12] = 38;
    out.transformCombineTable[18][13] = 36;
    out.transformCombineTable[18][14] = 39;
    out.transformCombineTable[18][15] = 37;
    out.transformCombineTable[18][16] = 2;
    out.transformCombineTable[18][17] = 0;
    out.transformCombineTable[18][18] = 3;
    out.transformCombineTable[18][19] = 1;
    out.transformCombineTable[18][20] = 6;
    out.transformCombineTable[18][21] = 4;
    out.transformCombineTable[18][22] = 7;
    out.transformCombineTable[18][23] = 5;
    out.transformCombineTable[18][24] = 42;
    out.transformCombineTable[18][25] = 40;
    out.transformCombineTable[18][26] = 43;
    out.transformCombineTable[18][27] = 41;
    out.transformCombineTable[18][28] = 46;
    out.transformCombineTable[18][29] = 44;
    out.transformCombineTable[18][30] = 47;
    out.transformCombineTable[18][31] = 45;
    out.transformCombineTable[18][32] = 10;
    out.transformCombineTable[18][33] = 8;
    out.transformCombineTable[18][34] = 11;
    out.transformCombineTable[18][35] = 9;
    out.transformCombineTable[18][36] = 14;
    out.transformCombineTable[18][37] = 12;
    out.transformCombineTable[18][38] = 15;
    out.transformCombineTable[18][39] = 13;
    out.transformCombineTable[18][40] = 26;
    out.transformCombineTable[18][41] = 24;
    out.transformCombineTable[18][42] = 27;
    out.transformCombineTable[18][43] = 25;
    out.transformCombineTable[18][44] = 30;
    out.transformCombineTable[18][45] = 28;
    out.transformCombineTable[18][46] = 31;
    out.transformCombineTable[18][47] = 29;
    out.transformCombineTable[19][0] = 19;
    out.transformCombineTable[19][1] = 17;
    out.transformCombineTable[19][2] = 18;
    out.transformCombineTable[19][3] = 16;
    out.transformCombineTable[19][4] = 23;
    out.transformCombineTable[19][5] = 21;
    out.transformCombineTable[19][6] = 22;
    out.transformCombineTable[19][7] = 20;
    out.transformCombineTable[19][8] = 35;
    out.transformCombineTable[19][9] = 33;
    out.transformCombineTable[19][10] = 34;
    out.transformCombineTable[19][11] = 32;
    out.transformCombineTable[19][12] = 39;
    out.transformCombineTable[19][13] = 37;
    out.transformCombineTable[19][14] = 38;
    out.transformCombineTable[19][15] = 36;
    out.transformCombineTable[19][16] = 3;
    out.transformCombineTable[19][17] = 1;
    out.transformCombineTable[19][18] = 2;
    out.transformCombineTable[19][19] = 0;
    out.transformCombineTable[19][20] = 7;
    out.transformCombineTable[19][21] = 5;
    out.transformCombineTable[19][22] = 6;
    out.transformCombineTable[19][23] = 4;
    out.transformCombineTable[19][24] = 43;
    out.transformCombineTable[19][25] = 41;
    out.transformCombineTable[19][26] = 42;
    out.transformCombineTable[19][27] = 40;
    out.transformCombineTable[19][28] = 47;
    out.transformCombineTable[19][29] = 45;
    out.transformCombineTable[19][30] = 46;
    out.transformCombineTable[19][31] = 44;
    out.transformCombineTable[19][32] = 11;
    out.transformCombineTable[19][33] = 9;
    out.transformCombineTable[19][34] = 10;
    out.transformCombineTable[19][35] = 8;
    out.transformCombineTable[19][36] = 15;
    out.transformCombineTable[19][37] = 13;
    out.transformCombineTable[19][38] = 14;
    out.transformCombineTable[19][39] = 12;
    out.transformCombineTable[19][40] = 27;
    out.transformCombineTable[19][41] = 25;
    out.transformCombineTable[19][42] = 26;
    out.transformCombineTable[19][43] = 24;
    out.transformCombineTable[19][44] = 31;
    out.transformCombineTable[19][45] = 29;
    out.transformCombineTable[19][46] = 30;
    out.transformCombineTable[19][47] = 28;
    out.transformCombineTable[20][0] = 20;
    out.transformCombineTable[20][1] = 22;
    out.transformCombineTable[20][2] = 21;
    out.transformCombineTable[20][3] = 23;
    out.transformCombineTable[20][4] = 16;
    out.transformCombineTable[20][5] = 18;
    out.transformCombineTable[20][6] = 17;
    out.transformCombineTable[20][7] = 19;
    out.transformCombineTable[20][8] = 36;
    out.transformCombineTable[20][9] = 38;
    out.transformCombineTable[20][10] = 37;
    out.transformCombineTable[20][11] = 39;
    out.transformCombineTable[20][12] = 32;
    out.transformCombineTable[20][13] = 34;
    out.transformCombineTable[20][14] = 33;
    out.transformCombineTable[20][15] = 35;
    out.transformCombineTable[20][16] = 4;
    out.transformCombineTable[20][17] = 6;
    out.transformCombineTable[20][18] = 5;
    out.transformCombineTable[20][19] = 7;
    out.transformCombineTable[20][20] = 0;
    out.transformCombineTable[20][21] = 2;
    out.transformCombineTable[20][22] = 1;
    out.transformCombineTable[20][23] = 3;
    out.transformCombineTable[20][24] = 44;
    out.transformCombineTable[20][25] = 46;
    out.transformCombineTable[20][26] = 45;
    out.transformCombineTable[20][27] = 47;
    out.transformCombineTable[20][28] = 40;
    out.transformCombineTable[20][29] = 42;
    out.transformCombineTable[20][30] = 41;
    out.transformCombineTable[20][31] = 43;
    out.transformCombineTable[20][32] = 12;
    out.transformCombineTable[20][33] = 14;
    out.transformCombineTable[20][34] = 13;
    out.transformCombineTable[20][35] = 15;
    out.transformCombineTable[20][36] = 8;
    out.transformCombineTable[20][37] = 10;
    out.transformCombineTable[20][38] = 9;
    out.transformCombineTable[20][39] = 11;
    out.transformCombineTable[20][40] = 28;
    out.transformCombineTable[20][41] = 30;
    out.transformCombineTable[20][42] = 29;
    out.transformCombineTable[20][43] = 31;
    out.transformCombineTable[20][44] = 24;
    out.transformCombineTable[20][45] = 26;
    out.transformCombineTable[20][46] = 25;
    out.transformCombineTable[20][47] = 27;
    out.transformCombineTable[21][0] = 21;
    out.transformCombineTable[21][1] = 23;
    out.transformCombineTable[21][2] = 20;
    out.transformCombineTable[21][3] = 22;
    out.transformCombineTable[21][4] = 17;
    out.transformCombineTable[21][5] = 19;
    out.transformCombineTable[21][6] = 16;
    out.transformCombineTable[21][7] = 18;
    out.transformCombineTable[21][8] = 37;
    out.transformCombineTable[21][9] = 39;
    out.transformCombineTable[21][10] = 36;
    out.transformCombineTable[21][11] = 38;
    out.transformCombineTable[21][12] = 33;
    out.transformCombineTable[21][13] = 35;
    out.transformCombineTable[21][14] = 32;
    out.transformCombineTable[21][15] = 34;
    out.transformCombineTable[21][16] = 5;
    out.transformCombineTable[21][17] = 7;
    out.transformCombineTable[21][18] = 4;
    out.transformCombineTable[21][19] = 6;
    out.transformCombineTable[21][20] = 1;
    out.transformCombineTable[21][21] = 3;
    out.transformCombineTable[21][22] = 0;
    out.transformCombineTable[21][23] = 2;
    out.transformCombineTable[21][24] = 45;
    out.transformCombineTable[21][25] = 47;
    out.transformCombineTable[21][26] = 44;
    out.transformCombineTable[21][27] = 46;
    out.transformCombineTable[21][28] = 41;
    out.transformCombineTable[21][29] = 43;
    out.transformCombineTable[21][30] = 40;
    out.transformCombineTable[21][31] = 42;
    out.transformCombineTable[21][32] = 13;
    out.transformCombineTable[21][33] = 15;
    out.transformCombineTable[21][34] = 12;
    out.transformCombineTable[21][35] = 14;
    out.transformCombineTable[21][36] = 9;
    out.transformCombineTable[21][37] = 11;
    out.transformCombineTable[21][38] = 8;
    out.transformCombineTable[21][39] = 10;
    out.transformCombineTable[21][40] = 29;
    out.transformCombineTable[21][41] = 31;
    out.transformCombineTable[21][42] = 28;
    out.transformCombineTable[21][43] = 30;
    out.transformCombineTable[21][44] = 25;
    out.transformCombineTable[21][45] = 27;
    out.transformCombineTable[21][46] = 24;
    out.transformCombineTable[21][47] = 26;
    out.transformCombineTable[22][0] = 22;
    out.transformCombineTable[22][1] = 20;
    out.transformCombineTable[22][2] = 23;
    out.transformCombineTable[22][3] = 21;
    out.transformCombineTable[22][4] = 18;
    out.transformCombineTable[22][5] = 16;
    out.transformCombineTable[22][6] = 19;
    out.transformCombineTable[22][7] = 17;
    out.transformCombineTable[22][8] = 38;
    out.transformCombineTable[22][9] = 36;
    out.transformCombineTable[22][10] = 39;
    out.transformCombineTable[22][11] = 37;
    out.transformCombineTable[22][12] = 34;
    out.transformCombineTable[22][13] = 32;
    out.transformCombineTable[22][14] = 35;
    out.transformCombineTable[22][15] = 33;
    out.transformCombineTable[22][16] = 6;
    out.transformCombineTable[22][17] = 4;
    out.transformCombineTable[22][18] = 7;
    out.transformCombineTable[22][19] = 5;
    out.transformCombineTable[22][20] = 2;
    out.transformCombineTable[22][21] = 0;
    out.transformCombineTable[22][22] = 3;
    out.transformCombineTable[22][23] = 1;
    out.transformCombineTable[22][24] = 46;
    out.transformCombineTable[22][25] = 44;
    out.transformCombineTable[22][26] = 47;
    out.transformCombineTable[22][27] = 45;
    out.transformCombineTable[22][28] = 42;
    out.transformCombineTable[22][29] = 40;
    out.transformCombineTable[22][30] = 43;
    out.transformCombineTable[22][31] = 41;
    out.transformCombineTable[22][32] = 14;
    out.transformCombineTable[22][33] = 12;
    out.transformCombineTable[22][34] = 15;
    out.transformCombineTable[22][35] = 13;
    out.transformCombineTable[22][36] = 10;
    out.transformCombineTable[22][37] = 8;
    out.transformCombineTable[22][38] = 11;
    out.transformCombineTable[22][39] = 9;
    out.transformCombineTable[22][40] = 30;
    out.transformCombineTable[22][41] = 28;
    out.transformCombineTable[22][42] = 31;
    out.transformCombineTable[22][43] = 29;
    out.transformCombineTable[22][44] = 26;
    out.transformCombineTable[22][45] = 24;
    out.transformCombineTable[22][46] = 27;
    out.transformCombineTable[22][47] = 25;
    out.transformCombineTable[23][0] = 23;
    out.transformCombineTable[23][1] = 21;
    out.transformCombineTable[23][2] = 22;
    out.transformCombineTable[23][3] = 20;
    out.transformCombineTable[23][4] = 19;
    out.transformCombineTable[23][5] = 17;
    out.transformCombineTable[23][6] = 18;
    out.transformCombineTable[23][7] = 16;
    out.transformCombineTable[23][8] = 39;
    out.transformCombineTable[23][9] = 37;
    out.transformCombineTable[23][10] = 38;
    out.transformCombineTable[23][11] = 36;
    out.transformCombineTable[23][12] = 35;
    out.transformCombineTable[23][13] = 33;
    out.transformCombineTable[23][14] = 34;
    out.transformCombineTable[23][15] = 32;
    out.transformCombineTable[23][16] = 7;
    out.transformCombineTable[23][17] = 5;
    out.transformCombineTable[23][18] = 6;
    out.transformCombineTable[23][19] = 4;
    out.transformCombineTable[23][20] = 3;
    out.transformCombineTable[23][21] = 1;
    out.transformCombineTable[23][22] = 2;
    out.transformCombineTable[23][23] = 0;
    out.transformCombineTable[23][24] = 47;
    out.transformCombineTable[23][25] = 45;
    out.transformCombineTable[23][26] = 46;
    out.transformCombineTable[23][27] = 44;
    out.transformCombineTable[23][28] = 43;
    out.transformCombineTable[23][29] = 41;
    out.transformCombineTable[23][30] = 42;
    out.transformCombineTable[23][31] = 40;
    out.transformCombineTable[23][32] = 15;
    out.transformCombineTable[23][33] = 13;
    out.transformCombineTable[23][34] = 14;
    out.transformCombineTable[23][35] = 12;
    out.transformCombineTable[23][36] = 11;
    out.transformCombineTable[23][37] = 9;
    out.transformCombineTable[23][38] = 10;
    out.transformCombineTable[23][39] = 8;
    out.transformCombineTable[23][40] = 31;
    out.transformCombineTable[23][41] = 29;
    out.transformCombineTable[23][42] = 30;
    out.transformCombineTable[23][43] = 28;
    out.transformCombineTable[23][44] = 27;
    out.transformCombineTable[23][45] = 25;
    out.transformCombineTable[23][46] = 26;
    out.transformCombineTable[23][47] = 24;
    out.transformCombineTable[24][0] = 24;
    out.transformCombineTable[24][1] = 28;
    out.transformCombineTable[24][2] = 25;
    out.transformCombineTable[24][3] = 29;
    out.transformCombineTable[24][4] = 26;
    out.transformCombineTable[24][5] = 30;
    out.transformCombineTable[24][6] = 27;
    out.transformCombineTable[24][7] = 31;
    out.transformCombineTable[24][8] = 40;
    out.transformCombineTable[24][9] = 44;
    out.transformCombineTable[24][10] = 41;
    out.transformCombineTable[24][11] = 45;
    out.transformCombineTable[24][12] = 42;
    out.transformCombineTable[24][13] = 46;
    out.transformCombineTable[24][14] = 43;
    out.transformCombineTable[24][15] = 47;
    out.transformCombineTable[24][16] = 8;
    out.transformCombineTable[24][17] = 12;
    out.transformCombineTable[24][18] = 9;
    out.transformCombineTable[24][19] = 13;
    out.transformCombineTable[24][20] = 10;
    out.transformCombineTable[24][21] = 14;
    out.transformCombineTable[24][22] = 11;
    out.transformCombineTable[24][23] = 15;
    out.transformCombineTable[24][24] = 32;
    out.transformCombineTable[24][25] = 36;
    out.transformCombineTable[24][26] = 33;
    out.transformCombineTable[24][27] = 37;
    out.transformCombineTable[24][28] = 34;
    out.transformCombineTable[24][29] = 38;
    out.transformCombineTable[24][30] = 35;
    out.transformCombineTable[24][31] = 39;
    out.transformCombineTable[24][32] = 0;
    out.transformCombineTable[24][33] = 4;
    out.transformCombineTable[24][34] = 1;
    out.transformCombineTable[24][35] = 5;
    out.transformCombineTable[24][36] = 2;
    out.transformCombineTable[24][37] = 6;
    out.transformCombineTable[24][38] = 3;
    out.transformCombineTable[24][39] = 7;
    out.transformCombineTable[24][40] = 16;
    out.transformCombineTable[24][41] = 20;
    out.transformCombineTable[24][42] = 17;
    out.transformCombineTable[24][43] = 21;
    out.transformCombineTable[24][44] = 18;
    out.transformCombineTable[24][45] = 22;
    out.transformCombineTable[24][46] = 19;
    out.transformCombineTable[24][47] = 23;
    out.transformCombineTable[25][0] = 25;
    out.transformCombineTable[25][1] = 29;
    out.transformCombineTable[25][2] = 24;
    out.transformCombineTable[25][3] = 28;
    out.transformCombineTable[25][4] = 27;
    out.transformCombineTable[25][5] = 31;
    out.transformCombineTable[25][6] = 26;
    out.transformCombineTable[25][7] = 30;
    out.transformCombineTable[25][8] = 41;
    out.transformCombineTable[25][9] = 45;
    out.transformCombineTable[25][10] = 40;
    out.transformCombineTable[25][11] = 44;
    out.transformCombineTable[25][12] = 43;
    out.transformCombineTable[25][13] = 47;
    out.transformCombineTable[25][14] = 42;
    out.transformCombineTable[25][15] = 46;
    out.transformCombineTable[25][16] = 9;
    out.transformCombineTable[25][17] = 13;
    out.transformCombineTable[25][18] = 8;
    out.transformCombineTable[25][19] = 12;
    out.transformCombineTable[25][20] = 11;
    out.transformCombineTable[25][21] = 15;
    out.transformCombineTable[25][22] = 10;
    out.transformCombineTable[25][23] = 14;
    out.transformCombineTable[25][24] = 33;
    out.transformCombineTable[25][25] = 37;
    out.transformCombineTable[25][26] = 32;
    out.transformCombineTable[25][27] = 36;
    out.transformCombineTable[25][28] = 35;
    out.transformCombineTable[25][29] = 39;
    out.transformCombineTable[25][30] = 34;
    out.transformCombineTable[25][31] = 38;
    out.transformCombineTable[25][32] = 1;
    out.transformCombineTable[25][33] = 5;
    out.transformCombineTable[25][34] = 0;
    out.transformCombineTable[25][35] = 4;
    out.transformCombineTable[25][36] = 3;
    out.transformCombineTable[25][37] = 7;
    out.transformCombineTable[25][38] = 2;
    out.transformCombineTable[25][39] = 6;
    out.transformCombineTable[25][40] = 17;
    out.transformCombineTable[25][41] = 21;
    out.transformCombineTable[25][42] = 16;
    out.transformCombineTable[25][43] = 20;
    out.transformCombineTable[25][44] = 19;
    out.transformCombineTable[25][45] = 23;
    out.transformCombineTable[25][46] = 18;
    out.transformCombineTable[25][47] = 22;
    out.transformCombineTable[26][0] = 26;
    out.transformCombineTable[26][1] = 30;
    out.transformCombineTable[26][2] = 27;
    out.transformCombineTable[26][3] = 31;
    out.transformCombineTable[26][4] = 24;
    out.transformCombineTable[26][5] = 28;
    out.transformCombineTable[26][6] = 25;
    out.transformCombineTable[26][7] = 29;
    out.transformCombineTable[26][8] = 42;
    out.transformCombineTable[26][9] = 46;
    out.transformCombineTable[26][10] = 43;
    out.transformCombineTable[26][11] = 47;
    out.transformCombineTable[26][12] = 40;
    out.transformCombineTable[26][13] = 44;
    out.transformCombineTable[26][14] = 41;
    out.transformCombineTable[26][15] = 45;
    out.transformCombineTable[26][16] = 10;
    out.transformCombineTable[26][17] = 14;
    out.transformCombineTable[26][18] = 11;
    out.transformCombineTable[26][19] = 15;
    out.transformCombineTable[26][20] = 8;
    out.transformCombineTable[26][21] = 12;
    out.transformCombineTable[26][22] = 9;
    out.transformCombineTable[26][23] = 13;
    out.transformCombineTable[26][24] = 34;
    out.transformCombineTable[26][25] = 38;
    out.transformCombineTable[26][26] = 35;
    out.transformCombineTable[26][27] = 39;
    out.transformCombineTable[26][28] = 32;
    out.transformCombineTable[26][29] = 36;
    out.transformCombineTable[26][30] = 33;
    out.transformCombineTable[26][31] = 37;
    out.transformCombineTable[26][32] = 2;
    out.transformCombineTable[26][33] = 6;
    out.transformCombineTable[26][34] = 3;
    out.transformCombineTable[26][35] = 7;
    out.transformCombineTable[26][36] = 0;
    out.transformCombineTable[26][37] = 4;
    out.transformCombineTable[26][38] = 1;
    out.transformCombineTable[26][39] = 5;
    out.transformCombineTable[26][40] = 18;
    out.transformCombineTable[26][41] = 22;
    out.transformCombineTable[26][42] = 19;
    out.transformCombineTable[26][43] = 23;
    out.transformCombineTable[26][44] = 16;
    out.transformCombineTable[26][45] = 20;
    out.transformCombineTable[26][46] = 17;
    out.transformCombineTable[26][47] = 21;
    out.transformCombineTable[27][0] = 27;
    out.transformCombineTable[27][1] = 31;
    out.transformCombineTable[27][2] = 26;
    out.transformCombineTable[27][3] = 30;
    out.transformCombineTable[27][4] = 25;
    out.transformCombineTable[27][5] = 29;
    out.transformCombineTable[27][6] = 24;
    out.transformCombineTable[27][7] = 28;
    out.transformCombineTable[27][8] = 43;
    out.transformCombineTable[27][9] = 47;
    out.transformCombineTable[27][10] = 42;
    out.transformCombineTable[27][11] = 46;
    out.transformCombineTable[27][12] = 41;
    out.transformCombineTable[27][13] = 45;
    out.transformCombineTable[27][14] = 40;
    out.transformCombineTable[27][15] = 44;
    out.transformCombineTable[27][16] = 11;
    out.transformCombineTable[27][17] = 15;
    out.transformCombineTable[27][18] = 10;
    out.transformCombineTable[27][19] = 14;
    out.transformCombineTable[27][20] = 9;
    out.transformCombineTable[27][21] = 13;
    out.transformCombineTable[27][22] = 8;
    out.transformCombineTable[27][23] = 12;
    out.transformCombineTable[27][24] = 35;
    out.transformCombineTable[27][25] = 39;
    out.transformCombineTable[27][26] = 34;
    out.transformCombineTable[27][27] = 38;
    out.transformCombineTable[27][28] = 33;
    out.transformCombineTable[27][29] = 37;
    out.transformCombineTable[27][30] = 32;
    out.transformCombineTable[27][31] = 36;
    out.transformCombineTable[27][32] = 3;
    out.transformCombineTable[27][33] = 7;
    out.transformCombineTable[27][34] = 2;
    out.transformCombineTable[27][35] = 6;
    out.transformCombineTable[27][36] = 1;
    out.transformCombineTable[27][37] = 5;
    out.transformCombineTable[27][38] = 0;
    out.transformCombineTable[27][39] = 4;
    out.transformCombineTable[27][40] = 19;
    out.transformCombineTable[27][41] = 23;
    out.transformCombineTable[27][42] = 18;
    out.transformCombineTable[27][43] = 22;
    out.transformCombineTable[27][44] = 17;
    out.transformCombineTable[27][45] = 21;
    out.transformCombineTable[27][46] = 16;
    out.transformCombineTable[27][47] = 20;
    out.transformCombineTable[28][0] = 28;
    out.transformCombineTable[28][1] = 24;
    out.transformCombineTable[28][2] = 29;
    out.transformCombineTable[28][3] = 25;
    out.transformCombineTable[28][4] = 30;
    out.transformCombineTable[28][5] = 26;
    out.transformCombineTable[28][6] = 31;
    out.transformCombineTable[28][7] = 27;
    out.transformCombineTable[28][8] = 44;
    out.transformCombineTable[28][9] = 40;
    out.transformCombineTable[28][10] = 45;
    out.transformCombineTable[28][11] = 41;
    out.transformCombineTable[28][12] = 46;
    out.transformCombineTable[28][13] = 42;
    out.transformCombineTable[28][14] = 47;
    out.transformCombineTable[28][15] = 43;
    out.transformCombineTable[28][16] = 12;
    out.transformCombineTable[28][17] = 8;
    out.transformCombineTable[28][18] = 13;
    out.transformCombineTable[28][19] = 9;
    out.transformCombineTable[28][20] = 14;
    out.transformCombineTable[28][21] = 10;
    out.transformCombineTable[28][22] = 15;
    out.transformCombineTable[28][23] = 11;
    out.transformCombineTable[28][24] = 36;
    out.transformCombineTable[28][25] = 32;
    out.transformCombineTable[28][26] = 37;
    out.transformCombineTable[28][27] = 33;
    out.transformCombineTable[28][28] = 38;
    out.transformCombineTable[28][29] = 34;
    out.transformCombineTable[28][30] = 39;
    out.transformCombineTable[28][31] = 35;
    out.transformCombineTable[28][32] = 4;
    out.transformCombineTable[28][33] = 0;
    out.transformCombineTable[28][34] = 5;
    out.transformCombineTable[28][35] = 1;
    out.transformCombineTable[28][36] = 6;
    out.transformCombineTable[28][37] = 2;
    out.transformCombineTable[28][38] = 7;
    out.transformCombineTable[28][39] = 3;
    out.transformCombineTable[28][40] = 20;
    out.transformCombineTable[28][41] = 16;
    out.transformCombineTable[28][42] = 21;
    out.transformCombineTable[28][43] = 17;
    out.transformCombineTable[28][44] = 22;
    out.transformCombineTable[28][45] = 18;
    out.transformCombineTable[28][46] = 23;
    out.transformCombineTable[28][47] = 19;
    out.transformCombineTable[29][0] = 29;
    out.transformCombineTable[29][1] = 25;
    out.transformCombineTable[29][2] = 28;
    out.transformCombineTable[29][3] = 24;
    out.transformCombineTable[29][4] = 31;
    out.transformCombineTable[29][5] = 27;
    out.transformCombineTable[29][6] = 30;
    out.transformCombineTable[29][7] = 26;
    out.transformCombineTable[29][8] = 45;
    out.transformCombineTable[29][9] = 41;
    out.transformCombineTable[29][10] = 44;
    out.transformCombineTable[29][11] = 40;
    out.transformCombineTable[29][12] = 47;
    out.transformCombineTable[29][13] = 43;
    out.transformCombineTable[29][14] = 46;
    out.transformCombineTable[29][15] = 42;
    out.transformCombineTable[29][16] = 13;
    out.transformCombineTable[29][17] = 9;
    out.transformCombineTable[29][18] = 12;
    out.transformCombineTable[29][19] = 8;
    out.transformCombineTable[29][20] = 15;
    out.transformCombineTable[29][21] = 11;
    out.transformCombineTable[29][22] = 14;
    out.transformCombineTable[29][23] = 10;
    out.transformCombineTable[29][24] = 37;
    out.transformCombineTable[29][25] = 33;
    out.transformCombineTable[29][26] = 36;
    out.transformCombineTable[29][27] = 32;
    out.transformCombineTable[29][28] = 39;
    out.transformCombineTable[29][29] = 35;
    out.transformCombineTable[29][30] = 38;
    out.transformCombineTable[29][31] = 34;
    out.transformCombineTable[29][32] = 5;
    out.transformCombineTable[29][33] = 1;
    out.transformCombineTable[29][34] = 4;
    out.transformCombineTable[29][35] = 0;
    out.transformCombineTable[29][36] = 7;
    out.transformCombineTable[29][37] = 3;
    out.transformCombineTable[29][38] = 6;
    out.transformCombineTable[29][39] = 2;
    out.transformCombineTable[29][40] = 21;
    out.transformCombineTable[29][41] = 17;
    out.transformCombineTable[29][42] = 20;
    out.transformCombineTable[29][43] = 16;
    out.transformCombineTable[29][44] = 23;
    out.transformCombineTable[29][45] = 19;
    out.transformCombineTable[29][46] = 22;
    out.transformCombineTable[29][47] = 18;
    out.transformCombineTable[30][0] = 30;
    out.transformCombineTable[30][1] = 26;
    out.transformCombineTable[30][2] = 31;
    out.transformCombineTable[30][3] = 27;
    out.transformCombineTable[30][4] = 28;
    out.transformCombineTable[30][5] = 24;
    out.transformCombineTable[30][6] = 29;
    out.transformCombineTable[30][7] = 25;
    out.transformCombineTable[30][8] = 46;
    out.transformCombineTable[30][9] = 42;
    out.transformCombineTable[30][10] = 47;
    out.transformCombineTable[30][11] = 43;
    out.transformCombineTable[30][12] = 44;
    out.transformCombineTable[30][13] = 40;
    out.transformCombineTable[30][14] = 45;
    out.transformCombineTable[30][15] = 41;
    out.transformCombineTable[30][16] = 14;
    out.transformCombineTable[30][17] = 10;
    out.transformCombineTable[30][18] = 15;
    out.transformCombineTable[30][19] = 11;
    out.transformCombineTable[30][20] = 12;
    out.transformCombineTable[30][21] = 8;
    out.transformCombineTable[30][22] = 13;
    out.transformCombineTable[30][23] = 9;
    out.transformCombineTable[30][24] = 38;
    out.transformCombineTable[30][25] = 34;
    out.transformCombineTable[30][26] = 39;
    out.transformCombineTable[30][27] = 35;
    out.transformCombineTable[30][28] = 36;
    out.transformCombineTable[30][29] = 32;
    out.transformCombineTable[30][30] = 37;
    out.transformCombineTable[30][31] = 33;
    out.transformCombineTable[30][32] = 6;
    out.transformCombineTable[30][33] = 2;
    out.transformCombineTable[30][34] = 7;
    out.transformCombineTable[30][35] = 3;
    out.transformCombineTable[30][36] = 4;
    out.transformCombineTable[30][37] = 0;
    out.transformCombineTable[30][38] = 5;
    out.transformCombineTable[30][39] = 1;
    out.transformCombineTable[30][40] = 22;
    out.transformCombineTable[30][41] = 18;
    out.transformCombineTable[30][42] = 23;
    out.transformCombineTable[30][43] = 19;
    out.transformCombineTable[30][44] = 20;
    out.transformCombineTable[30][45] = 16;
    out.transformCombineTable[30][46] = 21;
    out.transformCombineTable[30][47] = 17;
    out.transformCombineTable[31][0] = 31;
    out.transformCombineTable[31][1] = 27;
    out.transformCombineTable[31][2] = 30;
    out.transformCombineTable[31][3] = 26;
    out.transformCombineTable[31][4] = 29;
    out.transformCombineTable[31][5] = 25;
    out.transformCombineTable[31][6] = 28;
    out.transformCombineTable[31][7] = 24;
    out.transformCombineTable[31][8] = 47;
    out.transformCombineTable[31][9] = 43;
    out.transformCombineTable[31][10] = 46;
    out.transformCombineTable[31][11] = 42;
    out.transformCombineTable[31][12] = 45;
    out.transformCombineTable[31][13] = 41;
    out.transformCombineTable[31][14] = 44;
    out.transformCombineTable[31][15] = 40;
    out.transformCombineTable[31][16] = 15;
    out.transformCombineTable[31][17] = 11;
    out.transformCombineTable[31][18] = 14;
    out.transformCombineTable[31][19] = 10;
    out.transformCombineTable[31][20] = 13;
    out.transformCombineTable[31][21] = 9;
    out.transformCombineTable[31][22] = 12;
    out.transformCombineTable[31][23] = 8;
    out.transformCombineTable[31][24] = 39;
    out.transformCombineTable[31][25] = 35;
    out.transformCombineTable[31][26] = 38;
    out.transformCombineTable[31][27] = 34;
    out.transformCombineTable[31][28] = 37;
    out.transformCombineTable[31][29] = 33;
    out.transformCombineTable[31][30] = 36;
    out.transformCombineTable[31][31] = 32;
    out.transformCombineTable[31][32] = 7;
    out.transformCombineTable[31][33] = 3;
    out.transformCombineTable[31][34] = 6;
    out.transformCombineTable[31][35] = 2;
    out.transformCombineTable[31][36] = 5;
    out.transformCombineTable[31][37] = 1;
    out.transformCombineTable[31][38] = 4;
    out.transformCombineTable[31][39] = 0;
    out.transformCombineTable[31][40] = 23;
    out.transformCombineTable[31][41] = 19;
    out.transformCombineTable[31][42] = 22;
    out.transformCombineTable[31][43] = 18;
    out.transformCombineTable[31][44] = 21;
    out.transformCombineTable[31][45] = 17;
    out.transformCombineTable[31][46] = 20;
    out.transformCombineTable[31][47] = 16;
    out.transformCombineTable[32][0] = 32;
    out.transformCombineTable[32][1] = 34;
    out.transformCombineTable[32][2] = 36;
    out.transformCombineTable[32][3] = 38;
    out.transformCombineTable[32][4] = 33;
    out.transformCombineTable[32][5] = 35;
    out.transformCombineTable[32][6] = 37;
    out.transformCombineTable[32][7] = 39;
    out.transformCombineTable[32][8] = 16;
    out.transformCombineTable[32][9] = 18;
    out.transformCombineTable[32][10] = 20;
    out.transformCombineTable[32][11] = 22;
    out.transformCombineTable[32][12] = 17;
    out.transformCombineTable[32][13] = 19;
    out.transformCombineTable[32][14] = 21;
    out.transformCombineTable[32][15] = 23;
    out.transformCombineTable[32][16] = 40;
    out.transformCombineTable[32][17] = 42;
    out.transformCombineTable[32][18] = 44;
    out.transformCombineTable[32][19] = 46;
    out.transformCombineTable[32][20] = 41;
    out.transformCombineTable[32][21] = 43;
    out.transformCombineTable[32][22] = 45;
    out.transformCombineTable[32][23] = 47;
    out.transformCombineTable[32][24] = 0;
    out.transformCombineTable[32][25] = 2;
    out.transformCombineTable[32][26] = 4;
    out.transformCombineTable[32][27] = 6;
    out.transformCombineTable[32][28] = 1;
    out.transformCombineTable[32][29] = 3;
    out.transformCombineTable[32][30] = 5;
    out.transformCombineTable[32][31] = 7;
    out.transformCombineTable[32][32] = 24;
    out.transformCombineTable[32][33] = 26;
    out.transformCombineTable[32][34] = 28;
    out.transformCombineTable[32][35] = 30;
    out.transformCombineTable[32][36] = 25;
    out.transformCombineTable[32][37] = 27;
    out.transformCombineTable[32][38] = 29;
    out.transformCombineTable[32][39] = 31;
    out.transformCombineTable[32][40] = 8;
    out.transformCombineTable[32][41] = 10;
    out.transformCombineTable[32][42] = 12;
    out.transformCombineTable[32][43] = 14;
    out.transformCombineTable[32][44] = 9;
    out.transformCombineTable[32][45] = 11;
    out.transformCombineTable[32][46] = 13;
    out.transformCombineTable[32][47] = 15;
    out.transformCombineTable[33][0] = 33;
    out.transformCombineTable[33][1] = 35;
    out.transformCombineTable[33][2] = 37;
    out.transformCombineTable[33][3] = 39;
    out.transformCombineTable[33][4] = 32;
    out.transformCombineTable[33][5] = 34;
    out.transformCombineTable[33][6] = 36;
    out.transformCombineTable[33][7] = 38;
    out.transformCombineTable[33][8] = 17;
    out.transformCombineTable[33][9] = 19;
    out.transformCombineTable[33][10] = 21;
    out.transformCombineTable[33][11] = 23;
    out.transformCombineTable[33][12] = 16;
    out.transformCombineTable[33][13] = 18;
    out.transformCombineTable[33][14] = 20;
    out.transformCombineTable[33][15] = 22;
    out.transformCombineTable[33][16] = 41;
    out.transformCombineTable[33][17] = 43;
    out.transformCombineTable[33][18] = 45;
    out.transformCombineTable[33][19] = 47;
    out.transformCombineTable[33][20] = 40;
    out.transformCombineTable[33][21] = 42;
    out.transformCombineTable[33][22] = 44;
    out.transformCombineTable[33][23] = 46;
    out.transformCombineTable[33][24] = 1;
    out.transformCombineTable[33][25] = 3;
    out.transformCombineTable[33][26] = 5;
    out.transformCombineTable[33][27] = 7;
    out.transformCombineTable[33][28] = 0;
    out.transformCombineTable[33][29] = 2;
    out.transformCombineTable[33][30] = 4;
    out.transformCombineTable[33][31] = 6;
    out.transformCombineTable[33][32] = 25;
    out.transformCombineTable[33][33] = 27;
    out.transformCombineTable[33][34] = 29;
    out.transformCombineTable[33][35] = 31;
    out.transformCombineTable[33][36] = 24;
    out.transformCombineTable[33][37] = 26;
    out.transformCombineTable[33][38] = 28;
    out.transformCombineTable[33][39] = 30;
    out.transformCombineTable[33][40] = 9;
    out.transformCombineTable[33][41] = 11;
    out.transformCombineTable[33][42] = 13;
    out.transformCombineTable[33][43] = 15;
    out.transformCombineTable[33][44] = 8;
    out.transformCombineTable[33][45] = 10;
    out.transformCombineTable[33][46] = 12;
    out.transformCombineTable[33][47] = 14;
    out.transformCombineTable[34][0] = 34;
    out.transformCombineTable[34][1] = 32;
    out.transformCombineTable[34][2] = 38;
    out.transformCombineTable[34][3] = 36;
    out.transformCombineTable[34][4] = 35;
    out.transformCombineTable[34][5] = 33;
    out.transformCombineTable[34][6] = 39;
    out.transformCombineTable[34][7] = 37;
    out.transformCombineTable[34][8] = 18;
    out.transformCombineTable[34][9] = 16;
    out.transformCombineTable[34][10] = 22;
    out.transformCombineTable[34][11] = 20;
    out.transformCombineTable[34][12] = 19;
    out.transformCombineTable[34][13] = 17;
    out.transformCombineTable[34][14] = 23;
    out.transformCombineTable[34][15] = 21;
    out.transformCombineTable[34][16] = 42;
    out.transformCombineTable[34][17] = 40;
    out.transformCombineTable[34][18] = 46;
    out.transformCombineTable[34][19] = 44;
    out.transformCombineTable[34][20] = 43;
    out.transformCombineTable[34][21] = 41;
    out.transformCombineTable[34][22] = 47;
    out.transformCombineTable[34][23] = 45;
    out.transformCombineTable[34][24] = 2;
    out.transformCombineTable[34][25] = 0;
    out.transformCombineTable[34][26] = 6;
    out.transformCombineTable[34][27] = 4;
    out.transformCombineTable[34][28] = 3;
    out.transformCombineTable[34][29] = 1;
    out.transformCombineTable[34][30] = 7;
    out.transformCombineTable[34][31] = 5;
    out.transformCombineTable[34][32] = 26;
    out.transformCombineTable[34][33] = 24;
    out.transformCombineTable[34][34] = 30;
    out.transformCombineTable[34][35] = 28;
    out.transformCombineTable[34][36] = 27;
    out.transformCombineTable[34][37] = 25;
    out.transformCombineTable[34][38] = 31;
    out.transformCombineTable[34][39] = 29;
    out.transformCombineTable[34][40] = 10;
    out.transformCombineTable[34][41] = 8;
    out.transformCombineTable[34][42] = 14;
    out.transformCombineTable[34][43] = 12;
    out.transformCombineTable[34][44] = 11;
    out.transformCombineTable[34][45] = 9;
    out.transformCombineTable[34][46] = 15;
    out.transformCombineTable[34][47] = 13;
    out.transformCombineTable[35][0] = 35;
    out.transformCombineTable[35][1] = 33;
    out.transformCombineTable[35][2] = 39;
    out.transformCombineTable[35][3] = 37;
    out.transformCombineTable[35][4] = 34;
    out.transformCombineTable[35][5] = 32;
    out.transformCombineTable[35][6] = 38;
    out.transformCombineTable[35][7] = 36;
    out.transformCombineTable[35][8] = 19;
    out.transformCombineTable[35][9] = 17;
    out.transformCombineTable[35][10] = 23;
    out.transformCombineTable[35][11] = 21;
    out.transformCombineTable[35][12] = 18;
    out.transformCombineTable[35][13] = 16;
    out.transformCombineTable[35][14] = 22;
    out.transformCombineTable[35][15] = 20;
    out.transformCombineTable[35][16] = 43;
    out.transformCombineTable[35][17] = 41;
    out.transformCombineTable[35][18] = 47;
    out.transformCombineTable[35][19] = 45;
    out.transformCombineTable[35][20] = 42;
    out.transformCombineTable[35][21] = 40;
    out.transformCombineTable[35][22] = 46;
    out.transformCombineTable[35][23] = 44;
    out.transformCombineTable[35][24] = 3;
    out.transformCombineTable[35][25] = 1;
    out.transformCombineTable[35][26] = 7;
    out.transformCombineTable[35][27] = 5;
    out.transformCombineTable[35][28] = 2;
    out.transformCombineTable[35][29] = 0;
    out.transformCombineTable[35][30] = 6;
    out.transformCombineTable[35][31] = 4;
    out.transformCombineTable[35][32] = 27;
    out.transformCombineTable[35][33] = 25;
    out.transformCombineTable[35][34] = 31;
    out.transformCombineTable[35][35] = 29;
    out.transformCombineTable[35][36] = 26;
    out.transformCombineTable[35][37] = 24;
    out.transformCombineTable[35][38] = 30;
    out.transformCombineTable[35][39] = 28;
    out.transformCombineTable[35][40] = 11;
    out.transformCombineTable[35][41] = 9;
    out.transformCombineTable[35][42] = 15;
    out.transformCombineTable[35][43] = 13;
    out.transformCombineTable[35][44] = 10;
    out.transformCombineTable[35][45] = 8;
    out.transformCombineTable[35][46] = 14;
    out.transformCombineTable[35][47] = 12;
    out.transformCombineTable[36][0] = 36;
    out.transformCombineTable[36][1] = 38;
    out.transformCombineTable[36][2] = 32;
    out.transformCombineTable[36][3] = 34;
    out.transformCombineTable[36][4] = 37;
    out.transformCombineTable[36][5] = 39;
    out.transformCombineTable[36][6] = 33;
    out.transformCombineTable[36][7] = 35;
    out.transformCombineTable[36][8] = 20;
    out.transformCombineTable[36][9] = 22;
    out.transformCombineTable[36][10] = 16;
    out.transformCombineTable[36][11] = 18;
    out.transformCombineTable[36][12] = 21;
    out.transformCombineTable[36][13] = 23;
    out.transformCombineTable[36][14] = 17;
    out.transformCombineTable[36][15] = 19;
    out.transformCombineTable[36][16] = 44;
    out.transformCombineTable[36][17] = 46;
    out.transformCombineTable[36][18] = 40;
    out.transformCombineTable[36][19] = 42;
    out.transformCombineTable[36][20] = 45;
    out.transformCombineTable[36][21] = 47;
    out.transformCombineTable[36][22] = 41;
    out.transformCombineTable[36][23] = 43;
    out.transformCombineTable[36][24] = 4;
    out.transformCombineTable[36][25] = 6;
    out.transformCombineTable[36][26] = 0;
    out.transformCombineTable[36][27] = 2;
    out.transformCombineTable[36][28] = 5;
    out.transformCombineTable[36][29] = 7;
    out.transformCombineTable[36][30] = 1;
    out.transformCombineTable[36][31] = 3;
    out.transformCombineTable[36][32] = 28;
    out.transformCombineTable[36][33] = 30;
    out.transformCombineTable[36][34] = 24;
    out.transformCombineTable[36][35] = 26;
    out.transformCombineTable[36][36] = 29;
    out.transformCombineTable[36][37] = 31;
    out.transformCombineTable[36][38] = 25;
    out.transformCombineTable[36][39] = 27;
    out.transformCombineTable[36][40] = 12;
    out.transformCombineTable[36][41] = 14;
    out.transformCombineTable[36][42] = 8;
    out.transformCombineTable[36][43] = 10;
    out.transformCombineTable[36][44] = 13;
    out.transformCombineTable[36][45] = 15;
    out.transformCombineTable[36][46] = 9;
    out.transformCombineTable[36][47] = 11;
    out.transformCombineTable[37][0] = 37;
    out.transformCombineTable[37][1] = 39;
    out.transformCombineTable[37][2] = 33;
    out.transformCombineTable[37][3] = 35;
    out.transformCombineTable[37][4] = 36;
    out.transformCombineTable[37][5] = 38;
    out.transformCombineTable[37][6] = 32;
    out.transformCombineTable[37][7] = 34;
    out.transformCombineTable[37][8] = 21;
    out.transformCombineTable[37][9] = 23;
    out.transformCombineTable[37][10] = 17;
    out.transformCombineTable[37][11] = 19;
    out.transformCombineTable[37][12] = 20;
    out.transformCombineTable[37][13] = 22;
    out.transformCombineTable[37][14] = 16;
    out.transformCombineTable[37][15] = 18;
    out.transformCombineTable[37][16] = 45;
    out.transformCombineTable[37][17] = 47;
    out.transformCombineTable[37][18] = 41;
    out.transformCombineTable[37][19] = 43;
    out.transformCombineTable[37][20] = 44;
    out.transformCombineTable[37][21] = 46;
    out.transformCombineTable[37][22] = 40;
    out.transformCombineTable[37][23] = 42;
    out.transformCombineTable[37][24] = 5;
    out.transformCombineTable[37][25] = 7;
    out.transformCombineTable[37][26] = 1;
    out.transformCombineTable[37][27] = 3;
    out.transformCombineTable[37][28] = 4;
    out.transformCombineTable[37][29] = 6;
    out.transformCombineTable[37][30] = 0;
    out.transformCombineTable[37][31] = 2;
    out.transformCombineTable[37][32] = 29;
    out.transformCombineTable[37][33] = 31;
    out.transformCombineTable[37][34] = 25;
    out.transformCombineTable[37][35] = 27;
    out.transformCombineTable[37][36] = 28;
    out.transformCombineTable[37][37] = 30;
    out.transformCombineTable[37][38] = 24;
    out.transformCombineTable[37][39] = 26;
    out.transformCombineTable[37][40] = 13;
    out.transformCombineTable[37][41] = 15;
    out.transformCombineTable[37][42] = 9;
    out.transformCombineTable[37][43] = 11;
    out.transformCombineTable[37][44] = 12;
    out.transformCombineTable[37][45] = 14;
    out.transformCombineTable[37][46] = 8;
    out.transformCombineTable[37][47] = 10;
    out.transformCombineTable[38][0] = 38;
    out.transformCombineTable[38][1] = 36;
    out.transformCombineTable[38][2] = 34;
    out.transformCombineTable[38][3] = 32;
    out.transformCombineTable[38][4] = 39;
    out.transformCombineTable[38][5] = 37;
    out.transformCombineTable[38][6] = 35;
    out.transformCombineTable[38][7] = 33;
    out.transformCombineTable[38][8] = 22;
    out.transformCombineTable[38][9] = 20;
    out.transformCombineTable[38][10] = 18;
    out.transformCombineTable[38][11] = 16;
    out.transformCombineTable[38][12] = 23;
    out.transformCombineTable[38][13] = 21;
    out.transformCombineTable[38][14] = 19;
    out.transformCombineTable[38][15] = 17;
    out.transformCombineTable[38][16] = 46;
    out.transformCombineTable[38][17] = 44;
    out.transformCombineTable[38][18] = 42;
    out.transformCombineTable[38][19] = 40;
    out.transformCombineTable[38][20] = 47;
    out.transformCombineTable[38][21] = 45;
    out.transformCombineTable[38][22] = 43;
    out.transformCombineTable[38][23] = 41;
    out.transformCombineTable[38][24] = 6;
    out.transformCombineTable[38][25] = 4;
    out.transformCombineTable[38][26] = 2;
    out.transformCombineTable[38][27] = 0;
    out.transformCombineTable[38][28] = 7;
    out.transformCombineTable[38][29] = 5;
    out.transformCombineTable[38][30] = 3;
    out.transformCombineTable[38][31] = 1;
    out.transformCombineTable[38][32] = 30;
    out.transformCombineTable[38][33] = 28;
    out.transformCombineTable[38][34] = 26;
    out.transformCombineTable[38][35] = 24;
    out.transformCombineTable[38][36] = 31;
    out.transformCombineTable[38][37] = 29;
    out.transformCombineTable[38][38] = 27;
    out.transformCombineTable[38][39] = 25;
    out.transformCombineTable[38][40] = 14;
    out.transformCombineTable[38][41] = 12;
    out.transformCombineTable[38][42] = 10;
    out.transformCombineTable[38][43] = 8;
    out.transformCombineTable[38][44] = 15;
    out.transformCombineTable[38][45] = 13;
    out.transformCombineTable[38][46] = 11;
    out.transformCombineTable[38][47] = 9;
    out.transformCombineTable[39][0] = 39;
    out.transformCombineTable[39][1] = 37;
    out.transformCombineTable[39][2] = 35;
    out.transformCombineTable[39][3] = 33;
    out.transformCombineTable[39][4] = 38;
    out.transformCombineTable[39][5] = 36;
    out.transformCombineTable[39][6] = 34;
    out.transformCombineTable[39][7] = 32;
    out.transformCombineTable[39][8] = 23;
    out.transformCombineTable[39][9] = 21;
    out.transformCombineTable[39][10] = 19;
    out.transformCombineTable[39][11] = 17;
    out.transformCombineTable[39][12] = 22;
    out.transformCombineTable[39][13] = 20;
    out.transformCombineTable[39][14] = 18;
    out.transformCombineTable[39][15] = 16;
    out.transformCombineTable[39][16] = 47;
    out.transformCombineTable[39][17] = 45;
    out.transformCombineTable[39][18] = 43;
    out.transformCombineTable[39][19] = 41;
    out.transformCombineTable[39][20] = 46;
    out.transformCombineTable[39][21] = 44;
    out.transformCombineTable[39][22] = 42;
    out.transformCombineTable[39][23] = 40;
    out.transformCombineTable[39][24] = 7;
    out.transformCombineTable[39][25] = 5;
    out.transformCombineTable[39][26] = 3;
    out.transformCombineTable[39][27] = 1;
    out.transformCombineTable[39][28] = 6;
    out.transformCombineTable[39][29] = 4;
    out.transformCombineTable[39][30] = 2;
    out.transformCombineTable[39][31] = 0;
    out.transformCombineTable[39][32] = 31;
    out.transformCombineTable[39][33] = 29;
    out.transformCombineTable[39][34] = 27;
    out.transformCombineTable[39][35] = 25;
    out.transformCombineTable[39][36] = 30;
    out.transformCombineTable[39][37] = 28;
    out.transformCombineTable[39][38] = 26;
    out.transformCombineTable[39][39] = 24;
    out.transformCombineTable[39][40] = 15;
    out.transformCombineTable[39][41] = 13;
    out.transformCombineTable[39][42] = 11;
    out.transformCombineTable[39][43] = 9;
    out.transformCombineTable[39][44] = 14;
    out.transformCombineTable[39][45] = 12;
    out.transformCombineTable[39][46] = 10;
    out.transformCombineTable[39][47] = 8;
    out.transformCombineTable[40][0] = 40;
    out.transformCombineTable[40][1] = 44;
    out.transformCombineTable[40][2] = 42;
    out.transformCombineTable[40][3] = 46;
    out.transformCombineTable[40][4] = 41;
    out.transformCombineTable[40][5] = 45;
    out.transformCombineTable[40][6] = 43;
    out.transformCombineTable[40][7] = 47;
    out.transformCombineTable[40][8] = 24;
    out.transformCombineTable[40][9] = 28;
    out.transformCombineTable[40][10] = 26;
    out.transformCombineTable[40][11] = 30;
    out.transformCombineTable[40][12] = 25;
    out.transformCombineTable[40][13] = 29;
    out.transformCombineTable[40][14] = 27;
    out.transformCombineTable[40][15] = 31;
    out.transformCombineTable[40][16] = 32;
    out.transformCombineTable[40][17] = 36;
    out.transformCombineTable[40][18] = 34;
    out.transformCombineTable[40][19] = 38;
    out.transformCombineTable[40][20] = 33;
    out.transformCombineTable[40][21] = 37;
    out.transformCombineTable[40][22] = 35;
    out.transformCombineTable[40][23] = 39;
    out.transformCombineTable[40][24] = 8;
    out.transformCombineTable[40][25] = 12;
    out.transformCombineTable[40][26] = 10;
    out.transformCombineTable[40][27] = 14;
    out.transformCombineTable[40][28] = 9;
    out.transformCombineTable[40][29] = 13;
    out.transformCombineTable[40][30] = 11;
    out.transformCombineTable[40][31] = 15;
    out.transformCombineTable[40][32] = 16;
    out.transformCombineTable[40][33] = 20;
    out.transformCombineTable[40][34] = 18;
    out.transformCombineTable[40][35] = 22;
    out.transformCombineTable[40][36] = 17;
    out.transformCombineTable[40][37] = 21;
    out.transformCombineTable[40][38] = 19;
    out.transformCombineTable[40][39] = 23;
    out.transformCombineTable[40][40] = 0;
    out.transformCombineTable[40][41] = 4;
    out.transformCombineTable[40][42] = 2;
    out.transformCombineTable[40][43] = 6;
    out.transformCombineTable[40][44] = 1;
    out.transformCombineTable[40][45] = 5;
    out.transformCombineTable[40][46] = 3;
    out.transformCombineTable[40][47] = 7;
    out.transformCombineTable[41][0] = 41;
    out.transformCombineTable[41][1] = 45;
    out.transformCombineTable[41][2] = 43;
    out.transformCombineTable[41][3] = 47;
    out.transformCombineTable[41][4] = 40;
    out.transformCombineTable[41][5] = 44;
    out.transformCombineTable[41][6] = 42;
    out.transformCombineTable[41][7] = 46;
    out.transformCombineTable[41][8] = 25;
    out.transformCombineTable[41][9] = 29;
    out.transformCombineTable[41][10] = 27;
    out.transformCombineTable[41][11] = 31;
    out.transformCombineTable[41][12] = 24;
    out.transformCombineTable[41][13] = 28;
    out.transformCombineTable[41][14] = 26;
    out.transformCombineTable[41][15] = 30;
    out.transformCombineTable[41][16] = 33;
    out.transformCombineTable[41][17] = 37;
    out.transformCombineTable[41][18] = 35;
    out.transformCombineTable[41][19] = 39;
    out.transformCombineTable[41][20] = 32;
    out.transformCombineTable[41][21] = 36;
    out.transformCombineTable[41][22] = 34;
    out.transformCombineTable[41][23] = 38;
    out.transformCombineTable[41][24] = 9;
    out.transformCombineTable[41][25] = 13;
    out.transformCombineTable[41][26] = 11;
    out.transformCombineTable[41][27] = 15;
    out.transformCombineTable[41][28] = 8;
    out.transformCombineTable[41][29] = 12;
    out.transformCombineTable[41][30] = 10;
    out.transformCombineTable[41][31] = 14;
    out.transformCombineTable[41][32] = 17;
    out.transformCombineTable[41][33] = 21;
    out.transformCombineTable[41][34] = 19;
    out.transformCombineTable[41][35] = 23;
    out.transformCombineTable[41][36] = 16;
    out.transformCombineTable[41][37] = 20;
    out.transformCombineTable[41][38] = 18;
    out.transformCombineTable[41][39] = 22;
    out.transformCombineTable[41][40] = 1;
    out.transformCombineTable[41][41] = 5;
    out.transformCombineTable[41][42] = 3;
    out.transformCombineTable[41][43] = 7;
    out.transformCombineTable[41][44] = 0;
    out.transformCombineTable[41][45] = 4;
    out.transformCombineTable[41][46] = 2;
    out.transformCombineTable[41][47] = 6;
    out.transformCombineTable[42][0] = 42;
    out.transformCombineTable[42][1] = 46;
    out.transformCombineTable[42][2] = 40;
    out.transformCombineTable[42][3] = 44;
    out.transformCombineTable[42][4] = 43;
    out.transformCombineTable[42][5] = 47;
    out.transformCombineTable[42][6] = 41;
    out.transformCombineTable[42][7] = 45;
    out.transformCombineTable[42][8] = 26;
    out.transformCombineTable[42][9] = 30;
    out.transformCombineTable[42][10] = 24;
    out.transformCombineTable[42][11] = 28;
    out.transformCombineTable[42][12] = 27;
    out.transformCombineTable[42][13] = 31;
    out.transformCombineTable[42][14] = 25;
    out.transformCombineTable[42][15] = 29;
    out.transformCombineTable[42][16] = 34;
    out.transformCombineTable[42][17] = 38;
    out.transformCombineTable[42][18] = 32;
    out.transformCombineTable[42][19] = 36;
    out.transformCombineTable[42][20] = 35;
    out.transformCombineTable[42][21] = 39;
    out.transformCombineTable[42][22] = 33;
    out.transformCombineTable[42][23] = 37;
    out.transformCombineTable[42][24] = 10;
    out.transformCombineTable[42][25] = 14;
    out.transformCombineTable[42][26] = 8;
    out.transformCombineTable[42][27] = 12;
    out.transformCombineTable[42][28] = 11;
    out.transformCombineTable[42][29] = 15;
    out.transformCombineTable[42][30] = 9;
    out.transformCombineTable[42][31] = 13;
    out.transformCombineTable[42][32] = 18;
    out.transformCombineTable[42][33] = 22;
    out.transformCombineTable[42][34] = 16;
    out.transformCombineTable[42][35] = 20;
    out.transformCombineTable[42][36] = 19;
    out.transformCombineTable[42][37] = 23;
    out.transformCombineTable[42][38] = 17;
    out.transformCombineTable[42][39] = 21;
    out.transformCombineTable[42][40] = 2;
    out.transformCombineTable[42][41] = 6;
    out.transformCombineTable[42][42] = 0;
    out.transformCombineTable[42][43] = 4;
    out.transformCombineTable[42][44] = 3;
    out.transformCombineTable[42][45] = 7;
    out.transformCombineTable[42][46] = 1;
    out.transformCombineTable[42][47] = 5;
    out.transformCombineTable[43][0] = 43;
    out.transformCombineTable[43][1] = 47;
    out.transformCombineTable[43][2] = 41;
    out.transformCombineTable[43][3] = 45;
    out.transformCombineTable[43][4] = 42;
    out.transformCombineTable[43][5] = 46;
    out.transformCombineTable[43][6] = 40;
    out.transformCombineTable[43][7] = 44;
    out.transformCombineTable[43][8] = 27;
    out.transformCombineTable[43][9] = 31;
    out.transformCombineTable[43][10] = 25;
    out.transformCombineTable[43][11] = 29;
    out.transformCombineTable[43][12] = 26;
    out.transformCombineTable[43][13] = 30;
    out.transformCombineTable[43][14] = 24;
    out.transformCombineTable[43][15] = 28;
    out.transformCombineTable[43][16] = 35;
    out.transformCombineTable[43][17] = 39;
    out.transformCombineTable[43][18] = 33;
    out.transformCombineTable[43][19] = 37;
    out.transformCombineTable[43][20] = 34;
    out.transformCombineTable[43][21] = 38;
    out.transformCombineTable[43][22] = 32;
    out.transformCombineTable[43][23] = 36;
    out.transformCombineTable[43][24] = 11;
    out.transformCombineTable[43][25] = 15;
    out.transformCombineTable[43][26] = 9;
    out.transformCombineTable[43][27] = 13;
    out.transformCombineTable[43][28] = 10;
    out.transformCombineTable[43][29] = 14;
    out.transformCombineTable[43][30] = 8;
    out.transformCombineTable[43][31] = 12;
    out.transformCombineTable[43][32] = 19;
    out.transformCombineTable[43][33] = 23;
    out.transformCombineTable[43][34] = 17;
    out.transformCombineTable[43][35] = 21;
    out.transformCombineTable[43][36] = 18;
    out.transformCombineTable[43][37] = 22;
    out.transformCombineTable[43][38] = 16;
    out.transformCombineTable[43][39] = 20;
    out.transformCombineTable[43][40] = 3;
    out.transformCombineTable[43][41] = 7;
    out.transformCombineTable[43][42] = 1;
    out.transformCombineTable[43][43] = 5;
    out.transformCombineTable[43][44] = 2;
    out.transformCombineTable[43][45] = 6;
    out.transformCombineTable[43][46] = 0;
    out.transformCombineTable[43][47] = 4;
    out.transformCombineTable[44][0] = 44;
    out.transformCombineTable[44][1] = 40;
    out.transformCombineTable[44][2] = 46;
    out.transformCombineTable[44][3] = 42;
    out.transformCombineTable[44][4] = 45;
    out.transformCombineTable[44][5] = 41;
    out.transformCombineTable[44][6] = 47;
    out.transformCombineTable[44][7] = 43;
    out.transformCombineTable[44][8] = 28;
    out.transformCombineTable[44][9] = 24;
    out.transformCombineTable[44][10] = 30;
    out.transformCombineTable[44][11] = 26;
    out.transformCombineTable[44][12] = 29;
    out.transformCombineTable[44][13] = 25;
    out.transformCombineTable[44][14] = 31;
    out.transformCombineTable[44][15] = 27;
    out.transformCombineTable[44][16] = 36;
    out.transformCombineTable[44][17] = 32;
    out.transformCombineTable[44][18] = 38;
    out.transformCombineTable[44][19] = 34;
    out.transformCombineTable[44][20] = 37;
    out.transformCombineTable[44][21] = 33;
    out.transformCombineTable[44][22] = 39;
    out.transformCombineTable[44][23] = 35;
    out.transformCombineTable[44][24] = 12;
    out.transformCombineTable[44][25] = 8;
    out.transformCombineTable[44][26] = 14;
    out.transformCombineTable[44][27] = 10;
    out.transformCombineTable[44][28] = 13;
    out.transformCombineTable[44][29] = 9;
    out.transformCombineTable[44][30] = 15;
    out.transformCombineTable[44][31] = 11;
    out.transformCombineTable[44][32] = 20;
    out.transformCombineTable[44][33] = 16;
    out.transformCombineTable[44][34] = 22;
    out.transformCombineTable[44][35] = 18;
    out.transformCombineTable[44][36] = 21;
    out.transformCombineTable[44][37] = 17;
    out.transformCombineTable[44][38] = 23;
    out.transformCombineTable[44][39] = 19;
    out.transformCombineTable[44][40] = 4;
    out.transformCombineTable[44][41] = 0;
    out.transformCombineTable[44][42] = 6;
    out.transformCombineTable[44][43] = 2;
    out.transformCombineTable[44][44] = 5;
    out.transformCombineTable[44][45] = 1;
    out.transformCombineTable[44][46] = 7;
    out.transformCombineTable[44][47] = 3;
    out.transformCombineTable[45][0] = 45;
    out.transformCombineTable[45][1] = 41;
    out.transformCombineTable[45][2] = 47;
    out.transformCombineTable[45][3] = 43;
    out.transformCombineTable[45][4] = 44;
    out.transformCombineTable[45][5] = 40;
    out.transformCombineTable[45][6] = 46;
    out.transformCombineTable[45][7] = 42;
    out.transformCombineTable[45][8] = 29;
    out.transformCombineTable[45][9] = 25;
    out.transformCombineTable[45][10] = 31;
    out.transformCombineTable[45][11] = 27;
    out.transformCombineTable[45][12] = 28;
    out.transformCombineTable[45][13] = 24;
    out.transformCombineTable[45][14] = 30;
    out.transformCombineTable[45][15] = 26;
    out.transformCombineTable[45][16] = 37;
    out.transformCombineTable[45][17] = 33;
    out.transformCombineTable[45][18] = 39;
    out.transformCombineTable[45][19] = 35;
    out.transformCombineTable[45][20] = 36;
    out.transformCombineTable[45][21] = 32;
    out.transformCombineTable[45][22] = 38;
    out.transformCombineTable[45][23] = 34;
    out.transformCombineTable[45][24] = 13;
    out.transformCombineTable[45][25] = 9;
    out.transformCombineTable[45][26] = 15;
    out.transformCombineTable[45][27] = 11;
    out.transformCombineTable[45][28] = 12;
    out.transformCombineTable[45][29] = 8;
    out.transformCombineTable[45][30] = 14;
    out.transformCombineTable[45][31] = 10;
    out.transformCombineTable[45][32] = 21;
    out.transformCombineTable[45][33] = 17;
    out.transformCombineTable[45][34] = 23;
    out.transformCombineTable[45][35] = 19;
    out.transformCombineTable[45][36] = 20;
    out.transformCombineTable[45][37] = 16;
    out.transformCombineTable[45][38] = 22;
    out.transformCombineTable[45][39] = 18;
    out.transformCombineTable[45][40] = 5;
    out.transformCombineTable[45][41] = 1;
    out.transformCombineTable[45][42] = 7;
    out.transformCombineTable[45][43] = 3;
    out.transformCombineTable[45][44] = 4;
    out.transformCombineTable[45][45] = 0;
    out.transformCombineTable[45][46] = 6;
    out.transformCombineTable[45][47] = 2;
    out.transformCombineTable[46][0] = 46;
    out.transformCombineTable[46][1] = 42;
    out.transformCombineTable[46][2] = 44;
    out.transformCombineTable[46][3] = 40;
    out.transformCombineTable[46][4] = 47;
    out.transformCombineTable[46][5] = 43;
    out.transformCombineTable[46][6] = 45;
    out.transformCombineTable[46][7] = 41;
    out.transformCombineTable[46][8] = 30;
    out.transformCombineTable[46][9] = 26;
    out.transformCombineTable[46][10] = 28;
    out.transformCombineTable[46][11] = 24;
    out.transformCombineTable[46][12] = 31;
    out.transformCombineTable[46][13] = 27;
    out.transformCombineTable[46][14] = 29;
    out.transformCombineTable[46][15] = 25;
    out.transformCombineTable[46][16] = 38;
    out.transformCombineTable[46][17] = 34;
    out.transformCombineTable[46][18] = 36;
    out.transformCombineTable[46][19] = 32;
    out.transformCombineTable[46][20] = 39;
    out.transformCombineTable[46][21] = 35;
    out.transformCombineTable[46][22] = 37;
    out.transformCombineTable[46][23] = 33;
    out.transformCombineTable[46][24] = 14;
    out.transformCombineTable[46][25] = 10;
    out.transformCombineTable[46][26] = 12;
    out.transformCombineTable[46][27] = 8;
    out.transformCombineTable[46][28] = 15;
    out.transformCombineTable[46][29] = 11;
    out.transformCombineTable[46][30] = 13;
    out.transformCombineTable[46][31] = 9;
    out.transformCombineTable[46][32] = 22;
    out.transformCombineTable[46][33] = 18;
    out.transformCombineTable[46][34] = 20;
    out.transformCombineTable[46][35] = 16;
    out.transformCombineTable[46][36] = 23;
    out.transformCombineTable[46][37] = 19;
    out.transformCombineTable[46][38] = 21;
    out.transformCombineTable[46][39] = 17;
    out.transformCombineTable[46][40] = 6;
    out.transformCombineTable[46][41] = 2;
    out.transformCombineTable[46][42] = 4;
    out.transformCombineTable[46][43] = 0;
    out.transformCombineTable[46][44] = 7;
    out.transformCombineTable[46][45] = 3;
    out.transformCombineTable[46][46] = 5;
    out.transformCombineTable[46][47] = 1;
    out.transformCombineTable[47][0] = 47;
    out.transformCombineTable[47][1] = 43;
    out.transformCombineTable[47][2] = 45;
    out.transformCombineTable[47][3] = 41;
    out.transformCombineTable[47][4] = 46;
    out.transformCombineTable[47][5] = 42;
    out.transformCombineTable[47][6] = 44;
    out.transformCombineTable[47][7] = 40;
    out.transformCombineTable[47][8] = 31;
    out.transformCombineTable[47][9] = 27;
    out.transformCombineTable[47][10] = 29;
    out.transformCombineTable[47][11] = 25;
    out.transformCombineTable[47][12] = 30;
    out.transformCombineTable[47][13] = 26;
    out.transformCombineTable[47][14] = 28;
    out.transformCombineTable[47][15] = 24;
    out.transformCombineTable[47][16] = 39;
    out.transformCombineTable[47][17] = 35;
    out.transformCombineTable[47][18] = 37;
    out.transformCombineTable[47][19] = 33;
    out.transformCombineTable[47][20] = 38;
    out.transformCombineTable[47][21] = 34;
    out.transformCombineTable[47][22] = 36;
    out.transformCombineTable[47][23] = 32;
    out.transformCombineTable[47][24] = 15;
    out.transformCombineTable[47][25] = 11;
    out.transformCombineTable[47][26] = 13;
    out.transformCombineTable[47][27] = 9;
    out.transformCombineTable[47][28] = 14;
    out.transformCombineTable[47][29] = 10;
    out.transformCombineTable[47][30] = 12;
    out.transformCombineTable[47][31] = 8;
    out.transformCombineTable[47][32] = 23;
    out.transformCombineTable[47][33] = 19;
    out.transformCombineTable[47][34] = 21;
    out.transformCombineTable[47][35] = 17;
    out.transformCombineTable[47][36] = 22;
    out.transformCombineTable[47][37] = 18;
    out.transformCombineTable[47][38] = 20;
    out.transformCombineTable[47][39] = 16;
    out.transformCombineTable[47][40] = 7;
    out.transformCombineTable[47][41] = 3;
    out.transformCombineTable[47][42] = 5;
    out.transformCombineTable[47][43] = 1;
    out.transformCombineTable[47][44] = 6;
    out.transformCombineTable[47][45] = 2;
    out.transformCombineTable[47][46] = 4;
    out.transformCombineTable[47][47] = 0;
#endif
    return out;
}
